;;; -*- Mode:Common-Lisp; Package:QSIM; Syntax:COMMON-LISP; Base:10 -*-

;;;  $Id: glaucoma.lisp,v 1.3 91/09/16 14:11:44 clancy Exp $
(in-package :qsim)

; This model is directly taken from the CASNET/Glaucoma model [Weiss, et al, AIJ, 1978].
;   - low flow facility for intra-ocular fluid increases intra-ocular pressure.
;   - high intra-ocular pressure restricts blood flow to the optic nerve.
;   - restricted blood flow causes a decrease in optic nerve cell population.
;       = nerve cell population cannot regenerate (cf. population model)
;   - decreased optic nerve population causes visual field loss (not modeled).

(define-QDE glaucoma
  (text "Glaucoma as result of high intra-ocular pressure")
  (quantity-spaces
    (amt  (minf 0 inf)   "amt(fluid,eye)")
    (iop  (minf 0 inf)   "intraocular pressure")
    (ffac (minf 0 inf)   "flow facility (const)")
    (if   (minf 0 inf)   "inflow (const)")
    (of   (minf 0 inf)   "outflow")
    (nf   (minf 0 inf)   "net flow")
    (nbs  (0 inf)        "neural blood flow")
    (os   (minf 0 inf)   "O2 supply")
    (on   (minf 0 inf)   "O2 need")
    (om   (minf 0 inf)   "O2 margin")
    (cp   (0 inf)        "cell population")
    (cpr  (minf 0 inf)   "rate of cell pop change"))
  (constraints
    ((m+ amt iop)        (0 0) (inf inf))
    ((mult iop ffac of))
    ((add of nf if))
    ((d/dt amt nf))
    ((m- iop nbs)        (minf inf))
    ((m+ nbs os)         (0 0) (inf inf))
    ((m+ cp on)          (0 0) (inf inf))
    ((add on om os))
    ((s+ om cpr (minf minf) (0 0)))
    ((d/dt cp cpr))
    ((constant ffac))
    ((constant if)))
  (layout
    (amt if ffac)
    (iop of nf)
    (nbs cp cpr)
    (os om on)))

; Create a normal state with positive values for all state variables, and operating
; with a surplus of oxygen to the optic nerve.
; Initialize with decreased flow facility for intra-ocular fluid (ffac).
; Produces four behaviors:
;   1,4:  decrease oxygen surplus, but no deficit.
;   2,3:  oxygen deficit causes cell death; branch on process rates.

(defun low-flow-facility (&optional (hod t))
  (let* ((sim (make-sim :HOD-constraints hod
			:unreachable-values '((nbs 0))))
	 (normal (make-new-state :from-qde glaucoma
				 :assert-values '((ffac ((0 inf) std))	; constant
						  (if   ((0 inf) std))	; constant
						  (amt  ((0 inf) std))	; state
						  (cp   ((0 inf) std))	; state
						  (om   ((0 inf) std)))
				 :sim sim))
	 (init (make-new-state :from-state normal
			       :perturb '((ffac -))
			       :inherit '(amt cp if)
			       :text "Decreased flow facility")))
    (qsim init)
    (qsim-display init :reference-states `((normal ,normal)))))

