;;; -*- Syntax: Common-Lisp; Package: QSIM -*-
(in-package :qsim)
;;; Copyright (c) 1991, Benjamin Kuipers.

;;; Assumes that /u/kuipers/cc/lib/simple-interface.lisp already loaded.

(define-component RC electrical
  "Resistor-Capacitor Circuit"
   (components (B battery)
	       (R resistor)
	       (C capacitor)
	       (S switch)
	       (G ground))
   (connections (n1 (R t1) (S t1))
		(n2 (R t2) (C t1))
		(n3 (C t2) (B t2) (G t))
		(n4 (B t1) (S t2))))

(defun test-RC ()
  (setq qde (build-qde 'RC :initial-values '(((R mode)     (working std))
					     ((S mode)     (closed  std)))))
  (let* ((sim  (make-sim :HOD-constraints nil))
	 (init (make-CC-state  :from-qde qde
			       :sim sim
			       :assert-values '(((B v)        ((0 inf) std))
						((C c)        ((0 inf) std))
						((C q)        (0       nil)))
			       :text "Charge a capacitor")))
    (qsim init)			
    (qsim-display init) ))

;;; RC circuit with two resistors

(define-component RC2 electrical
  "Resistor-Capacitor Circuit"
   (components (B battery)
	       (R1 resistor)		; two resistors in series.
	       (R2 resistor)
	       (C capacitor)
	       (S switch)
	       (G ground))
   (connections (n1 (R1 t1) (S t1))
		(n0 (R1 t2) (R2 t1))
		(n2 (R2 t2) (C t1))
		(n3 (C t2) (B t2) (G t))
		(n4 (B t1) (S t2))))

(defun test-RC2 ()
  (setq qde (build-qde 'RC2 :initial-values '(((R1 mode)     (working std))
					      ((R2 mode)     (working std))
					      ((S mode)     (closed  std)))))
  (let* ((sim  (make-sim :HOD-constraints nil))
	 (init (make-CC-state  :from-qde qde
			       :sim sim
			       :assert-values '(((B v)        ((0 inf) std))
						((C c)        ((0 inf) std))
						((C q)        (0       nil)))
			       :text "Charge a capacitor")))
    (qsim init)			
    (qsim-display init) ))

;;; This RC circuit includes the more complex Resistor+ component.

;;; Results:
;;;   - When initialized with R=working, S=closed, BUILD-QDE returns two QDEs,
;;;     This requires the auxiliary loop, and extra control.  Needs to be cleaned.
;;;   - The correct model generates three initial states, but successfully
;;;     refutes the one with (R p) > max-power.
;;;   - One of the two QDEs implies R=burned-out, so BUILD-QDE isn't recognizing
;;;     the contradiction.

;;; Observations:
;;;   - The initial value asserted to BUILD-QDE is not provided to the QDE,
;;;     so it must be asserted again at the initial state.

(define-component RC+ electrical
  "Resistor-Capacitor Circuit"
   (components (B battery)
	       (R resistor+)
	       (C capacitor)
	       (S switch)
	       (G ground))
   (connections (n1 (R t1) (S t1))
		(n2 (R t2) (C t1))
		(n3 (C t2) (B t2) (G t))
		(n4 (B t1) (S t2)))
   )


(defun test-RC+ ()
  (setq qde (build-qde 'RC+ :initial-values '(((R mode)     (working std))
					      ((S mode)     (closed  std)))))
  (cond ((typep qde 'qde) (test-RC+aux qde))
	((typep qde 'cons)
	 (dolist (q qde)
	   (test-RC+aux q)))
	(t (error "Bad output ~a from BUILD-QDE." qde)))
  )

(defun test-RC+aux (qde)
  (let* ((sim  (make-sim :HOD-constraints nil))
	 (init (make-CC-state  :from-qde qde
			       :sim sim
			       :assert-values '(((B v)        ((0 inf) std))
						((C c)        ((0 inf) std))
						((C q)        (0       nil))
						((R mode)     (working std))
						((S mode)     (closed  std))
						)
			       :text "Charge a capacitor"))
	 )
    (qsim init)			
    (qsim-display init) ))


