;;; -*- Mode:Common-Lisp; Package:QSIM; Syntax:COMMON-LISP; Base:10 -*-
(in-package :qsim)

; The two cascaded tanks is a classic example of the "chattering" branch.
; This structure produces it both by filling the tanks to equilibrium from
; empty, and by perturbing an initial equilibrium state.
;   These tanks don't overflow.

(define-QDE Cascaded-tanks
  (text "Two cascaded tanks")
  (quantity-spaces
    (inflowa    (0 inf)       "flow(out->A)")
    (amounta    (0 inf)       "amount(A)")
    (outflowa   (0 inf)       "flow(A->b)")
    (netflowa   (minf 0 inf)  "d amount(A)")
    (amountb    (0 inf)       "amount(B)")
    (inflowb    (0 inf)       "flow(a->B)")
    (outflowb   (0 inf)       "flow(B->out)")
    (netflowb   (minf 0 inf)  "d amount(B)" ))
  (constraints
    ((M+ amounta outflowa)            (0 0) (inf inf))
    ((ADD outflowa netflowa inflowa))
    ((d/dt amounta netflowa))
    ((m+ outflowa inflowb)            (0 0) (inf inf))
    ((m+ amountb outflowb)            (0 0) (inf inf))
    ((add outflowb netflowb inflowb))
    ((d/dt amountb netflowb))
    ((constant inflowa)))
  (layout
     (nil amounta nil nil)
     (inflowa outflowa nil amountb)
     (netflowa nil inflowb outflowb)
     (nil nil netflowb))
  )

(defun increase-cascade-inflow-rate ()
  (let* ((sim (make-sim :HOD-constraints nil
			:state-limit 20))
	 (normal (make-new-state :from-qde cascaded-tanks
				 :assert-values '((inflowa ((0 inf) std))
						  (amounta ((0 inf) std))
						  (amountb ((0 inf) std)))
				 :sim sim))
	 (s0 (make-new-state :from-state normal
			     :inherit '(amounta amountb)
			     :perturb '((inflowa +))
			     :text "Perturb inflow rate from equilibrium")))
    (qsim s0)
    (qsim-display s0
		  :reference-states `((normal ,normal)))
    ))

(defun fill-cascade-from-empty ()
  (let* ((sim (make-sim :HOD-constraints nil
			:state-limit 20))
	 (s0 (make-new-state :from-qde cascaded-tanks
			     :assert-values '((inflowa ((0 inf) std))
					      (amounta (0 nil))
					      (amountb (0 nil)))
			     :text "Fill from empty"
			     :sim  sim)))
    (qsim s0)
    (qsim-display s0)
    ))


; 2. The structure is the same as before, but the SIM says
;    to ignore qdir distinctions on the problematical parameter NETFLOWB.
;    The chatter is eliminated.

(defun increase-cascade-inflow-rate-ign ()
  (let* ((sim (make-sim :ignore-qdirs '(netflowb)
			:state-limit 20))
	 (normal (make-new-state :from-qde cascaded-tanks
				 :assert-values '((inflowa ((0 inf) std))
						  (amounta ((0 inf) std))
						  (amountb ((0 inf) std)))
				 :sim sim))
	 (s0 (make-new-state :from-state normal
			     :inherit '(amounta amountb)
			     :perturb '((inflowa +)))))
    (qsim s0)
    (qsim-display s0
		  :reference-states `((normal ,normal)))))

(defun fill-cascade-from-empty-ign ()
  (let* ((sim (make-sim :ignore-qdirs '(netflowb)
			:HOD-constraints nil
			:state-limit 20))
	 (s0 (make-new-state :from-qde cascaded-tanks
			     :assert-values '((inflowa ((0 inf) std))
					      (amounta (0 nil))
					      (amountb (0 nil)))
			     :sim sim)))
    (qsim s0)
    (qsim-display s0)
    sim))



; This example tries a related technique of declaring that no new landmarks
; are to be created for a given parameter.  In this case, it doesn't
; eliminate chatter, but it might be useful elsewhere.
;
; For NO-NEW-LANDMARKS, the global *current-sim* must be set before creating
; initial states.

(defun increase-cascade-inflow-rate-nnl ()
  (let* ((sim (make-sim :HOD-constraints nil
			:no-new-landmarks '(netflowB)
			:state-limit 20))
	 (normal (make-new-state :from-qde cascaded-tanks
				 :assert-values '((inflowa ((0 inf) std))
						  (amounta ((0 inf) std))
						  (amountb ((0 inf) std)))
				 :sim  sim))
	 (s0 (make-new-state :from-state normal
			     :inherit '(amounta amountb)
			     :perturb '((inflowa +)))))
    (qsim s0)
    (qsim-display s0
		  :reference-states `((normal ,normal)))))

(defun fill-cascade-from-empty-nnl ()
  (let* ((sim (make-sim :HOD-constraints nil
			:no-new-landmarks '(netflowB)
			:state-limit 20))
	 (s0 (make-new-state :from-qde cascaded-tanks
			     :assert-values '((inflowa ((0 inf) std))
					      (amounta (0 nil))
					      (amountb (0 nil)))
			     :sim   sim)))
    (qsim s0)
    (qsim-display s0)))

; Apply higher-order-derivative (curvature) constraints.

(defun increase-cascade-inflow-rate-acc ()
  (setf (qde-derived-sd2-expressions cascaded-tanks) nil)
  (let* ((sim (make-sim :HOD-constraints t
			:state-limit 20))
	 (normal (make-new-state :from-qde cascaded-tanks
				 :assert-values '((inflowa ((0 inf) std))
						  (amounta ((0 inf) std))
						  (amountb ((0 inf) std)))
				 :sim  sim))
	 (s0 (make-new-state :from-state normal
			     :inherit '(amounta amountb)
			     :perturb '((inflowa +))
			     :text "Perturb inflow rate from equilibrium")))
    (qsim s0)
    (qsim-display s0 :reference-states `((normal ,normal)))))

(defun fill-cascade-from-empty-acc ()
  (setf (qde-derived-sd2-expressions cascaded-tanks) nil)
  (let* ((sim (make-sim :HOD-constraints t
			:state-limit 20))
	 (s0 (make-new-state :from-qde cascaded-tanks
			     :assert-values '((inflowa ((0 inf) std))
					      (amounta (0 nil))
					      (amountb (0 nil)))
			     :text "Fill from empty"
			     :sim sim)))
    (qsim s0)
    (qsim-display s0)
    ))

;  Cascaded-tanks with the explicit inclusion of curvature constraint clause
; in the QDE

(define-QDE Cascaded-tanks-with-curvature-constraint
  (text "Two cascaded tanks")
  (quantity-spaces
    (inflowa    (0 inf)       "flow(out->A)")
    (amounta    (0 inf)       "amount(A)")
    (outflowa   (0 inf)       "flow(A->b)")
    (netflowa   (minf 0 inf)  "d amount(A)")
    (amountb    (0 inf)       "amount(B)")
    (inflowb    (0 inf)       "flow(a->B)")
    (outflowb   (0 inf)       "flow(B->out)")
    (netflowb   (minf 0 inf)  "d amount(B)" ))
  (constraints
    ((M+ amounta outflowa)            (0 0) (inf inf))
    ((ADD outflowa netflowa inflowa))
    ((d/dt amounta netflowa))
    ((m+ outflowa inflowb)            (0 0) (inf inf))
    ((m+ amountb outflowb)            (0 0) (inf inf))
    ((add outflowb netflowb inflowb))
    ((d/dt amountb netflowb))
    ((constant inflowa)))
  (other
    (curvature-at-steady
      ((netflowa nil)
       (netflowb (deriv netflowa)))))
  (layout
     (nil amounta nil nil)
     (inflowa outflowa nil amountb)
     (netflowa nil inflowb outflowb)
     (nil nil netflowb))
  )

(defun increase-cascade-inflow-rate-with-curvature-constraint ()
  (let* ((sim (make-sim :HOD-constraints t
			:state-limit 20))
	 (normal (make-new-state :from-qde cascaded-tanks
				 :assert-values '((inflowa ((0 inf) std))
						  (amounta ((0 inf) std))
						  (amountb ((0 inf) std)))
				 :sim sim))
	 (s0 (make-new-state :from-state normal
			     :inherit '(amounta amountb)
			     :perturb '((inflowa +))
			     :text "Perturb inflow rate from equilibrium")))
    (qsim s0)
    (qsim-display s0 :reference-states `((normal ,normal)))))

(defun fill-cascade-from-empty-with-curvature-constraint ()
  (let* ((sim (make-sim :HOD-constraints t
			:state-limit 20))
	 (s0 (make-new-state :from-qde cascaded-tanks
			     :assert-values '((inflowa ((0 inf) std))
					      (amounta (0 nil))
					      (amountb (0 nil)))
			     :text "Fill from empty"
			     :sim sim)))
    (qsim s0)
    (qsim-display s0)
    ))
