;;; -*- Mode:Common-Lisp; Package:QSIM; Syntax:COMMON-LISP; Base:10 -*-
;;; $Id: bathtub-w-ranges.lisp,v 1.9 92/07/20 16:45:29 bert Exp $

(in-package :qsim)

;;; This file contains examples of applying quantitative information to a
;;; simple first order system.  The QDE is augmented with
;;;  1)  Bounding envelopes for monotonic functions
;;;  2)  Ranges for landmarks
;;; This added information can be used to prune behaviors that are otherwise
;;; plausible.  It can also be used to predict the values of model variables
;;; over time.

;;; Examples in this file :
;;; 1.  A simple example of Q2 and Nsim.
;;; 2.  Asserting a range for a Qsim-generated landmark.
;;; 3.  Using multivariate M constraints.
;;; 4.  Assorted debugging tools (description only).


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; 1.  A simple example of Q2 and Nsim.
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


;;; The simplest bathtub, with range constraints added.
;;;
(define-QDE Bathtub-w-ranges
   (text "Simple bathtub model with interval range information")
   (quantity-spaces
     (amount    (0 FULL inf)  "amt(water,tub)")
     (outflow   (0 inf)       "flow(water,tub->out)")
     (inflow    (0 if* inf)   "flow(water,in->tub)")
     (netflow   (minf 0 inf)  "net flow(water,out->tub)"))
   (constraints
     ((M+ amount outflow)    (0 0) (inf inf))
     ((ADD netflow outflow inflow))
     ((d/dt amount netflow))
     ((constant inflow)))
   (transitions
     ((amount (FULL inc)) -> t)			;tub-overflows
     )
   (layout (nil nil amount  nil)
	   (nil inflow outflow)
	   (nil time netflow))
   (m-envelopes
     ((M+ amount outflow) (upper-envelope (lambda (x) x))
                          (upper-inverse  (lambda (y) y))
                          (lower-envelope (lambda (x) (if (< x 20) (/ x 2)
							  (+ (/ x 5) 6))))
			  (lower-inverse  (lambda (y) (if (< y 10) (* y 2)
							  (* 5 (- y 6)))))))
   (initial-ranges ((inflow if*)  (5 10))
		   ((amount full) (80 100))
		   ((time t0)     (0 0))))


;;; Fill the bathtub using Q2.
;;; This generates one consistent behavior.
;;;
(defun q2-bathtub ()
  (let* ((sim (make-sim :Q2-constraints t
			:new-landmarks-always t))
	 (initial-state
	   (make-new-state :from-qde bathtub-w-ranges
			   :assert-values '((inflow (if* std))
					    (amount (0 nil)))
			   :text "Filling empty tub"
			   :sim sim)))

    (format *QSIM-Report* 
	    "~2%This is the simplest bathtub example, but augmented with ~
             ~%quantitative range information about some of the landmarks and ~
             ~%the M+ constraint, as described by Kuipers and Berleant [AAAI-88].~
             ~%In this case, we can use incomplete quantitative information ~
             ~%to refute two qualitatively plausible behaviors.  Read the ~
             ~%example file, EXAMPLES;bathtub-w-ranges.lisp, to see how to ~
             ~%represent range information. ~2%")

    (qsim initial-state)
    (qsim-display initial-state :show-inconsistent-successors t)
    ))

;;; This function demonstrates the Nsim dynamic envelope simulator.  Nsim
;;; runs together with Q2 to further refine the predicted values of the system
;;; by simulating bounding ODE systems.
;;;
(defun nsim-bathtub ()
  (let* ((*check-quantitative-ranges* t)
	(*use-dynamic-envelopes* T)
	(*new-landmarks-across-M-constraints* t)
	(initial-state
	  (make-initial-state bathtub-w-ranges
			      '((inflow (if* std))
				(amount (0 nil)))
			      "Filling empty tub (using NSIM)")))

    ;; Construct an initial simulation structure for the numerical simulator.
    ;; This call generates a set of extremal systems and compiles them into
    ;; lisp functions so that they may be called by the numerical simulator.
    ;; Note that this requires having an initial state defined so that 
    ;; the numerical simulator can derive its initial state.
    (nsim-initialize initial-state)

    ;; Run QSIM with Q2.  
    (qsim initial-state)

    ;; Run the numerical simulator to determine the bounding envelopes.
    (nsim initial-state)

    ;; Display the results for the amount variable in the QDE.
    (set-numeric-layout initial-state '((amount nil)))
    (qsim-display initial-state :plot-mode 'numeric-time-plot)
    ))



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; 2.  Asserting a range for a Qsim-generated landmark.
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; This QDE is the same as bathtub-w-ranges except that the initial range
;;; for the ifa* landmark of inflow is unspecified.  This change means that
;;; three behaviors are generated.
;;;
(define-QDE Bathtub-w-ranges-2
   (text "Simple bathtub model with interval range information")
   (quantity-spaces
     (amount    (0 FULL inf)  "amt(water,tub)")
     (outflow   (0 inf)       "flow(water,tub->out)")
     (inflow    (0 if* inf)   "flow(water,in->tub)")
     (netflow   (minf 0 inf)  "net flow(water,out->tub)"))
   (constraints
     ((M+ amount outflow)    (0 0) (inf inf))
     ((ADD netflow outflow inflow))
     ((d/dt amount netflow))
     ((constant inflow)))
   (transitions
     ((amount (FULL inc)) -> t)			;tub-overflows
     )
   (layout (nil nil amount  nil)
	   (nil inflow outflow)
	   (nil time netflow))
  
   (m-envelopes
     ((M+ amount outflow) (upper-envelope (lambda (x) x))
                          (upper-inverse  (lambda (y) y))
                          (lower-envelope (lambda (x) (if (< x 20) (/ x 2)
							  (+ (/ x 5) 6))))
			  (lower-inverse  (lambda (y) (if (< y 10) (* y 2)
							  (* 5 (- y 6)))))))
   (initial-ranges 
		   ((amount full) (80 100))
		   ((time t0)     (0 0))))

;;; This function runs Q2 on the model and then asserts the observation that
;;; the final value of amount is in the range [60 70] and then reruns Q2.
;;; This results in a single consistent behavior because it is clear that
;;; behaviors where the tub fills or overflows can be refuted.
;;;
(defun q2-bathtub-2 ()
  (qsim-cleanup)
  (let* ((sim (make-sim :Q2-constraints t
			:new-landmarks-always t))
	 (initial-state
	   (make-new-state :from-qde bathtub-w-ranges-2
			   :assert-values '((inflow (if* std))
					    (amount (0 nil)))
			   :text "Filling empty tub"
			   :sim sim)))

    
    (format *Qsim-Report*
	    "~2%This function demonstrates the assertion of a range to a ~
             ~%Qsim-generated landmark. ~
             ~%See the file EXAMPLES;bathtub-w-ranges for more details.")
    
    (qsim initial-state)

    ;; So far, this is identical to the code in bathtub-w-ranges, except that
    ;; because we have no longer specified a range on the inflow, three
    ;; consistent behaviors result.  

    ;; Now, for each behavior, we assert that the final value for amount is the
    ;; range [60 70].  
    (dolist (beh (get-behaviors initial-state)) 
      
      ;; Beh is bound to a single behavior (which is a list of states).
      ;; The last element of this list is the final state of the behavior.
      ;; We extract the landmark for the AMOUNT variable (note that in general
      ;; this is not guaranteed to be a landmark; it could be an interval
      ;; although in this case it isn't).
      (let* ((final-state (car (last beh)))
	     (lmark (qmag (qval 'amount final-state))))

	;; Now we rerun Q2 on the final state with the assertion that the 
	;; range of amount in this state is really [60 70].  
	(quantitative-range-reasoning final-state
				      :assert-ranges `(((amount ,lmark)
							(60 70))))
	))
    ;; Behavior 2 is the only consistent behavior that could end up in the
    ;; range [60 70] for amount.
    (qsim-display initial-state :show-inconsistent-successors t)
    ))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; 3.  Using multivariate M constraints.
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; The multivariate M constraint is an extension of the M+ and M- constraint.
;;; See the manual and the blurb in NQ:NQ;equations.lisp

;;; This example takes the qde bathtub-w-ranges and replaces the M+ and
;;; ADD constraints with M constraints.  The M constraints are treated as
;;; "one-way" envelope definitions (unlike M+ and M- which also include inverses
;;; in their envelope definitions).  Therefore, to get the effect of an ADD
;;; constraint, 3 M constraints are needed.  Try removing some of the 2-ary
;;; M constraints that mimic ADD and see what the resulting behaviors look
;;; like.

(define-QDE Bathtub-w-ranges-m
   (text "Simple bathtub model with interval range information")
   (quantity-spaces
     (amount    (0 FULL inf)  "amt(water,tub)")
     (outflow   (0 inf)       "flow(water,tub->out)")
     (inflow    (0 if* inf)   "flow(water,in->tub)")
     (netflow   (minf 0 inf)  "net flow(water,out->tub)"))
   (constraints
     ;; This constraint replaces the following M+ constraint.  Normally, the
     ;; reverse constraint ((M +) outflow amount) would also be needed,
     ;; but things work out OK without it in this example.
     (((M +) amount outflow)    (0 0) (inf inf))
;     ((M+ amount outflow)    (0 0) (inf inf))

     ;; These next three constraints replace the following add constraint
     ;; With a single (M + +), we refute two of the three behaviors, but
     ;; our bounds on netflow and outflow are weak
     (((M + +) netflow outflow inflow) (0 0 0))
     ;; This gives better bounds for netflow
     (((M + -) inflow outflow netflow) (0 0 0))
     ;; And this gives better bounds for outflow
     (((M + -) inflow netflow outflow) (0 0 0))

;     ((ADD netflow outflow inflow))
     ((d/dt amount netflow))
     ((constant inflow)))
   (transitions
     ((amount (FULL inc)) -> t)			;tub-overflows
     )
   (layout (nil nil amount  nil)
	   (nil inflow outflow)
	   (nil time netflow))
   (m-envelopes
     (((M + +) netflow outflow inflow)
      (upper-envelope  (lambda (A B) (+ A B)))
      (lower-envelope  (lambda (A B) (+ A B))))
     (((M + -) inflow outflow netflow)
      (upper-envelope (lambda (C A) (- C A)))
      (lower-envelope (lambda (C A) (- C A))))
     (((M + -) inflow netflow outflow)
      (upper-envelope (lambda (C B) (- C B)))
      (lower-envelope (lambda (C B) (- C B))))
     (((M +) amount outflow)
      (upper-envelope (lambda (x) x))
      (lower-envelope (lambda (x) (if (< x 20) (/ x 2)
				      (+ (/ x 5) 6)))))
;     (((M+ amount outflow)
;      (upper-envelope (lambda (x) x))
;      (upper-inverse  (lambda (y) y))
;      (lower-envelope (lambda (x) (if (< x 20) (/ x 2)
;				      (+ (/ x 5) 6))))
;      (lower-inverse  (lambda (y) (if (< y 10) (* y 2)
;				      (* 5 (- y 6)))))))
     )
   (initial-ranges ((inflow if*)  (5 10))
		   ((amount full) (80 100))
		   ((time t0)     (0 0))))


;;; Fill the bathtub using Q2.
;;; This generates one consistent behavior.
;;;
(defun q2-bathtub-m ()
  (let* ((sim (make-sim :Q2-constraints t
			:new-landmarks-always t))
	 (initial-state
	   (make-new-state :from-qde bathtub-w-ranges-m
			   :assert-values '((inflow (if* std))
					    (amount (0 nil)))
			   :text "Filling empty tub"
			   :sim sim)))

    (format *QSIM-Report* 
	    "~2%This is the simplest bathtub example, but augmented with ~
             ~%quantitative range information about some of the landmarks and ~
             ~%the M constraint to replace the ADD and M+ constraints. ~2%")

    (qsim initial-state)
    (qsim-display initial-state :show-inconsistent-successors t)
    ))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; 4.  Assorted debugging tools.
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; The functions (show-bindings <state>) and (show-equations <state>)
;;; will show some of the internal structures used by Q2.

;;; Monotonic function envelopes from the m-envelopes qde clause can be
;;; graphed using the graph-envelopes command.  It has the form
;;;
;;;     (graph-envelopes qde-or-state constraint
;;;                                   :envelopes envlist)
;;; where envlist is a list of the subclause names in an m-envelope
;;;       (the default is (upper-envelope lower-envelope)


