;;; -*- Package: qsim; Syntax: Common-lisp -*-
(in-package :qsim) ;changed DJC

; 1. Two tank system with neither ignore-qdirs nor curvature constraints.
; 2. Two tank system with ignore-qdirs.
; 3. Two tank system with curvature constraints. 

; 1. The two tank system. No curvature constraint and no ignore-qdirs. 
(define-QDE two-tank
	    (text "Two-tank system")
  (quantity-spaces
    (amounta (0 amta* inf))
    (amountb (0 amtb* inf))
    (pressurea (0 pa* inf))
    (pressureb (0 pb* inf))
    (delta-ab (minf 0 dab* inf))
    (crossflowab (minf 0 crf* inf))
    (inflowa (0 ifa* inf))
    (outflowb (0 ofb* inf))
    (netflowa (minf 0 inf))
    (netflowb (minf 0 inf)))
  (constraints
    ((M+ pressurea amounta)(0 0)(pa* amta*))
    ((M+ pressureb amountb)(0 0)(pb* amtb*))
    ((add delta-ab pressureb pressurea)(0 0 0)(dab* pb* pa*))
    ((M+ delta-ab crossflowab)(0 0)(dab* crf*)(inf inf))
    ((m+ pressureb outflowb)(0 0)(pb* ofb*)(inf inf))
    ((add netflowa crossflowab inflowa)(0 0 0)(0 crf* ifa*))
    ((add netflowb outflowb crossflowab)(0 0 0)(0 ofb* crf*))
    ((d/dt amounta netflowa))
    ((d/dt amountb netflowb)))
  (independent inflowa)
  (history amounta amountb)
  (layout
    (amounta amountb nil)
    (inflowa crossflowab nil)
    (crossflowab outflowb nil)
    (netflowa netflowb nil))    
  (print-names
    (amounta "amount (A)" amta)
    (amountb "amount (B)" amtb)
    (pressurea "pressure (A)" pa)
    (pressureb "pressure (B)" pb)
    (delta-ab "d press(Pa-Pb)" dab)

    (crossflowab "Crossflow (A->B)" crf)
    (inflowa "flow ( -> A)" ifa)
    (outflowb "flow (B->out)" ofb)    
    (netflowa "d amount (A)" nfa)
    (netflowb "d amount (B)" nfb)))


; 1a. Fill the two tank system, i.e. fill-two-tanks-from empty.
(defun fill-two-tanks-from-empty()
  (let*((init (make-initial-state two-tank
				    '((amounta (0 nil))
				      (amountb (0 nil))
				      (inflowa (ifa* std))))))

    (qsim init)
    (qsim-display init)))

; 1b. Note for the two tank model: pdmi designates
;     the initial amounta: plus and decreasing, and
;     the initial amountb: minus and increasing. 

(defun two-tank-pdmi()
  (let*((normal (make-initial-state two-tank
				    '((amounta (amta* nil))
				      (amountb (amtb* nil))
				      (crossflowab (crf* std))
				      (netflowa (0 std))
				      (netflowb (0 std))
				      (outflowb (ofb* std))
				      (inflowa (ifa* std)))))
	(init (make-modified-state normal
				   '((amounta ((amta* inf) nil))
				     (amountb ((0 amtb*) nil))
				     (crossflowab ((crf* inf) nil)) 				 
				     (netflowa ((minf 0) nil))
				     (netflowb ((0 inf) nil)) 				 
				     (inflowa (ifa* std))))))
    (qsim init)
    (qsim-display init
		  :reference-states `((normal ,normal)))))

; 2. The Two-Tank System, ignoring some qdirs.

; 4a. Fill two tank from empty ign.
(define-QDE Two-Tank-System-ign-NFB
  (text "Two tank system, ignoring some qdirs")
  (quantity-spaces
    (inflowa    (0 ifa* ifa+ inf))
    (amounta    (0 amta* inf))
    (pressurea  (0 pa* inf))
    (outflowa   (0 ofa* inf))
    (netflowa   (minf 0 inf))
    (amountb    (0 amtb* inf))
    (pressureb  (0 pb* inf))
    (delta-ab   (minf 0 dab* inf))
    (inflowb    (0 ifb* inf))
    (outflowb   (0 ofb* inf))
    (netflowb   (minf 0 inf))
    )
  (constraints
    ((M+ amounta pressurea)           (0 0)   (amta* pa*) (inf inf))
    ((ADD delta-ab pressureb pressurea)  (0 0 0)  (dab* pb* pa*))
    ((M+ delta-ab  outflowa)          (0 0)   (dab* ofa*) (inf inf))
    ((ADD outflowa netflowa inflowa)  (0 0 0) (ofa* 0 ifa*))
    ((d/dt amounta netflowa))
    ((m+ outflowa inflowb)            (0 0)   (ofa* ifb*) (inf inf))
    ((m+ amountb pressureb)           (0 0)   (amtb* pb*) (inf inf))
    ((m+ pressureb outflowb)          (0 0)   (pb* ofb*) (inf inf))
    ((add outflowb netflowb inflowb)  (0 0 0) (ofb* 0 ifb*))
    ((d/dt amountb netflowb)))
  (independent inflowa)
  (history amounta amountb)
  (layout
    (amounta amountb nil)
    (inflowa delta-ab nil)
    (delta-ab outflowb nil)
    (netflowa netflowb nil))    
  (print-names
    (inflowa   "flow(out->A)"  ifa)
    (amounta   "amount(A)"     amta)
    (pressurea "pressure(A)"   pa)
    (pressureb "pressure(B)"   pb)
    (delta-ab  "d press(A->B)" dab)
    (outflowa  "flow(A->b)"    ofa)
    (netflowa  "d amount(A)"   nfa)
    (inflowb   "flow(a->B)"    ifb)
    (amountb   "amount(B)"     amtb)
    (outflowb  "flow(B->out)"  ofb)
    (netflowb  "d amount(B)"   nfb))
  (other
    (ignore-qdirs netflowb
		  delta-ab outflowa netflowa inflowb
		  ))
)

; 2a Two tank with ignore.
(defun fill-two-tanks-from-empty-ign-NFB ()
  (let ((init (make-initial-state two-tank-system-ign-NFB
				   '((inflowa (ifa* std))
				     (amounta (0 nil))
				     (amountb (0 nil))))))
    (qsim init)
    (qsim-display init)
    ))

; 2b. Two-tank-ign-pdmi:

(defun two-tank-ign-pdmi()
  (let*((normal (make-initial-state two-tank-system-ign-NFB
				    '((amounta (amta* nil))
				      (amountb (amtb* nil))
				      (delta-ab (dab* std))
				      (netflowa (0 std))
				      (netflowb (0 std))
				      (outflowb (ofb* std))
				      (inflowa (ifa* std)))))
	(init (make-modified-state normal
				   '((amounta ((amta* inf) nil))
				     (amountb ((0 amtb*) nil))
				     (delta-ab ((dab* inf) nil)) 				 
				     (netflowa ((minf 0) nil))
				     (netflowb ((0 inf) nil)) 				 
				     (inflowa (ifa* std))))))
    (qsim init)
    (qsim-display init
		  :reference-states `((normal ,normal)))))




; 3. The two tank model: automatic determination of curvature constraints. A. C. C.
; two-tank-pdmi (amounta: plus and increasing, amountb minus and decreasing)

(define-QDE two-tank-acc
	    (text "Two-tank system with auto-curvature-constraints")
  (quantity-spaces
    (amounta (0 amta* inf))
    (amountb (0 amtb* inf))
    (pressurea (0 pa* inf))
    (pressureb (0 pb* inf))
    (delta-ab (minf 0 dab* inf))
    (crossflowab (minf 0 crf* inf))
    (inflowa (0 ifa* inf))
    (outflowb (0 ofb* inf))
    (netflowa (minf 0 inf))
    (netflowb (minf 0 inf)))
  (constraints
    ((M+ pressurea amounta)(0 0)(pa* amta*))
    ((M+ pressureb amountb)(0 0)(pb* amtb*))
    ((add delta-ab pressureb pressurea)(0 0 0)(dab* pb* pa*))
    ((M+ delta-ab crossflowab)(0 0)(dab* crf*)(inf inf))
    ((m+ pressureb outflowb)(0 0)(pb* ofb*)(inf inf))
    ((add netflowa crossflowab inflowa)(0 0 0)(0 crf* ifa*))
    ((add netflowb outflowb crossflowab)(0 0 0)(0 ofb* crf*))
    ((d/dt amounta netflowa))
    ((d/dt amountb netflowb)))
  (independent inflowa)
  (history amounta amountb)
  (other (curvature-at-steady nil))
  (layout
    (amounta amountb nil)
    (inflowa crossflowab nil)
    (crossflowab outflowb nil)
    (netflowa netflowb nil))
  (print-names
    (amounta "amount (A)" amta)
    (amountb "amount (B)" amtb)
    (pressurea "pressure (A)" pa)
    (pressureb "pressure (B)" pb)
    (delta-ab "d press(Pa-Pb)" dab)
    (crossflowab "Crossflow (A->B)" crf)
    (inflowa "flow ( -> A)" ifa)
    (outflowb "flow (B->out)" ofb)    
    (netflowa "d amount (A)" nfa)
    (netflowb "d amount (B)" nfb)))

; 3a. pdmi designates that initially amounta=plus and increasing, amountb=minus and decreasing.
(defun two-tank-pdmi-acc()
  (setf (qde-curvature-at-steady two-tank-acc) nil)
  (let*(( *perform-acc-analysis* t)
	(normal (make-initial-state two-tank-acc
				    '((amounta (amta* nil))
				      (amountb (amtb* nil))
				      (crossflowab (crf* std))
				      (netflowa (0 std))
				      (netflowb (0 std))
				      (outflowb (ofb* std))
				      (inflowa (ifa* std)))))
	(init (make-modified-state normal
				   '((amounta ((amta* inf) nil))
				     (amountb ((0 amtb*) nil))
				     (crossflowab ((crf* inf) nil)) 				 
				     (netflowa ((minf 0) nil))
				     (netflowb ((0 inf) nil)) 				 
				     (inflowa (ifa* std))))))
    (special  *perform-acc-analysis*)
    (qsim init)
    (qsim-display init
		  :reference-states `((normal ,normal)))))

; 3b. Fill the two tank system, i.e. fill-two-tanks-from empty.
(defun fill-two-tanks-from-empty-acc()
  (setf (qde-curvature-at-steady two-tank-acc) nil)
  (let*(( *perform-acc-analysis* t)
	(init (make-initial-state two-tank-acc
				    '((amounta (0 nil))
				      (amountb (0 nil))
				      (inflowa (ifa* std))))))
    (special  *perform-acc-analysis*)

    (qsim init)
    (qsim-display init)))

