;;; -*- Package: qsim; Syntax: Common-lisp -*-

; 1.  Two cascaded-tanks with neither curvature constraints nor ignore-qdirs.
; 2.  Two cascaded-tanks with ignore-qdirs.
; 3.  Two cascaded-tanks with input of curvature constraints.
; 4.  Two cascaded-tanks with automatic determination of curvature constraints.

; The two cascaded tanks is a classic example of the "chattering" branch.
; This structure produces it both by filling the tanks to equilibrium from
; empty, and by perturbing an initial equilibrium state. These tanks don't overflow.

; 1. The cascaded-tanks with neither curvature constraints nor ignore-qdirs.

(define-QDE Cascaded-tanks
  (text "Two cascaded tanks")
  (quantity-spaces
    (inflowa    (0 ifa* ifa+ inf))
    (amounta    (0 amta* inf))
    (outflowa   (0 ofa* inf))
    (netflowa   (minf 0 inf))
    (amountb    (0 amtb* inf))
    (inflowb    (0 ifb* inf))
    (outflowb   (0 ofb* inf))
    (netflowb   (minf 0 inf))
    )
  (constraints
    ((M+ amounta outflowa)            (0 0) (inf inf)   (amta* ofa*))
    ((ADD outflowa netflowa inflowa)  (0 0 0)           (ofa* 0 ifa*))
    ((d/dt amounta netflowa))
    ((m+ outflowa inflowb)            (0 0) (inf inf)   (ofa* ifb*))
    ((m+ amountb outflowb)            (0 0) (inf inf)   (amtb* ofb*))
    ((add outflowb netflowb inflowb)  (0 0 0)           (ofb* 0 ifb*))
    ((d/dt amountb netflowb)))
  (independent inflowa)
  (history amounta amountb)
  (layout
     (nil amounta nil nil)
     (inflowa outflowa nil amountb)
     (netflowa nil inflowb outflowb)
     (nil nil netflowb))
  (print-names
    (inflowa   "flow(out->A)"  ifa)
    (amounta   "amount(A)"     amta)
    (outflowa  "flow(A->b)"    ofa)
    (netflowa  "d amount(A)"   nfa)
    (inflowb   "flow(a->B)"    ifb)
    (amountb   "amount(B)"     amtb)
    (outflowb  "flow(B->out)"  ofb)
    (netflowb  "d amount(B)"   nfb))
  )

(defun increase-cascade-inflow-rate ()
  (let* ((normal (make-initial-state cascaded-tanks
				     '((inflowa (ifa* std))
				       (amounta (amta* nil))
				       (amountb (amtb* nil)))))
	 (init (make-modified-state normal
				    '((inflowa (ifa+ std))
				       (amounta (amta* nil))
				       (amountb (amtb* nil))))))
    (qsim init)
    (qsim-display init
		  :reference-states `((normal ,normal)))
    ))

(defun fill-cascade-from-empty ()
  (let ((init (make-initial-state cascaded-tanks
				   '((inflowa (ifa* std))
				     (amounta (0 nil))
				     (amountb (0 nil))))))
    (qsim init)
    (qsim-display init)
    ))

; 2. This version of the structure is the same as the above, but says
;    to ignore qdir distinctions on the problematical parameter NETFLOWB.
;    The chatter is eliminated.

(define-QDE Cascaded-tanks-ign-test
  (text "Two cascaded tanks (ignore qdirs on netflowb)")
  (quantity-spaces
    (inflowa    (0 ifa* ifa+ inf))
    (amounta    (0 amta* inf))
    (outflowa   (0 ofa* inf))
    (netflowa   (minf 0 inf))
    (amountb    (0 amtb* inf))
    (inflowb    (0 ifb* inf))
    (outflowb   (0 ofb* inf))
    (netflowb   (minf 0 inf))
    )
  (constraints
    ((M+ amounta outflowa)            (0 0) (inf inf)   (amta* ofa*))
    ((ADD outflowa netflowa inflowa)  (0 0 0)           (ofa* 0 ifa*))
    ((d/dt amounta netflowa))
    ((m+ outflowa inflowb)            (0 0) (inf inf)   (ofa* ifb*))
    ((m+ amountb outflowb)            (0 0) (inf inf)   (amtb* ofb*))
    ((add outflowb netflowb inflowb)  (0 0 0)           (ofb* 0 ifb*))
    ((d/dt amountb netflowb)))
  (independent inflowa)
  (history amounta amountb)
  (layout
     (nil amounta nil nil)
     (inflowa outflowa nil amountb)
     (netflowa nil inflowb outflowb)
     (nil nil netflowb))
  (print-names
    (inflowa   "flow(out->A)"  ifa)
    (amounta   "amount(A)"     amta)
    (outflowa  "flow(A->b)"    ofa)
    (netflowa  "d amount(A)"   nfa)
    (inflowb   "flow(a->B)"    ifb)
    (amountb   "amount(B)"     amtb)
    (outflowb  "flow(B->out)"  ofb)
    (netflowb  "d amount(B)"   nfb))
  (other
    (ignore-qdirs netflowb))
)


(defun increase-cascade-inflow-rate-ign ()
  (let* ((normal (make-initial-state cascaded-tanks-ign-test
				     '((inflowa (ifa* std))
				       (amounta (amta* nil))
				       (amountb (amtb* nil)))))
	 (init (make-modified-state normal
				    '((inflowa (ifa+ std))
				       (amounta (amta* nil))
				       (amountb (amtb* nil))))))
    (qsim init)
    (qsim-display init :reference-states `((normal ,normal)))
    ))

(defun fill-cascade-from-empty-ign ()
  (let ((init (make-initial-state cascaded-tanks-ign-test
				   '((inflowa (ifa* std))
				     (amounta (0 nil))
				     (amountb (0 nil))))))
    (qsim init)
    (qsim-display init)
    ))


; 3. Cascaded-tanks with the explicit inclusion of curvature constraint clause. 

(define-QDE Cascaded-tanks-with-curvature-constraint
	    (text "Two cascaded tanks")
  (quantity-spaces
    (inflowa    (0 ifa* ifa+ inf))
    (amounta    (0 amta* inf))
    (outflowa   (0 ofa* inf))
    (netflowa   (minf 0 inf))
    (amountb    (0 amtb* inf))
    (inflowb    (0 ifb* inf))
    (outflowb   (0 ofb* inf))
    (netflowb   (minf 0 inf)))
  (constraints
    ((M+ amounta outflowa)            (0 0) (inf inf)   (amta* ofa*))
    ((ADD outflowa netflowa inflowa)  (0 0 0)           (ofa* 0 ifa*))
    ((d/dt amounta netflowa))
    ((m+ outflowa inflowb)            (0 0) (inf inf)   (ofa* ifb*))
    ((m+ amountb outflowb)            (0 0) (inf inf)   (amtb* ofb*))
    ((add outflowb netflowb inflowb)  (0 0 0)           (ofb* 0 ifb*))
    ((d/dt amountb netflowb)))
  (independent inflowa)
  (history amounta amountb)
  (layout
    (nil amounta nil nil)
    (inflowa outflowa nil amountb)
    (netflowa nil inflowb outflowb)
    (nil nil netflowb))
  (print-names
    (inflowa   "flow(out->A)"  ifa)
    (amounta   "amount(A)"     amta)
    (outflowa  "flow(A->b)"    ofa)
    (netflowa  "d amount(A)"   nfa)
    (inflowb   "flow(a->B)"    ifb)
    (amountb   "amount(B)"     amtb)
    (outflowb  "flow(B->out)"  ofb)
    (netflowb  "d amount(B)"   nfb))
  (other (curvature-at-steady ((netflowa nil)(netflowb (deriv netflowa))))))

(defun increase-cascade-inflow-rate-with-curvature-constraint ()
  (let* (( *perform-acc-analysis* t)
	 (normal (make-initial-state cascaded-tanks-with-curvature-constraint
				     '((inflowa (ifa* std))
				       (amounta (amta* nil))
				       (amountb (amtb* nil)))))
	 (init (make-modified-state normal
				    '((inflowa (ifa+ std))
				       (amounta (amta* nil))
				       (amountb (amtb* nil))))))
    (special *perform-acc-analysis*)
    (qsim init)
    (qsim-display init :reference-states `((normal ,normal)))
    ))

(defun fill-cascade-from-empty-with-curvature-constraint ()
  (let ((*perform-acc-analysis* t)
	(init (make-initial-state cascaded-tanks-with-curvature-constraint
				   '((inflowa (ifa* std))
				     (amounta (0 nil))
				     (amountb (0 nil))))))
    (special *perform-acc-analysis*)
    (qsim init)
    (qsim-display init)
    ))


; 4. Automatic determination of curvature constraints. A. C. C.

; The two cascaded tanks is a classic example of the "chattering" branch.
; This structure produces it both by filling the tanks to equilibrium from
; empty, and by perturbing an initial equilibrium state.
;   These tanks don't overflow.

(define-QDE Cascaded-tanks-acc
  (text "Two cascaded tanks with auto-curvature-constraints")
  (quantity-spaces
    (inflowa    (0 ifa* ifa+ inf))
    (amounta    (0 amta* inf))
    (outflowa   (0 ofa* inf))
    (netflowa   (minf 0 inf))
    (amountb    (0 amtb* inf))
    (inflowb    (0 ifb* inf))
    (outflowb   (0 ofb* inf))
    (netflowb   (minf 0 inf))
    )
  (constraints
    ((M+ amounta outflowa)            (0 0) (inf inf)   (amta* ofa*))
    ((ADD outflowa netflowa inflowa)  (0 0 0)           (ofa* 0 ifa*))
    ((d/dt amounta netflowa))
    ((m+ outflowa inflowb)            (0 0) (inf inf)   (ofa* ifb*))
    ((m+ amountb outflowb)            (0 0) (inf inf)   (amtb* ofb*))
    ((add outflowb netflowb inflowb)  (0 0 0)           (ofb* 0 ifb*))
    ((d/dt amountb netflowb)))
  (independent inflowa)
  (history amounta amountb)
  (other (curvature-at-steady nil))
  (layout
     (nil amounta nil nil)
     (inflowa outflowa nil amountb)
     (netflowa nil inflowb outflowb)
     (nil nil netflowb))
  (print-names
    (inflowa   "flow(out->A)"  ifa)
    (amounta   "amount(A)"     amta)
    (outflowa  "flow(A->b)"    ofa)
    (netflowa  "d amount(A)"   nfa)
    (inflowb   "flow(a->B)"    ifb)
    (amountb   "amount(B)"     amtb)
    (outflowb  "flow(B->out)"  ofb)
    (netflowb  "d amount(B)"   nfb))
  )

(defun increase-cascade-inflow-rate-acc ()
  (setf (qde-curvature-at-steady cascaded-tanks-acc) nil)
  (let* (( *perform-acc-analysis* t)
	 (normal (make-initial-state cascaded-tanks-acc
				     '((inflowa (ifa* std))
				       (amounta (amta* nil))
				       (amountb (amtb* nil)))))
	 (init (make-modified-state normal
				    '((inflowa (ifa+ std))
				       (amounta (amta* nil))
				       (amountb (amtb* nil))))))
    (special  *perform-acc-analysis*)
    (qsim init)
    (qsim-display init :reference-states `((normal ,normal)))
    ))

(defun fill-cascade-from-empty-acc ()
  (setf (qde-curvature-at-steady cascaded-tanks-acc) nil)
  (let (( *perform-acc-analysis* t)
	(init (make-initial-state cascaded-tanks-acc
				   '((inflowa (ifa* std))
				     (amounta (0 nil))
				     (amountb (0 nil))))))
    (special  *perform-acc-analysis*)
    (qsim init)
    (qsim-display init)
    ))

