;;; -*- Syntax: Common-Lisp; Package: QSIM -*-
;;; Copyright (c) 1992, Benjamin Kuipers
;;; $Id: hydraulic.lisp,v 1.2 92/06/09 13:54:51 clancy Exp $
;;;
;;; This is the CC component library for hydraulic and compartmental models.
;;; Compartmental models [Jacquez, 1985] are also the basic elements for most
;;; system dynamics models.
;;;
;;; Modeling assumptions:
;;;   - Flow variables point *into* the component.
;;;
;;;
;;;      Compartment:  abstract container
;;;

(in-package :qsim)
(define-component Compartment hydraulic
  "Compartmental modeling tank"
  (terminal-variables (in (inflow flow)
			  (Pin pressure))
		      (out (outflow flow)
			   (Pout pressure)))
  (component-variables (netflow flow)
		       (amount pressure))
  (constraints ((equal amount Pout))	; for Q2 test
	       ((add inflow outflow netflow))
	       ((d/dt amount netflow))))

;;; A compartment is drained by fractional transfer coefficients, so the
;;; effective pressure is simply the amount it contains.


;;;
;;;      Tank
;;;
(define-component Tank hydraulic
  "Gravity-draining tank of fluid"
  (terminal-variables (top (inflow flow)
			   (Pin pressure))
		      (bot (outflow flow)
			   (Pout pressure)))
  (component-variables (netflow flow)
		       (amount pressure))
  (constraints ((M+ amount Pout) (minf minf) (0 0) (inf inf))
	       ((add inflow outflow netflow))
	       ((d/dt amount netflow))))

;;; For a tank, the dependence of Pout on Amount is a property of the tank.


;;;
;;;      Arrow
;;;
(define-component Arrow hydraulic
  "flow with fractional transfer coefficient"
  (terminal-variables (in (Q flow)
			  (source pressure))
		      (out (-Q flow)
			   ))
  (component-variables (k resistance (quantity-space (0 k* inf))))
  (constraints ((minus Q -Q))
	       ((mult source k Q))
	       ((constant k k*))))

;;;
;;;      Pipe:  a bidirectional arrow, with a single coefficient
;;;
(define-component Pipe hydraulic
  "bidirectional flow, with single resistance"
  (terminal-variables (in (q  flow)
			  (p1 pressure))
		      (out (-q flow)
			   (p2 pressure)))
  (component-variables (k resistance (quantity-space (0 k* inf)))
		       (dp pressure))
  (constraints ((minus q -q))
	       ((add dp p2 p1))
	       ((mult k q dp))
	       ((constant k k*))))


;;;
;;;      QPump:  Pump with constant flow (Q)
;;;
(define-component QPump hydraulic
  "one-directional pumped flow, with constant flow rate"
  (terminal-variables (in  (Q flow (quantity-space (minf 0 Q* inf)))
			  )
		      (out (-Q flow)
			   ))
  (constraints ((constant Q Q*))
	       ((minus Q -Q))))

;;;
;;;      PPump:  Pump with Pressure Head
;;;
(define-component PPump hydraulic
  "bidirectional pumped flow, with resistance and pressure head"
  (terminal-variables (in (Q flow)
			  (p1 pressure))
		      (out (-Q flow)
			   (p2 pressure)))
  (component-variables (k resistance  (quantity-space (0 k* inf)))
		       (head pressure (quantity-space (0 H* inf)))
		       (ep pressure)
		       (dp pressure))
  (constraints ((minus Q -Q))
	       ((add p1 head ep))
	       ((add dp p2 ep))			; dp = p1 + h - p2
	       ((mult dp k Q))
	       ((constant head H*))
	       ((constant k k*))))

;;;
;;;      Source
;;;
(define-component Source hydraulic
  "Infinite tank for source of flow"
  (terminal-variables (out (P pressure (quantity-space (minf 0 P* inf)))))
  (constraints ((constant P P*))))

;;; P* is some constant positive pressure for the Source.

;;;
;;;      Sink
;;;
(define-component Sink hydraulic
  "Infinite tank for destination of flow"
  (terminal-variables (in  (P pressure)))
  (constraints				; no constraints from a Sink
	       ))

;;; The inflow to a bathtub can be modeled as a QPump from a Source.
;;; The drain can be modeled as an Arrow or Pipe to a Sink.

;;;
;;;      Stopper
;;;
(define-component Stopper hydraulic
  "Attach to a compartment terminal to prevent flow"
  (terminal-variables (in  (Q flow)
			   ))
  (constraints ((constant Q 0))))


