;;; -*- Syntax: Common-Lisp; Package: QSIM -*-
;;; Copyright (c) 1991, Benjamin Kuipers
;;;  $Id: electrical.lisp,v 1.2 92/06/09 13:54:45 clancy Exp $
;;;
;;;  Component definitions for the Electrical Domain.

(in-package :qsim)


;;;  The {\bf Resistor} obeys Ohm's law:  $v_1 - v_2 = I * R$.  

(define-component Resistor electrical
  "Resistor:  Ohm's law"
  (terminal-variables (t1 (v1 voltage)
                          (i  current))
                      (t2 (v2 voltage)
                          (i2 current)))
  (component-variables (v voltage)
                       (r resistance (quantity-space (0 R* inf))))
  (mode-variables
   (mode working burnout))
  (constraints ((add v v2 v1))
               ((mult i r v))
               ((minus i i2))
	       ((mode working) -> ((constant r R*)))
	       ((mode burnout) -> ((constant r inf)))))

;;;  We define the implicit variable power --- $p = v*i$ --- in order 
;;;  to be able to express the resistor's capacity limit.

(define-component Resistor+ electrical
  "Resistor+:  Ohm's law plus maximum power limit"
  (terminal-variables (t1 (v1 voltage)
                          (i  current))
                      (t2 (v2 voltage)
                          (i2 current)))
  (component-variables (v voltage)
                       (p power (quantity-space (0 max-power inf)))
                       (r resistance (quantity-space (0 R* inf))))
  (mode-variables
   (mode (working <- (and (p ((0 max-power) nil))
			  (r ((0 inf) nil))))
	 (burnout <- (p ((max-power inf) nil)))))
  (constraints ((add v v2 v1))
               ((mult i r v))
               ((minus i i2))
               ((mult v i p))
	       ((mode working) -> ((constant r R*)))
	       ((mode burnout) -> ((constant r inf)))
	       )
  ;; trying to implement:  (transitions ((p (max-power inc)) => burn-out))
  )


;  The {\bf Battery} maintains constant voltage:  $v_{bat} = v_1 - v_2$.
;  (A battery viewed as an exhaustible reservoir of charge is essentially identical to
;  the capacitor model below.)

(define-component Battery electrical
  "Battery:  constant voltage source"
  (terminal-variables (t1 (v1 voltage)
                          (i  current))
                      (t2 (v2 voltage)
                          (i2 current)))
  (component-variables (v voltage (quantity-space (0 vbat inf))))
  (constraints ((add v v2 v1))
               ((minus i i2))
               ((constant v vbat))))


;;;  The {\bf Capacitor} obeys the equation:  $I = C * \frac{d}{dt} (v_1 - v_2)$.

(define-component Capacitor electrical
  "Capacitor:  container for charge"
  (terminal-variables (t1 (v1 voltage)
                          (i  current))
                      (t2 (v2 voltage)
                          (i2 current)))
  (component-variables (v voltage)
                       (c capacitance (quantity-space (0 C* inf)))
                       (q charge))
  (constraints ((add v v2 v1))
               ((mult v c q))
               ((d/dt q i))
               ((minus i i2))
               ((constant c C*))))


;;; The {\bf Inductor} obeys the equation:  $v_1 - v_2 = L * \frac{d}{dt} I$.

(define-component Inductor electrical
  "Inductor:  container for flux"
  (terminal-variables (t1 (v1 voltage)
                          (i  current))
                      (t2 (v2 voltage)
                          (i2 current)))
  (component-variables (v voltage)
                       (L inductance (quantity-space (0 L* inf)))
                       (f flux-linkage))
  (constraints ((add v v2 v1))
               ((mult L i f))
               ((d/dt f v))
               ((minus i i2))
               ((constant L L*))))

;;;  {\bf Ground} is a component that acts as a reference voltage, $v=0$.
;;;  Any number of components can be connected to ground.

(define-component Ground electrical
  "Ground:  constant voltage (current sink)"
  (terminal-variables (t (v voltage)
                         (i current)))
  (constraints ((constant v 0))
	       ((constant i 0))))

;;;  A {\bf Reference Point} is a component that does nothing, but allows
;;;  voltage to be measured.

(define-component Reference-Point electrical
  "Reference point:  test-point for voltage; zero current"
  (terminal-variables (t (v voltage)
                         (i current)))
  (constraints ((constant i 0))))

;;;  The {\bf Switch} has two states, and either guarantees that current does not flow,
;;;  or that two voltages are identical.  This model is also appropriate to
;;;  treating a wire as a component, rather than as an ideal connection.

(define-component Switch electrical
  "Switch:  externally opened or closed"
  (mode-variables (mode open closed))
  (terminal-variables (t1 (v1 voltage)
                          (i1 current))
                      (t2 (v2 voltage)
                          (i  current)))
  (component-variables (v voltage))
  (constraints ((add v v2 v1))
	       ((minus i i1))
               ((mode open)    ->  ((constant i 0)))
               ((mode closed)  ->  ((constant v 0)))))

;;;  The ideal {\bf Diode} allows current to flow in one direction,
;;;  but not in the other.  This is essentially the same as the {\tt Switch}
;;;  component, except that {\tt transitions} clauses allow it to change from
;;;  one mode to the other itself.

(define-component Diode electrical
  "Diode:  wire in one direction, open in the other"
  (mode-variables
   (mode (open   <- (v ((minf 0) ign)))
	 (closed <- (v (0 ign)))))
  (terminal-variables (t1 (v1 voltage)
                          (i1 current))
                      (t2 (v2 voltage)
                          (i  current)))
  (constraints ((add v v2 v1))
	       ((minus i i1))
               ((mode open)    ->  ((constant i 0)))
               ((mode closed)  ->  ((constant v 0)))
	       ))

;;;  Q:  Will the correct transitions compile from this?
;;;      What are the correct transitions, anyway?

;;;  The {\bf Transistor} model still needs to be supplied.
