/*
 *  'token' reads the QDE input file, identifies the next "token" (which can
 *  be a word, a quoted string, or a punctuation character), and returns the
 *  identity of the new token.  If the token was a word or string, it is
 *  placed in the symbol array, unless it matches an existing symbol.
 *  In either case, the index of the token in the symbol array is returned.
 */

#include  <stdio.h>
#include  <ctype.h>

#ifdef ATTUNIX
#include  <string.h>
#else
#include  <strings.h>
#endif

#include  "defs.h"
#define   EXTERN extern
#include  "global.h"
#include  "io.h"


/*
 *  These are the possible values for 'mode'.
 */
#define  SEARCHING  0	/* searching for start of a token */
#define  COMMENT    1	/* inside a comment, looking for newline */
#define  TOKEN      2	/* inside a alphanumeric token, looking for end */
#define  STRING     3	/* inside a string token, looking for end */


static int  mode;	/* mode = SEARCHING, TOKEN, or COMMENT */


/*
 *  'mopen' opens the model input file for reading.
 */
mopen(filename)
char *filename;
{
    mode = SEARCHING;
    mstream = fopen( filename, "r");
    if (mstream == NULL)
    {
	printf("Can't open file %s\n", filename);
	exit(1);
    }
}


/*
 *  'token' gets the next "token" from the input file and returns a value
 *  identifying it as being any of the following:
 *     -- left parenthesis (LPAREN)
 *     -- right parenthesis (RPAREN)
 *     -- end-of-file (MEOF)
 *     -- a symbol (an index value 0-n in the array 'symbol').
 *
 *  When 'token' encounters a symbol it looks for the symbol in the
 *  symbol table.  If found, it returns that index.  If not found, it
 *  is appended to the symbol table and the new index returned.
 */
token()
{
    register char  *cp;
    register int   c, i;
    char buf[80];

    while(1)
    {
	c = getc(mstream);
	switch(mode)
	{
	case COMMENT:
	    switch(c)
	    {
	    case '\n':  mode = SEARCHING;  continue;
	    case EOF:   return(MEOF);
	    default:    continue;
	    }

	case SEARCHING:
	    switch(c)
	    {
	    case EOF:  return(MEOF);
	    case '(':  return(LPAREN);
	    case ')':  return(RPAREN);
	    case ';':  mode = COMMENT;  continue;
	    case '"':  mode = STRING;  cp = &buf[0];  continue;
	    default:
		if (isspace(c))  continue;
		mode = TOKEN;
		cp = &buf[0];
		*cp++ = (char) c;
		continue;
	    }

	case TOKEN:
	    switch(c)
	    {
	    case EOF:  break;
	    case '(':  break;
	    case ')':  break;
	    case ';':  break;
	    default:
		if (isspace(c))  break;
		*cp++ = (char) c;
		continue;
	    }
	    /* now have a complete token in buf */
	    ungetc(c,mstream);
	    mode = SEARCHING;
	    *cp = '\0';		/* terminate string with a null */
	    /* search for string in symbol table */
	    for (i = 0;  i < nsymbols;  i++)
		if ( strcci( buf, symbol[i]) == 0)  return(i);
	    /* string not found, so add it to symbol table. */
	    if (nsymbols >= MAXSYMBOLS)
	    {
		printf("Exceeded limit of %d symbols\n", MAXSYMBOLS);
		exit(1);
	    }
	    symbol[nsymbols] = symp;
	    cp = &buf[0];
	    while ( *symp++ = *cp++);	/* copy string */
	    if (symp >= sympmax)
	    {
		printf("Exceeded limit of %d symbol characters\n", MAXSYMCHARS);
		exit(1);
	    }
	    return(nsymbols++);

	case STRING:
	    switch(c)
	    {
	    case EOF:  printf("EOF while inside a string!");  return(MEOF);
	    case '"':  break;
	    default:
		*cp++ = (char) c;
		continue;
	    }
	    /* now have a complete string in buf */
	    *cp = '\0';		/* terminate string with a null */
	    mode = SEARCHING;
	    /* Add string to symbol table. */
	    symbol[nsymbols] = symp;
	    cp = &buf[0];
	    while ( *symp++ = *cp++);	/* copy string */
	    return(nsymbols++);
	}
    }
}
