;;; -*- Mode:Common-Lisp; Package:POS; Syntax:COMMON-LISP; Default-character-style:(FIX BOLD NORMAL); Base:10 -*-
;;; Copyright 1988 David Throop at the University of Texas at Austin
;;; X Windows additions by Kee Kimbrell Spring 1991
;;; Allegro CL additions by Mike Chien and Kee Kimbrell Summer 1991
;;; Lispview extensions by BKay 27Jun92

(in-package  :pos)

#+:ccl
(use-package :ccl :pos)

#+symbolics
(use-package :scl :pos)

#+:pos-lispview
(use-package :clos)

#+symbolics
(import '(self send defmethod))

(proclaim '(special *alu* *thickness* *qplot-output-file-name*
		    *window-number*))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Queue management
;;;
;;; Queues are used by both qgraph (to store datasets) and the lispview
;;; version of the pos (to store drawing commands in-order).
;;;
;;; This code added 27Jun92 by BKay
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; This is an implementation of a FCFS queue.

(defstruct (queue (:print-function display-q))
  "An FCFS queue"
  (head nil)  ; the first conspair of the queue
  (tail nil)  ; the last conspair of the queue
  (len 0)     ; the length of the queue
  )

(defun display-q (q stream ignore)
  (declare (ignore ignore))
  (format stream "#<QUEUE: len=~d  ~a>" (queue-len q) (queue-head q)))
	  
;;; Returns a queue that is empty
(defun make-q (&key (init nil))
  (let ((q (make-queue)))
    (when init (qpush init q))
    q))


;;; Push an entry onto its proper place in the queue
;;; Returns: the queue.
;;;
(defun qpush (e q)
  (let ((new (list e)))
    (if (qempty q)
	(progn (setf (queue-tail q) new) (setf (queue-head q) new))
	(progn (setf (cdr (queue-tail q)) new) (setf (queue-tail q) new))))
  (incf (queue-len q))
  q)


;;; Pop the top off of the queue.
;;; Returns: an element or NIL if the queue is empty
;;;
(defun qpop (q)
  (when (not (qempty q))
    (let ((e (car (queue-head q))))
      (setf (queue-head q) (cdr (queue-head q)))
      (when (null (queue-head q)) (setf (queue-tail q) nil))
      (decf (queue-len q))
      e)))

(defun qempty (q)
  (null (queue-head q)))

(defun qtop (q)
  (when (not (qempty q))
    (car (queue-head q))))

(defmacro qlist (q)
  `(queue-head ,q))

(defmacro qlength (q)
  `(queue-len ,q))

;;; Pop the last element pushed off of the queue.
(defun qpop-backend (q)
  (when (not (qempty q))
    (cond
      ((= (qlength q) 1)
       (qpop q))
      (T
       (let ((e (first (queue-tail q))))
	 (setf (queue-tail q)
	       (nthcdr (- (qlength q) 2) (queue-head q)))
	 (setf (cdr (queue-tail q))
	       NIL)
	 (decf (queue-len q))
	 e)))))
      

; DEVICE DEPENDENT PLOTTING ROUTINES.
 
;;; Each function has two parts.  They may send their output to the
;;; screen, a postscript file, or both.  The first section sends the
;;; image to the PS file. The second section is
;;; conditionalized,depending on the hardware, using the #+ readmacro.
;;; Graphics commands for different machines go in this code.

;;; Each output device will have a different set of tasks to do when
;;; starting a new behavior


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; LUCID SPECIFIC CODE
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

#+(and l-windows lcl3.0)
  (defmacro positions-of (points)
            `(loop for (xpt  ypt) on ,points by #'cddr
	                collect (make-position :x (round xpt)
		   	        	       :y (round ypt))))



#+(and l-windows  (not lcl3.0))
(defun positions-of (points &aux pile)
       (do ((psstk points (cddr psstk)))
           ((null psstk))
           (push  (make-position :x (round (first psstk))
		   	         :y (round (second psstk)))
	          pile))
       (reverse pile))



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; X WINDOWS SPECIFIC CODE
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

#+(and unix x-windows)
  (defun get-output-filename ()
         (format nil "/tmp/xbatchtmp~d" (get-internal-real-time)))



#+x-windows
(defmacro set-color (alu)
         `(unless (equalp *alu* ,alu) 
                  (format *qplot-output* "setcolor	~a~%" ,alu)
                  (setq *alu* ,alu)))


#+x-windows
(defmacro set-thickness (thickness)
          `(unless (= *thickness* ,thickness)     
                   (format *qplot-output* "setlinewidth	~d~%" ,thickness)
                   (setq *thickness* ,thickness)))



#+x-windows
(defun x-set-font (pos-font)
       (let ((x-font-face		(get-x-font-face (car pos-font) (cadr pos-font)))
             (x-font-size		(get-x-font-size   (caddr pos-font))))
            (format *qplot-output* "setfont		*~a-*-~d*~%" x-font-face x-font-size)))


#+x-windows
(defun get-x-font-face (pos-family pos-face)
       (case pos-family
       		(:swiss		(case pos-face
					(:italic	"helvetica-medium-o")
                                        (:bold		"helvetica-bold-r")
                                        (:bold-italic	"helvetica-bold-o")
                                        (t		"helvetica-medium-r")))
                (:fix		(case pos-face
					(:italic	"courier-medium-o")
                                        (:bold		"courier-bold-r")
                                        (:bold-italic   "courier-bold-o")
                                        (t		"courier-medium-r")))
                (t              (case pos-face
                                        (:italic	"times-medium-i")
                                        (:bold		"times-bold-r")
                                        (:bold-italic	"times-bold-i")
                                        (t		"times-medium-r"))) ))


#+x-windows
(defun get-x-font-size (pos-size)
       (case pos-size
		(:tiny		80)
                (:very-small	80)
                (:smaller	100)
                (:small		100)
                (:normal	120)
                (:large		140)
                (:larger	180)
                (:very-large	240)
                (t		120) ))

;;; *helvetica-medium-r-*-120*


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; LISPVIEW SPECIFIC CODE
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; Most of the lispview code is in XPOS:XPOS;lispview-extensions.lisp.
;;; Since this file is only loaded when lispview is loaded, those 
;;; functions and macros which are used in parts of POS that get loaded
;;; under either X or lispview need to be defined in either case.
;;; The functions listed here are therefore defined twice -- once for
;;; the lispview version and once for the X version.

;;; A structure that holds the POS version of the graphics context.
;;; This is used for quick comparisons, since extracting font and
;;; color info from the x graphics context is very slow.
;;;
#+:pos-lispview
(defstruct posgcon
  (color        "black")
  (font         *plain-font*)
  (dashed       nil)
  (dash-pattern '(10 10))
  (thickness    1))


;;; Set the graphics context of a plot.
;;; Using a graphics context rather than passing specific
;;; contex settings in a call is MUCH more efficient.
;;; 
#+:pos-lispview
(defmacro with-qplot-gcon ((&key (color *black*)
				 (thickness 1) (dashed nil)
				 (font *plain-font*)
				 (dash-pattern '(quote (10 10)))) &body body)
  `(let ((old-posgcon (pos-gcon *qplot-output*)))
    (unwind-protect
       (progn
	 (setf (pos-gcon *qplot-output*)
	       (make-posgcon :color ,color
			     :font (quote ,font)
			     :dashed ,dashed
			     :dash-pattern ,dash-pattern
			     :thickness ,thickness))
	 (lv:with-graphics-context ((gcon *qplot-output*)
				    :line-width ,thickness
				    :line-style (if ,dashed :dash :solid)
				    :font (get-x-font (quote ,font))
				    :dashes ,dash-pattern
				    :foreground (lv:find-color :name ,color))
	   ,@body)))
    (setf (pos-gcon *qplot-output*) old-posgcon)))


;;; The macro is a no-op without lispview
;;;
#-:pos-lispview
(defmacro with-qplot-gcon ((&key (color *black*) (alu *black*)
				 (thickness 1) (dashed nil)
				 (dash-pattern '(quote (10 10)))) &body body)
  (declare (ignore color alu thickness dashed dash-pattern))
  `(progn ,@body))




;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; SYMBOLICS SPECIFIC CODE
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


;;; #+symbolics
;;; (defmacro gll (graphcall)
;;;           `(+ ,graphcall (if (< gray-level .875) x 0)))

;;;#+(or symbolics ti)
;;;(defmacro get-gray-level (alu)
;;;          `(cond ((equal ,alu *gray-one*)    .1)
;;;                 ((equal ,alu *gray-two*)    .2)
;;;                 ((equal ,alu *gray-three*)  .3)
;;;                 ((equal ,alu *gray-four*)   .4)
;;;                 ((equal ,alu *gray-five*)   .5)
;;;                 ((equal ,alu *gray-six*)    .6)
;;;                 ((equal ,alu *gray-seven*)  .7)
;;;                 ((equal ,alu *gray-eight*)  .8)
;;;                 ((equal ,alu *gray-nine*)   .9)
;;;                 ((equal ,alu *gray-ten*)    1)
;;;                 (t                          1)))


;;;#+(or symbolics ti)
;;;(defun get-gray-level (alu)
;;;       (cond ((equal alu *gray-one*)    .1)
;;;             ((equal alu *gray-two*)    .2)
;;;             ((equal alu *gray-three*)  .3)
;;;             ((equal alu *gray-four*)   .4)
;;;             ((equal alu *gray-five*)   .5)
;;;             ((equal alu *gray-six*)    .6)
;;;             ((equal alu *gray-seven*)  .7)
;;;             ((equal alu *gray-eight*)  .8)
;;;             ((equal alu *gray-nine*)   .9)
;;;             ((equal alu *gray-ten*)    1)
;;;             (t                         10)))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; TI SPECIFIC CODE
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

#+ti       
(defun arrays-from-pointlist (points numpoints)
  (loop with xpoints = (make-array numpoints)
	  with ypoints = (make-array numpoints)
	  for (x y) on points by #'cddr
	  for count from 0
	  do (setf (aref xpoints count) x)
	  (setf (aref ypoints count) y)
	  finally (return (list xpoints ypoints))))



#+ti
(defun ellp (center-x center-y x-radius y-radius
		      alu  thickness)
    (loop with points = (min (round (* 2 (max center-x center-y))) 20)
	  with xarray = (make-array (1+ points))
	  with yarray = (make-array (1+ points))
	  for count from 0 to points
	  for theta = (* count 2 (/ pi points))
	  do (setf (aref xarray count)
		   (+ center-x (* x-radius  (cos theta)))) 
	  (setf (aref yarray count)
		(+ center-y (*  y-radius (sin theta))))
	  finally (send *qplot-output* :draw-cubic-spline xarray yarray 4
			thickness w:black alu :cyclic)))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;  CCL Specific Code
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

#+:ccl
(proclaim '(object-variable *MWindow*))

#+:ccl
(defobject *auto-redraw-window* *window*)

#+:ccl
(defobfun (exist *auto-redraw-window*) (init-list)
  "Create slots for initialization and handle for a saved picture."
  (declare (object-variable init-flag saved-picture))
  (usual-exist init-list)
  (without-interrupts
   (have 'init-flag nil)
   (have 'saved-picture)
   (start-picture)
   (setq init-flag T)
   (pen-show)))

#+:ccl
(defobfun (window-update-event-handler *auto-redraw-window*) ()
  "Get current picture, handle update, then redraw picture and kill it."
  (declare (object-variable init-flag saved-picture))
  (if (and (ownp 'init-flag) init-flag)
    (without-interrupts
     (setq saved-picture (get-picture))
     (usual-window-update-event-handler)
     (erase-rect 0 (window-size))
     (start-picture)
     (pen-show)
     (draw-picture saved-picture)
     (setq saved-picture (kill-picture saved-picture)))
    (usual-window-update-event-handler)))

#+:ccl
(defobfun (window-close *auto-redraw-window*) ()
   "Release storage used for holding picture."
   (declare (object-variable init-flag saved-picture))
   (without-interrupts
    (if init-flag
      (progn
        (setq saved-picture (get-picture))
        (setq saved-picture (kill-picture saved-picture))))
    (usual-window-close)))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; COMMON CODE
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

       
(defmacro get-color (alu color)
         #-symbolics `(if (equal ,color *black*) ,alu ,color)
         #+symbolics `(cond ((not (numberp ,alu))   1)
                            ((not (numberp ,color)) 1)
                            (t  (if (equal ,color *black*) ,alu ,color))))
                
(defun string-conversion (string)
       (typecase string				; We handle these types individually
           (string      string)			; to aviod an expensive call to FORMAT.
           (null        "NIL")
           (symbol      (symbol-name string))
           (t           (let ((*print-pretty*    nil))
	                     (princ-to-string string)))))



(defparameter *symbol-ps-size* 12)


#+(or l-windows x-windows :ccl)
(defun qvec (x1 y1 x2 y2 &key (alu *black*) (color *black*))
       #+l-windows      (qplot-vector x1 y1 x2 y2 :arrow-head-length  4 
                                                  :arrow-base-width   4
		                                  :thickness          2 
                                                  :shaftthick         2
                                                  :alu                alu
                                                  :color              color)
       #+x-windows
                        (qplot-vector x1 y1 x2 y2 :arrow-head-length  4 
                                                  :arrow-base-width   4
		                                  :shaftthick         2
                                                  :alu                alu
                                                  :color              color)
       #+:ccl           (qplot-vector x1 y1 x2 y2 :arrow-head-length  2   ;modified DJC 21Oct91
                                                  :arrow-base-width   4
		                                  :shaftthick         1
                                                  :alu                alu
                                                  :color              color))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; QPLOT-PRIMITIVES CODE
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; New-window - Tells systems capable of making multiple windows to
;;;              use a new one.
;;; Size       - Requests that the new window be of a particular size.
;;;              (X Y) : make it be X x Y pixels.
;;;              :fit  : make it fit the maximum extent of the data.
;;;              NIL   : use a system default size.
;;;
(defun qplot-new-behavior (&key (box nil)
				(gcon nil)
				(new-window T) (size :fit))
       #-:pos-lispview (declare (ignore new-window size gcon))
       #+lispm (setq *qplot-output* *standard-output*)
       (when (image-to-postscript-p)
             (ps-new-behavior :box box))
       (when (image-to-screen-p)
             #+symbolics         (send *qplot-output* :new-behavior)
             #+ti                (send *qplot-output* :clear-screen)
             #+(or symbolics ti) (send *qplot-output* :set-cursorpos
	                                              0 (round (- yscreen bmargin)))
             #+l-windows         (progn (slb-cl::clear-window-stream *qplot-output*)
	                                (setf xscreen (slb-cl::window-stream-inner-width *qplot-output*)
		                              yscreen (slb-cl::window-stream-inner-height *qplot-output*)))
             #+x-windows         (progn (setq *qplot-output-file-name* (get-output-filename))
                                        (setq *thickness* 1)
                                        (setq *alu* *black*)
                                        (setq *current-font* nil)
                                        (setq *qplot-output* (open *qplot-output-file-name* :direction :output))
                                        (format *qplot-output* "setlinewidth	1~%"))

             #+:ccl              (setf (objvar *MWindow*)
                                           (oneof *auto-redraw-window*
                                                       :window-title (format nil "Qsim - ~a" *window-number*)
                                                       :window-type :tool
                                                       :window-size (make-point (- *screen-width* 20) 
                                                                                (- *screen-height* 75))))
	     #+:pos-lispview     (progn
				   (when
				       (or (not (typep *qplot-output* 'qplot-display))
					   (not (use-lv-call *qplot-output*)))
				     ;; Set up the window only if we aren't
				     ;; actually plotting.
				     (cond
				       ;; Make a new qplot-display if we want a new window.
				       (new-window
					(setq *qplot-output*
					      (make-instance 'qplot-display
							     :size size))
					(when gcon
					  (format t "~%Resetting context--THIS IS WRONG")
					  (setf (gcon *qplot-output*)
						(apply #'make-instance 'lv:graphics-context
						       gcon))))
				       
				       ;; Otherwise, clear the current display (via qplot-new-behavior).
				       (T
					(when (not (typep *qplot-output* 'qplot-window))
					  (error "QPLOT-NEW-BEHAVIOR called with :new-window NIL, but there~
                                             ~%is no window currently bound to *QPLOT-OUTPUT*"))
					(setf (commands *qplot-output*) NIL)
					)))
				   (if (use-lv-call *qplot-output*)
				       (lv-new-behavior (window *qplot-output*) box)
				       (qpush `(qplot-new-behavior :box box)
					      (commands *qplot-output*)))
				   NIL)
				   
	     ))
                         








(defun qplot-end-display (&optional (newy (- yscreen 50)))
       #-(or ti symbolics)(declare (ignore newy))
       (when (image-to-postscript-p)
             (ps-end-image))
       (when (image-to-screen-p)
             #+symbolics         (send *qplot-output* :ADJUST-CURSOR-POSITION-AND-VISIBILITY 
		                                      (- (send *qplot-output* :cursor-x))
		                                      (- (round newy) (send *qplot-output* :cursor-y)))
             #+ti                (send *qplot-output* :set-cursorpos
	                                              0 (round newy))
             #+l-windows         t		; 50 pixels sets this the same as QSIMs BMARGIN
	     #+x-windows         (let ((arg-one (format nil "~a/xbatch  -e -f ~a -n qplot-~d &" 
                                                        *obj-dir* *qplot-output-file-name* *window-number*)))
				   (setq *window-number* (1+ *window-number*))
				   (close *qplot-output*)
				   #+kcl	(system arg-one)
				   #+lucid	(user::shell arg-one)
				   #+allegro    (user::shell arg-one)
				   #+:ccl       (setq *window-number* (1+ *window-number*)))
	     #+:pos-lispview
	     (when (null (window *qplot-output*))
	       ;; Assuming that the window is not already defined, 
	       ;; make the window and add a control panel with a
	       ;; print button.
	       (let (br display control-panel quit-button print-button
			(cntrl-panel-height 25))
		 (multiple-value-bind (width height)
		     (lv-window-size *qplot-output*)
		   (setf (base-window *qplot-output*)
			 (make-instance
			  'lv:base-window
			  :mapped nil
			  :width (round width)
			  :height (round
				   (+ height cntrl-panel-height))
			  :label (format nil "Qplot ~a"
					 *window-number*)
			  :icon (lv-make-icon
				 (format nil "Qplot ~a"
					 *window-number*)))))
;		 (push 	(make-instance 'repaint-qplot) (lv:interests (base-window *qplot-output*)))

		 (setf br
		       (lv:bounding-region (base-window *qplot-output*)))
		 (setf control-panel
		       (make-instance
			'lv:panel
			:parent (base-window *qplot-output*)
			:mapped t
			:left 0 :top 0
			:width (lv:region-width br)
			:height cntrl-panel-height))
		 (setf (panel *qplot-output*) control-panel)
		 (setf (window *qplot-output*)
		       (make-instance
			'qplot-window
			:qplot-display *qplot-output*
			:parent (base-window *qplot-output*)
			:mapped t
			:top 27 :left 0
;			:border-width 1
			:width (lv:region-width br)
			:height (- (lv:region-height br)
				   25)))

		 ;; This builds a closure for the quit and print buttons.
		 (setf display *qplot-output*)
		 
		 ;; Put a "quit" button on the display
		 (setf quit-button
		       (make-instance
			'lv:command-button
			:parent control-panel
			:mapped t
			:command #'(lambda ()
				     (lv:withdraw (base-window display)
						  (lv:parent
						   (base-window display)))
				     (setf (window display) nil))
			:foreground (lv:find-color :name
						   :darkslategray)
			:label "Quit"))
		 (when (eq *image-disposal* :screen)
		   ;; Allow printing from screen only if
		   ;; we aren't printing by default.
		   (setf print-button
			 (make-instance
			  'lv:command-button
			  :parent control-panel
			  :mapped t
			  :command #'(lambda ()
				       (lv-postscript display))
			  :foreground (lv:find-color :name
						     :darkslategray)
			  :label "Print")))
		 (setf (lv:mapped (base-window display)) t)
		 ;; Wait until the window appears.
		 (loop until
		       (eq (lv:status (window *qplot-output*))
			   :realized)) )
	       (incf *window-number*)

	       ;; Set the busy bit so that :immediate commands are ignored
	       (setf (status *qplot-output*) :busy)

	       ;; We don't push a qplot-end-display since this
	       ;; will always be added automatically at the end
	       ;; for printing and isn't used by lv.
	       )
	     ))




(defun qplot-dot (x y &rest keys
		    &key (alu   #-:pos-lispview *black*
			        #+:pos-lispview (posgcon-color (pos-gcon *qplot-output*)))
		         (color #-:pos-lispview *black*
				#+:pos-lispview (posgcon-color (pos-gcon *qplot-output*)))
			 )
  #-:pos-lispview (declare (ignore keys))
  (when (image-to-postscript-p)
    (ps-draw-point  x y :alu        (get-color alu color) 
		    :font       *dot-font*))
  (when (image-to-screen-p)
    #+symbolics           (graphics:draw-point (round x) (round y) 
					       :alu        tv:alu-seta
					       :gray-level (get-color alu color)
					       :stream     *qplot-output*)
    #+ti                  (send *qplot-output* :draw-point (round x) (round y))
    #+(and l-windows lcl3.0)  
    (windows::draw-rectangle *qplot-output* (round x) (round y)
			     :width 2 :height 2)
    #+(and l-windows (not lcl3.0))
    (draw-line-xy *qplot-output* (round x) (round y) (+ (round x) 1) (+ (round y) 1)
		  :brush-width 2)
    #+x-windows           (progn (set-color (get-color alu color))
				 (format *qplot-output* "drawpoint	~d ~d~%" (round x) (round y)))
    #+:ccl                (let ((x1 (round x))(y1 (round y)))
			    (objvar *MWindow*)
			    (ask *MWindow* (set-pen-pattern (get-color alu color)))
			    (ask *MWindow* (set-pen-size (make-point 1 1)))
			    (ask *MWindow* (frame-oval (- x1 1) (- y1 1)
					;((<- to balance parens (+ x1 1) (+ y1 1)))
						       x1 y1))
                                        ;(ask *MWindow* (move-to (- x1 1) (+ y1 1)))
                                        ;(ask *MWindow* (line-to (+ x1 1) (+ y1 1)))
                                        ;(ask *MWindow* (line-to (+ x1 1) (- y1 1)))
                                        ;(ask *MWindow* (line-to (- x1 1) (- y1 1)))
                                        ;(ask *MWindow* (line-to (- x1 1) (+ y1 1)))
			    )

    #+:pos-lispview       (progn
			    (if (use-lv-call *qplot-output*)
				(lv-qplot-dot (window *qplot-output*) x y
					      alu color)
				(qpush
				 `(qplot-dot ,x ,y ,@keys)
				 (commands *qplot-output*)))
			    NIL)
    ))





;;; Oddly, specifying thickness = 1 explicitly 
;;; moves the line 1, at least on Peano. 5/11/90 D.B.

(defun qplot-hline (x y length &rest keys
		      &key (alu   #-:pos-lispview *black*
				  #+:pos-lispview (posgcon-color (pos-gcon *qplot-output*)))
		           (color #-:pos-lispview *black*
				  #+:pos-lispview (posgcon-color (pos-gcon *qplot-output*)))
		           (thickness #-:pos-lispview 1
				      #+:pos-lispview (posgcon-thickness (pos-gcon *qplot-output*)))
		           (dashed    #-:pos-lispview nil
				      #+:pos-lispview (posgcon-dashed (pos-gcon *qplot-output*)))
		           (dash-pattern #-:pos-lispview '(10 10)
					 #+:pos-lispview (posgcon-dash-pattern
							  (pos-gcon *qplot-output*))))
  #-:pos-lispview (declare (ignore keys))
  (when (image-to-postscript-p)
    (if dashed	(ps-draw-dashed-line x y (+ x length) y 
				     :alu          (get-color alu color) 
				     :dash-pattern dash-pattern)
	(ps-draw-line x y (+ x length) y 
		      :thickness thickness 
		      :alu       (get-color alu color))))
  (when (image-to-screen-p)
    #+symbolics (if (= thickness 1)                         
		    (graphics:draw-line (round x) (round y) (round (+ x length)) (round y)
					:alu        tv:alu-seta
					:gray-level (get-color alu color)
					:stream     *qplot-output*)
		    (graphics:draw-line (round x) (round y) (round (+ x length)) (round y)
					:alu        tv:alu-seta
					:gray-level (get-color alu color)
					:stream     *qplot-output*
					:thickness  (max 1 (round thickness))))
    #+:pos-lispview
    (progn
      (if (use-lv-call *qplot-output*)
	  (lv-qplot-hline
	   (window *qplot-output*) x y length
	   alu color thickness dashed dash-pattern)
	  (qpush
	   `(qplot-hline ,x ,y ,length ,@keys)
	   (commands *qplot-output*)))
      NIL)
    #-(or symbolics :pos-lispview)
    (qplot-line (round x) (round y) (round (+ x length)) (round y) :alu alu 
		:color color :thickness thickness
		:dashed dashed :dash-pattern dash-pattern)
    ))




(defun qplot-vline (x y length &rest keys
		      &key (alu   #-:pos-lispview *black*
				  #+:pos-lispview (posgcon-color (pos-gcon *qplot-output*)))
		           (color #-:pos-lispview *black*
				  #+:pos-lispview (posgcon-color (pos-gcon *qplot-output*)))
		           (thickness #-:pos-lispview 1
				      #+:pos-lispview (posgcon-thickness (pos-gcon *qplot-output*)))
		           (dashed    #-:pos-lispview nil
				      #+:pos-lispview (posgcon-dashed (pos-gcon *qplot-output*)))
		           (dash-pattern #-:pos-lispview '(10 10)
					 #+:pos-lispview (posgcon-dash-pattern
							  (pos-gcon *qplot-output*))))
       #-:pos-lispview (declare (ignore keys))
       (when (image-to-postscript-p)
             (if dashed	(ps-draw-dashed-line x y x (+ y length) 
                                             :alu          (get-color alu color) 
                                             :dash-pattern dash-pattern)
	                (ps-draw-line x y x (+ y length) 
                                      :thickness thickness 
                                      :alu (get-color alu color))))
       (when (image-to-screen-p)
             #+symbolics (graphics:draw-line (round x) (round y) (round x) (round (+ y length)) 
                                             :alu        tv:alu-seta
                                             :gray-level (get-color alu color)
                                             :stream     *qplot-output* 
                                             :thickness  (max 1 (round thickness)))
	     #+:pos-lispview (progn
			       (if (use-lv-call *qplot-output*)
				   (lv-qplot-vline
				    (window *qplot-output*) x y length
				    alu color thickness dashed dash-pattern)
				 (qpush
				  `(qplot-vline ,x ,y ,length ,@keys)
				  (commands *qplot-output*)))
			       NIL)
             #-(or symbolics :pos-lispview)
	                 (qplot-line (round x) (round y) (round x) (round (+ y length)) :alu alu 
				     :color color :thickness thickness
				     :dashed dashed :dash-pattern dash-pattern)))



(defun qplot-line (x1 y1 x2 y2 &rest keys
		      &key (alu   #-:pos-lispview *black*
				  #+:pos-lispview (posgcon-color (pos-gcon *qplot-output*)))
		           (color #-:pos-lispview *black*
				  #+:pos-lispview (posgcon-color (pos-gcon *qplot-output*)))
		           (thickness #-:pos-lispview 1
				      #+:pos-lispview (posgcon-thickness (pos-gcon *qplot-output*)))
		           (dashed    #-:pos-lispview nil
				      #+:pos-lispview (posgcon-dashed (pos-gcon *qplot-output*)))
		           (dash-pattern #-:pos-lispview '(10 10)
					 #+:pos-lispview (posgcon-dash-pattern
							  (pos-gcon *qplot-output*))))
       #-:pos-lispview (declare (ignore keys))
       (when (image-to-postscript-p)
             (if dashed	(ps-draw-dashed-line x1 y1 x2 y2 :alu          (get-color alu color) 
                                                         :thickness    thickness
                                                         :dash-pattern dash-pattern)
	                (ps-draw-line x1 y1 x2 y2 :thickness  thickness 
                                                  :alu        (get-color alu color))))
       (when (image-to-screen-p)
             #+symbolics  (graphics:draw-line (round x1) (round y1) (round x2) (round y2)
				             :stream       *qplot-output*     
                                             :alu          tv:alu-seta
                                             :gray-level   (get-color alu color)
				             :dash-pattern dash-pattern 
                                             :dashed       dashed
				             :thickness    (max 0 (round thickness)))

             #+ti	  (if dashed (send *qplot-output* :draw-dashed-line  (round x1) (round y1)
		                                  (round x2) (round y2) thickness w:black (get-color alu color)
		                                  (apply #'+ dash-pattern) t 0 (first dash-pattern))
	                             (send *qplot-output* :draw-line (round x1) (round y1)
		                                  (round x2) (round y2) thickness w:black (get-color alu color)))

             #+l-windows  (draw-line-xy  *qplot-output* (round x1) (round y1) (round x2) (round y2)
			                       :brush-width thickness :color (get-color alu color))

             #+x-windows  (progn (set-thickness thickness)
                                 (set-color (get-color alu color))
                                 (when dashed (format *qplot-output* "setlinestyle	Dashed~%")
                                              (format *qplot-output* "setdashes	~d" (length dash-pattern))
                                              (mapcar #'(lambda (x) (format *qplot-output* " ~d" x)) dash-pattern)
                                              (format *qplot-output* "~%"))
                                 (format *qplot-output* "drawline	~d ~d ~d ~d~%" (round x1) (round y1) 
                                                                                       (round x2) (round y2))
                                 (if dashed (format *qplot-output* "setlinestyle	Solid~%")))
             #+:ccl       (progn
                                (objvar *MWindow*)
                                (ask *MWindow* (set-pen-pattern (get-color alu color)))
;;;                               (if dashed (ask *MWindow* (set-pen-pattern *light-gray-pattern*)))
                                (ask *MWindow* (set-pen-size (make-point thickness thickness)))
                                (ask *MWindow* (move-to (round x1) (round y1)))
                                (ask *MWindow* (line-to (round x2) (round y2)))
                                (if dashed (ask *MWindow* (set-pen-pattern *black-pattern*))))
	     #+:pos-lispview (progn
			       (if (use-lv-call *qplot-output*)
				   (lv-qplot-line
				    (window *qplot-output*) x1 y1 x2 y2
				    alu color thickness dashed dash-pattern)
				 (qpush
				  `(qplot-line ,x1 ,y1 ,x2 ,y2 ,@keys)
				  (commands *qplot-output*)))
			       NIL)
	     ))

                       


;;; POINTS is an list of (x y x y ...).

(defun qplot-lines (points &rest keys
		      &key (alu   #-:pos-lispview *black*
				  #+:pos-lispview (posgcon-color (pos-gcon *qplot-output*)))
		           (color #-:pos-lispview *black*
				  #+:pos-lispview (posgcon-color (pos-gcon *qplot-output*)))
		           (thickness #-:pos-lispview 1
				      #+:pos-lispview (posgcon-thickness (pos-gcon *qplot-output*)))
		           (dashed    #-:pos-lispview nil
				      #+:pos-lispview (posgcon-dashed (pos-gcon *qplot-output*)))
		           (dash-pattern #-:pos-lispview '(10 10)
					 #+:pos-lispview (posgcon-dash-pattern
							  (pos-gcon *qplot-output*))))
       #-:pos-lispview (declare (ignore keys))
       (when (image-to-postscript-p)
             (ps-draw-lines points :alu          (get-color alu color)       
                                   :thickness    thickness
		                   :dashed       dashed 
                                   :dash-pattern dash-pattern))
       (when (image-to-screen-p)
             #+symbolics (graphics:draw-lines (mapcar 'round points)
		                              :stream       *qplot-output*
		                              :alu          tv:alu-seta
                                              :gray-level   (get-color alu color)
                                              :thickness    thickness
		                              :dashed       dashed 
                                              :dash-pattern dash-pattern)

             #+ti        (let* ((numpoints  (/ (length points) 2))
		                (parrays    (arrays-from-pointlist points numpoints)))
	                 (send *qplot-output* :draw-polyline (first parrays) (second parrays)
		                              thickness w:black numpoints (get-color alu color)))

             #+l-windows
                         (draw-lines *qplot-output* (positions-of points)
			                            :brush-width thickness :color (get-color alu color))

             #+x-windows (progn (set-thickness thickness)
                                (set-color (get-color alu color))
                                (when dashed (format *qplot-output* "setlinestyle	dashed~%")
                                             (format *qplot-output* "setdashes	~d" (length dash-pattern))
                                             (mapcar #'(lambda (x) (format *qplot-output* " ~d" x)) dash-pattern)
                                             (format *qplot-output* "~%"))
                                (format *qplot-output* "drawlines	~d " (/ (length points) 2))
                                (mapcar #'(lambda (x) (format *qplot-output* "~d " x)) (mapcar #'round points))
                                (format *qplot-output* "~%")
                                (if dashed (format *qplot-output* "setlinestyle	LineSolid~%")))
             #+:ccl      (labels ((d-lines (f-x f-y pts) 
                                         (cond ((null pts) nil)
                                               (t   (qplot-line f-x f-y (car pts) (cadr pts)
                                                               :alu            alu
                                                               :color          color
                                                               :thickness      thickness
                                                               :dashed         dashed
                                                               :dash-pattern   dash-pattern)
                                                    (d-lines (car pts) (cadr pts) (cddr pts))))))
                               (cond ((oddp (length points))  (error "List must be even"))
                                     (t     (d-lines (car points) (cadr points) (cddr points)))))

	     #+:pos-lispview (progn
			       (if (use-lv-call *qplot-output*)
				   (lv-qplot-lines
				    (window *qplot-output*) points
				    alu color thickness dashed dash-pattern)
				 (qpush
				  `(qplot-lines ,points ,@keys)	
				  (commands *qplot-output*)))
			       NIL)
	     ))




(defun qplot-string (string x y &rest keys
		      &key (alu   #-:pos-lispview *black*
				  #+:pos-lispview (posgcon-color (pos-gcon *qplot-output*)))
		           (color #-:pos-lispview *black*
				  #+:pos-lispview (posgcon-color (pos-gcon *qplot-output*)))
		           (font  #-:pos-lispview *plain-font*
				  #+:pos-lispview (posgcon-font (pos-gcon *qplot-output*))))
       #-:pos-lispview (declare (ignore keys))
       (setf string (string-conversion string))
       (when (image-to-postscript-p)
             (ps-draw-string string x y :alu        (get-color alu color) 
                                        :font       font))
       (when (image-to-screen-p)
             #+symbolics (graphics:draw-string string (round x) (round y) 
                                                      :character-style font
				                      :alu             tv:alu-seta
                                                      :gray-level      (get-color alu color)
                                                      :stream          *qplot-output*)
             #+ti        (send *qplot-output* :string-out-explicit
	                                       string (round x) (round (- y 9)) nil nil
	                                       font tv:alu-xor 0 (zlc:string-length string) nil)
             #+l-windows
                         (windows::stringblt *qplot-output*
				             (windows::make-position
				             (round x) (round y)) font string
				             :operation (if (equal (get-color alu color) *black*))
					                    boole-ior boole-andc1)
             #+x-windows (progn (set-color (get-color alu color))
                                (unless (equal *current-font* font)
                                        (x-set-font font)
                                        (setq *current-font* font))
                                (format *qplot-output* "drawstring	")
                                (format *qplot-output* "~d ~d " (round x) (round y))
				(format *qplot-output* "~a~%" string))
             #+:ccl      (progn (objvar *MWindow*)
                                (let ((original-font (ask *MWindow* (window-font))))
                                     (ask *MWindow* (set-pen-pattern (get-color alu color)))
                                     (ask *MWindow* (set-window-font font))
                                     (ask *MWindow* (move-to (round x) (round y)))
                                     (princ string *MWindow*)
                                     (ask *MWindow* (set-window-font original-font))))
	     #+:pos-lispview (progn
			       (if (use-lv-call *qplot-output*)
				   (lv-qplot-string
				    (window *qplot-output*) string x y
				    alu color font)
				 (qpush
				  `(qplot-string ,string ,x ,y ,@keys)
				  (commands *qplot-output*)))
			       NIL)
	     ))




(defun qplot-circle (x y radius &rest keys
		      &key (alu   #-:pos-lispview *black*
				  #+:pos-lispview (posgcon-color (pos-gcon *qplot-output*)))
		           (color #-:pos-lispview *black*
				  #+:pos-lispview (posgcon-color (pos-gcon *qplot-output*)))
		           (thickness  #-:pos-lispview 1
				       #+:pos-lispview (posgcon-thickness (pos-gcon *qplot-output*)))
			   (filled nil) (start-angle 0) (end-angle (* 2 pi)))
       #-:pos-lispview (declare (ignore keys))
       (when (image-to-postscript-p)
             (ps-draw-circle x y radius :alu (get-color alu color) 
                                        :filled      filled      
                                        :thickness   thickness
		                        :start-angle start-angle 
                                        :end-angle   end-angle))
       (when (image-to-screen-p)			; tolerates floating point

             #+symbolics   (graphics:draw-circle  x y radius 
                                                 :alu         tv:alu-seta
                                                 :gray-level  (get-color alu color)
                                                 :thickness   thickness	
				                 :stream      *qplot-output* :filled filled
				                 :start-angle start-angle 
                                                 :end-angle   end-angle)

             #+ti          (if filled 
                               (send *qplot-output*  :draw-filled-circle	; Arcs not currently implemented for TI.
		                     (round x)(round y) (round radius) w:black (get-color alu color))
	                       (send *qplot-output*  :draw-circle
		                     (round x) (round y) (round radius) thickness w:black (get-color alu color)))


    ;; Currently, Lucid-CL has no support for :filled.  Drawing a cicle of half the
    ;; radius with a thick brush is a poor hack, and it looks it.  Complain to Lucid.

             #+l-windows   (if filled 
                               (windows::draw-circle *qplot-output* (windows::make-position (round x)(round y))
				                     (round radius) :width (round radius)
				                     :operation (if (equal (get-color alu color) *black*)
						                    windows::boole-1 windows::boole-c1))
		               (draw-circle-xy *qplot-output*  (round x)(round y) (round radius)
				                               :brush-width thickness 
                                                               :color       (get-color alu color)))

             #+x-windows   (progn (set-thickness thickness)
                                  (set-color (get-color alu color))
                                  (if filled
                                       (format *qplot-output* "fillarc		")
                                       (format *qplot-output* "drawarc		"))
                                  (format *qplot-output* "~d ~d ~d ~d " (round (- x radius))
                                                                        (round (- y radius))
                                                                        (* 2 radius) (* 2 radius))
                                  (format *qplot-output* "~d ~d~%" (round (* 64 (* start-angle (/ 360 (* 2 pi)))))
                                                                   (round (* 64 (* end-angle (/ 360 (* 2 pi)))))))
             #+:ccl        (let* ((x1         (round x))
                                  (y1         (round y))
                                  (s          (round (- 90 (* start-angle  (/ 360 (* 2 pi))))))
                                  (e          (round (- (- 90 (* end-angle (/ 360 (* 2 pi)))) s))))
                                (objvar *MWindow*)
                                (ask *MWindow* (set-pen-size (make-point thickness thickness)))
                                (ask *MWindow* (set-pen-pattern (get-color alu color)))
                                (if filled
                                    (ask *MWindow*
                                         (paint-arc s e
                                                    (- x1 radius) (- y1 radius)
                                                    (+ x1 radius) (+ y1 radius)))
                                    (ask *MWindow*
                                         (frame-arc s e
                                                    (- x1 radius) (- y1 radius)
                                                    (+ x1 radius) (+ y1 radius)))))
	     #+:pos-lispview (progn
			       (if (use-lv-call *qplot-output*)
				   (lv-qplot-circle
				    (window *qplot-output*) x y radius
				    alu color filled start-angle end-angle thickness)
			           (qpush
				    `(qplot-circle ,x ,y ,radius ,@keys)
				    (commands *qplot-output*)))
			       NIL)
	     ))

                       


(defun qplot-ring (x y minor-r major-r &rest keys
		      &key (alu   #-:pos-lispview *black*
				  #+:pos-lispview (posgcon-color (pos-gcon *qplot-output*)))
		           (color #-:pos-lispview *black*
				  #+:pos-lispview (posgcon-color (pos-gcon *qplot-output*))))
       #-:pos-lispview (declare (ignore keys))
       (when (< major-r minor-r) (Error "Wrong size radii"))
       (when (image-to-postscript-p)
             (ps-draw-ring x y minor-r major-r 
                           :alu        (get-color alu color)))
       (when (image-to-screen-p)
             #+symbolics    (graphics:draw-circle (round x) (round y) (round minor-r) 
                                                  :inner-radius (round major-r)
		                                  :alu          tv:alu-seta
                                                  :gray-level   (get-color alu color)
                                                  :stream       *qplot-output*)

             #+ti           (let ((thickness      (round (- major-r minor-r)))
                                  (r              (round (+ minor-r (/ (- major-r minor-r) 2)))))
                                 (send *qplot-output* :draw-circle (round x) (round y) r thickness))

             #+l-windows    (draw-circle-xy *qplot-output* (round x) (round y)
			                    (round (/ (+ major-r minor-r) 2))
			                    :brush-width (- major-r minor-r)
			                    :color       (get-color alu color))

             #+x-windows    (let ((thickness      (round (- major-r minor-r)))
                                  (r              (round (+ minor-r (/ (- major-r minor-r) 2)))))
                                 (format *qplot-output* "setlinewidth	~d~%" thickness)
                                 (set-color (get-color alu color))
                                 (format *qplot-output* "drawarc	~d ~d ~d ~d " (round (- x r)) (round (- y r))
                                                                                      (round (* 2 r)) (round (* 2 r)))
                                 (format *qplot-output* "~d ~d~%" 0 23040)
                                 (format *qplot-output* "setlinewidth	~d~%" *thickness*))
             #+:ccl         (let ((thickness      (round (- major-r minor-r)))
                                  (r              (round major-r)))
                                 (objvar *MWindow*)
                                 (ask *MWindow* (set-pen-size (make-point thickness thickness)))
                                 (ask *MWindow* (set-pen-pattern (get-color alu color)))
                                 (ask *MWindow* (frame-oval (round (- x r)) (round (- y r))
                                                            (round (+ x r)) (round (+ y r)))))
	     #+:pos-lispview (progn
			      (if (use-lv-call *qplot-output*)
				  (lv-qplot-ring
				   (window *qplot-output*) x y minor-r major-r
				   alu color)
				  (qpush
				   `(qplot-ring ,x ,y ,minor-r ,major-r ,@keys)
				   (commands *qplot-output*)))
			      NIL)
	     ))



(defun qplot-ellipse (center-x center-y x-radius y-radius &rest keys
		      &key (alu   #-:pos-lispview *black*
				  #+:pos-lispview (posgcon-color (pos-gcon *qplot-output*)))
		           (color #-:pos-lispview *black*
				  #+:pos-lispview (posgcon-color (pos-gcon *qplot-output*)))
		           (thickness  #-:pos-lispview 1
				       #+:pos-lispview (posgcon-thickness (pos-gcon *qplot-output*)))
			   (filled nil) (start-angle 0) (end-angle (* 2 pi)) (rotation 0))
       #-:pos-lispview (declare (ignore keys))
       (when (image-to-postscript-p)
             (ps-draw-ellipse center-x center-y x-radius y-radius 
                              :alu         (get-color alu color) 
                              :filled      filled
		              :start-angle start-angle 
                              :end-angle   end-angle 
                              :thickness   thickness
		              :rotation    rotation))
       (when (image-to-screen-p)			; tolerates floating point
             #+symbolics    (graphics:draw-ellipse 0 0 x-radius y-radius 
                                                   :alu         tv:alu-seta
                                                   :gray-level  (get-color alu color)
                                                   :rotation    rotation
				                   :stream      *qplot-output* 
                                                   :filled      filled 
                                                   :thickness   thickness
				                   :translation (list center-x center-y)
				                   :start-angle start-angle 
                                                   :end-angle   end-angle)

             #+ti           (ellp center-x center-y x-radius y-radius (get-color alu color) thickness)

             #+l-windows    (error "qplot-ellipse not supported for lucid")

             #+x-windows    (progn (set-thickness thickness)
                                   (set-color (get-color alu color))
                                   (format *qplot-output* "drawarc	~d ~d ~d ~d " (round (- center-x x-radius))
                                                                                      (round (- center-y y-radius))
                                                                                      (round (* 2 x-radius)) 
                                                                                      (round (* 2 y-radius)))
                                   (format *qplot-output* "~d ~d~%" (round (* 64 (* start-angle (/ 360 (* 2 pi)))))
                                                                    (round (* 64 (* end-angle (/ 360 (* 2 pi)))))))
             #+:ccl         (let* ((x1         (round (- center-x x-radius)))
                                   (y1         (round (- center-y y-radius)))
                                   (x2         (round (+ center-x x-radius)))
                                   (y2         (round (+ center-y y-radius)))
                                   (s          (round (- 90 (* start-angle  (/ 360 (* 2 pi))))))
                                   (e          (round (- (- 90 (* end-angle (/ 360 (* 2 pi)))) s))))
                                (objvar *MWindow*)
                                (ask *MWindow* (set-pen-size (make-point thickness thickness)))
                                (ask *MWindow* (set-pen-pattern (get-color alu color)))
                                (if filled
                                    (ask *MWindow* (paint-arc s e x1 y1 x2 y2))
                                    (ask *MWindow* (frame-arc s e x1 y1 x2 y2))))
	     #+:pos-lispview (progn
			      (if (use-lv-call *qplot-output*)
				  (lv-qplot-ellipse
				   (window *qplot-output*) center-x center-y
				   x-radius y-radius
				   alu color filled start-angle end-angle thickness
				   rotation)
				  (qpush
				   `(qplot-ellipse ,center-x ,center-y
						   ,x-radius ,y-radius ,@keys)
				   (commands *qplot-output*)))
			      NIL)
	     ))
                                                   



;;; POINTS is a list of (x y x y x y ...)


(defun qplot-polygon (points &rest keys
		      &key (alu   #-:pos-lispview *black*
				  #+:pos-lispview (posgcon-color (pos-gcon *qplot-output*)))
		           (color #-:pos-lispview *black*
				  #+:pos-lispview (posgcon-color (pos-gcon *qplot-output*)))
		           (thickness  #-:pos-lispview 1
				       #+:pos-lispview (posgcon-thickness (pos-gcon *qplot-output*)))
			   (filled nil))
       #-:pos-lispview (declare (ignore keys))
       (when (image-to-postscript-p)
             (ps-draw-polygon  points
		               :filled      filled 
                               :thickness   thickness 
                               :alu         (get-color alu color)))
       (when (image-to-screen-p)			; graphics:draw-polygon tolerates floating point
             #+symbolics   (graphics:draw-polygon points 
                                                  :stream     *qplot-output*
			                          :alu        tv:alu-seta
                                                  :filled     filled
                                                  :gray-level (get-color alu color))

             #+ti          (qplot-lines points :alu alu :color color :dashed nil :thickness thickness)

             #+l-windows   (draw-polygon *qplot-output* (positions-of points)
			                 :color         (get-color alu color) 
                                         :filled        nil 
                                         :brush-width   thickness)

             #+x-windows   (let ((pts  (append points (list (car points) (cadr points)))))
			     (set-thickness thickness)
			     (set-color (get-color alu color))
			     (if filled 
				 (format *qplot-output* "fillpolygon	~d " (/ (length points) 2))
				 (format *qplot-output* "drawlines	~d " (/ (length points) 2)))
			     (mapcar #'(lambda (x) (format *qplot-output* "~d " x)) (mapcar #'round points))
			     (format *qplot-output* "~%"))
             #+:ccl        (progn (objvar *MWindow*)
                                  (ask *MWindow* (start-polygon))
                                  (with-plotting-to-postscript-inhibited
                                        (qplot-lines (append points (list (car points) 
                                                                    (cadr points)))
                                                     :alu       alu
                                                     :color     color
                                                     :dashed    nil
                                                     :thickness thickness))
                                  (let ((poly (ask *MWindow* (get-polygon))))
                                       (if filled (ask *MWindow* (paint-polygon poly))
                                                  (ask *MWindow* (frame-polygon poly)))
                                       (ask *MWindow* (kill-polygon poly))))
	     #+:pos-lispview (progn
			       (if (use-lv-call *qplot-output*)
				   (lv-qplot-polygon
				    (window *qplot-output*) points
				    alu color filled thickness)
				 (qpush
				  `(qplot-polygon ,points ,@keys)
				  (commands *qplot-output*)))
			       NIL)
	     ))
                                
                               


;; Draw a box.  (XPOS, YPOS) is upper left corner

(defun qplot-box (xpos ypos xwide yheight &rest keys
		      &key (alu   #-:pos-lispview *black*
				  #+:pos-lispview (posgcon-color (pos-gcon *qplot-output*)))
		           (color #-:pos-lispview *black*
				  #+:pos-lispview (posgcon-color (pos-gcon *qplot-output*)))
		           (thickness  #-:pos-lispview 1
				       #+:pos-lispview (posgcon-thickness (pos-gcon *qplot-output*)))
			   (filled nil))
       #-:pos-lispview (declare (ignore keys))
       (when (image-to-postscript-p)
             (ps-draw-box
                  xpos ypos (+ xpos xwide) (+ yheight ypos)	; left top right bottom
                  :thickness   thickness 
                  :alu         (get-color alu color) 
                  :filled      filled))
       (when (image-to-screen-p)
             #+symbolics   (graphics:draw-rectangle	; left top right bottom
		                (round xpos) (round ypos) (round (+ xpos xwide)) (round (+ yheight ypos))
		                :thickness  thickness 
                                :filled     filled 
                                :stream     *qplot-output* 
                                :alu        tv:alu-seta
                                :gray-level (get-color alu color))

             #+ti          (if filled
	                       (send *qplot-output* :draw-filled-rectangle 
		                    xpos ypos xwide yheight w:black (get-color alu color) t)
	                       (send *qplot-output* :draw-rectangle 
		                    xpos ypos xwide yheight thickness w:black (get-color alu color)))

             #+l-windows   (if filled
		               (draw-line-xy *qplot-output* (+ xpos (/ xwide 2.0)) ypos
			            (+ xpos (/ xwide 2.0)) (+ ypos yheight)  
                                    :color       (get-color alu color)
			            :brush-width xwide)
		               (draw-rectangle-xy *qplot-output* xpos ypos xwide yheight 
                                    :color       (get-color alu color)
				    :filled      nil 
                                    :brush-width thickness))

             #+x-windows   (progn (set-thickness thickness)
                                  (set-color (get-color alu color))
                                  (if filled 
                                       (format *qplot-output* "fillrectangle	")
                                       (format *qplot-output* "drawrectangle	"))
                                  (format *qplot-output* "~d ~d ~d ~d~%" (round xpos)  (round ypos) 
                                                                         (round xwide) (round yheight)))
             #+:ccl        (progn (objvar *MWindow*)
                                  (ask *MWindow* (set-pen-size (make-point thickness thickness)))
                                  (ask *MWindow* (set-pen-pattern (get-color alu color)))
                                  (ask *MWindow* (frame-rect (round xpos)
                                                             (round ypos)
                                                             (round (+ xpos xwide))
                                                             (round (+ ypos yheight)))))
	     #+:pos-lispview (progn
			       (if (use-lv-call *qplot-output*)
				   (lv-qplot-box
				    (window *qplot-output*) xpos ypos xwide yheight
				    alu color thickness filled)
			           (qpush
				    `(qplot-box ,xpos ,ypos ,xwide ,yheight ,@keys)
				    (commands *qplot-output*)))
			       NIL)
	     ))







; Special symbols.  If you add any new symbols to this list; they must be added to 
; the list of exported functions from the POS package, (the parameter *EXPORTS*).

(defun qplot-symbol (x y symbol &rest keys
		      &key (alu   #-:pos-lispview *black*
				  #+:pos-lispview (posgcon-color (pos-gcon *qplot-output*)))
		           (color #-:pos-lispview *black*
				  #+:pos-lispview (posgcon-color (pos-gcon *qplot-output*))))
  #-:pos-lispview (declare (ignore keys))
  (let ((xloc (round (+ x *symbol-x-offset*)))
	(yloc (round (+ y *symbol-y-offset*)))
	(small-sym-ps (round (* .66 *symbol-ps-size*)))
	#+ti (xti (round (+ x -5)))
	#+ti (yti (round (+ y -4)))
	#+(or lucid x-windows) (asiz 5)
        #+:ccl (asiz 3))
    (when (image-to-postscript-p)
      (case symbol
	((std :std) (ps-special-char (- x 2.5) (+ y 6.5) 176 
                                     :alu        (get-color alu color)))
	((inc :inc) (ps-special-char xloc yloc 173 
                                     :size       small-sym-ps
                                     :alu        (get-color alu color)))
	((dec :dec) (ps-special-char xloc yloc 175 
                                     :size       small-sym-ps
                                     :alu        (get-color alu color)))
	((ign :ign) (ps-draw-string "*" xloc yloc
                                        :alu        (get-color alu color)))
	((left :left) (ps-special-char xloc yloc 172 
                                       :size       small-sym-ps
                                       :alu        (get-color alu color)))
	((right :right) (ps-special-char xloc yloc 174 
                                         :size       small-sym-ps
                                         :alu        (get-color alu color)))
	(T (ps-draw-string "?"  (- x 1)(+ y 2)
                                :alu        (get-color alu color)))))
    (when (image-to-screen-p)
      #+symbolics (case symbol
		    ((std :std) (graphics:draw-circle
			   (round x) (round y) 2 :filled nil
			   :stream     *qplot-output* 
                           :alu        tv:alu-seta
                           :gray-level (get-color alu color)))
		    ((inc :inc) (graphics:draw-string
			   "" xloc yloc :stream     *qplot-output* 
                                          :alu        tv:alu-seta
                                          :gray-level (get-color alu color)))
		    ((dec :dec) (graphics:draw-string
			   "" xloc yloc :stream     *qplot-output* 
                                          :alu        tv:alu-seta
                                          :gray-level (get-color alu color)))
		    ((ign :ign) (graphics:draw-string
			   "*" xloc yloc :stream     *qplot-output* 
                                         :alu        tv:alu-seta
                                         :gray-level (get-color alu color)))
		    ((left :left) (graphics:draw-string
			    "" xloc yloc :stream     *qplot-output* 
                                           :alu        tv:alu-seta
                                           :gray-level (get-color alu color)))
		    ((right :right) (graphics:draw-string
			     "" xloc yloc :stream     *qplot-output* 
                                            :alu        tv:alu-seta
                                            :gray-level (get-color alu color)))
		    (t (graphics:draw-string "?" xloc yloc
					     :stream     *qplot-output* 
                                             :alu        tv:alu-seta
                                             :gray-level (get-color alu color))))

      #+ti (case symbol
	     ((std :std) (send *qplot-output* :draw-circle  (round x)(round y) 2))
	     ((inc :inc) (send *qplot-output* :draw-string w:cptfont-font  ""
			(1+ xti) yti w:black 0 8 1 (get-color alu color)))
	     ((dec :dec) (send *qplot-output* :draw-string w:cptfont-font ""
			(1+ xti) yti w:black 0 8 1 (get-color alu color)))
	     ((ign :ign) (send *qplot-output* :draw-string w:cptfont-font "*"
			xti yti w:black 0 8 1 (get-color alu color)))
	     ((left :left) (send *qplot-output* :draw-string w:cptfont-font ""
			 xti yti w:black 0 8 1 (get-color alu color)))
	     ((right :right) (send *qplot-output* :draw-string w:cptfont-font ""
			  xti yti w:black 0 8 1 (get-color alu color)))
	     (t (send *qplot-output* :draw-string w:cptfont-font "?"
		      (+ 2 xti) yti w:black 0 8 1 (get-color alu color))))
      #+(or x-windows l-windows)
      (with-plotting-to-postscript-inhibited
	      (case symbol
	        ((std :std)	(qplot-circle (round x) (round y) 2 :alu alu :color color))
	        ((inc :inc)	(qvec x (+ y asiz) x (- y asiz)     :alu alu :color color))
	        ((dec :dec)	(qvec x (- y asiz) x (+ y asiz)     :alu alu :color color))
	        ((ign :ign)	(qplot-string "*" xloc yloc         :alu alu :color color))
	        ((left :left)	(qvec (+ x asiz) y (- x asiz) y     :alu alu :color color))
	        ((right :right)	(qvec (- x asiz)  y (+ x asiz) y    :alu alu :color color))
	        (t		(qplot-string "?" xloc yloc         :alu alu :color color))))

      #+:ccl   
      (with-plotting-to-postscript-inhibited
	      (case symbol
	        ((std :std)	(qplot-circle (round x) (round y) 2 :alu alu :color color))
	        ((inc :inc)	(qvec x (+ y asiz) x (- y asiz)     :alu alu :color color))
	        ((dec :dec)	(qvec x (- y asiz) x (+ y asiz)     :alu alu :color color))
	        ((ign :ign)	(qplot-string "*" xloc yloc         :alu alu :color color))
	        ((left :left)	(qvec (+ x asiz) y (- x asiz) y     :alu alu :color color))
	        ((right :right)	(qvec (- x asiz)  y (+ x asiz) y    :alu alu :color color))
	        (t		(qplot-string "?" xloc yloc         :alu alu :color color))))
      #+:pos-lispview
      (progn
	(if (use-lv-call *qplot-output*)
	    (lv-qplot-symbol
	     (window *qplot-output*) x y symbol
	     alu color)
	    (qpush
	     `(qplot-symbol ,x ,y ,symbol ,@keys)
	     (commands *qplot-output*)))
	NIL)
      )))







(defun qplot-special-char (x y char ascii &rest keys
		      &key (alu   #-:pos-lispview *black*
				  #+:pos-lispview (posgcon-color (pos-gcon *qplot-output*)))
		           (color #-:pos-lispview *black*
				  #+:pos-lispview (posgcon-color (pos-gcon *qplot-output*)))
		           (font-info  #-:pos-lispview *plain-font*
				  #+:pos-lispview (posgcon-font (pos-gcon *qplot-output*)))
			   (size 12))
  #-:pos-lispview (declare (ignore keys))
  (when (image-to-postscript-p)
    (ps-special-char x y ascii 
		     :size       size 
		     :alu        (get-color alu color)))
  (when (image-to-screen-p)
    ;; Symbolics machines use the Char-info argument, all others ignore it.
    #+symbolics 	(graphics:draw-string (merge-font-info font-info char)
		  			      x y :stream     *qplot-output* 
                                                  :alu        tv:alu-seta
                                                  :gray-level (get-color alu color))

    #+ti		(send *qplot-output* :string-out-explicit
	       			(format nil "~a" char) (round x) (round y)
	       			nil nil font-info (get-color alu color) 0 1 nil)

    #+l-windows         (qplot-string "X" x y :color (get-color alu color) :font *plain-font*)

    #+x-windows		(with-plotting-to-postscript-inhibited
                                 (qplot-string char x y :alu alu :color color :font font-info))
    #+:ccl              (with-plotting-to-postscript-inhibited
                                 (qplot-string char x y :alu alu :color color :font font-info))
    #+:pos-lispview     (progn
			  (if (use-lv-call *qplot-output*)
			      (lv-qplot-special-char
			       (window *qplot-output*) x y char ascii
			       size font-info alu color)
			      (qpush
			       `(qplot-special-char ,x ,y ,char ,ascii ,@keys)
			       (commands *qplot-output*)))
			  NIL)
    ))






(defun qplot-vector (from-x from-y to-x to-y &rest keys
		      &key (alu   #-:pos-lispview *black*
				  #+:pos-lispview (posgcon-color (pos-gcon *qplot-output*)))
		           (color #-:pos-lispview *black*
				  #+:pos-lispview (posgcon-color (pos-gcon *qplot-output*)))
		           (dashed #-:pos-lispview nil
				  #+:pos-lispview (posgcon-dashed (pos-gcon *qplot-output*)))
		           (dash-pattern #-:pos-lispview '(10 10)
					 #+:pos-lispview (posgcon-dash-pattern
							  (pos-gcon *qplot-output*)))
		           #+:lucid (thickness #-:pos-lispview *black*
					       #+:pos-lispview (posgcon-thickness
								(pos-gcon *qplot-output*)))
			   (arrow-head-length #+symbolics tv:*default-arrow-length* #-symbolics 10)
			   (arrow-base-width #+symbolics tv:*default-arrow-width* #-symbolics 5)
			   (filled t) (shaftthick 1))
  #-:pos-lispview (declare (ignore keys thickness))
  #-(or pos-lispview ccl) (declare (ignore dash-pattern dashed))
  (when (image-to-postscript-p)
    (ps-draw-vector from-x from-y to-x to-y
		    :arrow-head-length arrow-head-length
		    :arrow-base-width  arrow-base-width
		    :shaftthick        shaftthick 
		    :alu               (get-color alu color)
		    :filled            filled))
  (when (image-to-screen-p)
    #+ti           (multiple-value-bind (points xbas ybas)
		       (triangle-point-translation from-x from-y to-x to-y
						   arrow-base-width arrow-head-length)
		     (with-plotting-to-postscript-inhibited
		       (qplot-line from-x from-y xbas ybas :thickness shaftthick 
				   :alu (get-color alu color))
		       (send *qplot-output* :draw-filled-triangle
			     (first points)(second points)(third points)
			     (fourth points)(fifth points)(sixth points)
			     w:black (get-color alu color))))
    
    #+l-windows    (multiple-value-bind (points  xbas ybas)
		       (triangle-point-translation
			 from-x from-y to-x to-y arrow-base-width arrow-head-length)
		     (with-plotting-to-postscript-inhibited
		       (qplot-line from-x from-y xbas ybas :thickness shaftthick 
				   :alu (get-color alu color))
		       (draw-polygon *qplot-output* (positions-of points)
				     :filled      nil 
				     :color       (get-color alu color)	; filling not supported yet   
				     :brush-width thickness)))
    
    #+(or x-windows symbolics)
    (multiple-value-bind (points xbas ybas)
	(triangle-point-translation from-x from-y to-x to-y 
				    arrow-base-width arrow-head-length)
      (with-plotting-to-postscript-inhibited
	(qplot-line from-x from-y xbas ybas :thickness shaftthick 
		    :alu alu
		    :color color)
	(qplot-polygon points :alu alu 
		       :color color
		       :filled filled 
		       :thickness shaftthick)))
    #+:ccl
    (multiple-value-bind (points xbas ybas)
	(triangle-point-translation from-x from-y to-x to-y 
				    arrow-base-width arrow-head-length)
      (with-plotting-to-postscript-inhibited
       (qplot-line (round (- from-x (/ shaftthick 2))) 
		   (round (- from-y (/ shaftthick 2)))
		   (round (- xbas (/ shaftthick 2)))   
		   (round (- ybas (/ shaftthick 2)))
		   :thickness      shaftthick 
		   :alu            alu
		   :color          color
		   :dash-pattern   dash-pattern
		   :dashed         dashed)
       (qplot-polygon points :alu alu 
		      :color color
		      :filled filled 
		      :thickness shaftthick)))
    #+:pos-lispview
    (progn
      (if (use-lv-call *qplot-output*)
	  (lv-qplot-vector
	   (window *qplot-output*) from-x from-y to-x to-y
	   alu color arrow-head-length arrow-base-width
	   filled #+lucid thickness dashed dash-pattern
	   shaftthick)
	(qpush
	 `(qplot-vector ,from-x ,from-y ,to-x ,to-y
			,@keys)
	 (commands *qplot-output*)))
      NIL)
    ))
      




#|
(defun qplot-arrowhead (from-x from-y to-x to-y &optional (drawlinep t)
			(shaftthick 1) (alu "black") (arrow-head-length 10) (arrow-base-width 5))
       (multiple-value-bind (points  xbas ybas)
            (triangle-point-translation
                  from-x from-y to-x to-y arrow-base-width arrow-head-length)
;;;	          from-x from-y to-x to-y (* shaftthick arrow-base-width)
;;;	                                  (* shaftthick arrow-head-length))
            (when  drawlinep
                   (qplot-line from-x from-y xbas ybas
		               :thickness shaftthick :alu alu))
            #+ti (send *qplot-output* :draw-filled-triangle
	               (first points)(second points)(third points)
	               (fourth points)(fifth points)(sixth points)
	               w:black (sym-color alu))
            #-ti (qplot-polygon points :alu alu :filled t)))

|#



;;; Under lispview, this takes the last element off of the commands stack.
;;;
(defun qplot-undo ()
  #+:pos-lispview (when (and (typep *qplot-output* 'qplot-display)
			     (not (qempty (commands *qplot-output*))))
		    (format t "~%Undoing")
		    (qpop-backend (commands *qplot-output*))
		    NIL)
  #-:pos-lispview (format t "QPLOT-UNDO unimplemented without lisview")
  )
