;;; -*- Mode:Common-Lisp; Package:QSIM; Base:10 -*-
;;;  $Id: xedge.lisp,v 1.5 1992/07/23 09:36:19 clancy Exp $

(in-package :qsim)

(export '(with-envisioning with-further-envisioning with-global-filter
	  state-cross-edge-identity state-cross-edge?
	  reset-state-table
	  state-cross-edge-predecessors))

#|
 Implement attainable envisionments.

    (with-envisioning (simulate-complex-system))

where simulate-complex-system creates an initial state and qsims from
it, will result in an attainable envisionment being built, rather than
the normal qsim behavior tree.

The envisionment is done with landmark generation turned off.
Therefore, the set of possible states is guarateed to be finite.

The basic representation for a qsim behavior tree is a tree with
back edges.  E.g.
	A --|-- B -- A'
            |-- C
would be a normal behavior tree, where A' is a state equivalent to A,
and is marked as a cycle, with a "cycle-identity" link to A.

The envisionment tree adds cross edges to this basic representation.
A cross edge connects nodes in two branches of a tree [cousins].
Suppose that the behavior tree is:
	A --|-- B -- A'
            |-- C -- B' -- A''
Where B=B' and A=A'.  Then the corresponding envisionment tree [with
cross edges] would be:
	A --|-- B -- A'
            |-- C -- B'
The state B' has a cross-edge-identity link to B.  This representation
is clearly much more compact.  If there had been a large subgraph
attached to A', then it would have been repeated below A''.

This representation is equivalent in expressiveness to a graph
representation of the envisionment:
        |------|
 	v      |
	A --|--B<-|
	    |     |
	    |--C--|
For tree-based algorithms, however, it is somewhat easier to use.

Note:  Get-behaviors will return the "wrong" answer on an
envisionment tree.   It would yield
	((A B A')
	 (A C B'))
on the above envisionment tree, rather than
        ((A B A')
         (A C B' A'))
I.e. the final states of the behaviors may either be "real" final
states, or cross edges, linking the state to the continuation of the
behavior.  [I say "real" because the standard representation allows
for cycles, which are not really final.]
|#

;;; Implement a global filter that looks for an identical state any
;;; where in the behavior state graph.  It only makes sense when lmark
;;; generation is turned off.
;;;
;;; The unique states are stored in the xedge-state-table slot of the sim.
;;;
;;; Useful interface functions:
;;;
;;; state-cross-edge-identity
;;; state-cross-edge?
;;; state-cross-edge-predecessors
;;; with-global-filter
;;; with-envisioning
;;;
;;;
;;;  (setq *plot-state-indices* :right-of-node)
;;;  
;;;         ^^^^^^^^^^^^^^^^^^^ for much more informative tree plotting!
;;;
;;; -----------
;;; This turns the state tree [really a graph with back edges] into a
;;; graph with both back and cross edges -- that is a graph of
;;; the atainable envisionment rooted in the initial state.  Without
;;; landmark generation, this is guaranteed to be finite [and probably
;;; reasonably sized] and smaller than the simple behavior graph.
;;;
;;; We might want to check if the proposed cross edge is really to an
;;; ancestor, in which case it is a back edge.  The back edges will be
;;; labeled as weak cycles.
;;;
;;; Question: what should get-behaviors do when applied to such a
;;; graph?  It could do the complete unrolling, or it could treat the
;;; cross edges like cycle or transition edges and clip the graph there.
;;;
;;;
;;; BUGS:
;;; Since the state-table is a global, it is a disaster to try to
;;; continue any simulation other than the most recent one.
;;; It should be stored on the sim, and probably be indexed by qde.
;;;

(pushnew 'cross-edge-filter *global-state-filters*)



(defmacro with-global-filter (filter &body body)
  "Ensure that FILTER is a global filter when BODY is executed."
  `(let ((*global-state-filters*
	  (if (member ',filter *global-state-filters*)
	      *global-state-filters*
	      (cons ',filter *global-state-filters*))))
    (declare (special *global-state-filters*))
    ,@body))

(defparameter *trace-cross-edge-filter* nil)

(defother state cross-edge-predecessors
  :description
  "Return a list of states whose cross-edge-identity links point here.")

(defother state cross-edge-identity
  :description
  "Return the state [or nil] that this has a cross edge to.")

(defun state-cross-edge? (s)
  "T if a state ends in a cross edge.  I.e. it's successors are the
same as the successors of another [previously generated] state."
  (member 'cross-edge (state-status s)))

;(defun reset-state-table ()
;  "Must be called to reset the state table after each simulation."
;  (clrhash *xedge-state-table*))

(defun cross-edge-filter (s)
  "Filter S if an equivalent state [same qvalues] has already been
seen.  Label S as a cycle [to fool eligable-for-successors], and add a
cross-edge-identity link between S and the equivalent state.  Using
this filter turns the state tree into an attainable envisonment.  See
the macros with-envisioning, and with-further-envisioning."
  (when *cross-edge-envisionment*
    (let* ((key (state-key s))
	   (state-hash-table (sim-xedge-state-table (state-sim s)))
	   (known (gethash key state-hash-table)))
      (cond
	((eq s known) s)		; filter may be wrongly
					; applied to a state twice!
	((and known (inconsistent-p known))
	 (setf (state-cross-edge-predecessors s) nil)
	 (setf (state-cross-edge-identity s) nil)
	 (setf (gethash key state-hash-table) s))
	(known
	 ;; An old state.
	 (trace-cross-edge-filter s known)
	 ;; Label as a cross-edge: s0...s->known... and  provide a link
	 ;; in either direction.
	 (pushnew 'CROSS-EDGE (state-status s))
	 (setf (state-cross-edge-identity s) known)
	 (pushnew s (state-cross-edge-predecessors known))
	 (when (inconsistent-p known)
	   (mark-inconsistent-state s (state-predecessor s)
				    "No consistent successors (xedge)"))
	 (return-from cross-edge-filter nil)
	 )
	(T;; A new state
	 ;; The following two [other] slots MUST BE CLEARED because of
	 ;; the brain-dead initialization of states in
	 ;; successor-state-from-predecessor, which does a
	 ;; copy-tree!!! of other.
	 (setf (state-cross-edge-predecessors s) nil)
	 (setf (state-cross-edge-identity s) nil)
	 (setf (gethash key state-hash-table) s)))))
  s)



;; Not used, and can be very inefficient.  Should be memoized.
(defun ancestor? (a s)
  "Is state A an ancestor of state S?  Recognizes the cross edges."
  (cond ((null s) NIL)
	((eq a s) T)
	((ancestor? a (state-predecessor s)))
	((some #'(lambda (p)
		   (ancestor? a p))
	       (state-cross-edge-predecessors s)))))


(defun trace-cross-edge-filter (s known)
  (when *trace-cross-edge-filter*
       (format *qsim-trace*
	       "~&FILTER: The state, ~a, is equal to ~a.~%" s known)))


(defmacro with-envisioning (&body body)
  "Reset the state-table, turn on the cross-edge filter and suppress
landmark generation for the duration of body."
  `(with-further-envisioning
    ;(reset-state-table)
    ,@body))

(defmacro with-further-envisioning (&body body)
  "Continue the current envisionment.  This is useful for wrapping
around a call to q-continue."
  `(with-global-filter cross-edge-filter
    (let ((*enable-landmark-creation* nil)
	  (*cross-edge-envisionment* t)
	  (*simulation-type* :depth-first)
	  (*plot-state-indices* :above-node))
      
      (declare (special *enable-landmark-creation* *cross-edge-envisionment*
			*simulation-type*))
      ,@body)))



;;;======================================================================
;;; Utilities.
;;;

(defun state-key (state)
  "Return: a list (time-point? (var (qmag qdir)...) for all vars,
excluding time, where var, qmag, qdir are symbols, not the qsim
structures."
  (cons (point-p (qmag (state-time state)))
	(cons (user-qde (state-qde state))
	      (mapcar #'(lambda (var.qval)
			  (list (car var.qval)
				(user-qval (cdr var.qval))))
		      (cdr (state-qvalues state))))))

(defun user-qde (qde)
  "REturn the name of the qde"
  (and qde
       (qde-name qde)))

(defun user-lmark (lmark)
  "Return the name of a landmark."
  (and lmark
       (lmark-name lmark)))

(defun user-qval (qval)
  "Return a list (qmag qdir) of symbols for qmag and qdir."
  (list (user-qmag qval)
	(qval-qdir qval)))

(defun user-qmag (qval)
  "Return either the symbol for qval or a list (lm1 lm2) of symbols."
  (let ((qmag (qmag qval)))
    (if (consp qmag)
	(list (user-lmark (car qmag))
	      (user-lmark (cadr qmag)))
	(user-lmark qmag))))

(defun print-hash (table)
  (format *Qsim-Report* "~&HASH TABLE")
  (maphash #'(lambda (key value)
	       (format *Qsim-Report* "~&  Key: ~a Entry: ~a" key value))
	   table))
