;;; -*- Mode:Common-Lisp; Package:QSIM; Syntax:COMMON-LISP; Base:10 -*-
;;;  $Id: time-scale.lisp,v 1.2 1991/07/22 15:45:29 clancy Exp $
;;; Copyright (c) 1987, Benjamin Kuipers.

(in-package 'QSIM)


;---------------------------------------------------------------------------------------------
;   Contents:   Source code for time-scale abstraction in QSIM.
;
;   Author:     Benjamin Kuipers
;
;   References: "Abstraction by Time-Scale in Qualitative Simulation", Benjamin Kuipers,
;               Proceedings of AAAI-87 (The Sixth National Conference on Artificial
;               Intelligence), July 13-17, 1987.
;
;   Synopsis:   "Time-scale abstraction" is a method for structuring a complex system
;               as a hierarchy of smaller, interacting equilibrium mechanisms.  Within
;               this hierarchy, a given mechanism views a slower one as being constant,
;               and a faster one as being instantaneous.  A perturbation to a fast
;               mechanism may be seen by a slower mechanism as a displacement of a
;               monotonic function constraint.  Where the structure of a large system
;               permits decomposition by time-scale, this abstraction method permits
;               qualitative simulation of otherwise intractibly complex systems.
;
;   Examples:   To demonstrate time-scale abstraction in QSIM, call q, then select "TSA"
;               from the catalog menu, then select "SODIUM-AND-WATER" from the structure
;               menu.  The file containing this example is in:
;               "...>examples>sodium-and-water.lisp"
;
;   Discussion: Consider the following system having 3 mechanisms at 3 different time
;               scales where "A" is the fastest and "C" is the slowest mechanism:
;                     _____                  _____                  _____
;                     |   | --- slower --- |   | --- slower --- |   |
;                     | A |                  | B |                  | C |
;                     |___| --- faster --- |___| --- faster --- |___|
;
;               Using the notation that A1,...,An represents states of mechanism A,
;               then simulation of the entire system (across 3 time scales) could
;               produce a sequence of states (a behavior) in the following order:
;
;                  slowest mechanism C:                       C8--C9--C10
;                                                                       |
;                                                              |         
;                  medium mechanism B:              B4--B5--B6        B11
;                                                             |         |
;                                                    |                  
;                  fastest mechanism A:   A1--A2--A3        A7        A12
;
;               Simulation begins with state A1 and proceeds until mechanism A reaches
;               equilibrium at A3.  Attention then turns to the next slower mechanism, B.
;               Using parameters shared between A and B, B4 is initialized from A3 and
;               simulation proceeds until B reaches equilibrium at B6.  At this point,
;               the faster mechanism A is updated from B6, producing A7 by propagation.
;               Then, values from B6 and A7 are available to initialize mechanism C,
;               producing C8.  Simulation proceeds until C reaches equilibrium at C10.
;               The faster mechanism B is updated from C10, producing B11, and then the
;               still faster mechanism A is updated from B11 and C10, producing A12.
;               End of simulation.
;---------------------------------------------------------------------------------------------


;---------------------------------------------------------------------------------------------
;  Function:  (tsa-simulation init)
;
;  Given:     init, an initial state (presumed to be at equilibrium);
;
;  Returns:   a set of hierarchical behaviors (hbehavior) of the form:
;
;         <behavior>  ::=  (<state> ... <state>)
;         <sbehavior> ::=  <behavior>  |
;                          (SBEH <mech-name> <behavior> <set of (PROP <fmech-name> <state>)>)
;         <hbehavior> ::=  <sbehavior>  |
;                          (FBEH <mech-name> <sbehavior> <hbehavior>)  |
;
;     where:  (FBEH <mech-name> <sbehavior> <hbehavior>) indicates that <sbehavior>
;             was generated by the "fast" mechanism <mech-name>, followed by the
;             <hbehavior> of a slower mechanism; and
;
;             (SBEH <mech-name> <behavior> <set of (PROP <fmech-name> <state>)>)
;             indicates that <behavior> was generated by a "slow" mechanism <mech-name>,
;             followed by propagation to complete the state of each faster mechanism
;             <fmech-name>.
;
;             Thus, the hierarchical behavior of the 3-mechanism system shown on the
;             previous page is:
;                       (FBEH A (A1 A2 A3)
;                           (FBEH B (SBEH B (B4 B5 B6) ((PROP A A7)))
;                               (SBEH C (C8 C9 C10) ((PROP B B11) (PROP A A12)))))
;
;  Comments:  -- In the special case where there is only one mechanism and therefore
;                no use of time-scale abstraction, then <hbehavior>  <behavior>.
;             -- This function calls initialize-and-run-slower-mechanisms, which in turn
;                calls simulate-across-time-scales (i.e., it's recursive).
;
;  Design:    Given a final state of one mechanism, and a normal state of one at a
;             different time-scale, determine the initial values for the new one.
;             If the new one is faster, just propagate to a complete equilibrium state;
;             if it is slower, simulate to a new equilibrium state.
;             The hard part about initializing the slower mechanism is neutralizing
;             problem corresponding values.
;---------------------------------------------------------------------------------------------

;;; Given a behavior, find faster mechanisms that share parameters with this one.
;;; Initialize them with values from the final state of this behavior, assume they
;;; are in equilibrium, and propagate to determine a complete state.
;;; TSA-SIMULATION separates simulation from building a structure for display.

;;; Modified so that *current-sim* was bound to the SIM of the initial state
;;; so that newly created states will use this SIM.  DJC 07/21/91

(defun tsa-simulation (init)
  (let* ((*current-sim* (state-sim init))
	 (faster (get-faster-mechanisms init))	; BUG?  get from final state?
	 (slower (get-slower-mechanisms init))
	 (behaviors nil))
    (qsim init)

    (cond ((and faster
		(or (not *ask-before-each-tsa-step*)
		    (y-or-n-p (format nil "After ~a, complete state of faster mechanisms ~a? "
				      (qde-name (state-qde init)) faster))))
	   (setq behaviors
		 (mapcar #'(lambda (sbeh)
			     (tsa-faster-mechanisms sbeh))
			 (get-behaviors init))))
	  (t (setq behaviors (get-behaviors init))))

    (cond ((and slower
		(or (not *ask-before-each-tsa-step*)
		    (y-or-n-p (format nil "After ~a, simulate slower mechanisms ~a? "
				      (qde-name (state-qde init)) slower))))
	   (setq behaviors
		 (mapcan #'(lambda (beh)
			     (tsa-slower-mechanisms beh))
			 behaviors))))
    behaviors))

(defun tsa-trace-display (hbeh)			; transparent displayer
  (when *show-behaviors-during-tsa*
    (display-behavior nil hbeh
		      (layout-from-state (first hbeh)) nil))
  hbeh)

;---------------------------------------------------------------------------------------------
;  Function:  (get-faster-mechanisms state)
;
;  Given:     state, a state of some mechanism;
;
;  Returns:   a list of faster mechanisms in increasing order of speed.
;
;  Design:    This function performs a breadth-first search through the mechanisms
;             to insure that the returned list of mechanisms is in a non-decreasing
;             order.  This order is essential to performing the propagation from
;             slower to faster mechanisms in the correct sequence.
;---------------------------------------------------------------------------------------------
; Retrieve the faster and slower mechanisms.

(defun get-faster-mechanisms (state)		; get ALL faster mechanisms
  (let ((flist  (lookup-set 'faster (qde-other (state-qde state)))))
    (append flist (get-faster-mechs flist))))

(defun get-faster-mechs (mlist)
  (cond ((null mlist) nil)
	(t (let ((flist (apply 'append
		   (mapcar #'(lambda (mech) (lookup-set 'faster (qde-other (eval mech))))
			   mlist))))
	     (append flist (get-faster-mechs flist))))))

;---------------------------------------------------------------------------------------------
;  Function:  (get-slower-mechanisms state)
;
;  Given:     state, a state of some mechanism;
;
;  Returns:   a list of the next-slowest mechanism(s).
;
;  Notes:     This function only finds the next-slowest mechanisms; unlike its cousin
;             get-faster-mechanisms, it does NOT recurse and try to find even slower
;             mechanisms.  (This is intentional).
;---------------------------------------------------------------------------------------------

(defun get-slower-mechanisms (state)		; get locally-known slower mechanisms
  (lookup-set 'slower (qde-other (state-qde state))))

; Keep pointers to faster and slower behaviors in the state.successors.tsa-id slot

(defun get-faster-states (final)
  (if (eql (car (state-successors final)) 'tsa-id)         
                                       ;;  used to call state-successor 02/14/91 DJC
      (lookup-set 'faster-states (cdr (filtered-successor-slot-contents final)))))

(defun get-slower-states (final)
  (if (eql (car (state-successors final)) 'tsa-id)         
                                      ;;  used to call state-successor 02/14/91 DJC
      (lookup-set 'slower-states (cdr (filtered-successor-slot-contents final)))))

(defun put-faster-states (final faster-states)
  (let ((succ (filtered-successor-slot-contents final)))     ;;  used to call state-successor 02/14/91 DJC
    (cond ((null succ)
	   (setq succ (list 'tsa-id (cons 'faster-states faster-states)))
	   (setf (state-successors final) succ))
	  ((null (lookup 'faster-states (cdr succ)))
	   (nconc succ (list (cons 'faster-states faster-states))))
	  (t (error "Something is already there!  ~a.successor = ~a." final succ)))))

(defun put-slower-states (final slower-states)
  (let ((succ (filtered-successor-slot-contents final)))     ;;  used to call state-successor 02/14/91 DJC
    (cond ((null succ)
	   (setq succ (list 'tsa-id (cons 'slower-states slower-states)))
	   (setf (state-successors final) succ))
	  ((null (lookup 'slower-states (cdr succ)))
	   (nconc succ (list (cons 'slower-states slower-states))))
	  (t (error "Something is already there!  ~a.successor = ~a." final succ)))))

;---------------------------------------------------------------------------------------------
;  Function:  (get-final-state beh)
;
;  Given:     beh, a behavior as defined above;
;
;  Returns:   a final state of behavior.
;---------------------------------------------------------------------------------------------

; Generic function for getting last state of a behavior (plus syntax check).

(defun get-final-state (beh)
  (cond ((and (listp beh)
	      (state-p (car beh)))
	 (car (last beh)))
	(t (error "Bad behavior syntax:  ~a" beh))))

;---------------------------------------------------------------------------------------------
;  Function:  (tsa-faster-mechanisms sbeh)
;
;  Given:     sbeh, a behavior of a "slow" mechanism (any mechanism other than the fastest).
;
;  Returns:   the behavior of the "slow" mechanism followed by the updated state of each
;             faster mechanisms, in the form:
;                 (SBEH  <mech-name>  <sbeh>  <set of (PROP <mech-name> <state>)> )
;
;  Notes:     After a "slow" mechanism has reached an equilibrium state, this function
;             is called to update the state of a faster mechanism by propagating values
;             obtained from the slow mechanism's final state.  Exactly what values are
;             applied to the fast mechanism?  Here is the algorithm:
;             1.  Determine what parameters are shared between the two mechanisms by
;                 taking the intersection of the define-normal parameters of the fast
;                 mechanism with the 
;---------------------------------------------------------------------------------------------
; Puts created states of faster mechanisms into final.successors.faster-states.
; Those states might be incomplete and branch to their completions.

(defun tsa-faster-mechanisms (sbeh)
  (let* ((final (get-final-state sbeh))
	 (faster (get-faster-mechanisms final))
	 (faster-states nil))
    (put-latest-state (state-qde final) final)
    (tsa-trace-display sbeh)
    (setq faster-states
	  (mapcar #'(lambda (mech)
		      (let* ((fast-normal (get-normal-state (eval mech)))
			     (shared (initial-values-for-new-state final fast-normal))
			     (qvalues (convert-user-values shared fast-normal))
			     (nstate (new-state-from-old-state
				       fast-normal qvalues
				       (format nil "Initialized from ~a as faster than ~a"
					       (state-name final)
					       (qde-name (state-qde final))))))
			(make-values-constant (qde-history (state-qde nstate)) nstate)
			(tsa-trace-display nstate)
			(complete-states-from-partial-state nstate)
			; => nil if inconsistent
			))
		  faster))
    (put-faster-states final faster-states)
    (if (member nil faster-states)		; i.e. some faster state was inconsistent
	(prune-inconsistent-state final "faster state with no completions"))
    t))

; => Failure to complete the faster state should not be an error.
;    It should refute the final slow state, from which it was created.


;---------------------------------------------------------------------------------------------
;  Function:  (tsa-slower-mechanisms fbeh)
;
;  Given:     fbeh, a behavior of a "fast" mechanism (any mechanism other than the slowest).
;
;  Returns:   set of:  (FBEH <mech-name> <behavior> <hbehavior>)
;---------------------------------------------------------------------------------------------
; Puts created states of slower mechanisms into final.successors.slower-states.
; Those states might be incomplete and branch to their completions.
; Calls tsa-simulation on the slow states.
;   Fixed to apply global state filters, and to make sure the list is flat.  (BJK:  10-8-90)

(defun tsa-slower-mechanisms (fbeh)
  (let* ((final (get-final-state fbeh))
	 (slower (get-slower-mechanisms final))
	 (slower-states nil))
    (put-latest-state (state-qde final) final)
    (tsa-trace-display fbeh)
    (setq slower-states
	  (mapcan
	    #'(lambda (mech)
		(let* ((slow-normal (get-normal-state (eval mech)))
		       (shared (initial-values-for-new-state final slow-normal))
		       (qvalues (convert-user-values shared slow-normal))
		       (nstate (new-state-from-old-state
				 slow-normal qvalues
				 (format nil "Initialized from ~a as slower than ~a"
					 (state-name final)
					 (qde-name (state-qde final))))))
		  (translate-cvts-when-possible final nstate)
		  (complete-states-from-partial-state nstate)
		  (mapcan #'apply-global-state-filters
			  (get-list-of-initial-states nstate))
		  ; => nil if inconsistent
		  ))
	    slower))
    (put-slower-states final slower-states)
    (if (member nil slower-states)		; i.e. some slower state was inconsistent
	(prune-inconsistent-state final "slower state with no completions"))
    (mapc #'tsa-simulation slower-states)
    t))

;---------------------------------------------------------------------------------------------
;  Function:  (initial-values-for-new-state  from-state  to-state)
;
;  Given:     from-state, the final state of one mechanism; and
;             to-state, the normal state of a faster or slower mechanism to be initialized.
;
;  Returns:   an alist of (param qval) where:
;                qval = (qmag std)  for independent parameters, and
;                qval = (qmag qdir) for history parameters.
;             
;             For independent parameters, qmag is taken from from-state if possible,
;             otherwise from the normal-values of to-state.
;             For history parameters, (qmag qdir) is taken from from-state if possible,
;             otherwise (qmag nil) is returned, with qmag taken from the normal-values
;             of to-state.
;
;  Notes:     This function is used to obtain initial values for a mechanism at one
;             time scale from the final values of a mechanism at another time scale.
;             This function is used for both fast-to-slow and slow-to-fast time jumps.
;             This function gets values only for the "independent" and "history" parameters
;             of the to-state (since, by definition, the history and independent parameters
;             taken together should provide enough information for local constraint
;             propagation to determine the values for the rest of the dependent parameters).
;---------------------------------------------------------------------------------------------

; Assemble the initial values to be provided for a new state.
;   - Get all variables shared with faster mechanisms.
;   - make sure all independent variables are std.
;   - any independent variable not specified gets its normal value.
;   - any history variable not specified gets its normal value.  (?)
; Revised to fix problem:
;  - don't take all shared variables;
;  - iterate down history variables, and look only for them.
;     (unless they are constant in fast mechanism; then default to normal value)
;  - take constants from faster mechanisms if available; else default to normal.

(defparameter *trace-tsa-initial-values* t)

(defun initial-values-for-new-state (from-state to-state)
  (let ((normal-values (state-qvalues (get-normal-state (state-qde to-state))))
	(alist nil))
    (mapc #'(lambda (param)
	      (push
		(list param
		      (list (or (search-for-parameter-value param to-state (list from-state))
				(qmag (alookup param normal-values)))
			    'std))
	      alist)
	      (if *trace-tsa-initial-values*
		  (format *QSIM-Trace* "~%Pulling initial constant value ~a to state ~a." (car alist) to-state)))
	  (qde-independent (state-qde to-state)))
    (mapc #'(lambda (param)
	      (push
		(list param
		      (list (or (search-for-parameter-value param to-state (list from-state))
				(qmag (alookup param normal-values)))
			    nil))
	      alist)
	      (if *trace-tsa-initial-values*
		  (format *QSIM-Trace* "~%Pulling initial history value ~a to state ~a." (car alist) to-state)))
	  (qde-history (state-qde to-state)))
    (mapc #'(lambda (constraint-entry)
	      (mapc #'(lambda (param)
			(cond ((assoc param alist))
			      ((push (list param
					   (list (search-for-parameter-value param to-state
									     (list from-state))
						 nil))
				     alist)
			       (if *trace-tsa-initial-values*
				   (format *QSIM-Trace* "~%Pushing initial value ~a to state ~a from ~a."
					   (car alist) to-state from-state)))))
		    (cdr (car constraint-entry))))   ;;;???? constraint-varnames ?
	  (get-abstracted-constraints from-state))
    alist))



;---------------------------------------------------------------------------------------------
;  Function:  (search-for-parameter-value  param  to-state  states)
;
;  Given:     param:     a parameter name;
;             to-state:  the destination state defining the qspace which the returned qmag
;                        must be meaningful in; and
;             states:    a list of states (of increasingly faster mechanisms) from which
;                        a value for param will be found.
;
;  Returns:   a qmag of param.
;
;  Notes:     This function is used in the process of initializing a slower mechanism
;             from a faster one.  The purpose of this function is to retrieve the most
;             appropriate value for "param" that can be found in "states".  Here is how
;             it works:
;
;             1.  Normally, get the parameter's qmag from the latest state of the next
;                 faster mechanism (and then convert that to the corresponding qmag in
;                 the qspace of the slower mechanism.
;
;             2.  If the parameter does not exist in the next faster mechanism,
;                 then look in the latest states of still faster mechanisms.
;
;             3.  Rule 1 is skipped if the faster mechanism is the same as the slower
;                 mechanism AND the parameter is a dependent parameter.  Thus, the slow
;                 mechanism can take a value from a previous run of itself only if that
;                 param is an independent parameter.
;                 (DLD: personally, I don't see how rule 3 can ever get invoked since
;                  the search is always towards faster and faster mechanisms).
;---------------------------------------------------------------------------------------------

(defun search-for-parameter-value (param to-state states)	; => (param qval) or nil
  (let ((value nil))
    (cond ((null states) nil)
	  ((and (not (and (eql (state-qde to-state) (state-qde (car states)))
			  (not (member param (qde-independent (state-qde to-state))))))
		(setq value (alookup param (state-qvalues (car states)))))
	   (corresponding-description-of-value
	     (qmag value) param (car states) to-state))
	  (t (search-for-parameter-value
	       param	to-state
	       (append (cdr states)
		       (mapcar #'(lambda (name) (get-latest-state (eval name)))
			       (get-faster-mechanisms (car states)))))))))

; UGH!  A destination mechanism can take a value from a previous run of itself
; only if that value is an independent value.
;   Improve by keeping a global set of assumptions.

;---------------------------------------------------------------------------------------------
;  Note:      To set up the initial state of any mechanism, get the most recently set
;             state of that mechanism.  For this purpose, *latest-states* is an alist
;             of each mechanism and its most recent state.
;
;  Function:  (put-latest-state qde state)
;             updates *latest-states* to save 'state' as the latest state of 'qde'.
;
;  Function:  (get-latest-state qde)
;             returns the latest state for 'qde' that was saved in *latest-states*.
;             If no state has ever been saved for 'qde', then return the QDE's normal-state.
;---------------------------------------------------------------------------------------------

; To set up the initial state of anything, get the most recently set value,
; if one is available on a related mechanism.

(defparameter *latest-states* nil)		; alist of (qde state)

(defun put-latest-state (qde state)
  (let ((pair (assoc qde *latest-states*)))
    (cond ((null pair) (push (list qde state) *latest-states*))
	  (t (setf (cadr pair) state)))
    state))

(defun get-latest-state (qde)
  (let ((pair (assoc qde *latest-states*)))
    (cond ((null pair)
	   (setq pair (list qde (get-normal-state qde)))
	   (push pair *latest-states*)))
    (cadr pair)))

; LANDMARKS HAVE MEANINGS.

; The problem in using values of one mechanism to initialize another is
; that different physical landmarks in the quantity spaces represent the
; same "meanings" such as the "normal value" of a parameter.  We need to
; match meanings, not the physical symbols.
;    Current meanings are:  MINF, 0, INF, and NORMAL.

; landmark -> meaning.  (The quantity space is specified as param + state.)

(defun meaning-of-landmark (landmark param state)
  (cond ((member landmark (list *minf-lmark* *zero-lmark* *inf-lmark*)) landmark)
	((eql landmark (normal-qmag param state)) 'normal)
	(t nil)))

(defun normal-qmag (varname state)
  (let* ((normal-state (get-normal-state (state-qde state)))
	 (qval   (alookup varname (state-qvalues normal-state))))
    (qmag qval)))

; meaning -> landmark.

(defun landmark-with-meaning (meaning param state)
  (if (member meaning (list *minf-lmark* *zero-lmark* *inf-lmark*))
      meaning
      (find meaning (alookup param (state-qspaces state))
	    :key #'(lambda (lmark)
		     (meaning-of-landmark lmark param state)))))



; Given a value, return the best description with the same meaning in the
; destination qspace.

(defun corresponding-description-of-value (qmag param from-state to-state)
  (let ((meaning nil))
    (cond ((interval-p qmag)
	   (list (search-down-for-meaning (car qmag) param from-state to-state)
		 (search-up-for-meaning  (cadr qmag) param from-state to-state)))
	  ((and (setq meaning (meaning-of-landmark qmag param from-state))
		(landmark-with-meaning meaning param to-state)))
	  (t (list (search-down-for-meaning qmag param from-state to-state)
		   (search-up-for-meaning   qmag param from-state to-state))))))

(defun search-down-for-meaning (landmark param from-state to-state)
  (let ((meaning (meaning-of-landmark landmark param from-state))
	(pred nil))
    (cond ((and meaning
		(landmark-with-meaning meaning param to-state)))
	  ((setq pred (pred landmark (cdr (assoc param (state-qspaces from-state)))))
	   (search-down-for-meaning pred param from-state to-state))
	  (t (error "No corresponding meanings for ~a=~a, from ~a to ~a."
		     param landmark (state-name from-state) (state-name to-state))))))

(defun search-up-for-meaning (landmark param from-state to-state)
  (let ((meaning (meaning-of-landmark landmark param from-state))
	(succ nil))
    (cond ((and meaning
		(landmark-with-meaning meaning param to-state)))
	  ((setq succ (succ landmark (cdr (assoc param (state-qspaces from-state)))))
	   (search-up-for-meaning succ param from-state to-state))
	  (t (error "No corresponding meanings for ~a=~a, from ~a to ~a."
		     param landmark (state-name from-state) (state-name to-state))))))

; When activating a slower process, the problem is that the corresponding values
; of an abstracted constraint may have changed.  We do the following:
;  - eliminate existing corresponding values containing landmarks other than (minf 0 inf).
;  - map corresponding value tuples, via their meanings, from the fast mechanism
;    (which stores them on qde.other.abstracted-to) to the slower one.

(defun translate-cvts-when-possible (fast slow)	;states
  (let ((abstractions (get-abstracted-constraints fast))
	(constraints (state-constraints slow)))
    (if (and *trace-landmark-translation* (not (null abstractions)))
	(format *QSIM-Trace* "~%Translating landmarks (~a -> ~a) along shared constraints ~a."
		fast slow (mapcar #'car abstractions)))
    (dolist (con constraints)
      (when (assoc (constraint-name con) abstractions :test #'equal)
	(let ((source-cvts (get-or-record-abstracted-cvts con fast)))
	  (do ((cs source-cvts (cdr cs))
	       (ncvt nil)
	       (ncvts (mapcan #'(lambda (cvt) (unless (contains-created-landmark cvt)
						(list cvt)))
			      (cdr (assoc con (state-cvalues slow))))))
	      ((null cs) (setf (cdr (assoc con (state-cvalues slow))) ncvts))
	    (setq ncvt (translate-cvt-via-meaning (car cs) (constraint-varnames con) fast slow))
	    (or (null ncvt)
		(member ncvt ncvts :test #'equal)
		(setq ncvts (cons ncvt ncvts)))))))))

; Translate a corresponding-value tuple from one state to another via meaning.

(defun translate-cvt-via-meaning (cvt param-list from-state to-state)
  (let ((meaning-tuple
	  (mapcar #'(lambda (landmark param)
		      (meaning-of-landmark landmark param from-state))
		  cvt param-list))
	(target-tuple nil))
    (if *trace-landmark-translation*
	(format *QSIM-Trace* "~% Translating ~a: ~a -> ~a " param-list cvt meaning-tuple))
    (cond ((member nil meaning-tuple) nil)
	  ((setq target-tuple
		 (mapcar #'(lambda (meaning param)
			     (landmark-with-meaning meaning param to-state))
			 meaning-tuple param-list))
	   (if *trace-landmark-translation*
	       (format *QSIM-Trace* "-> ~a." target-tuple))
	   (cond ((not (member nil target-tuple))
		  target-tuple))))))

(defun contains-created-landmark (cv)
  (do ((L cv (cdr L)))
      ((null L) nil)
    (cond ((not (member (car L) (list *minf-lmark* *zero-lmark* *inf-lmark*)))
	   (return t)))))


; CORRESPONDING VALUES OF ABSTRACTED CONSTRAINT

; The assumptions behind an abstracted constraint are the values of the
; independent variables which are not included in the constraint.

; state.other:
;    abstracted-to:  alist of (constraint .
;                              alist of (assumptions corr-vals))

; The state.other.abstracted-to form is copied from qde.other.abstracted-to
; the first time access is attempted.  This prevents abstractions from different
; behaviors from interacting on their shared qde.  At that time, if there is a
; normal state, corresponding values from it will be added to the list, under
; the appropriate assumptions.

(defun get-abstracted-constraints (state)
  (let ((abs-cons (lookup-set 'abstracted-to (state-other state)))
	(from-qde nil)
	(normal (lookup 'normal-state (qde-other (state-qde state)))))
    (cond (abs-cons)
	  ((setq from-qde
		 (copy-tree (lookup-set 'abstracted-to
					(qde-other (state-qde state)))))
	   (push (cons 'abstracted-to from-qde)
		 (state-other state))
	   (if normal
	       (abstract-cvs-from-one-state-to-another normal state))
	   from-qde)
	  (t nil))))

; Determine current assumptions about abstracted constraints.
;  - the state is an equilibrium state of the mechanism, and
;  - the constraint is an abstracted constraint.
;  - returns an alist of the assumption variables and their values.


(defun get-current-assumption-values (state constraint-params)
  (let ((values (state-qvalues state)))
    (cond ((quiescent state)
	   (mapcar #'(lambda (param) (alookup param values))
		   (set-difference (qde-independent (state-qde state))
				   constraint-params)))
	  (t nil))))

; Look over the current quiescent state and extract the corresponding values
; of its abstracted constraints, if any.
;   This is now called from qsim-global-filters.

(defun check-for-abstracted-corresponding-values (state)
  (abstract-cvs-from-one-state-to-another state state)
  state)

(defun abstract-cvs-from-one-state-to-another (from-state to-state)
  (if (and *check-abstracted-constraints*
	   (quiescent from-state))
      (do ((L (get-abstracted-constraints to-state) (cdr L)))
	  ((null L))
	(map-abstracted-corresponding-values (car (car L)) from-state to-state)))
  to-state)

; Look at the parameters in the from-state, specified by the given constraint.
; Get or create the corresponding values for the current assumptions, storing
; them in the to-state.   If nothing is there, create from current values.
; (Distinguishing from-state and to-state is to get correspondences from the
; normal state and store them in t0.)

(defun get-or-record-abstracted-cvts (constraint state)
  (map-abstracted-corresponding-values constraint state state))

;; "constraint" is the name, i.e., '(M+ A B).
(defun map-abstracted-corresponding-values (constraint from-state to-state)
  (if (constraint-p constraint)
      ;; CONSTRAINT is a structure.
      (let ((current-cvt (get-landmark-tuple (constraint-variables constraint) from-state))
	    (current-assumptions (get-current-assumption-values from-state (constraint-varnames constraint)))
	    (abstracted-cvs (assoc (constraint-name constraint)
				   (get-abstracted-constraints to-state)
				   :test #'equal)))
	(do ((L (cdr abstracted-cvs) (cdr L)))
	    ((null L) (nconc abstracted-cvs
			     (list (list current-assumptions (list current-cvt))))
	     (list current-cvt))
	  (cond ((assumption-sets-match (caar L) current-assumptions)
		 (cond ((member current-cvt (cadr (car L)) :test #'equal))
		       (t (setf (cadr (car L)) (cons current-cvt (cadr (car L))))))
		 (return (cadr (car L)))))))

      ;; CONSTRAINT is a list of names.
      (let* ((vars (mapcar #'(lambda (p) (alookup p (qde-var-alist (state-qde from-state)))) (cdr constraint)))
	     (current-cvt (get-landmark-tuple vars from-state))
	     (current-assumptions (get-current-assumption-values from-state (cdr constraint)))
	     (abstracted-cvs (assoc constraint
				    (get-abstracted-constraints to-state)
				    :test #'equal)))
	(do ((L (cdr abstracted-cvs) (cdr L)))
	    ((null L) (nconc abstracted-cvs
			     (list (list current-assumptions (list current-cvt))))
	     (list current-cvt))
	  (cond ((assumption-sets-match (caar L) current-assumptions)
		 (cond ((member current-cvt (cadr (car L)) :test #'equal))
		       (t (setf (cadr (car L)) (cons current-cvt (cadr (car L))))))
		 (return (cadr (car L)))))))))


; Tests whether two alists consist of exactly the same bindings, in any order.
; This is just set-equal!
; Could probably use set-exclusive-or here!

(defun assumption-sets-match (A B)
  (cond ((and (null A) (null B)) t)
	((or  (null A) (null B)) nil)
	(t (let ((B-qval (find (variable-name (qval-variable (car A))) B
			       :key #'(lambda (qval) (variable-name (qval-variable qval))))))
	     (if (qval-equal (car A) B-qval)
		 (let ((B-list (remove B-qval B)))
		   (assumption-sets-match (cdr A) B-list))
		 nil)))))


; Check against all constraints.  Filter out a state that violates it.

(defun check-abstracted-constraints (state)
  (cond ((not *check-abstracted-constraints*) state)
	((not (quiescent state)) state)
	(t (do ((L (get-abstracted-constraints state)
		   (cdr L)))
	       ((null L) state)
	     (cond ((not (test-against-abstracted-constraint (car (car L)) state))
		    (prune-inconsistent-state state "violates abstracted constraint")
		    (return nil)))))))

; Construct the appropriate form of the abstracted constraint, and test using
; the standard machinery (from qsim-constraints).

(defun test-against-abstracted-constraint (constraint state)
  (let* (;(qspaces (state-qspaces state))
	 (values (state-qvalues state))
	 (current-values (mapcar #'(lambda (param) (cdr (assoc param values)))
				 (cdr constraint)))
	 (corr-vals (get-or-record-abstracted-cvts constraint state))
	            ; this returns the currently valid ones
	 (newcon (make-constraint :name constraint
				  :type (contype-from-name (car constraint))
				  :variables (mapcar #'(lambda (varname)
							 (cdr (assoc varname
								     (qde-var-alist (state-qde state)))))
						     (cdr constraint))
				  :-cvals corr-vals)))
    (check-qsim-constraint current-values newcon)))

; Useful functions for manipulating states and QDEs.

;---------------------------------------------------------------------------------------------
;  Function:  (get-normal-state  qde)
;
;  Returns:   the "normal" state of 'qde'.
;
;  Design:    If something is stored in the QDE's normal-state, that is returned.
;             Otherwise, we must generate and store the normal-state before returning it.
;             So, if the QDE contains define-normal, then that supplies initial values for
;             normal-state.  If there is no define-normal, then assume that the
;             independent and history parameters are all ((0 inf) std).  Yes, it's a
;             dubious default -- model-builders should use define-normal!
;---------------------------------------------------------------------------------------------
; Retrieve a stored "normal" state, making one if it's not there.
;   BUG:  assumes that all initialization parameters are ((0 inf) std).
;  Now has capability of explicitly storing the definition of "normal"

(defun get-normal-state (qde)
  (let ((normal-pair (assoc 'normal-state (qde-other qde)))
	(inits nil))
    (cond ((null normal-pair)
	   (setq normal-pair (list 'normal-state nil))
	   (push normal-pair (qde-other qde))))
    (cond ((null (cadr normal-pair))
	   (cond ((setq inits (copy-tree (lookup 'define-normal (qde-other qde)))))
		 (t (dolist (var (qde-variables qde) inits)
		      (if (or (variable-independent-p var)
			      (variable-history-p var))
			  (push `(,(variable-name var) ((0 inf) std)) inits)))))
	   (setf (cadr normal-pair)
		 (make-initial-state qde inits :text "Normal state" :sim *current-sim*)))) ; added keywords DJC 7/21/91
    (cadr normal-pair)))


;---------------------------------------------------------------------------------------------
;  Function:  (make-values-constant  param-list  state)
;
;  Returns:   nil.  This function is executed for its side effect, which is to set
;             qdir = 'std' for each of the "param-list" parameters in "state".
;---------------------------------------------------------------------------------------------

(defun make-values-constant (param-list state)
  (dolist (param param-list)
    (let ((qval (cdr (assoc param (state-qvalues state)))))
      (if (null (qmag qval))
	  (error "No value for ~a in ~a." qval state)
	  (setf (qdir qval) 'std)))))


