;;; -*- Mode:Common-Lisp; Package:Qsim; Syntax:Common-Lisp; Base:10 -*-
;;;  $Id: syntax.lisp,v 1.6 1992/05/05 19:08:40 bert Exp $

;;; Just a comment to show that Mallory edited on 91/03/26 thru RCS.  RSM

(in-package 'Qsim)


;;; This file holds functions for CHECKING THE SYNTAX OF THE DEFINE-QDE FORM.
;;; These functions are executed at the time that the DEFINE-QDE form is
;;; executed.  The syntax of constraints are not checked at this time - they
;;; are checked during INITIALIZE-CONSTRAINT-NETWORK.


(defparameter *OTHERS* nil
  "Collects names of new subslots in the OTHER slot.
   Used with the comparison utility to notice new OTHER slots which should,
   perhaps, have syntax checkers written.")


(defparameter *COMMON-TYPOS*
  '((ignore-qdir . ignore-qdirs))
  "A-list of commonly misspelled clause headings and the correct heading.")


(defun WRN (fmt-string &rest args)
  (declare (special qde-name syntax-warnings))
  (let ((*detailed-printing* nil))
     (unless syntax-warnings
       (setq syntax-warnings t)
       (format t "~%SYNTAX ERRORS (WARNINGS) in QDE ~A:~%" qde-name))
     (apply #'format t (concatenate 'string "~&WARNING: " fmt-string "~&") args)))


;;; CHECK-SYNTAX is the main entry function for syntax checking. 

(defun CHECK-SYNTAX (qde clauses)
  "Main function for checking the syntax of QDEs."
  (let ((qde-name (qde-name qde))
	(syntax-warnings nil)
	(qde-dyn-others (dynamic-qde-others)))
    (declare (special qde-name syntax-warnings qde-dyn-others))   
    (if (check-quantity-space-syntax qde (alookup 'quantity-spaces clauses))
	(wrn "Error(s) in QUANTITY-SPACE(S) preclude checking other clauses.")
	(dolist	(clause clauses)
	  (check-clause clause clauses qde t)))
    (if syntax-warnings (format t "~%"))))


;;; Added a new argument to check-initial-ranges and a new syntax clause
;;; for the Initial-values slot used by numsim.  BKay 18Oct91
;;;
(defun CHECK-CLAUSE (clause clauses qde &optional top-level-p)
  "Check an individual top-level or Other clause in the QDE."
  (declare (special qde-dyn-others))
  (let* ((clause-head (if (consp clause) (first clause)))
	 (typo (assoc clause-head *COMMON-TYPOS*)))
    (cond
      ((null clause-head)
       (wrn "Malformed clause ~:[in OTHER clause ~;~]being ignored:~%  ~S"
	    top-level-p clause))
      (typo
       (wrn "You probably mean ~A instead of ~A.~%  ~A clause being ignored."
	    (cdr typo) clause-head clause-head))
      ((and top-level-p
	    (not (or (member clause-head *define-qde-clauses*)
		     (member clause-head qde-dyn-others)
		     (member clause-head *other-alist*))))
       (wrn "Unrecognized top-level clause being ignored:~%  ~S" clause))
      (t
       (case clause-head
	 (Quantity-Spaces nil)		; Already done
	 (Constraints nil)		; Done by Initialize-Constraint-Network
	 (Dependent (wrn "Obsolete clause being ignored:~%  ~A" clause))
	 ((Text Discrete-Variables Layout Print-Names
		Coeffs Sd3-Constraint Abstracted-From Faster Slower
		Normal-State Abstracted-To Qspace-Hierarchy
		Curvature-At-Steady Define-Normal)
	  nil)				; No syntax check for these clauses
	 (Other			(dolist (subclause (rest clause))
				  (check-clause subclause clauses qde)))
	 ((Independent History No-New-Landmarks Ignore-Qdirs Ignore-Qvals)
	  (variables-only qde clauses clause))
	 (Transitions		(check-transition-syntax qde clauses clause))
	 (Energy-Constraint	(check-energy-constraint-syntax qde clauses clause))
	 (M-Envelopes		(check-menvelope-syntax qde clauses clause))
	 (Initial-Ranges	(check-initial-ranges-syntax qde
							     clauses clause
							     'INITIAL-RANGES))
	 (Expected-Values	(check-initial-ranges-syntax qde
							     clauses clause
							     'EXPECTED-VALUES))
	 (Unreachable-Values	(check-unreachable-values-syntax qde clauses clause))
	 (t (unless top-level-p
	      (pushnew (car clause) *Others*))))))))



;;; Unlike the other syntax checkers, CHECK-QUANTITY-SPACE-SYNTAX is called
;;; for both side effect (printing warnings) and value.  It returns NIL iff it
;;; finds no errors.  Otherwise it returns :ERROR because if the quantity
;;; space is malformed, all of the other syntax checkers will probably break.
;;; The syntax of the quantity space is thus checked first and if an error is
;;; found, the other syntax checking is skipped.
;;;   Modified RSM 16 Feb 91 to check all quantity spaces, not just until the
;;; first error.

(defun CHECK-QUANTITY-SPACE-SYNTAX (qde qclause)
  "Check the syntax of the Quantity Space clause
   and return :Error in case of errors."
  (declare (ignore qde))
  (loop
    with result = nil
    for (entry . tail) on qclause
    for okstructure = (and (consp entry)
			   (symbolp (car entry))
			   (consp (second entry))
			   (or (null (third entry))
			       (stringp (third entry)))
			   (null (cdddr entry)))
    for var = (if okstructure (car entry))
    do (cond ((not okstructure)
	      (wrn "Malformed QUANTITY-SPACES clause:~%  ~S" entry)
	      (setq result :error))
	     ((eql 'time var)
	      (wrn "The variable TIME is built-in.  Do not give it a Q-space.")
	      (setq result :error))
	     ((assoc var tail)
	      (wrn "The variable ~S has two QUANTITY-SPACES." var)
	      (setq result :error))
	     (t (loop
		  for lmark in (second entry)
		  unless (and lmark (or (symbolp lmark)
					(numberp lmark)))
		  do (wrn "Bad landmark ~S in QUANTITY-SPACE ~S." lmark entry)
		     (setq result :error))))
    finally (return result)))


(defun CHECK-MENVELOPE-SYNTAX (qde clauses subclause)
  "Check the syntax of M-Envelope clauses."
  (declare (ignore qde))
  (loop
    with constraintspec = (alookup 'constraints clauses)
    for menvclause in (cdr subclause)
    for okstructure   = (and (consp menvclause)
			     (every 'consp menvclause)
			     ;; There are 6 clause types and the constraint in
			     ;; an menvclause (including expected-function and
			     ;; expected-inverse)  BKay 18Oct91
			     (< (length menvclause) 8))
    if (not okstructure)
    do (wrn "Malformed M-Envelope clause:~%  ~S." menvclause)
    else if (not (find (car menvclause) constraintspec
		       :key 'car :test 'equal))
    do (wrn "~S does not name a constraint in the M-Envelopes clause:~%  ~S)"
	    (car menvclause) menvclause)
    else do (loop
	      for (envclause . tail) on (cdr menvclause)
	      with knowns = '(upper-envelope upper-inverse
			      lower-envelope lower-inverse
			      expected-function expected-inverse) ; BKay 18Oct91
	      unless (and (consp envclause)
			  (member (car envclause) knowns))
	      do (wrn "Unknown ~S clause:~%    ~S~%  in M-Envelope clause:~%    ~S"
		       (car envclause) envclause menvclause)
	      when (assoc (car envclause) tail)
	      do (wrn "Two ~S clauses in M-Envelope clause:~%  ~S"
		      (car envclause) menvclause))))


;;; Added a range-clause type argument which can be either INITIAL-RANGES
;;; (for the Q2 clause) or INITIAL-VALUES (for the numsim clause).
;;; BKay 18Oct91
;;;
(defun CHECK-INITIAL-RANGES-SYNTAX (qde clauses subclause range-clause-type)
  "Check that the syntax of initial-ranges clauses is
  (initial-ranges ((var lmark) (lbound ubound))+). or
  the syntax of expected-values is ((var lmark) val)"
  (declare (ignore qde))
  (loop
    with qspacespec = (alookup 'quantity-spaces clauses)
    for range in (cdr subclause)
    for okstructure = (and (consp range)
			   (consp (first range))
			   (if (eq range-clause-type 'initial-ranges)
			       (consp (second range))
			       (not (consp (second range))))
			   (null (cddr range)))
    for qspace = (and okstructure (assoc (caar range) qspacespec))
    do (cond ((not okstructure)
	      (wrn "Malformed ~S ~S~
		    ~%  Expected the form ((var lmark) (lbound ubound))."
		   range-clause-type range))
	     ((and (null qspace) (not (eq (caar range) 'time)))
	      (wrn "Unknown variable ~S in ~S ~S."
		   (caar range) range-clause-type range))
	     ((and (not (member (cadar range) (second qspace))) (not (eq (caar range) 'time)))
	      (wrn "Unknown landmark ~S for variable ~S in ~S ~S."
		   (cadar range) (caar range) range-clause-type range))
	     ((cddar range) (wrn "Garbage: ~S in ~S." (cddar range) range))
	     ;; Changed by BKay 3Sept91 so that initial range limits are
	     ;; evaluated before being checked for being numbers.
	     ;; This is done because the initial ranges are now evaled
	     ;; before being stored in the qde.initial-ranges slot.
	     ((or (and (eq range-clause-type 'expected-values)
		       (not (numberp (eval (cadr range)))))
		  (and (eq range-clause-type 'initial-ranges)
		       (let ((lo (eval (caadr range)))
			     (hi (eval (cadadr range))))
			 (or (not (numberp lo))
			     (not (numberp hi))
			     (> lo hi)
			     (cddadr range)))))
	      (wrn "Bad initial range numbers in ~S ~S."
		   range-clause-type range)))))


(defun VARIABLES-ONLY (qde clauses subclause)
  "Check that all of the VARIABLES listed in a subclause are known variables
   (they have their own qspace) and that they appear only once."
  (declare (ignore qde))
  (loop
    with qspacespec = (alookup 'quantity-spaces clauses)
    for (var . vartail) on (cdr subclause)
    if (consp var)
    do (wrn "Found a CONS where expected a variable name in ~S clause."
	    (car subclause))
    else if (not (or (assoc var qspacespec) (eq var 'time)))
    do (wrn "Unknown variable ~S in ~S clause." var (car subclause))
    else if (member var vartail)
    do (wrn "~S appears twice in ~S clause." var (car subclause))))


;;; Make sure that, for each TRANSITION, there are no gross malformations.
;;; Then make sure that each transition is of the form ([...] means optional)
;;;	(condition [->] function)   or
;;;	(testfn [->] function)
;;; where
;;;	condition is checked by Critique-Condition,
;;;	function is T, NIL, a keyword, a non-nil symbol, or a function, and
;;;	testfn is a non-nil symbol or a function.

(defun CHECK-TRANSITION-SYNTAX (qde clauses clause)
  (loop
    with qspacespec = (append (alookup 'discrete-variables clauses)
			      (alookup 'quantity-spaces clauses))
    for tran in (cdr clause)
    for arrowp = (and (consp tran) (eq (second tran) '->))
    for okstructure = (and (consp tran)
			   (null (if arrowp (cdddr tran) (cddr tran))))
    for function = (and okstructure (if arrowp (third tran) (second tran)))
    for testfn = (car tran)
    do (cond ((not okstructure)
	      (wrn "Malformed transition:~%    ~S~
		    ~%  Expected something of the form --~
		    ~%    (condition [->] function)  or  (testfn [->] function)."
		   tran))
	     ((not (or (functionp function)	; Lambda forms and symbols with existing function defns
		       (symbolp function)	; Nil and symbols (whose functions may be defined later)
		       (keywordp function)	; Keywords are labeled halt transitions.
		       (eq function t)))
	      (wrn "~S is not a function, keyword, t, or nil in a TRANSITION:~%  ~S"
		   function tran))
	     ((or (functionp testfn)			; Accept functions and non-nil symbols
		  (and testfn (symbolp testfn))))	;  (whose functions may be defined later).
	     (t					; Testfn is really a condition
	      (critique-condition testfn qspacespec qde)))))


;;; Check that CONDITION is of the form shown in the flet statement.
;;; Check recursively if the condition begins with AND, OR, or NOT.
;;; Call Critique-Single-Condtion for a non-compound condition.

(defun CRITIQUE-CONDITION (condition qspacespec qde)
  (flet ((malrpt ()
	   (wrn "Malformed condition:~%    ~S~
		 ~%  Expected something of the form --~
		 ~%    (var (point-mag dir))  or~
		 ~%    (var (point-mag NIL))  or~
		 ~%    (NOT condition)  or~
		 ~%    (AND condition+)  or~
		 ~%    (OR condition+)."
		condition)
	   t))
    (cond ((not (consp condition))			; Who knows?  Functions and symbols were 
	   (malrpt))					;   tested for in Check-Transition-Syntax.
	  ((equal (car condition) 'NOT)			; Case (NOT condition)
	   (if (or (cddr condition) (null (cdr condition)))	; Other than one arg
	       (malrpt)
	       (critique-condition (second condition) qspacespec qde)))
	  ((member (car condition) '(AND OR))		; Case (AND|OR condition*)
	   (if (null (cdr condition))			; No args
	       (malrpt)
	       (loop for cond in (cdr condition)
		     do (critique-condition cond qspacespec qde))))
	  ((not (and (symbolp (car condition))		; Other possible malformations
		     (consp (cadr condition))))
	   (malrpt))
	  (t (critique-single-condtion condition qspacespec qde)))))


;;; Critique the syntax of a non-compound condtion.  NOT recursive.

(defun CRITIQUE-SINGLE-CONDTION (condition qspacespec qde)
  (declare (ignore qde))
  (let* ((var (car condition))
	 (qspace (lookup var qspacespec))
	 (mag (caadr condition))
	 (dir (cadadr condition)))
    (cond ((null qspace)				; Unknown variable
	   (wrn "Unknown variable ~S in condition ~S in a TRANSITION."
		var condition))
	  ((cond ((atom mag)
		  (not (member mag qspace)))		; Unknown landmark
		 ((consp mag)
		  (not (and (member (second mag)	; Bad interval
				    (cdr (member (first mag) qspace)))
			    (null (cddr mag)))))
		 (t))					; Junk in magnitude slot
	   (wrn "Unknown landmark ~S for variable ~S~%  in condition ~S in a TRANSITION."
		mag var condition))
	  ((not (member dir '(inc dec ign std nil)))	; Unknown direction
	   (wrn "Unknown direction ~S in condition ~S in a TRANSITION."
		dir condition)))))



(defun CHECK-UNREACHABLE-VALUES-SYNTAX (qde clauses subclause)
  "Check the Unreachable Values clause for the syntax --
   (unreachable-values (var lmark+)+)."
  (declare (ignore qde))
  (loop
    with qspacespec = (alookup 'quantity-spaces clauses)
    for exp in (cdr subclause)
    for okstructure = (and (consp exp) (cdr exp))
    for var = (if okstructure (car exp))
    for qspace = (if okstructure (lookup var qspacespec))
    do (cond ((not okstructure)
	      (wrn "Malformed clause ~S in UNREACHABLE-VALUES." exp))
	     ((not (symbolp var))
	      (wrn "Malformed UNREACHABLE-VALUES clause ~S.~
		    ~%  Found ~S where a variable name was expected.~
		    ~%  Expected (unreachable-values (var landmark+)+).~
		    ~%  This syntax changed in Nov '89.  See manual for explanation."
		   exp var))
	     ((null qspace)
	      (wrn "Unknown variable ~S in UNREACHABLE-VALUES." var))
	     (t (loop
		  for lmark in (cdr exp)
		  unless (member lmark qspace)
		  do (wrn "Unknown landmark ~S for variable ~S in UNREACHABLE-VALUES."
			  lmark var))))))


;;; Does not build the entry in the Qde-Other slot.

(defun CHECK-ENERGY-CONSTRAINT-SYNTAX (qde clauses clause)
  "Check the syntax of the Energy Constraint clause."
  (declare (ignore qde))
  (if (alookup 'energy-constraint (alookup 'other clauses))
      (wrn "Cannot have both a toplevel clause ~S~
	    ~%  and an OTHER clause ~S."
	   clause (lookup 'energy-constraint (lookup 'other clauses)))
      (loop
	with qspace = (alookup 'quantity-spaces clauses)
	for econstraint in (cdr clause)
	unless (consp econstraint)
	do (wrn "Energy constraint is of the wrong form; should be a list:~%  ~S"
		econstraint)
	else do (loop
		  for varname in (subseq econstraint 0 2)
		  unless (or (numberp varname)
			     (alookup varname qspace))
		  do (wrn "Unknown variable ~S in energy constraint ~S."
			  varname econstraint)))))


;;; Not called by Check-Syntax:

(defun CRITIQUE-QDE (qde)
  (let ((qde-name (qde-name qde))
	(syntax-warnings nil))
    (declare (special qde-name syntax-warnings))   
    (unless (= (length (qde-variables qde))
	       (length (qde-var-alist qde)))
      (wrn "~10T(length (qde-variables qde)) = ~D~@
	    ~10T(length (qde-var-alist qde)) = ~D"
	   (length (qde-variables qde)) (length (qde-var-alist qde))))
    (loop for var in (qde-variables qde)
	  unless (or (eq (variable-name var) 'time)
		     (some #'(lambda (constraint)
			       (member var (constraint-variables constraint)))
			   (qde-constraints qde))
		     (variable-discrete-p var))   ; added DJC 07/21/91
	  do (wrn "Variable ~S appears in no constraint.  CFILTER will make malformed tuples."
		  var))
    (loop for (con . tail) on (qde-constraints qde)
	  when (member (constraint-name con)
		       tail :test #'equal :key #'constraint-name)
	  do (wrn "The constraint ~S appears twice." con))
    (if syntax-warnings (format t "~%"))))

