;;; -*- Mode:Common-Lisp; Package:QSIM; Syntax:COMMON-LISP; Base:10 -*-
;;;  $Id: structures.lisp,v 1.31 1992/05/28 15:33:20 clancy Exp $


(in-package 'QSIM)

;(require 'dynamic-slots)
;(import  '(defother defslot undefslot))

;;;=============================================================================
;;;
;;;			Q S I M	   S T R U C T U R E S
;;;
;;;  This file contains definitions of the major structures of QSIM:
;;;  qde, variable, constraint, contype, qval, lmark, state, time, and mode.
;;;
;;;-----------------------------------------------------------------------------
;;;
;;;  Pointers:  The following "pointer diagram" shows which structures refer to
;;;		(point to) other structures.  In this diagram, "X ---> Y" or
;;;		"X ---> list of Ys" means that structure X has a slot that 
;;;		either points to an instance of structure Y or a list of
;;;		instances of Y.
;;;
;;;		       QDE ---> list of VARIABLEs 
;;;			   ---> list of CONSTRAINTs
;;;			   ---> alist of LMARKs (initial qspaces)
;;;			   ---> alist of LMARKs (initial cval tuples)
;;;
;;;		  VARIABLE ---> list of CONSTRAINTs
;;;			   ---> QVAL	   
;;;			   ---> list of QVALs (the possible values)
;;;			   ---> list of LMARKs (the qspace)
;;;
;;;		CONSTRAINT ---> CONTYPE (the type of constraint)
;;;			   ---> list of VARIABLEs (the constraint's arguments)
;;;			   ---> list of CONSTRAINTs (neighboring constraints)
;;;			   ---> list of lists of LMARKs (cval tuples)
;;;
;;;		     STATE ---> QDE   
;;;			   ---> alist of (varname . QVAL) [the state values]
;;;			   ---> alist of (varname . list of LMARKs) [the qspaces]
;;;			   ---> alist of (CONSTRAINT . list of LMARKs) [cvals]
;;;			   ---> list of STATEs (successor states)
;;;
;;;		      MODE ---> VARIABLE
;;;			   ---> alist of (qmag . list of CONSTRAINTs)
;;;
;;;		       SIM ---> QDE
;;;			   ---> STATE
;;;			   ---> control switch settings for current simulation
;;;			   ---> agenda, limits, etc., for current simulation
;;;
;;;-----------------------------------------------------------------------------
;;;
;;;  Working Storage Slots:
;;;	     As a convention, defstruct slot names that begin with "-", as in
;;;	     "variable--qval" and "constraint--cvals", refer to slots whose
;;;	     values may change during simulation.  These slots contain "working
;;;	     storage", and the appearance of the "--" in the name of the
;;;	     accessor function serves as a visual warning/reminder that this
;;;	     is changeable data (as opposed to the static data stored in the
;;;	     QDE structure, for instance).
;;;
;;;  Dynamic Slots:
;;;	     To facilitate private/experimental extensions to QSIM, a dynamic
;;;	     slots facility is included, which is best explained by example.
;;;	     Suppose you want to add a numeric range slot to the QVAL structure.
;;;	     Evaluating the top-level form:
;;;
;;;			    (defother qval range)
;;;
;;;	     creates a pseudo-slot named qval-range which can be accessed and
;;;	     SETF'ed just like any real slot.  In reality, the range pseudo-
;;;	     slot is in an alist stored in qval-other, and access to it is
;;;	     therefore slightly slower than for a real slot.  But the beauty of
;;;	     dynamic slots is that:
;;;	     -- you can add a pseudo-slot to a structure and you don't have
;;;		to recompile all the QSIM code that uses that structure;
;;;	     -- if, after suitable experimentation, it is decided to
;;;		incorporate your extension into standard QSIM, then your
;;;		pseudo slots can become real slots and your code that uses
;;;		those slots doesn't have to change.
;;;
;;;=============================================================================


;;;-----------------------------------------------------------------------------
;;;  Structure:  QDE
;;;
;;;  Purpose:	 This structure is the "anchor" for all information associated
;;;		 with a set of qualitative differential equations.  All of the
;;;		 information about the constraint network is contained in
;;;		 subordinate structures VARIABLE and CONSTRAINT.
;;;
;;;  Note:	 A variable can be recognized as being independent in either of
;;;		 two ways.  If you have the variable name, then you can test for
;;;		 membership in the list in the qde.independent slot.  If you
;;;		 have the <variable> instance, then check the truth value of
;;;		 the variable.independent-p slot.  Although redundant, both
;;;		 forms are useful at different times.
;;;-----------------------------------------------------------------------------


(defstruct (QDE (:print-function qde-printer))
  "QDE specification: qspaces, constraints, transitions, layout, etc."
  name			 ; name of this QDE
  (text nil)		 ; text describing this QDE
  (variables nil)	 ; list of <variable>s
  (constraints nil)	 ; list of <constraint>s
  (qspaces nil)		 ; initial qspaces, alist of (<varname> . <list of lmarks>)
  (cvalues nil)		 ; initial corresponding values
  (transitions nil)	 ; list of <transition>s
  (layout nil)		 ; list of lists of variable names
  (var-alist nil)	 ; alist of (<var name> . <variable>)
  (independent nil)	 ; list of names of independent variables (used in ACC)
  (history nil)		 ; list of names of history variables (used in TSA)
  (constraints-within-modes nil)  ; constraints inside of mode specifications
  (other nil)		 ; dynamic slots
  )


(defun QDE-PRINTER (qde stream ignore)
  (declare (special *detailed-printing*) (ignore ignore))   ;added DJC porting to Sun
  (if *detailed-printing*
      (format stream "#<QDE ~a: ~a>" (qde-name qde) (qde-text qde))
      (format stream "~a" (qde-name qde))))

;  Used for derivation and application of HOD constraints


;;;-----------------------------------------------------------------------------
;;;  Structure:  LMARK
;;;
;;;  Purpose:	 Every landmark of every quantity space is represented as an
;;;		 instance of this structure.  There are three things that point
;;;		 to instances of LMARK:
;;;		 1.  A quantity space (the qspace slot in VARIABLE) is a list
;;;		     of LMARKs.
;;;		 2.  A qualitative magnitude (the qmag slot in QVAL) is either
;;;		     an LMARK or a list of two LMARKs (i.e., an interval).
;;;		 3.  A corresponding value tuple (an element of the cvals list
;;;		     in CONSTRAINT) is a list of LMARKs.
;;;-----------------------------------------------------------------------------


(defstruct (LMARK (:print-function lmark-printer))
  "a landmark of a qspace."
  name			; name of the landmark, such as 'full
  (when-defined nil)	; <state>
  (where-defined nil)	; (<lmark> <lmark>)
  (why-defined nil)	; a string explaining why
  (other nil)		; dynamic slots
  ;;range		; numeric range of this landmark
  )

;(defother lmark range)	    ; for Q2 numeric ranges


(defun LMARK-PRINTER (lmark stream ignore)
  (declare (special *detailed-printing*) (ignore ignore))  ; added DJC porting to Sun
  (if *detailed-printing*
      (format stream "#<Lm ~(~a~)>" (lmark-name lmark))
      (format stream "~(~a~)" (lmark-name lmark))))


;;; LMARK-EQUAL may be extended later to compare ranges.

(defun LMARK-EQUAL (L1 L2)
  (eql (lmark-name L1) (lmark-name L2)))

;;; THis lmark-equal will return nil if something other than an
;;; lmark strucutre is passed to it.

(defun robust-lmark-equal (l1 l2)
  (and (lmark-p l1)
       (lmark-p l2)
       (lmark-equal l1 l2)))

;;;-----------------------------------------------------------------------------
;;;  Normally, unique lmarks are created for each landmark in a qspace.
;;;  However, the landmarks for 'minf, 0, and 'inf are special in that
;;;  they are created once (here) and shared among all qspaces.  Having
;;;  a single lmark for all occurences of 0, for instance, facilitates
;;;  testing for 0, as in check-derivative-constraint.
;;;-----------------------------------------------------------------------------

(defvar *ZERO-LMARK* (make-lmark :name 0 :why-defined "DEFVAR in Structures.lisp")
  "Landmark 0")

(defvar *MINF-LMARK* (make-lmark :name 'minf :why-defined "DEFVAR in Structures.lisp")
  "Landmark minf")

(defvar *INF-LMARK*  (make-lmark :name 'inf :why-defined "DEFVAR in Structures.lisp")
  "Landmark inf")


;;;-----------------------------------------------------------------------------
;;;  Note:	 The Constraint Network
;;;
;;;		 The VARIABLE and CONSTRAINT structures defined below are the
;;;		 building blocks of a QDE's "constraint network".  The define-qde
;;;		 macro takes as input a nice declarative representation of the
;;;		 set of qspaces and constraints, and outputs an interconnected
;;;		 set of structures that represent the constraint network in a
;;;		 form that is designed for efficient simulation.  The 'cfilter'
;;;		 function and its subordinates are the main beneficiaries.
;;;
;;;		 An instance of VARIABLE is created for each variable of the QDE,
;;;		 and an instance of CONSTRAINT is created for each constraint
;;;		 of the QDE.  These instances are created at define-QDE time
;;;		 and are interconnected to facilitate rapid access to any
;;;		 information needed during simulation.
;;;-----------------------------------------------------------------------------


;;;-----------------------------------------------------------------------------
;;;  Structure:  VARIABLE
;;;
;;;  Purpose:	 This structure has two main purposes:
;;;		 (1) to collect in one place all of the static (unchanging)
;;;		     information associated with a variable (things like name,
;;;		     title, whether or not this is an independent variable,
;;;		     what constraints it participates in, etc); and
;;;		 (2) to provide "working storage" during simulation (for things
;;;		     like the current qval, current qspace, etc).
;;;
;;;  Note:	 Slot names beginning with "-" are values that change during
;;;		 simulation, so use them with care!
;;;-----------------------------------------------------------------------------


(defstruct (VARIABLE (:print-function variable-printer))
  "a variable of a QDE"
  name			     ; print-name
  (title nil)		     ; long title for this variable
  (prefix nil)		     ; prefix name for created landmarks
  (-qval nil)		     ; <qval>, current qualitative value.
  (-qspace nil)		     ; (<lmark> <lmark> ...) current quantity space.
  (-pvals nil)		     ; (<qval> <qval> ...) list of possible values.
  (-npvals nil)		     ; number of pvals
  (-onpvals nil)	     ; old number of pvals
  (constraints nil)	     ; list of attached constraints
  (unreachable-values nil)   ; list of unreachable landmarks
  (flags 0)		     ; Flags: see below
  ; independent-p	     ; T if this is an independent variable
  ; history-p		     ; T if this is a history variable
  ; ignore-qdir-p	     ; T if should ignore this qdir
  ; no-new-landmarks-p	     ; T if no new landmarks to be created
  ; discrete-p		     ; T if this is a discrete (not continuous) variable
  ; invisible-p		     ; T if this variable is to be "invisible"
;  Modes slot removed by BKay 10Sept91.  There is now a
;  constraint.mode-expressions slot that takes its place
;  (modes nil)		     ; alist of (qmag . list of CONSTRAINTs)
  (other nil)		     ; dynamic slots:
  ; domain-name		     ; name of domain quantity, such as 'voltage
  ; bond-name		     ; name of bond-graph quantity, such as 'pressure
  ; units		     ; units of measurement, such as meters/second.
  )


(defun VARIABLE-PRINTER (variable stream ignore)
  (declare (special *detailed-printing*) (ignore ignore))  ;added DJC porting to Sun
  (if *detailed-printing*
      (format stream "#<Var ~(~a~)>" (variable-name variable))
      (format stream "~a" (variable-name variable))))


(defconstant *INDEPENDENT*	(byte 1 0))	; bit 0 of flags
(defconstant *HISTORY*		(byte 1 1))	; bit 1 of flags
(defconstant *IGNORE-QDIR*	(byte 1 2))	; bit 2 of flags
(defconstant *NO-NEW-LANDMARKS* (byte 1 3))	; bit 3 of flags
(defconstant *DISCRETE*		(byte 1 4))	; bit 4 of flags
(defconstant *V-DONE*		(byte 1 5))	; bit 5 of flags
(defconstant *IGNORE-QVAL*	(byte 1 6))	; bit 6 of flags


;;; BKay 3Sept91
;;; Quoted first arg to ldb-test calls for use with allegro.
;;; This shouldn't be a problem with other common lisps, but it
;;; should be checked out.

(defmacro VARIABLE-INDEPENDENT-P (var)
  `(ldb-test ',*INDEPENDENT* (variable-flags ,var)))

(defmacro VARIABLE-HISTORY-P (var)
  `(ldb-test ',*HISTORY* (variable-flags ,var)))

(defmacro VARIABLE-IGNORE-QDIR-P (var)
  `(ldb-test ',*IGNORE-QDIR* (variable-flags ,var)))

(defmacro VARIABLE-NO-NEW-LANDMARKS-P (var)
  `(ldb-test ',*NO-NEW-LANDMARKS* (variable-flags ,var)))

(defmacro VARIABLE-DISCRETE-P (var)
  `(ldb-test ',*DISCRETE* (variable-flags ,var)))

(defmacro VARIABLE-DONE-P (var)
  `(ldb-test ',*V-DONE* (variable-flags ,var)))

(defmacro VARIABLE-IGNORE-QVAL-P (var)
  `(ldb-test ',*IGNORE-QVAL* (variable-flags ,var)))


;(defother variable domainname)  ; name of domain quantity, such as 'voltage
;(defother variable bondname)	 ; name of bond-graph quantity, such as 'pressure
;(defother variable units)	 ; units of measurement, such as meters/second.



;;;-----------------------------------------------------------------------------
;;;  Structure:  CONTYPE
;;;
;;;  Purpose:	 This structure holds the information common to a given type of
;;;		 constraint, such as the constraint-check-function.  In object-
;;;		 oriented programming terminology, a CONTYPE holds class
;;;		 information and a CONSTRAINT holds instance information.
;;;		 In effect, CONSTRAINT inherits the information in its CONTYPE.
;;;-----------------------------------------------------------------------------


(defstruct (CONTYPE (:print-function contype-printer))
  "definitions for one type of constraint"
  name				; name such as 'M+, 'ADD, etc.
  (nargs nil)			; maximum number of arguments.
  (propagator nil)		; propagator function.
  (checkfcn nil)		; constraint check function.
  (buildfcn 'build-constraint)	;  for building constraint from user supplied values
  (qmag-relation nil)		; +/0/- table of qmag relations.
  (qdir-relation nil)		; inc/std/dec table of qmag relations.
  (cvals-allowed-p nil)		; corresponding values allowed?: T or NIL.
  (bend-points-p nil)		; bend-points expected?: T or NIL.
  (implied-cvals nil)		; e.g. '((0 0 0)) for ADD.
  (disallowed-lmarks nil)	; list of lmarks disallowed for a cv-tuple, e.g. 'inf.
  (other nil)			; dynamic slots
  )


(defun CONTYPE-PRINTER (contype stream ignore)
  (declare (ignore ignore))  ;;  added DJC porting to the Sun
  (format stream "~a" (contype-name contype)))


;;;-----------------------------------------------------------------------------
;;;  Structure:  CONSTRAINT
;;;
;;;  Purpose:	 This structure has two main purposes:
;;;		 (1) to collect in one place all of the static (unchanging)
;;;		     information associated with a constraint (things like the
;;;		     type of constraint and its variables); and
;;;		 (2) to provide "working storage" during simulation (for things
;;;		     like value tuples and current corresponding values).
;;;
;;;  Note:	 Slot names beginning with "-" are values that change during
;;;		 simulation, so use them with care!
;;;-----------------------------------------------------------------------------


(defstruct (CONSTRAINT (:print-function constraint-printer))
  "a constraint of a QDE"
  name		     ; name, e.g. '(M+ LEVEL PRESSURE)
  type		     ; pointer to type of constraint
  (variables nil)    ; ordered list of variables of this constraint
  (neighbors nil)    ; list of neighboring constraints
  (bend-points nil)  ; only used by S+, S-, U+, U-
  (-cvals nil)	     ; list of corresponding values (changes during simulation)
  (-tuples nil)	     ; current list of tuples (used during prediction)
  (flags 0)	     ; (see below)
  ;; done	     ; temporary storage
  ;; active-p	     ; true if this constraint is active
  (other nil)	     ; dynamic slots
  ; Added a defother for constraint.mode-expressions that holds a list
  ; of mode expressions for a constraint.  BKay 10Sept91.  The defother call
  ; is in define-qde.lisp.
  )


(defun CONSTRAINT-PRINTER (con stream ignore)
  (declare (special *detailed-printing*) (ignore ignore))  ;; added DJC porting to the Sun
  (if *detailed-printing*
      (format stream "#<Con ~a>"
	      (constraint-name con))
      (format stream "~a" 
	      (constraint-name con))))


(defconstant *DONE*   (byte 1 0))	   ; bit 0 of flags
(defconstant *ACTIVE* (byte 1 1))	   ; bit 1 of flags

;;; ldb-test fist arg quoted for use with allegro BKay 3Sept91
(defmacro CONSTRAINT-DONE (con)
  `(ldb-test ',*DONE* (constraint-flags ,con)))

(defmacro CONSTRAINT-ACTIVE-P (con)
  `(ldb-test ',*ACTIVE* (constraint-flags ,con)))

(defmacro CONSTRAINT-VARNAMES (con)
  `(cdr (constraint-name ,con)))

(defmacro CONSTRAINT-CHECKFCN (con)		; pseudo-accessor
  `(contype-checkfcn (constraint-type ,con)))

(defmacro CONSTRAINT-TYPENAME (con)		; pseudo-accessor
  `(car (constraint-name ,con)))

(defmacro CONTYPE-ISA (name con)
  `(eql (contype-name (constraint-type ,con)) ,name))

(defun CONSTRAINT-PROPAGATOR (con)
  (unless (constraint-p con)
    (error "~%~a is not a constraint." con))
  (contype-propagator (constraint-type con)))


;;;-----------------------------------------------------------------------------
;;;  Structure:  QVAL
;;;
;;;  Purpose:	 This structure holds a qualitative value for a variable (and a
;;;		 numeric range, when applicable).  Normally, a QVAL instance is
;;;		 pointed to by the variable.qval slot or the variable.pvals slot
;;;		 or the state.qvalues slot.
;;;
;;;  Note:	 The qval.variable slot exists solely for efficiency reasons.
;;;		 During prediction, constraint checking, and global filtering,
;;;		 it often arises that while a QVAL is being processed, there is
;;;		 a need to check its qspace.  With the qval.variable slot, it is
;;;		 easy to access the variable.qspace slot.  This has made several
;;;		 sections of code simpler and faster (they used to do a lookup
;;;		 in the list of all qspaces).
;;;-----------------------------------------------------------------------------


(defstruct (QVAL (:print-function qval-printer))
  "qualitative value consisting of a qmag and qdir."
  (variable nil)     ; variable that this is a value for.
  (qmag nil)	     ; qualitative magnitude: <lmark> or (<lmark> <lmark>).
  (qdir nil)	     ; qualitative direction-of-change: inc, std, dec, or ign.
  (other nil)	     ; dynamic slots:
  ;;range	     ;	 numeric range of this qval
  )


;(defother qval range)


;;; Baffes, Mallory 10/17/90.  Updated to provide short qval printing.
;;; Parameters %up-arrow-char and %down-arrow-char are used by
;;; Qval-Printer only.

(defparameter %UP-ARROW-CHAR
   (string (or (name-char "Up-Arrow")	(name-char "UpArrow")	"^")))

(defparameter %DOWN-ARROW-CHAR
   (string (or (name-char "Down-Arrow") (name-char "DownArrow") "v")))


(defun QVAL-PRINTER (qval stream ignore)
  (declare (ignore ignore))  ;;  added DJC porting to the Sun
  "Print-function for structure qval."
  (let ((qmag (qval-qmag qval))
	(qdir (qval-qdir qval)))
    (cond (*detailed-printing*
	   (format stream "#<Qval ~(~A~): (~A ~A)>"
		   (variable-name (qval-variable qval)) qmag qdir))
	  (*short-qval-printing*
	   (format stream "~A,~A"
		   (cond ((null qmag) "?")
			 ((listp qmag) (substitute "?" nil qmag))
			 (t qmag))
		   (case qdir
		     (std "=")
		     (inc %up-arrow-char)
		     (dec %down-arrow-char)
		     (ign "*")
		     ((nil) "?")
		     (t (format nil "~(~A~)" qdir)))))
	  (t ; Ordinary printing
	   (format stream "(~A ~(~A~))" qmag qdir)))))


(defmacro QMAG (qval)		; (re)defined for convenience of existing code
  `(qval-qmag ,qval))

(defmacro QDIR (qval)		; (re)defined for convenience of existing code
  `(qval-qdir ,qval))


;;; Perhaps there should be a qval= and a qval-compatible,
;;; which could be added to as people add on features such as ranges.
;;; But then we would lose "soundness".  Oh well.

(defun QVAL-EQUAL (qv1 qv2)
  "True if two qvals have equal qmags and qdirs."
  (and (eql   (qval-qdir qv1) (qval-qdir qv2))
       (equal (qval-qmag qv1) (qval-qmag qv2))))


(defvar *UNSPECIFIED-VALUE* (list nil 'ign))


;;; QVAL-COMPATIBLE compares two qvals, one or both of which may be partially-
;;; specified.  If the two qvals do not conflict, then it returns true.

(defun QVAL-COMPATIBLE (qv1 qv2)
  "True if two qvals have compatible qmags and qdirs."
  (if (and (qval-p qv1) (qval-p qv2))
      ;; Both arguments are qvals.
      (and
	;; See if qdirs are compatible.
	(or (eql   (qval-qdir qv1) (qval-qdir qv2))
	    (member (qdir qv1) *unspecified-value*)
	    (member (qdir qv2) *unspecified-value*))
	;; See if qmags are compatible.
	(or (equal (qval-qmag qv1) (qval-qmag qv2))
	    (member (qmag qv1) *unspecified-value*)
	    (member (qmag qv2) *unspecified-value*)))
      ;; Error.
      (error "~a or ~a is not a qval!" qv1 qv2)))


(defmacro QMAG-POINT-P (qmag)
  `(atom ,qmag))

(defmacro QMAG-INTERVAL-P (qmag)
  `(listp ,qmag))


;;;-----------------------------------------------------------------------------
;;;  Structure:  STATE
;;;
;;;  Purpose:	 This structure is used for storing data that are unique to a
;;;		 single state of a simulation, such as the qualitative value of
;;;		 each variable, the qspace, and the corresponding values.
;;;		 Static (unchanging) data from the QDE are accessible via the
;;;		 state.qde slot.
;;;
;;;  Notes:	 1.  The "qvalues" alist is indexed by variable NAME rather
;;;		     than by <variable> to simplify comparisons between two
;;;		     states having different <qde>s.  Such 2-state processing
;;;		     is done in global filtering, in time-scale abstraction,
;;;		     and in qplot.  Note that the <variable> is available in
;;;		     (qval-variable (cdr qvalue)) and is used in much of the
;;;		     simulation code.
;;;		 2.  The three lists "state.qvalues", "state.qspaces", and 
;;;		     "qde.variables" are expected to be in the same order,
;;;		     i.e., the n'th element of each list must refer to the same
;;;		     variable.  This allows some functions to walk down the
;;;		     lists in parallel.
;;;		 3.  The "cvalues" alist must include a (<constraint>.<cvals>)
;;;		     pair for every constraint that can possibly have
;;;		     corresponding values, even if it does not have any in the
;;;		     initial state.  Thus, the only constraints that can be
;;;		     omitted from this alist are either the constraints that
;;;		     can never have cvals (such as DERIV) or constraints that
;;;		     initially don't have any cvals AND have at least one
;;;		     variable for which no new landmarks are to be created.
;;;-----------------------------------------------------------------------------


(defstruct (STATE (:print-function state-printer))
  "State specific data: time, status, qvals, qspaces, cvals, successors, etc."
  name		       ; atom (e.g. S-47)
  (qde nil)	       ; qde structure (has pointers to variables & constraints).
  (qvalues nil)	       ; alist of (<varname> . <qval>), one for each variable.
  (qspaces nil)	       ; alist of (<varname> . <list of lmarks>)
  (cvalues nil)	       ; alist of (<constraint> . <list of cvals>)
  (status nil)	       ; nil | incomplete | OK | quiescent | final-state |
		       ;    transition | cycle | inconsistent | self-intersection
  (justification nil)  ; (initialized-with <values>)  |
		       ; (one-of-several-completions-of <state>) |
		       ; (unique-successor-of <state>)  |
		       ; (one-of-several-successors-of <state> <states>)
		       ; (perturbation-of <state>)
		       ; (transition-from <state> <qde>)
  (successors nil)     ; (successors <states>) | (trajectory-intersection)
		       ; (completions <states>) |
		       ; (transition-identity <states>) |
		       ; (inconsistent) | (cycle-identity <state>) |
		       ; (partial-match) | (branches)  ;used by Q3
		       ; | (final-state) | (quiescent) | (tsa-id)
  (text	nil)	       ; string describing the state
  (time-label nil)     ; 't=inf  |  't<inf  |  nil
  (other nil)	       ; dynamic slots
  (misc nil)	       ; alist that is not copied when successors are created.
  )


;;;-----------------------------------------------------------------------------
;;; Other slots for structure STATE 
;;;	- PREDECESSORS stores the predecessors of the state,
;;;	- COARSENING is a pointer to a more abstract state,
;;;	- REFININGS is a list that the state is a refining of.
;;;
;;; Used in envisionment and aggregation of chatter and occurrence branching
;;; PF 08/30/90 
;;;-----------------------------------------------------------------------------

(defother state predecessors)

(defother state level)
(defother state coarsening)
(defother state refinings)


(defmacro STATE-TIME (state)
  "Returns QVAL of time in a given state."
  `(cdar (state-qvalues ,state)))

; BJK:  This should not have been named as it is, and it should not be a macro. (11-7-90)
 
(defun STATE-PREDECESSOR (state)
  "Returns predecessor of this state"
  (unless (member (car (state-justification state)) '(initialized-with copy-of))
     (cadr (state-justification state))))


(defun STATE-PRINTER (state stream ignore)
  (declare (special *detailed-printing*) (ignore ignore))  ;; added DJC porting to the Sun
  (if *detailed-printing*
      (format stream "#<State ~a>" (state-name state))
      (format stream "~a" (state-name state))))


(defun COPY-QVALUES (qvalues)
  (mapcar #'(lambda (qvalue)
	      (cons (car qvalue) (copy-qval (cdr qvalue))))
	  qvalues))


;; These functions used to be in qutils.lisp, but they used state-status
;; before it was defined, and lost.

(defun INCOMPLETE-P (state)
  (member 'incomplete (state-status state)))

(defun INCONSISTENT-P (state)
  (member 'inconsistent (state-status state)))

(defun QUIESCENT-P (state)
  (member 'quiescent (state-status state)))

(defun PERTURBED-P (state)
  (eq (first (filtered-successor-slot-contents state)) 'perturbation))

(defun CONTYPE-FROM-NAME (name)
  (or (find name *known-constraint-types* :key #'contype-name)
      (error "Invalid constraint name: ~a" name)))


;;;-----------------------------------------------------------------------------
;;;  A WORD ABOUT TIME:
;;;
;;;  "Time" is now represented as a variable, just like other variables of a 
;;;  QDE or STATE.  Thus, "time" has a qspace [initially (T0 inf)] and a QVAL
;;;  [initially (T0 inc)], and the name "time" can appear in the QDE, such as
;;;  in a constraint or in the `layout' clause or in the `other' clause.
;;;  Define-QDE creates the time variable and qspace automatically, and
;;;  the initial qval of T0 is set by the function new-state-from-qde.
;;;  
;;;  Internally, for both QDE and STATE, "time" occupies the first variable,
;;;  the first qspace, and the first qvalue --- and several functions depend
;;;  on this.  For some functions in QSIM, it is convenient to process the time
;;;  qvalue and/or time qspace just like for any other variable; for other
;;;  functions, such as no-change-filter, it is necessary to exclude the time
;;;  variable when processing the list of qvalues.  Since time is always the
;;;  first element of such lists, it is easy to exclude it from consideration
;;;  by saying "(cdr <list>)".
;;;
;;;  The following access functions help to hide the internal representation
;;;  details from the programmer.
;;;----------------------------------------------------------------------------

; time-interval-p is obsolete. Use qintervalp instead. DB 8/14/90.
(defmacro time-interval-p (time)
  `(listp (qmag ,time)))
(defmacro qintervalp (qval)
  `(listp (qmag ,qval)))

; time-point-p is obsolete. Use qpointp instead. DB 8/14/90.
(defmacro time-point-p (time)
  `(atom (qmag ,time)))
(defmacro qpointp (qval)
  `(atom (qmag ,qval)))


(defun TIME-VARIABLE (state-or-qde)
  (etypecase state-or-qde
    (QDE    (car (qde-variables state-or-qde)))
    (STATE  (car (qde-variables (state-qde state-or-qde))))))


(defun TIME-QSPACE (state-or-qde)
  (etypecase state-or-qde
    (QDE    (cdar (qde-qspaces state-or-qde)))
    (STATE  (cdar (state-qspaces state-or-qde)))))


(defmacro QSPACE (varname state)
  `(cdr (assoc ,varname (state-qspaces ,state))))

(defmacro QVAL (varname state)
  `(cdr (assoc ,varname (state-qvalues ,state))))

(defmacro VAR (varname state)
  `(cdr (assoc ,varname (qde-var-alist (state-qde ,state)))))

(defmacro BINDINGS (varname state)
  `(cdr (assoc ,varname (cadr (assoc '*bindings* (state-other ,state))))))


;;;-----------------------------------------------------------------------------
;;; (P-STATE-P <state>) is T if <state> is instantaneous.
;;; (I-STATE-P <state>) is T if <state> lasts over an interval of time.
;;;-----------------------------------------------------------------------------

(defun p-state-p (state)
  (qpointp (state-time state)))

(defun i-state-p (state)
  (qintervalp (state-time state)))



;;;-----------------------------------------------------------------------------
;;;  Structure:  SIM
;;;
;;;  Purpose:	 Simulation control:  (BJK:  10-9-90)
;;;		 All switches and status variables are stored in the SIM.  
;;;		 - Simulation should be controlled by specifying the SIM.
;;;		 - Default values for the SIM are specified in the
;;;		   control-variables menu.
;;;
;;;  To add a new control switch:
;;;	During debugging:
;;;	   Include the switch variable in your own code, defaulting to OFF.
;;;	   Include :other-variables '((my-switch t)) in the make-sim.
;;;	When the switch becomes permanent:
;;;	   Document it for the QMAN.
;;;	   Add it to the control menu, to set default value.
;;;	   Add a new slot to the (defstruct sim ... ) form.
;;;-----------------------------------------------------------------------------


(defstruct (SIM (:print-function sim-printer))
  (name (genname 'sim))
  qde 			; QDE for the initial state
  state			; initial state
  display-block
  (time-limit			   *time-limit*)
  (state-limit			   *state-limit*)
  (state-count 1)
  (agenda nil)		; saved to support resumption of simulation

  ;    A hash table of unique states.  The key is the symbolic version of
  ;    the qvalues [excluding time] returned by state-values.  Used in the Cross edge code.
  ;    added 03/15/92  DJC
  (xedge-state-table                (make-xedge-hash-table))
				   
  ;
  ; Semantic control of simulation:
  ;   Relations on variables
  ;
  (ignore-qdirs nil)		; set of variables
  (no-new-landmarks nil)	; set of variables
  (phase-planes nil)		; set of (var1 var2) pairs defining phase planes
  (unreachable-values nil)	; alist of (var lm1 lm2 ... )
  ;
  (cycle-detection		   *check-for-cycles*)	; nil | :weak | :strong
  ;
  ;   Booleans to control semantic filter checks.
  ;
  (analytic-functions-only	   *analytic-functions-only*)
  (HOD-constraints		   *perform-acc-analysis*)
  (SD3-constraints		   *auto-sd3-evaluation*)
  (Q2-constraints		   *check-quantitative-ranges*)
  (KET-constraint		   *check-energy-constraint*)
  (NIC-constraint		   *check-NIC*)
  (cross-edge-envisionment         *cross-edge-envisionment*) ; added DJC
  
  ;
  ;   Booleans to control landmark creation.
  ;
  (enable-landmark-creation	   *enable-landmark-creation*)
  (new-landmarks-at-initial-states *new-landmarks-at-initial-state*)
  (new-landmarks-at-final-states   *new-landmarks-on-transition*)
  (new-landmarks-across-constraints *new-landmarks-across-M-constraints*)
  (new-landmarks-always		   *always-create-landmarks*)
  ;
  ;   Semantic experiments, debugging tools, or bad ideas.
  ;
  (check-for-unreachable-values	   *check-for-unreachable-values*)
  (check-abstracted-constraints	   *check-abstracted-constraints*)
  (check-for-similarity		   *check-for-similarity*)
  (ask-about-multiple-completions  *ask-about-multiple-completions*)
  ;
  ; Procedural control of simulation
  ;
  (simulation-type                 *simulation-type*)
  (fire-transitions		   *fire-transitions*)
  ;
  (other-variables nil)		; alist of (var val) to be bound
  other)


(defun SIM-PRINTER (sim stream ignore)
  (declare (special *detailed-printing*) (ignore ignore))  ;; added DJC porting to the Sun
  (if (not (boundp (sim-name sim)))
      (set (sim-name sim) sim))
  (if *detailed-printing*
      (format stream "#<Sim ~a>" (sim-name sim))
      (format stream "~a" (sim-name sim))))

;;;  This will copy all of the values from a SIM except those that
;;;  depend upon the given simulation like the initial state.  DJC 07/15/91

(defun make-sim-from-sim (orig-sim)
  (if orig-sim
      (let ((sim (copy-sim orig-sim)))
	(setf (sim-name sim)               (genname 'sim)
	      (sim-qde sim)                nil
	      (sim-state sim)              nil
	      (sim-display-block sim)      nil
	      (sim-state-count sim)        1
	      (sim-agenda sim)             nil
	      (sim-xedge-state-table sim)  (make-xedge-hash-table)) ; added DJC 03/15/92
	sim)
      (error "A SIM of NIL was passed to make-sim-from-sim.")))

;;;-----------------------------------------------------------------------
;;;
;;;  Structure: PRINT-BLOCK
;;;
;;;  Purpose:   Maintains information about the simulation that is used
;;;             in the displaying of the behavior.  It is designed to be
;;;             analogous to the SIM.   The bindings within the SIM are
;;;             only valid during simulation.  This block will provide
;;;             information that is necessary for display.
;;;
;;;             Most of this information should be obtainable from the
;;;             behavior tree.  Maintaining it in this block is
;;;             decreases the need to search the tree for information.


(defstruct (display-block (:print-function display-block-printer))
  (name (genname 'disp))
  (initial-state nil)                ; the state that this block is attatched to
  (qdes-in-beh-tree nil)             ; a list of the qdes that are used in the tree
  (layout nil)                       ; a layout for the model
  other)


(defun display-block-printer (disp-block stream ignore)
  (declare (special *detailed-printing*) (ignore ignore))
  (if *detailed-printing*
      (format stream "~a  Initial State: ~a  QDES: ~a"
	      (display-block-name disp-block)
	      (display-block-initial-state disp-block)
	      (display-block-qdes-in-beh-tree disp-block))
      (format stream "~a" (display-block-name disp-block))))



(defun make-xedge-hash-table ()
  (make-hash-table :test #'equal))

#|
*****************************************************************************
	     D E F I N I N G   A B S T R A C T I O N   L E V E L S

Pierre Fouche <fouche@frutc51.bitnet>
Spring 1991

Aggregation algorithms (see "Abstracting Irrelevant Distinctions in Qualitative
Simulation" by Pierre Fouche and Benjamin Kuipers, Proceedings of QR'91,
Austin, TX, May 1991) build new states, which are clusters of already existing
states. Clusters provide a more abstract, less detailed view of a qualitative
simulation result. However, it is useful to keep the initial result on top
of which clusters have been defined. Thus, several levels of abstraction to
represent an envisionment graph have to be defined.

Everytime an aggregation algorithm is performed on an envisionment graph G0, a
new graph G1 is produced and a new level of abstraction is defined. A level is
represented as a structure:

(defstruct (level (:print-function level-printer))
  name		; name of the level, such as LEVEL-0
  description   ; a string such as:
		; "<level-name>(<position>)<reason>/<lower-level>"
		; where <position> is the level's position in the hierarchy of
		; levels, <reason> explains why the level was created, and
		; <lower-level> is the level on which the current level was
		; created.
  position	; see above
  states	; states belonging to the level
  lower-level	; pointer to the lower level
  upper-levels	; list of upper levels
  other)	; dynamic slot

A state can belong to several levels of abstractions, and the slot LEVELS
stores a list of all the levels a state belongs to.  States produced by QSIM
are in level *basic-level*.  Successors and predecessors of a state in level
*basic-level* are stored in SUCCESSORS and PREDECESSORS slots (they are set by
QSIM during simulation).  States in other levels are stored in slots NEW-
SUCCESSORS and NEW-PREDECESSORS.  These slots hold a-lists such as ((LEVEL-1
S-2 S-3) (LEVEL-2 S-4 S-5) (LEVEL-4 S-7)).  Keys are the levels of abstraction
and values list of successors or predecessors at the given level.  (Note:
NEW-SUCCESSORS and NEW- PREDECESSORS slots had to be defined not to change the
format of the existing SUCCESSORS and PREDECESSORS slots)

This allows one to define a tree of abstraction levels, as shown in the example 
below:
Algorithm A1 applied on level-0 -> level-1 defined
Algorithm A2 applied on level-0 -> level-2 defined
Algorithm A2 applied on level-1 -> level-3 defined
Algorithm A1 applied on level-2 -> level-4 defined
Algorithm A2 applied on level-3 -> level-5 defined
Algorithm A3 applied on level-0 -> level-6 defined


                  /-> 1 --> 3 --> 5
Resulting tree: 0 --> 2 --> 4
		  \-> 6


A state S in an upper level is a cluster of several states S1, S2, ...  S in a
lower level.  S is called a COARSENING of states S1, S2, ...  S.  Similarly,
S1, S2, ...  S are refinings of state S.  States in upper levels hold a list of
their refinings, in the slot REFININGS.  States in lower levels have a pointer
to their coarsenings, in the slot COARSENING.  As one state S can have
different coarsenings (if S's level has been refined twice or more),
COARSENING's value is an a-list such as ((1 .  S10) (2 .  S12)).

Those new slots are defined as dynamic slots in the file nq:nq;defother.lisp
Functions that handle them are in nq:nq;states.lisp
Aggregation algorithms are implemented in nq:nq;focusing-techniques.lisp
*****************************************************************************
|#

(defstruct (level (:print-function level-printer))
  name		; name of the level, such as LEVEL-0
  description   ; a string such as:
		; "<level-name>(<position>)<reason>/<lower-level>"
		; where <position> is the level's position in the hierarchy of
		; levels, <reason> explains why the level was created, and
		; <lower-level> is the level on which the current level was
		; created.
  position	; same as above
  states	; states belonging to the level
  lower-level	; pointer to the lower level
  upper-levels	; list of upper levels
  other)	; dynamic slot

(defun level-printer (level stream junk)
  (declare (ignore junk))
  (format stream "*~a*" (level-description level)))






;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;
;;;              NEW STRUCTURES USED FOR OCCURRENCE BRANCH ELIMINATION
;;;
;;;  A complete description of these structures is supplied in occ-branch-elimination.lisp.
;;;
;;;         EQUIVALENCE-NODE


(defstruct (equivalence-node (:print-function equivalence-node-printer))
  name
  (equiv-set nil)                 ; list of STATEs which are equivalent
  (equiv-lmarks nil)              ; alist of lmarks indexed by lmarks in the ABSTRACT-STATE
                                  ;  It contains the lmarks from other states which are 
                                  ;  equivalent to the index lmark.
  (abstract-state nil)            ; A state which is an anstraction of all of the
                                  ;   states in the equivalence set.
  (combined-from nil)             ; A list of the EQ nodes which were used to form this node
  (used? nil)                      ; True if this node is pointed to by an agregate structure
  (successor-node nil)            ; Points to an equivalence node which abstracts states
                                  ;   which are successors of the states in the equivalence-set.
  (terminal? nil)
  (histories nil)
  )




;;;
;;;      AGGREGATE-INTERVAL

(defstruct (aggregate-interval (:print-function aggregate-interval-printer))
  name
  (start-state nil)           ; the interval at which the aggregate state eminates
  (equiv-nodes nil)           ; an ordered set of equivalence nodes which define the
                              ;   state at which the aggregate ends.  It is ordered
                              ;   such that the set of states farthest from the start-state
                              ;   is first in the list
  (cur-level nil)             ;  a pointer into the equiv-nodes list identifying the current
                              ;   level of abstraction being displayed.
  )



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;
;;;                 PRINTERS FOR THE NEW STRUCTURES
;;;
;;;



(defun equivalence-node-printer (eq-node stream ignore)
  (declare (ignore ignore))
  (when *detail-print-eq-node*
    (format stream "~%EQUIVALENCE-NODE: ~a" (equivalence-node-name eq-node))
    (format stream "~%     Equivalent Set: ~a~%     Abstract-state: ~a~%     Equiv-lmarks: ~a"
	    (equivalence-node-equiv-set eq-node)
	    (equivalence-node-abstract-state eq-node)
	    (equivalence-node-equiv-lmarks eq-node))
    (setf *detail-print-eq-node* nil)
    (setf *detail-print-eq-node* t)
    (format stream "~%     Used?: ~a~%"                          
	    (if (equivalence-node-used? eq-node) t nil)))
  (unless *detail-print-eq-node*
    (format stream "~a" (equivalence-node-name eq-node))))
  



(defun aggregate-interval-printer (agg stream ignore)
  (declare (ignore ignore))
  (let (tmp)
    (when *detail-print-agg*
      (format stream "~%AGGREGATE-INTERVAL:  ~a~%~%    Start-state: ~VT~a"
	      (aggregate-interval-name agg)
	      25
	      (aggregate-interval-start-state agg))
      (setf tmp *detail-print-eq-node*)
      (setf *detail-print-eq-node* nil)
      (format stream "~%    Equiv-nodes/states:")
      (pr-equiv-nodes/states (aggregate-interval-equiv-nodes agg) 8)
      (format stream "~%    Cur-level: ~VT~a~%" 25 (car (aggregate-interval-cur-level agg))) 
      (setf *detail-print-eq-node* tmp))
    (unless *detail-print-agg*
      (format stream "~a" (aggregate-interval-name agg)))))


(defun pr-equiv-nodes/states (eq-nodes indent)
  (cond ((null eq-nodes) nil)
	(t (format t "~%~VTNode: ~a~VTStates: ~a" indent (equivalence-node-name (car eq-nodes))
		   (+ indent 12) (equivalence-node-equiv-set (car eq-nodes)))
	   (pr-equiv-nodes/states (cdr eq-nodes) indent))))


;;;  Returns true if state is an aggregate-interval node

(defmacro agg-interval-p (state)
  `(typep ,state 'aggregate-interval))



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Structures used by the dynamic envelope simulator Nsim.
;;; These structures are used by both nsim and the numeric plotting facilities.
;;; Added 25Sept91 by BKay.

;;; The nsim-struct hangs off of a state.
;;;
(defother state nsim-struct)

;;; Each numerical simulation is controled by an nsim-struct.  This
;;; structure contains the equations, and simulation info needed to
;;; run the numerical simulation.  The results are accumulated in 
;;; a matrix and pointers to each variable's results are placed in
;;; an nsim-struct.  Note that all references to equations, variables, etc.
;;; assume that the lists are broken up into lo - hi segments.
;;;
(defstruct (nsim-struct (:print-function nsim-struct-printer))
  "Data for a numerical simulaton"
  qde               ; qde that defines this struct
  state-vars        ; a list of var-names for the state params.
  other-vars        ; a list of other (non-state) variables to be computed.
  constant-vars     ; a list of the constant variables in the model.
  state-function    ; a compiled lisp function that computes the derivatives
  other-function    ; a compiled lisp function that computes the ovars
  exprTable         ; the expression table for the QDE.
  sVector           ; an array of [s1LB s1UB ... snLB snUB] for the svars.
                    ; This is initialized to the values at t0 (i.e., start)
  cVector           ; an array of [c1LB c1UB ... cmLB cmUB] for the cvars.
  start             ; simulation start time (default 0)
  stop              ; simulation stop time
  simStep           ; dT for simulation
  save-step         ; dT for display
  results           ; A result store for the values of all state and other vars
  (status NIL)      ; partial state in a step-by-step simulation
  )

(defun nsim-struct-printer (nsim-struct stream ignore)
  (declare (special *detailed-printing*) (ignore ignore))
  (if *detailed-printing*
      (format stream "#<NSIM ~a : state=~a other=~a>" 
	      (nsim-struct-qde nsim-struct) 
	      (nsim-struct-state-vars nsim-struct)
	      (nsim-struct-other-vars nsim-struct))
      (format stream "#<NSIM ~a>" (nsim-struct-qde nsim-struct))))


;;; The nsim-status structure keeps track of the current state of the nsim
;;; simulation.
;;;
(defstruct (nsim-status (:print-function nsim-status-printer))
  "Incremental nsim structure"
  nsim-struct     ; the nsim-struct that this is for
  num-svars       ; number of state vars 
  sVector         ; state vector [LB1 UB1 LB2 UB2 ...]
  oVector         ; state vector [LB1 UB1 LB2 UB2 ...]
  cVector         ; constant vector [LB1 UB1 LB2 UB2 ...]
  rk              ; Runge-Kutta structure for the state vector
  other-function  ; the function for computing the oVector
  time            ; how far we have simulated so far.
  other
  )

(defun nsim-status-printer (nsim-status stream ignore)
  (declare (ignore ignore))
  (format stream "#<NSIM-STATUS ~a" (nsim-status-nsim-struct nsim-status)))



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;; Structures used by the numerical Qde simulator numsim.
;;; These structures are used by both numsim and the numeric plotting
;;; facilities.
;;; Added 29Oct91 by BKay.

;;; There are two new slots defined for numsim :
;;;
(defother qde expected-values) ; This is just like initial-ranges but it maintains 
                               ; a constant for each var.
(defother state numsim-struct) ; This is where the numsim struct is attached to a 
                               ; a simulation.

;;; Also, expected-function and expected-inverse are new subclauses of 
;;; M-envelopes clause.  They are used to store an expected functional 
;;; value for each envelope.

;;; This structure records the simulation progress.  
;;;
(defstruct (numsim (:print-function numsim-printer))
  "Numerical simulation structure"
  (stateFctn  nil)  ; lisp function to compute sv'
  (otherFctn  nil)  ; lisp function to compute ov
  (sv         nil)  ; state vector
  (ov         nil)  ; other vector
  (cv         nil)  ; vector of constants
  (svars      nil)  ; state variable index (same order as sv)
  (ovars      nil)  ; other variable index (same order as ov)
  (cvars      nil)  ; constant variable index (same order as cv)
  (start      0.0)  ; sim start time
  (stop       10.0) ; sim stop time
  (simStep    0.1)  ; sim step    (only for fixedstep simulators)

  (method     #'rk45-run) ; the simulation method
  (fixedstep  nil)    ; If this is a fixedstep method, this holds
                      ; the value of the step.  Otherwise it is NIL.

  (time       0.0)  ; the time we have simulated to so far
  (results    NIL)  ; a list of the simulation so far
  (rk         NIL)  ; the rk-struct for the simulation.
  )

(defun numsim-printer (numsim stream ignore)
  (declare (special *detailed-printing*) (ignore ignore))
  (if *detailed-printing*
      (format stream "#<NUMSIM svars=~a ovars=~a time=~a"
	      (numsim-svars numsim) (numsim-ovars numsim) (numsim-time numsim))
      (format stream "#<NUMSIM time=~a>" (numsim-time numsim))))


;;; This structure simplifies repeated "get the next entry" calls to
;;; an NSIM result store.  This code is used by both NSIM and qgraph.
;;; Code that uses this should first make a call to position-rspointer
;;; and then make calls to retrieve-nsim-record-var.
;;;
(defstruct (rspointer (:print-function rspointer-printer))
  (type 0)           ; cvar, svar, ovar -- cvars are not stored in the array
  (results NIL)      ; the results array
  (recordIndex 0)    ; index for the current record
  (recordVarIndex 0) ; index to the LB for the var in the current record
  (recordLen 0)      ; length of a record
  (maxlen    0)      ; max size of the array
  )

(defun rspointer-printer (rspointer stream ignore)
  (declare (special *detailed-printing*) (ignore ignore))
  (format stream "#<RSPOINTER ~a: Index=~d  Maxlen=~d>"
	  (rspointer-type rspointer) (rspointer-recordindex rspointer)
	  (rspointer-maxlen rspointer)))


