;;; -*- Mode:Lisp; Package:QSIM; Syntax:COMMON-LISP; Base:10 -*-
;;;  $Id: states.lisp,v 1.53 1992/08/07 14:46:05 clancy Exp $
;;; Copyright (c) 1986 by Benjamin Kuipers.

(in-package :qsim)



;;;-----------------------------------------------------------------------------
;;;  WITH-BINDINGS-FROM-SIM is a macro that binds the globals from the sim.
;;;  It is intended to be wrapped around the relevant calls, in qsimulate, etc.
;;;      (Simplified, 12-6-90. -BJK)
;;;  Added *envisionment*, PF 03 Jun 1991
;;;-----------------------------------------------------------------------------


(defmacro with-bindings-from-sim (sim &rest body)
  #+symbolics (declare (zwei:indentation 1 1))

  `(let ((*current-sim*                    ,sim)
	 ;
	 (*check-for-cycles*               (sim-cycle-detection ,sim))
	;
	 (*current-display-block*          (sim-display-block ,sim))
	 (*analytic-functions-only*        (sim-analytic-functions-only ,sim))
	 (*perform-acc-analysis*           (sim-HOD-constraints ,sim))
	 (*auto-sd3-evaluation*            (sim-SD3-constraints ,sim))
	 (*check-quantitative-ranges*      (sim-Q2-constraints ,sim))
	 (*check-energy-constraint*        (sim-KET-constraint ,sim))
	 (*check-NIC*                      (sim-NIC-constraint ,sim))
	 ;
	 (*enable-landmark-creation*       (sim-enable-landmark-creation ,sim))
	 (*new-landmarks-at-initial-state* (sim-new-landmarks-at-initial-states ,sim))
	 (*new-landmarks-on-transition*    (sim-new-landmarks-at-final-states ,sim))
	 (*new-landmarks-across-M-constraints* (sim-new-landmarks-across-constraints ,sim))
	 (*always-create-landmarks*        (sim-new-landmarks-always ,sim))
	 ;
	 (*check-for-unreachable-values*   (sim-check-for-unreachable-values ,sim))
	 (*check-abstracted-constraints*   (sim-check-abstracted-constraints ,sim))
	 (*check-for-similarity*           (sim-check-for-similarity ,sim))
	 (*ask-about-multiple-completions* (sim-ask-about-multiple-completions ,sim))
	 ;
	 (*simulation-type*                (sim-simulation-type ,sim))
	 (*envisionment*		   (sim-envisionment-p ,sim))
	 (*fire-transitions*               (sim-fire-transitions ,sim))	; ??
	 (*xedge-state-table*              (sim-xedge-state-table ,sim)) ; added DJC 03/15/92 with xedge code
	 (*cross-edge-envisionment*        (sim-cross-edge-envisionment ,sim))
	 )
     (progv (mapcar #'car  (sim-other-variables ,sim))
	    (mapcar #'cadr (sim-other-variables ,sim))
       ,@body)))


;;;-----------------------------------------------------------------------------
;;; This function permits specialized copying of the other slot of
;;; a state.  Some slots do not want a full copy-tree, and so using this
;;; call rather than a copy-tree will copy just the right part of
;;; those structures.  It should be in place of copy-tree when
;;; copying the state.other slot.
;;; Inputs:  other-slots  - The contents of the state-other slot.
;;; Returns: a copy of other-slots for use in a new state.
;;; Note: This function is meant to behave like copy-tree.
;;; BKay 23May92
;;;   Modified to have an explicit list of dont-copy slots.
;;;   The default should probably be *not* to copy, but this preserves
;;;   compatibility.  BJK 8-5-92
;;;-----------------------------------------------------------------------------
;;;
(defvar *state-other-dont-copy*		; don't copy these other-slots
  '(predecessors stability-test abstracted-constraints
    corresponding-states corresponding-landmarks
    fast-completions slow-successors))

(defun copy-state-other (other-slots)
  (loop for slot in other-slots
	unless (member (car slot) *state-other-dont-copy*)
	collect (case (car slot)
		  (eqn-index (list 'eqn-index (copy-eqn-index (second slot))))
		  ;; Add other slot-specific copying functions here
		  (T         (copy-tree slot)) ; default = copy
		  )))


;;;-----------------------------------------------------------------------------
;;; Duplicate the Q2 eqn-index slot of a state.  Its format is :
;;; ( (var1 . ((lm1 . (eq1 eq2 ...))
;;;            (lm2 . (eq3 eq4 ...))
;;;            ...
;;;           ))
;;;   ...
;;;   (varn . ((lm3 . (eq5 eq6 ...))
;;;            ...
;;;           )) )
;;; Since we are uniqifying the eq entries, we don't want them copied.
;;; However, we do want to duplicate the higher level list structure.
;;; Inputs:  eqn-index - The equation index for the state.
;;; Returns: a copy of eqn-index, but without copying the equation
;;;          structures.
;;; BKay 23May92
;;;-----------------------------------------------------------------------------
;;;
(defun copy-eqn-index (eqn-index)
  (loop for var-entry in eqn-index
	collect (cons (car var-entry)
		      (loop for lm-entry in (cdr var-entry)
			    collect (cons (car lm-entry)
					  (copy-list (cdr lm-entry)))))))


;;;-----------------------------------------------------------------------------
;;;  MAKE-NEW-STATE creates an initial state from a QDE or previous state.
;;;  The new state gets its quantity spaces and corresponding values from the state or qde.
;;;  The creation of new initial states has been decomposed into two steps:
;;;  1.  create a partially-specified new state from given information,
;;;        keyword :assert-ranges <ranges> to add quantitative information (BK:10-2-89)
;;;        added :inherit and :perturb keywords (BJK: 10-1-90)
;;;  2.  complete that state description, if possible.
;;;-----------------------------------------------------------------------------


;;; DJC 07/15/91  Added the code which handles the SIM as an arguments and
;;; sets its slot as required.  
;;; BKay 27May92  Made all references to *current-qde* local.  The idea is
;;; that *current-qde* should not be used as a global variable, but only
;;; as a dynamically scoped local variable.

(defun make-new-state (&key (from-qde nil)	; specify either QDE or state
			    (from-state nil)
			    (inherit nil)	; variables whose values to inherit
			    (perturb nil)	; alist of (var sign) for perturbations.
			    (assert-values nil)	; assert qualitative values
			    (assert-ranges nil)	; assert quantitative values
			    (completions T)	; compute completions from partial state?
			    (text nil)		; comment text
			    (sim nil))          ; SIM structure to record information
  (reset-timers)
  (unless sim
    (cond (from-qde (setq sim (make-sim)))
	  (from-state (setq sim (make-sim-from-sim (state-sim from-state))))))
  (let ((nstate nil)
	(*current-sim* sim)
	(*current-qde* from-qde)
	qvalues)
    (declare (special *current-qde*))
    (with-bindings-from-sim sim
      (cond ((and from-qde (null from-state))
	     (unless (and (null inherit) (null perturb))
	       (error "Make-New-State :from-qde <qde> can't inherit or perturb values."))
;Removed	     (setq *current-qde* from-qde)
	     (initialize-network *current-qde*)
	     (setq qvalues (convert-user-values assert-values *current-qde*))
	     (setq nstate (new-state-from-qde from-qde qvalues text)))
	    ((and from-state (null from-qde))
	     (cond ((incomplete-p from-state)
		    (error "MAKE-NEW-STATE received an incomplete state ~a." from-state))
		   ((inconsistent-p from-state)
		    (error "MAKE-NEW-STATE received an inconsistent state ~a." from-state)))
	     (setq *current-qde* (state-qde from-state))
	     (initialize-network *current-qde*)
	     (setq qvalues (updated-values inherit perturb assert-values from-state))
	     (setq nstate (new-state-from-old-state from-state qvalues text)))
	    (t (error "Must specify exactly one:  QDE (~a) and state (~a)" from-qde from-state))))
      
    ;;  Initialize the rest of the SIM   DJC  07/16/91
    (setf (sim-qde sim) *current-qde*)
    (setf (sim-state sim) nstate)
    (initialize-SIM-from-QDE sim)
    (setf (sim-display-block sim)
	  (initialize-display-block nstate))
    
    (with-bindings-from-sim sim
      (progn			    
	(assert-ranges-to-state nstate assert-ranges)
	(dolist (init-fn *new-state-initializations*)	; This funcall loop adds the
	  (funcall init-fn nstate))                       ; possibility to change inits.
	(set-ignore-qdirs nstate (sim-ignore-qdirs sim))
	(when completions
	  (complete-states-from-partial-state nstate))
	nstate))))

;;;  MAKE-INITIAL-STATE and MAKE-MODIFIED-STATE are implemented in terms of MAKE-NEW-STATE
;;;  for compatibility.
;;;  The additional functions ADD-TEXT-KEYWORD, MAKE-INITIAL-STATE2,
;;;  and MAKE-MODIFIED-STATE2 were written 
;;;  because of problems with having both optional and keyword arguments.  This allows
;;;  the user to use text as either a keyword argument or an optional argument.
;;;   DJC  07/12/91

(defun make-initial-state (state-or-qde user-values &rest other-args)
  (apply #'make-initial-state2 state-or-qde user-values (add-text-keyword other-args)))

(defun make-initial-state2 (state-or-qde user-values &key (text nil) (sim nil))
  (etypecase state-or-qde
    (qde   (make-new-state :from-qde      state-or-qde
			   :assert-values user-values
			   :text          text
			   :sim           sim))
    (state (make-new-state :from-state    state-or-qde
			   :assert-values user-values
			   :text          text
			   :sim           sim))))

;;;  If the first word in the lamda-list is a string, then append the
;;;  keyword :text tot he beginning of the lamda-list.

(defun add-text-keyword (lambda-list)
  (if (stringp (car lambda-list))
      (cons :text lambda-list)
      lambda-list))

(defun make-modified-state (state-or-qde user-values &rest other-args)
  (apply #'make-modified-state2 state-or-qde user-values (add-text-keyword other-args)))

(defun make-modified-state2 (state user-values &key (text nil) (sim nil))
  (make-new-state :from-state    state
		  :assert-values user-values
		  :completions   t
		  :text          text
		  :sim           sim))

;;;-----------------------------------------------------------------------------
;;;  Combine the values from :assert-values, :inherit, and :perturb keywords
;;;  into a single list of values for make-new-state.   (BJK:  10-1-90)
;;;       :assert-values  =  alist of (varname (qmag qdir))
;;;       :inherit        =  set of varnames
;;;       :perturb        =  alist of (varname sign), where sign = + | -
;;;  These are intended to replace use of the qvalue=, qvalue+, qvalue- macros.
;;;-----------------------------------------------------------------------------

(defun updated-values (inherits perturbs asserts from-state)
  (convert-user-values
    (nconc (mapcar #'(lambda (varname)
		       (inherit-value-from-state varname from-state))
		   inherits)
	   (mapcar #'(lambda (pair)
		       (perturb-value-from-state (car pair) (cadr pair) from-state))
		   perturbs)
	   asserts)
    from-state))

(defun inherit-value-from-state (varname state)
  (let* ((qval   (or (qval varname state)
		     (error "Can't inherit ~a from ~a." varname state)))
	 (qmag   (qmag qval)))
    (list varname
	  (list (if (qmag-point-p qmag)
		    (lmark-name qmag)
		    (list (lmark-name (first qmag)) (lmark-name (second qmag))))
		nil))))

(defun perturb-value-from-state (varname sign state)
  (let* ((qspace (or (qspace varname state)
		     (error "Can't perturb ~a from ~a." varname state)))
	 (qval   (qval varname state))
	 (qmag   (qmag qval)))
    (list varname
	  (list (case sign
		  (+ (if (qmag-interval-p qmag)
			 (list (lmark-name (first qmag))
			       (lmark-name (second qmag)))
			 (list (lmark-name qmag)
			       (lmark-name (succ qmag qspace)))))
		  (- (if (qmag-interval-p qmag)
			 (list (lmark-name (first qmag))
			       (lmark-name (second qmag)))
			 (list (lmark-name (pred qmag qspace))
			       (lmark-name qmag))))
		  (0 (if (qmag-point-p qmag)
			 (lmark-name qmag)
			 (list (lmark-name (first qmag)) (lmark-name (second qmag)))))
		  (t (error "Perturbation (~a ~a) not legal." varname sign)))
		nil))))

;;;-----------------------------------------------------------------------------
;;; --af,dd

(defun new-state-from-qde (qde qvalues &optional (text nil))
  (let* ((nstate   (make-state :qde           qde
			       :qvalues       qvalues
			       :qspaces       (copy-alist (qde-qspaces qde))
			       :cvalues       (copy-alist (qde-cvalues qde))
			       :text          text
			       :justification `(initialized-with ,qvalues)
			       :name          (genname 'S)
			       )))
    (setf (state-sim nstate) *current-sim*)     ; added DJC 07/15/91
    (setf (state-time nstate) (make-qval :qmag (convert-qmag (initial-time) (time-qspace qde))
					 :qdir 'inc
					 :variable (time-variable qde)
					 ))
    (set (state-name nstate) nstate)
    nstate))


;;;-----------------------------------------------------------------------------
;;; --af,dd

(defun new-state-from-old-state (state qvalues &optional (text nil))
  (let* ((nstate 
	   (make-state :qde           (state-qde state)
		       :qvalues       qvalues
		       :qspaces       (copy-alist (state-qspaces state))
		       :cvalues       (copy-alist (state-cvalues state))
		       :other         (copy-state-other (state-other state)) ; BKay 26May92
		       :text          text
		       :justification `(initialized-with ,qvalues)
		       :name          (genname 'S)
		       )))

    (setf (state-sim nstate) (or *current-sim*       ; added DJC 07/15/91
				 (state-sim state))) ; added BJK 08/05/92
    ;; The statement to set time (below) is not actually necessary because
    ;; time is included in the qvalues list.
    ;; (setf (state-time nstate) (state-time state))
    (set (state-name nstate) nstate)
    nstate))

;;;-----------------------------------------------------------------------------
;;; --dd

(defun successor-state-from-predecessor (state qvalues)
  ;; No need to copy-alist qvalues here since caller is providing a fresh alist.
  (let* ((nstate
	  (make-state :qde       (state-qde state)
		      :qvalues   qvalues
		      :qspaces   (copy-alist (state-qspaces state))
		      :cvalues   (copy-alist (state-cvalues state))
		      :other     (copy-state-other (state-other state)) ; BKay 26May92
		      :name      (genname 'S)
		      )))
  
    (setf (state-sim nstate) *current-sim*)     ; added DJC 07/15/91
    (set (state-name nstate) nstate)    ;; Note: SET, not SETF
    nstate))


;   User-asserted quantitative ranges are alists of ((<varname> <lmark-name>) (lb ub))
;   They are stored in state.other.assert-ranges, but need to be transformed into
;   structures before being used.

(defun assert-ranges-to-state (state ranges)
  (if ranges
      (setf (state-assert-ranges state) ranges)))

(defun initialize-network (qde)
  ;; Install initial qspaces.
  (mapc #'(lambda (var qspace-pair)
	    (setf (variable--qspace var) (cdr qspace-pair)))
	(qde-variables qde)
	(qde-qspaces qde))
  ;; Install initial sets of corresponding values.
  (dolist (cvals-pair (qde-cvalues qde))
    (let ((con   (car cvals-pair))
	  (cvals (cdr cvals-pair)))
      (setf (constraint--cvals con) cvals))))


;;;-----------------------------------------------------------------------------
;;;  CONVERT-USER-VALUES takes as input a partial list of user-specified values
;;;  of the form: ((varname1 (qmag qdir)) (varname2 (qmag qdir)) ...)
;;;  and returns a full list of qvalues of the form:
;;;  ((varname1 . <qval1>) (varname2 . <qval2>) ...).
;;;  For those variables not included in the user-specified values, a <qval> of
;;;  (nil nil) is assumed.  The returned list is deliberately in the same order
;;;  as the list of variables in (qde-variables qde).
;;;-----------------------------------------------------------------------------

;;;  Dan Dvorak made a change to check that all variables specified are contained
;;;  in the model.  THis change was incorporated into the baseline 08/14/91  DJC

(defun convert-user-values (values context)
  (let (variables qspaces)
    (if (qde-p context)
	;; Context is a QDE.
	(setq variables (qde-variables context)
	      qspaces   (qde-qspaces context))
	;; Context is a STATE.
	(setq variables (qde-variables (state-qde context))
	      qspaces   (state-qspaces context)))

    ;; Verify that each varname in user values is in this qde/state.
    (dolist (item values)
      (let ((varname (car item)))
	(unless (member varname variables :key #'variable-name)
	  (format t "~%WARNING: ~a is not a variable in ~a"
		  varname (qde-name (if (qde-p context) context (state-qde context)))))))

    (mapcar #'(lambda (var)
		(let* ((varname (variable-name var))
		       (pair    (assoc varname values))
		       (qspace  (alookup varname qspaces)))
		  (if pair
		      (cons varname (convert-user-qval pair var qspace))
		      (cons varname (if (eql 'time varname)
					(make-qval :variable var
						   :qmag (convert-qmag (initial-time) qspace)
						   :qdir 'inc)
					(make-qval :variable var
						   :qmag nil
						   :qdir (if (variable-independent-p var)
							     'std
							     nil)))))))
	    variables)))



;;;-----------------------------------------------------------------------------
;;;  CONVERT-USER-QVAL converts a user-specified varname-and-value, such as
;;;  '(amount ((0 full) inc)), to a QVAL instance.
;;;-----------------------------------------------------------------------------

(defun convert-user-qval (varname-and-value var qspace)
  (let* (;(varname    (first  varname-and-value))
	 (user-value (second varname-and-value))
	 (user-qmag  (first  user-value))
	 (user-qdir  (second user-value)))
    (make-qval :variable var
	       :qmag (convert-user-qmag user-qmag qspace)
	       :qdir (cond (user-qdir)
			   ((variable-independent-p var) 'std)
			   ((variable-discrete-p var)    'std)
			   (t nil)))))


;;;-----------------------------------------------------------------------------
;;;  CONVERT-USER-QMAG converts a user-specified qualitative magnitude
;;;  into a either a landmark or a list of two landmarks.
;;;-----------------------------------------------------------------------------

(defun convert-user-qmag (uqmag qspace)
  (etypecase uqmag
    (null nil)  ;E.g. a simulation initialization might not specify the qmag but use nil instead.
    ((or symbol number lmark) (convert-qmag uqmag qspace))
    (list (list
	     (convert-qmag (first  uqmag) qspace)
	     (convert-qmag (second uqmag) qspace)))))
     
(defun convert-qmag (uqmag qspace)
  (if (lmark-p uqmag)
      (or (find (lmark-name uqmag) qspace :key #'lmark-name)
	  (error "Didn't find ~s in ~s" uqmag qspace))
      (case uqmag
	((NIL) NIL)  ; added DJC per AF's request 03/30/92
	(0     *zero-lmark*)
	(minf  *minf-lmark*)
	(inf   *inf-lmark*)
	(t     (cond ((find uqmag qspace :key #'lmark-name))
		     (t (error "~%  ~a not found in ~a" uqmag qspace)))))))



; State.status may include multiple attributes:
;   inconsistent   --  state is inconsistent; also (inconsistent <reason>).
;   complete       --  state is known to be complete.
;   incomplete     --  state.successors.completions points to completions
;   nil            --  status not yet determined
;   OK             --  state.successors points to its successors
;   GF             --  global filters have been applied
;   quiescent      --  all values std  (subcase of OK)
;   stable         --  quiescent state, diagnosed as stable equilibrium
;   unstable       --  quiescent state, diagnosed as unstable equilibrium
;   final-state    --  transition to infinity 
;   transition     --  transition identity detected
;   cycle          --  cycle detected; also (cycle <match-strength>)


; State.successors may contain:
;   nil                         --  no successors, or not yet determined.
;   (completions         . <states>)  --  all completions of incomplete state
;   (successors          . <states>)  --  all successors of complete state
;   (transition-identity . <states>)  --  states produced by transition
;   (cycle-identity      . <state>)   --  the matching cycle state
;   (tsa-id
;      (slower-states    . <states>)   --  states in slower mechanisms
;      (faster-states    . <states>))  --  states in faster mechanisms



; Allow an apparently initial state to be an incomplete state pointing to a
; list of completions.

;;; --af      modified 11/9/89 D.B.
;;;
;;;  Keyword argument added so that when called from DISPLAYS this function will not
;;;  complete a partial state.  DJC  08/14/91

(defun get-list-of-initial-states (state-or-list &key (complete-partial-state t))
  (when state-or-list               ;; added DJC 05/30/91 to handle error when transition state is incomplete
    (cond ((listp state-or-list)
	   (cerror "Return the list anyway? "
		   "Get-list-of-initial-states should not be called on lists anymore.")
	   state-or-list)
	  ((and (null (state-status state-or-list))
		complete-partial-state)
	   (complete-states-from-partial-state state-or-list)))
    (cond ((member 'complete (state-status state-or-list)) (list state-or-list))
	  ((member 'incomplete (state-status state-or-list))
	   (if (member (car (state-successors state-or-list))
		       '(completions))
	       ;;  Eliminated direct access of state.successor slot 02/14/91  DJC
	       (copy-list (successor-states state-or-list))
	       (error "Unknown successor type: ~a, for incomplete state ~a"
		      (car (state-successors state-or-list)) state-or-list)))
	  (t (list state-or-list)))))




;;; Check whether a state should go onto the agenda or not.

(defun filter-for-agenda (state)
  (let ((status (state-status state)))
    (cond ((null status) (list state))
	  ((intersection status '(inconsistent self-intersection cycle final-state))
	   nil)
	  ((intersection status '(incomplete transition OK))
	   (error "Agenda shouldn't see state ~a with status ~a."
		  state (state-status state)))
	  ((member 'quiescent status) (when *quiescent-ok-for-agenda*
					(list state)))
	  ((member 'complete status) (list state))
	  (t (error "State ~a has unknown status ~a."
		    state (state-status state))))))


(defvar *show-completion-steps* nil)
  

;;;-----------------------------------------------------------------------------
;;;  COMPLETE-STATES-FROM-PARTIAL-STATE takes a state having an incompletely
;;;  specified set of values, and completes those values.  If completion is
;;;  unique, returns the state.  If there are multiple completions, returns
;;;  the list of completions.
;;;-----------------------------------------------------------------------------

(defun complete-states-from-partial-state (nstate)
  (let* ((start-time        (get-internal-run-time))
	 (*current-qde*               (state-qde nstate))
	 (initial-values    (copy-qvalues (state-qvalues nstate)))
	 (current-state     nstate)
	 (*filtering-condition* :completions)
	 after-propagation
	 new-states
	 value-completions1 value-completions2)
    (declare (special current-state *current-qde*))		; make visible to cfilter, etc.

    (constraint-net-for-state nstate)		; Install the constraint network
    (setq after-propagation (propagate-if-possible nstate))
    
    ;; Compute possible values for each variable.
    (dolist (var (qde-variables *current-qde*))
      (setf (variable--pvals var)
	    (eliminate-unreachable-values var (all-qvalues (variable--qval var)))))
            ;; (all-qvalues (variable--qval var))

    (setq value-completions1 (cfilter *current-qde* #'check-qsim-constraint))
    (setq value-completions2 (filter-alists-at-init *current-qde* nstate value-completions1))

    ;;  NEW-STATES will be non-nil iff nstate had multiple completions.
    ;;  Install the completion(s) in Nstate or as its successors.
    (setf new-states
	  (install-value-completions nstate initial-values after-propagation value-completions2))

    ;; Possibly reduce the number of new states if two or more states are
    ;; similar (with respect to occurrence branching and "ignored" variables).
    (setq new-states (filter-for-similarity nstate new-states))

    (print-initialization-time-trace start-time new-states)    
    (if *show-completion-steps*
	(print-completions nstate initial-values after-propagation value-completions2))
    (cond ((member 'inconsistent (state-status nstate)) nil)
	  ((member 'incomplete   (state-status nstate)) new-states)
	  (t nstate))))

(defparameter *unreachables-early* nil
  "Eliminates unrechable values at an earlier point")

(defun eliminate-unreachable-values (var qvals)
   (if *unreachables-early*
       (delete-if #'(lambda (qval)
		      (member (qmag qval) (variable-unreachable-values var)))
		  qvals)
       qvals))



;;; CONSTRAINT-NET-FOR-STATE installs initial values in the constraint network and
;;; sets which constraints are active and inactive.

(defun constraint-net-for-state (nstate)
  (install-corresponding-values nstate)
  ;; Install qspaces and qvals in constraint network.
  (mapc #'(lambda (qspace-pair qvalue-pair)
	    (let* ((qval   (cdr qvalue-pair))
		   (var    (qval-variable qval))
		   (qspace (cdr qspace-pair)))
	      (setf (variable--qspace var) qspace
		    (variable--qval   var) qval)))
	(state-qspaces nstate)
	(state-qvalues nstate))
  ;; Activate/deactivate constraints within modes.
  (activate-moded-constraints (state-qde nstate)))


;;; Install corresponding values into constraint network.
(defun install-corresponding-values (nstate)
  (dolist (cvalue (state-cvalues nstate))
    (setf (constraint--cvals (car cvalue)) (cdr cvalue))))


;;; INSTALL-VALUE-COMPLETIONS and MAYBE-ASK-USER-ABOUT-MULTIPLE-COMPLETIONS
;;; changed by JMV 6 Feb 92, so that some completions can be continued
;;; while others are left in the dust
(defun install-value-completions
       (nstate initial-values after-propagation value-completions
	&aux (new-states nil))
  (cond ((null value-completions)
	 (prune-inconsistent-state nstate "no completions"))

	((singleton-p value-completions)
	 (fill-in-state-values nstate (car value-completions))
	 (push 'complete (state-status nstate)))

	(t (let ((new-val-comps
		  (maybe-ask-user-about-multiple-completions
		   nstate initial-values after-propagation value-completions)))
	     (setq new-states
		   (mapcar #'(lambda (alist)
			       (fill-in-state-values (new-state-from-old-state nstate alist)
						     alist
						     `(one-of-several-completions-of
						       ,nstate)))
			   new-val-comps))
	     (push 'incomplete (state-status nstate))
	     (mapc #'(lambda (s) (push 'complete (state-status s))) new-states)
	     (setf (state-successors nstate) (cons 'completions (copy-list new-states))))))
  new-states)

;;; JMV: This now returns the list of value-completions, possibly edited when
;;; the user wants to continue with a few of the completions
(defun maybe-ask-user-about-multiple-completions
       (nstate initial-values after-propagation value-completions)
  (cond (*ask-about-multiple-completions*
	 (format *QSIM-Trace* "~%The initial values are consistent with ~a complete states."
		 (length value-completions))
	 (if (y-or-n-p "Would you like to see the derivation? ")
	     (print-completions nstate initial-values
				after-propagation value-completions))
	 (if (y-or-n-p "Continue with all completions? ")
	     value-completions
	   (keep-some-completions value-completions)))
	(t value-completions)))



;;; This asks the use which ones s/he wants to keep, simply by number.
;;; Hopefully, they have looked at them already (or they know)
(defun keep-some-completions (value-completions)
  (loop for val-comp in value-completions
	for i from 1 to (length value-completions)
	when (y-or-n-p "Keep completion number ~a?" i)
	collect val-comp))


;(defun install-value-completions
;       (nstate initial-values after-propagation value-completions
;	&aux (new-states nil))
;  (cond ((null value-completions)
;	 (prune-inconsistent-state nstate "no completions"))
;
;	((singleton-p value-completions)
;	 (fill-in-state-values nstate (car value-completions))
;	 (push 'complete (state-status nstate)))
;
;	(t (maybe-ask-user-about-multiple-completions
;	     nstate initial-values after-propagation value-completions)
;	   (setq new-states
;		 (mapcar #'(lambda (alist)
;			     (fill-in-state-values (new-state-from-old-state nstate alist)
;						   alist
;						   `(one-of-several-completions-of
;						      ,nstate)))
;			 value-completions))
;	   ;; not needed anymore.
;	   ;;(mapc #'(lambda (s) (setf (state-time s) (state-time nstate))) new-states)
;	   (push 'incomplete (state-status nstate))
;	   (mapc #'(lambda (s) (push 'complete (state-status s))) new-states)
;	   (setf (state-successors nstate) (cons 'completions (copy-list new-states)))))
;  new-states)
;
;
;(defun maybe-ask-user-about-multiple-completions
;       (nstate initial-values after-propagation value-completions)
;  (when *ask-about-multiple-completions*
;    (format *QSIM-Trace* "~%The initial values are consistent with ~a complete states."
;	    (length value-completions))
;    (if (y-or-n-p "Would you like to see the derivation? ")
;	(print-completions nstate initial-values
;			   after-propagation value-completions))
;    (unless (y-or-n-p "Continue with all completions? ")
;      (error "Underdetermined initial state."))))

;;;--af
(defun propagate-if-possible (nstate)
  (if *propagate-first*
      (propagation nstate))
  (copy-tree (state-qvalues nstate)))       ;;;; !!!!!!! fix up after getting prop code.
  
;;;--af
(defun fill-in-state-values (nstate qvalues &optional (justification nil))
  (setf (state-qvalues nstate) qvalues)
  (when justification
      (setf (state-justification nstate) justification)
      (setf (state-predecessors nstate) (cdr justification))
      )
  (apply-global-state-analysis nstate)
  (create-new-landmarks-at-initial-state nstate)
  nstate)

;;;-----------------------------------------------------------------------------
;;;  UNREACHABLE-VALUES-FILTER takes a qvalues alist and checks to see if any
;;;  variable has reached an "unreachable value" (actually, an unreachable
;;;  landmark).  If so, NIL is returned, otherwise the qvalues are returned.
;;;  This is like a global alist filter, but it is only called by
;;;  FILTER-ALISTS-AT-INIT.
;;;-----------------------------------------------------------------------------
;;;--af, dd

(defun unreachable-values-filter (qde incstate qvalues)
  (declare (ignore qde incstate))
  (if (some #'(lambda (qvalue)
		(let* ((qval (cdr qvalue))
		       (var  (qval-variable qval)))
		  (member (qmag qval) (variable-unreachable-values var))))
	    qvalues)
      nil
      qvalues))

(defparameter *alists-at-init-filter*		; The introduction of *alists-at-init-filter*  makes
	      nil   ; changing unreachable values to a global state filter
	      ;;'(unreachable-values-filter)	; filter-alists-at-init more like APPLY-GLOBAL-ALIST-FILTERS
						; and APPLY-GLOBAL-STATE-FILTERS.
		"These functions are called on each alist when completing an incomplete state" )

;;; Although this loop only acts on a single function
;;; (UNREACHABLE-VALUES-FILTER) in the base version of QSIM, it allows
;;; hooks into the filtering the initial alists.

(defun filter-alists-at-init (qde ostate alists)	; => (list alist) | nil
  (do ((L *alists-at-init-filter*  (cdr L)))
      ((null L) alists)
    (setq alists
	  (delete-if-not #'(lambda (alist)
			     (funcall (car L) qde ostate alist))
			 alists))))

;;;;-----------------------------------------------------------------------------
;;;;  ELIMINATE-UNREACHABLE-VALUES is called in SUCCESSORS-OF-STATE to eliminate
;;;;  unreachable values at the earliest possible moment -- the moment when
;;;;  possible successor values are generated for each variable via
;;;;  P-successors or I-successors.
;;;;-----------------------------------------------------------------------------
;;;;--af  --dd
;
;(defun eliminate-unreachable-values (var qvals)
;  (let (unreachable-values)
;    (if (and *check-for-unreachable-values*
;	     (setq unreachable-values (variable-unreachable-values var)))
;	;; Return pruned list of qvals
;	(delete-if
;	  #'(lambda (qval)
;	      (and (point-p (qmag qval))
;		   (member (qmag qval) unreachable-values)))
;	  qvals)
;	;; Return unchanged list of qvals.
;	qvals)))

  
;;;-----------------------------------------------------------------------------
;;;  QSIM starts simulation from an initial state and continues until either
;;;  normal termination (due to quiescence, cycles, or boundary transitions)
;;;  or until the *state-limit* is reached.
;;;  -- Initial state can be a list of states.
;;;  -- Global filters are applied to the incoming state(s).
;;;  -- If state.status = incomplete, the state is replaced by its completions.
;;;-----------------------------------------------------------------------------

; QSIM2 takes a state and a sim, sets switches from the sim, and simulates.
; It returns the sim.
; Modified so that the SIM is obtained from the sim slot of the state passed as an
; argument.  I retained the ability to pass the SIM as an argument; however, a
; warning is issued if it is used.  In addition, a continueable error is
; initiated if the SIM passed is not the same as the one attatched to the
; state.  DJC  07/16/91

(defun qsim (state &optional (sim-arg nil))
  (let* ((sim (state-sim state))
	 (*current-sim* sim)
	 (*current-qde* (state-qde state)))  ; now a local def BKay 27May92
    (declare (special *current-qde*))
    
    ;;  Eventually the optional argument of sim-arg should be removed. When it is this
    ;;  warning can also be removed.  DJC  07/16/91
    (when sim-arg
      (report-sim-arg-warning sim sim-arg state)
      (setq sim sim-arg)
      (setq *current-sim* sim))

    (setf (sim-state sim) state)
    (setq *initial-state* state)
    (push *initial-state* *initial-state-stack*)    ;
    (check-control-variable-consistency sim)  ; added DJC 03/15/92
    ; Here I copy the code from (qsim state) in states.lisp.
    ;
    (let ((start-time  (get-internal-run-time)))
      (setq *envisionment* nil) 
      ; expanding (agenda-setup state sim-states) in place
      (reset-timers)
      ; modified so that it performs the same processing on these states as qsimulate-state
      ; DJC 05/25/92
      (when (sim-agenda sim)
	(error "The sim ~a already has states on the agenda." sim))
      (with-bindings-from-sim sim
	(let ((initial-states (get-list-of-initial-states state))
		(*trace-main-agenda* nil))
	  (activate-moded-constraints (state-qde state) state)
	  (dolist (state initial-states)
	    (apply-global-state-analysis state)
	    (apply-global-state-filters  state)
	    (pushnew 'GF (state-status state)))
	  (add-states-to-agenda (mapcan #'filter-for-transitions
					  initial-states)
				sim)))
      (trace-agenda-setup (sim-agenda sim))
      
					;
      (qsimulate sim)
      (final-agenda-tracing start-time))

    sim))

;;;  This function will issue a warning.  It is called from qsim when the
;;;  optional sim argument is included.  Furthermore, if the argument passed
;;;  is not equal to the SIM attatched to the initial state, then a
;;;  continueable error is issued.  If the user continues, then the
;;;  SIM which was passed is used.  Ceretain slots in this SIM are updated..
;;;  DJC  07/16/91

(defun report-sim-arg-warning (sim sim-arg state)
  (format *Qsim-Report* "~%QSIM2 called with a SIM as an argument.  A modification has been made and the SIM ")
  (format *Qsim-Report* "~%is obtained from the sim slot of the state argument.  Please remove this argument ")
  (format *Qsim-Report* "in the future. ~%The simulation is being continued.~%")
  (unless (equal sim sim-arg)
    (cerror "Use the SIM passed as an argument and modify the state to point to this SIM."
	    "The SIM passed as an argument (~a) is not equal to the SIM in the state-SIM slot of ~a." 
	    sim-arg state)
    (setf (state-sim state) sim-arg)
    (setf (sim-qde sim-arg) (state-qde state))
    (setf (sim-state sim-arg) state)
    (setf (sim-display-block sim-arg)
	  (initialize-display-block state))))


(defun check-control-variable-consistency (sim)
  "Issues warnings when there are possible conflicts in the control
   settings"
  (when (and (sim-cross-edge-envisionment sim)
	     (sim-enable-landmark-creation sim))
    (format *qsim-report* "~%~%WARNING:   When a cross edge envisionment is ~
             performed, ~%    *enable-landmark-creation* should be set to nil.~
             ~%    It has a value of T in the SIM. ~%")))

; This is now superceded by QSIM2, which wraps it with an SIM.
	    
;(defun qsim (initial-state)
;  (qsim2 initial-state)
;  initial-state)

(defun qsim2 (initial-state &optional (sim nil))
  (cerror "Call QSIM with initial-state and SIM." 
	  "QSIM2 has been eliminated.  Please change this call to QSIM.")
  (qsim initial-state sim))

; QSIM-COMPLETIONS takes a (possibly incomplete) state (as produced by MAKE-NEW-STATE)
; and returns the list of its completions, after filtering with the global filters.
; This requires a SIM, in order to handle unreachable-values.   (BJK:  10-24-90)
;   For use by QPC.

(defun qsim-completions (state &optional (sim (make-sim)))
  (let ((*current-qde* (state-qde state))) ; Now a local def BKay 27May92
    (declare (special *current-qde*))
    (setf (sim-qde sim) (state-qde state))
    (setf (sim-state sim) state)
    (setq *current-sim* sim)
    (setq *initial-state* state)
    (initialize-SIM-from-QDE sim)
    (mapcan #'apply-global-state-filters
	    ;; Used to call get-list-of-initial-states - JC 10/24/90.
	    ;; added listify, as complete... may return a state, not a list.  AF 10/29/90
	    (listify (complete-states-from-partial-state state)))))



; Initialize certain slots in the SIM as the union of what the user put into the SIM,
; and what is in QDE.other.  For compatibility.

(defun initialize-SIM-from-QDE (sim)
  (initialize-unreachable-values sim)
  (initialize-ign-qdirs sim)
  (initialize-NIC-phase-planes sim)
  (initialize-NNL-variables sim)
  ; intern the SIM name  DJC  08/23/91
  (set (sim-name sim) sim))

; Sets the qdirs of some variables in the value alist to IGN.
; Qvalues is of the form: ((varname . <qval>) ...)
;   It's slower than the variable-flag version, but accesses SIM.  (BJK:  10-5-90)


(defun initialize-ign-qdirs (sim)
  (let ((vars-for-ign (union (sim-ignore-qdirs sim)
			     (cdr (assoc 'ignore-qdirs (qde-other (sim-qde sim)))))))
    (unless (null vars-for-ign)
      (setf (sim-ignore-qdirs sim) vars-for-ign)
      (cond ((member 'incomplete (state-status (sim-state sim)))
	     (mapc #'(lambda (s) (set-qdirs-to-ign s vars-for-ign))
		   ;;  Eliminated direct access of state.successor slot 02/14/91  DJC
		   (successor-states (sim-state sim))))
	    (t (set-qdirs-to-ign (sim-state sim) vars-for-ign))))))


(defun set-qdirs-to-ign (state vars-for-ign)
  (dolist (qvalue (state-qvalues state))
    (let* ((qval (cdr qvalue)))
      (if (member (car qvalue) vars-for-ign)	; (variable-ignore-qdir-p var)
	  (setf (qdir qval) 'ign)))))

; Set qdirs to IGN for some variables (normally from *current-sim*.ignore-qdirs).
;    (BJK:  10-31-90)

(defun set-ignore-qdirs (state vars)		; called by complete-states-from-partial-state
  (dolist (v vars t)
    (let ((qval (qval v state)))
      (cond ((null qval)
	     (cond ((assoc v (state-qspaces state))
		    (push (cons v (make-qval :qdir 'ign))
			  (state-qvalues state)))
		   (t nil)))
	    (t (setf (qval-qdir qval) 'ign))))))

;;; Actually, the second embedded case should be an error:
;;; (error "Variable ~a not defined for ~a.  Probably *current-sim* ~a is obsolete."
;;;        v state *current-sim*)
;;; but the examples have not yet been rewritten to reset *current-sim* to NIL.


; Let the phase planes be the union of the phase planes from SIM and QDE.

(defun initialize-NIC-phase-planes (sim)
  (let ((pp-for-NIC (union (sim-phase-planes sim)
			   (cdr (assoc 'phase-planes (qde-other (sim-qde sim))))
			   :test #'equal)))
    (unless (null pp-for-NIC)
      (setf (sim-phase-planes sim) pp-for-NIC))))

(defun initialize-NNL-variables (sim)
  (let ((vars-for-NNL (union (sim-no-new-landmarks sim)
			     (cdr (assoc 'no-new-landmarks (qde-other (sim-qde sim)))))))
    (unless (null vars-for-NNL)
      (setf (sim-no-new-landmarks sim) vars-for-NNL))))

(defun initialize-unreachable-values (sim)
  (let ((unreachable-vals (union (sim-unreachable-values sim)
				 (cdr (assoc 'unreachable-values (qde-other (sim-qde sim))))
				 :test #'equal)))
    (unless (null unreachable-vals)
      (setf (sim-unreachable-values sim) unreachable-vals))))


;;; Creates a display block for this state.  Added DJC 05/14/91
;;; Modified PF 03 June 1991, to handle envisionment

(defun initialize-display-block (state)
  (let* ((sim (state-sim state))
	 (name (genname 'disp))
	 ;; creates the basic abstraction-level
	 (level (make-basic-level state))
	 (dp (make-display-block :name name 
				 :qdes-in-beh-tree (list (state-qde state)))))
    ;; Those are dynamic slots and cannot be set in the function
    ;; make-display-block. 
    (setf (display-block-basic-level dp) level
	  (display-block-levels dp) (list level)
	  (display-block-envisionment-p dp) (sim-envisionment-p sim)
	  (display-block-plot-intervals dp) (sim-envisionment-p sim)
	  (state-levels state) (list level)
	  *abstraction-level* level
	  *current-display-block* dp
	  (state-display-block state) dp)
    (set name dp)
    dp))




;;;-----------------------------------------------------------------------------
;;;  QSIMULATE takes the existing QSIM agenda and simulates until either
;;;  the agenda becomes empty or the state limit is reached.
;;;  QSIMULATE-STATE is separated out, but the modularity is not yet ideal.
;;;      (Revised, 12-6-90. -BJK)
;;;-----------------------------------------------------------------------------

(defun qsimulate (sim)
  (with-bindings-from-sim sim

    (loop while (sim-agenda sim)

	  for end-p = (end-simulation-p sim)
	  when end-p 
	    do (trace-simulation-stop end-p)
	       (return)

	  do (let ((S (pop (sim-agenda sim))))
	       (trace-qsimulate-count S (sim-agenda sim))
	       (qsimulate-state S sim))

	     )))



(defun qsimulate-state (S sim)
  (let ((*current-qde* (state-qde s)))
    ;; This function is wrapped with a *current-qde* call because 
    ;; q-continue calls this and the local binding of
    ;; *current-qde* by qsim will have been broken.  BKay 29May92
    (declare (special *current-qde*))
    (when (member 'incomplete (state-status S))
      (trace-qsimulate-incompletes s)
      ;;  Eliminated direct access of state.successor slot 02/14/91  DJC
      (add-states-to-agenda (successor-states S) sim :type :depth-first)
      (return-from qsimulate-state))
    
    (cond ((not (member 'complete (state-status S)))
	   (error "State ~a should be complete by now." S))
	  ((intersection '(incomplete OK) (state-status S))
	   (error "State ~a should not have status ~a by now." S (state-status S))))
    
    (trace-qsimulate-count S (sim-agenda sim))
    
    (unless (member 'GF (state-status S))
      (apply-global-state-analysis  S)
      (apply-global-state-filters   S)
      (pushnew 'GF (state-status S)))
    
    (if (member 'inconsistent (state-status S))
	(return-from qsimulate-state))
    
    (let ((tstates (remove S (filter-for-transitions S))))
      (if tstates
	  (add-states-to-agenda tstates sim)))
    
    (when (eligable-for-successors S)
      (let ((new-states (successors-of-state S)))
	(dolist (ns new-states)
	  (apply-global-state-analysis ns)
	  (apply-global-state-filters  ns)
	  (pushnew 'GF (state-status ns)))      
	(add-states-to-agenda new-states sim)))
    ))



; Removed the following two calls from qsimulate:
;    (setq *states* (union new-states *states*))
;    (filter-for-similarity S new-states)

(defun eligable-for-successors (state)
  (let ((status (state-status state)))
    (cond ((null status) t)
	  ((intersection status '(inconsistent self-intersection cycle final cross-edge)) nil)
	  ((member 'quiescent status) (if *quiescent-ok-for-agenda* t nil))
	  ((member 'complete status) t)
	  (t (error "State ~a has unknown status ~a." state status)))))


(defun add-states-to-agenda (states sim &key (type *simulation-type*))
  (let ((agenda (sim-agenda sim))
	(new-states (mapcan #'filter-for-agenda states)))
    (trace-qsimulate-agenda agenda new-states)
    (ecase type
      (:breadth-first (setf (sim-agenda sim) (append agenda new-states)))
      (:depth-first   (setf (sim-agenda sim) (append new-states agenda))))
    (incf (sim-state-count sim) (length new-states))
    (setf (sim-states sim) (union new-states (sim-states sim)))))


;;; Check to see if it's time to end the simulation, either because of
;;; state-limit or time-limit.
;;;
;;;  Modified so that it simulates all states to the time limit when
;;;  performing a depth first simulation.  This change was requested by Jack Vinson.
;;;  DJC  07/2/92

(defun end-simulation-p (sim)
  (or (and (> (sim-state-count sim) (sim-state-limit sim)) (sim-state-limit sim))
      (if (equal (sim-simulation-type sim) :breadth-first)
	  (let ((time-bound (sim-time-limit sim))
		(states (sim-agenda sim)))
	    (and (every #'(lambda (state)
			    (later-time? (qmag (state-time state))
					 time-bound
					 :or-equal))
			states)
		 (sim-time-limit sim)))
	  (let ((time (state-time (first (sim-agenda sim)))))
	    (and (sim-time-limit sim)
		 (qpointp time)
		 (eq (lmark-name (qmag time)) (sim-time-limit sim))
		 (sim-time-limit sim))))))
  

(defun trace-qsimulate-agenda (agenda states)
  (if *trace-main-agenda*
    (format *QSIM-trace* "~%Agenda = ~a.  Adding ~a." agenda states)))

;;;-----------------------------------------------------------------------------
;;;  SUCCESSORS-OF-STATE takes an existing state, generates the next possible
;;;  successor values for each variable, runs cfilter to find the set of
;;;  consistent value completions, and then runs the global alist filters
;;;  before committing any value completions to statehood.  This function is
;;;  called only by function QSIMULATE, which performs all other global filtering.
;;;-----------------------------------------------------------------------------

(defun successors-of-state (current-state)
  (declare (special Current-State))		; make visible to cfilter, etc.
  (let* ((*current-qde* (state-qde current-state))  ; Now a local def BKay 27May92
	 (old-time (state-time current-state))
	 (new-time (copy-qval old-time))
	 (time-var (car (qde-variables *current-qde*)))
	 (successor-function (if (qpointp old-time)
				 #'P-successors
				 #'I-successors)))
    (declare (special *current-qde*))

    ;; Compute next time value and install it as the only possible value
    ;; for the time variable.
    (setf (qmag new-time) (if (qpointp old-time)
			      (list (qmag old-time) (succ (qmag old-time) (time-qspace current-state)))
			      (cadr (qmag old-time)))
	  (variable--qspace time-var) (time-qspace current-state)
	  (variable--pvals  time-var) (list new-time))

    ;; Do P- or I-successor, setting each variable's possible values,
    ;; plus its qspace.
    (mapc #'(lambda (qvalue qspace-pair)
	      (let* ((qval   (cdr qvalue))
		     (qspace (cdr qspace-pair))
		     (var    (qval-variable qval)))
		(setf (variable--qspace var) qspace
		      (variable--pvals var)
		      (if (or (variable-independent-p var)
			      (variable-discrete-p var))
			  (list qval)
			  (funcall successor-function qval qspace var)
			  ;;(eliminate-unreachable-values     ; removed -> global filter
			  ;;  var
			  ;;  (funcall successor-function qval qspace var))
			  ))))
	  (cdr (state-qvalues current-state))		; skip over time qvalue
	  (cdr (state-qspaces current-state))))		; skip over time qspace
  (install-corresponding-values current-state)

  ;; Run cfilter, then filter the resulting value completions.
  (let* ((*filtering-condition* :successors)
	 (candidates (cfilter (state-qde current-state) #'check-qsim-constraint))
	 (alists (mapcan #'(lambda (alist)
			     (apply-global-alist-filters current-state alist))
			 candidates)))
    (cond ((null alists)
	   (prune-inconsistent-state current-state "No successors") nil)
	  (t (build-successors-from-alists current-state alists)))))


(defun build-successors-from-alists (state alists)
  (let ((new-states (mapcar #'(lambda (alist)
				(successor-state-from-predecessor state alist))
			    alists)))
    (mapc #'(lambda (nstate)
	      (setf (state-justification nstate)
		    (if (singleton-p alists)
			`(unique-successor-of ,state)
			`(one-of-several-successors-of ,state ,new-states)))
	      (setf (state-predecessors nstate) (list state))
	      (push 'complete (state-status nstate))
	      )
	  new-states)
    ;; Set attributes of the "parent" state.
    (setf (state-successors state) (cons `successors new-states))
    (push 'OK (state-status state))
    ;; Return the newly created states.
    new-states))

;;;-----------------------------------------------------------------------------
;;;  Q-CONTINUE may be called manually by the user to continue a simulation
;;;  that was stopped because of the *current-state-limit*
;;;  or the *current-time-limit*
;;;  Pfouche 11/30/89 
;;;
;;;  Modified so that it could be called on a simulation other than the most
;;;  recently executed one by passing either an initial-state or a sim
;;;  DJC  9Oct91
;;;-----------------------------------------------------------------------------

(defun q-continue (&key new-state-limit new-time-limit initial-state sim)
  "Continues simulation with a new time and/or state limit"
  (cond ((and initial-state sim)          ; DJC 9Oct91
	 (error "Provide either a sim or an initial-state, but not both"))
	(initial-state 
	 (setf sim (state-sim initial-state)))
	((null sim)
	 (setf sim (state-sim *initial-state*))))
    (reset-timers)
    (when new-state-limit
      (setf (sim-state-limit sim) new-state-limit))
    (when new-time-limit
      (setf (sim-time-limit sim) new-time-limit))
    (let ((start-time (get-internal-run-time))
	  (old-state-count (sim-state-count sim)))
      (qsimulate sim)
      (if *print-timing*
	  (format *QSIM-Trace* "~%Run time: ~,3f seconds to simulate ~d more states."
		  (/ (- (get-internal-run-time) start-time)
		     internal-time-units-per-second)
		  (- (sim-state-count sim) old-state-count))))
    (print-timers)
    sim)

  

;;;-----------------------------------------------------------------------------
;;; QCD combines q-continue and qsim-display 
;;;-----------------------------------------------------------------------------

(defun qcd (&key new-state-limit new-time-limit
	    (reference-states *reference-states*)
	    (trees t)
	    layout
	    (show-inconsistent-successors *show-inconsistent-successors*))
  "Qsim continue and display"
  (let ((initial (q-continue :new-time-limit new-time-limit
			    :new-state-limit new-state-limit)))
    (qsim-display initial
		  :reference-states reference-states
		  :trees trees
		  :show-inconsistent-successors show-inconsistent-successors
		  :layout (or layout
			      (layout-from-state initial)))))


;;;-----------------------------------------------------------------------------
;;;  QSIM-CONTINUE may be called manually by the user to continue a simulation
;;;  that was stopped because of the *state-limit*.
;;;-----------------------------------------------------------------------------

(defparameter
  *qsim-continue-warning-message*
  "~% *** Why don't you use q-continue that allows you to specify either a new time
or a new state limit ? ***")

(defun qsim-continue (new-state-limit)
  "You'd better use Q-CONTINUE instead !!"
  (warn *qsim-continue-warning-message*)
  (assert (numberp new-state-limit) () "Must supply a number for new-state-limit.")
  (q-continue :new-state-limit new-state-limit))

;;;-----------------------------------------------------------------------------
;;;  STRUCT-CLEANUP sets all slots of a structure to nil.  This is called by
;;;  qsim-cleanup in an attempt to make it possible for ephemeral garbage
;;;  collection to recover memory that would otherwise not be recovered until
;;;  a full GC occurs.  This function relies on the fact that in most Common
;;;  Lisps, structures are usually represented internally as arrays or lists.
;;;  If not, then this function will have no effect.
;;;-----------------------------------------------------------------------------

(defun struct-cleanup (s)
  "Sets all slots of a structure to nil"
  (when s
    (typecase s
      (STRING nil)
      (ARRAY  (dotimes (i (length s))
		(setf (aref s i) nil)))
      (LIST   (dotimes (i (length s))
		(setf (nth i s) nil))))))

;;;-----------------------------------------------------------------------------
;;;  QSIM-CLEANUP may be called manually by a user to "clean up" storage that
;;;  was allocated during simulation.  It simply uninterns the symbols, and
;;;  it sets to NIL the value of a few global parameters.
;;;-----------------------------------------------------------------------------

(defun qsim-cleanup ()
  "Uninterns symbols created during simulation."
  (let (sim)
    (when *initial-state*
      (setf sim (state-sim *initial-state*))
      (dolist (state (sim-agenda sim))
	(struct-cleanup state)))
    (struct-cleanup *initial-state*)
    (setq *current-sim*  nil
	  *initial-state* nil
;	  *current-qde*   'foo        ; Test code!! Should be removed.  BKay 27May92
	  *initial-state-stack* nil)
    (dolist (symbol *interned-symbols*)
      (setf (symbol-plist symbol) nil)
      (when (boundp symbol)
	(struct-cleanup symbol)
	(setf (symbol-value symbol) nil)))
    (dolist (symbol *interned-symbols*)
      (unintern symbol))
    (setq *interned-symbols* nil)
    (reset-roots)))
  
(defun reset-roots ()
  (dolist (atom *genname-roots*)
    (when (symbolp atom)		; TI explorers get mad when numbers are
      					; treated like symbols BKAY 9/13/89
      (setf (get atom 'gennum) 0))))


; Timing calls.

;;;--af
(defun print-initialization-time-trace (start &optional (state))
  (if *print-timing*
      (format *QSIM-Trace* "~%Run time: ~,3f seconds to initialize ~a."
	      (/ (- (get-internal-run-time) start)
		 internal-time-units-per-second)
	      (or state "a state"))))


; Trace functions for the central qsimulate loop.

(defun trace-agenda-setup (a)
  (if *trace-main-agenda*
      (format *QSIM-Trace* "~%Initial agenda: ~a" a)))

(defun final-agenda-tracing (start-time)
  (when *print-timing*
    (format *QSIM-Trace* "~%Run time: ~,3f seconds to simulate ~d states."
	    (/ (- (get-internal-run-time) start-time)
	       internal-time-units-per-second)
	    (sim-state-count *current-sim*)))
  (when *intersection-count-p*
    (format *QSIM-Trace* "~&Intersection counts:          Portrait                    Count")
    (mapc #'(lambda (alist) (format *QSIM-Trace* "~&~30a~28a~5a" " " (car alist) (cadr alist)))
	  *intersection-count*))
  (print-timers))

(defun trace-simulation-stop  (limit)
  (when *trace-simulation-stop*
    (if (numberp limit)
	(format *qsim-report* "~%---> Reached limit of ~d states.~
                               ~%---> To continue, type (q-continue :new-state-limit <new-state-limit>)"
		limit)
	(format *qsim-report* "~%---> Reached limit of time ~a.
                               ~%---> To continue, type (q-continue :new-time-limit <new-time-limit>)"
		limit))))

(defun trace-qsimulate (new-states)
  (if  *trace-main-agenda*
       (format *QSIM-Trace* "~%Adding ~a to agenda; now ~a entries."
	       new-states (length (sim-agenda *current-sim*)))))



(defun trace-qsimulate-incompletes (s)
  (when *trace-main-agenda*
    (format *QSIM-Trace* "~%Replacing incomplete ~a on agenda with its completions ~a."
	    S (successor-states S))))  ;;  Eliminated direct access of state.successor slot 02/14/91  DJC
 

(defun trace-qsimulate-count (S agenda)
  (when *trace-main-agenda*
    (format *QSIM-Trace* "~%Taking ~a from agenda for simulation, leaving ~a."
	    S agenda)))

;;;-----------------------------------------------------------------------------
;;;  Quasi-equilibrium-solve takes a qde or reference state, and completes it
;;;  wrt the quasi-equilibrium assumption.  The global switch
;;;  *quasi-equilibrium-reasoning* controls the constraint and propagation
;;;  semantics of D/DT.
;;;-----------------------------------------------------------------------------
;;;--af   ;;; updated BJK:  10-24-90

(defun quasi-equilibrium-solve (state-or-qde user-values &optional (text nil))
  (let* ((*quasi-equilibrium-reasoning* t)
	 (nstate (typecase state-or-qde
		   (qde    (make-new-state :from-qde state-or-qde
					   :assert-values user-values
					   :text text))
		   (state  (make-new-state :from-state state-or-qde
					   :assert-values user-values
					   :text text))
		   (t      (error "Can't solve from ~a." state-or-qde)))))
    (if nstate
	(get-list-of-initial-states nstate)
	(error "Inconsistent state ~a had no completions." nstate))))

;;;-----------------------------------------------------------------------------
;;; ENVISION is the main function to perform envisionment. If values is nil, a 
;;; total envisionment is performed. If values is a partial state specification,
;;; an attainable envisionment is performed, from that state or its completion.
;;;-----------------------------------------------------------------------------

(defun envision (&key qde values (sim (make-sim)))
  (let* ((start-time  (get-internal-run-time))
	 (*current-qde* qde)   ; Now a local def BKay 27May92
	 init-states)
    (declare (special *current-qde*))
    
    ;; set sim slots appropriately
    (setf *current-sim* sim
	  (sim-envisionment-p sim) t
	  (sim-time-limit sim) nil 
	  (sim-enable-landmark-creation sim) nil
	  (sim-KET-constraint sim) nil
	  (sim-Q2-constraints sim) nil
	  (sim-NIC-constraint sim) nil
	  (sim-simulation-type sim) :depth-first
	  (sim-cycle-detection sim) nil
	  (sim-other-variables sim) '((*plot-state-indices* :above-node)))
    (with-bindings-from-sim sim
      ;; make the initial state
      (setq *initial-state* (make-new-state :from-qde qde
				       :assert-values values
				       :sim sim))
	    ;;sim)
      (push *initial-state* *initial-state-stack*)  ; added DJC 02/11/91
      (setf init-states (mapcan #'apply-global-state-filters
				(get-list-of-initial-states *initial-state*))
	    (sim-state sim) *initial-state*)
      (reset-timers)
      (setf (sim-agenda sim) init-states
	    (sim-display-block sim) (initialize-display-block *initial-state*)
	    (sim-states sim) (copy-list init-states))
      (trace-agenda-setup (sim-agenda sim))
      ;; enter the main simulation loop
      (qsimulate sim)
      (final-agenda-tracing start-time)
      (sim-state sim))))



(defun eliminate-occurrence-branching (ignore)
  (declare (ignore ignore))
  (format *qsim-trace*
	  "~%
*****************************************************************************
You tried to call the function ELIMINATE-OCCURRENCE-BRANCHING.

Sorry, I didn't have time to incorporate that piece of code into QSIM before
leaving. I will email it as soon as possible.

				Pierre Fouche, August 30, 1990
*****************************************************************************
"))


(defun eliminate-chatter (ignore)
  (declare (ignore ignore))
  (format *qsim-trace*
	  "~%
*****************************************************************************
You tried to call the function ELIMINATE-CHATTER.

Sorry, I didn't have time to incorporate that piece of code into QSIM before
leaving. I will email it as soon as possible.

				Pierre Fouche, August 30, 1990
*****************************************************************************
"))
	  

;;;-----------------------------------------------------------------------------
;;;  GET-BEHAVIORS returns a list of all behaviors following from the current
;;;  state.  A behavior is a list of states.  The tree of behaviors is implicit
;;;  in the state.successors slot.
;;;
;;;  Modified by PFouche 08/30/90: It can now be used to display an envisionment
;;;  graph as well. In that case you can specify another argument, final-state,
;;;  to display all the behaviors from some initial state to that final state.
;;;-----------------------------------------------------------------------------


(defun get-behaviors (&optional (initial-state *initial-state*)
		      &key (*final-state* nil)
		      (successor-function #'get-successors)
		      (visited-nodes nil)) ; added DJC to work with xedge code so that
                                           ; the special variable *visited-nodes* can be initialized
                                           ; with a set of nodes when calling mapgraph.
  (declare (special *final-state*))
  (if (listp initial-state)			;Dan B. 10/22/90. Reimplementing previous capability.
						;Don't know what the optional & key args are for; being cautious.
      (if (and (null *final-state*) (eql successor-function #'get-successors)) 
	  (mapcan #'get-behaviors initial-state))
      (let ((incomplete-p (incomplete-p initial-state)))
	(mapcan #'(lambda (states)
		    (let* ((new-states (if *final-state*
					   (if (eq *final-state* (car states))
					       states nil) 
					   states))
			   (beh (reverse new-states))
			   (new-beh (if incomplete-p (cdr beh) beh)))
		      (when new-beh (list new-beh))))			  
		(mapgraph
		  #'(lambda (x) (values (list x) (not (eq x *final-state*))))
		  (list initial-state)
		  :apply-when-cycle t
		  :successor-function successor-function
		  :visited-nodes visited-nodes)))))

;;;-----------------------------------------------------------------------------
;;;  SUCCESSOR-STATES returns a list of successor states of state, or NIL.
;;;-----------------------------------------------------------------------------

(defconstant *states-without-successors* '(inconsistent
					    cycle-identity
					    partial-match
					    final-state
					    quiescent
					    tsa-id
					    trajectory-intersection))


;; Modified to filter inconsistent states  02/15/91  DJC
;; Inconsistent states are only included in *show-inconsistent-states* is set
;;
;;; Modified to handle the existence of aggregate intervals
;;; 15 May 1991
;;;
;;; Modified by adding the *traverse-xedges* special variable which causes this function
;;; to traverse an xedge to determine the successors of a state

(defun successor-states (state)        ;;Modified 11/9/89 D.B.
  (when state
  (if (typep state 'aggregate-interval)     ;; added DJC                       
      (list (equivalence-node-abstract-state 
	      (car (aggregate-interval-cur-level state))))
      (let ((ss (if (and *traverse-xedges*
			 (state-cross-edge-identity state)
			 (not (equal (car (state-successors state))
				     'cycle-identity)))
		    (cons 'successors
		     (successor-states (state-cross-edge-identity state)))
		    (state-successors state))))
	(cond ((null ss) nil)
	      ((member (car ss) *states-without-successors*) nil)
	      ((eql (car ss) 'transition-identity)
	       (cond ((null (cadr ss)) nil)
		     (t (if *show-inconsistent-successors*     ;  28 Feb DJC
			    (cdr ss)
			    (filter-inconsistent-successors (cdr ss))))))
	      ((member (car ss) '(successors completions perturbation branches))
	       (if *show-inconsistent-successors*
		   (cdr ss)
		   (filter-inconsistent-successors (cdr ss))))
	      (t (error "State ~a has invalid successors type:  ~a." state (car ss))))))))



;(defconstant *states-without-successors* '(inconsistent
;					    cycle-identity
;					    partial-match
;					    final-state
;					    quiescent
;					    tsa-id
;					    trajectory-intersection))
;;; Modified to filter inconsistent states  02/15/91  DJC
;;; Inconsistent states are only included in *show-inconsistent-states* is set
;;;
;;;; Modified to handle the existence of aggregate intervals
;;;; 15 May 1991

;(defun successor-states (state)        ;;Modified 11/9/89 D.B.
;  (if (typep state 'aggregate-interval)     ;; added DJC                       
;      (list (equivalence-node-abstract-state 
;	      (car (aggregate-interval-cur-level state))))
;      (let ((ss (state-successors state)))
;	(cond ((null ss) nil)
;	      ((member (car ss) *states-without-successors*) nil)
;	      ((eql (car ss) 'transition-identity)
;	       (cond ((null (cadr ss)) nil)
;		     (t (if *show-inconsistent-successors*     ;  28 Feb DJC
;			    (cdr ss)
;			    (filter-inconsistent-successors (cdr ss))))))
;	      ((member (car ss) '(successors completions perturbation branches))
;	       (if *show-inconsistent-successors*
;		   (cdr ss)
;		   (filter-inconsistent-successors (cdr ss))))
;	      (t (error "State ~a has invalid successors type:  ~a." state (car ss)))))))


;;;-----------------------------------------------------------------------------
;;;  FILTERED-SUCCESSOR-SLOT-CONTENTS returns the contents of successor slot of
;;;    state with the inconsistent states filtered out.  If 
;;;    *show-inconsistent-successors* is true then they are not filtered out.  
;;;        02/14/91   DJC
;;;-----------------------------------------------------------------------------
;;; Modified to handle the existence of aggregate-intervals
;;;  15 May 1991  DJC

(defun filtered-successor-slot-contents (state)
  (if (typep state 'aggregate-interval)
      (list 'successors 
	    (equivalence-node-abstract-state (car (aggregate-interval-cur-level state))))
      (let ((ss (state-successors state)))
	(if (or (member (car ss) *states-without-successors*)
		*show-inconsistent-successors*)     ;  28 Feb DJC
	    ss
	    (cons (car ss)
		  (filter-inconsistent-successors (cdr ss)))))))



;;;-----------------------------------------------------------------------------
;;; FILTER-INCONSISTENT-STATES receives a list of states.  It filters out the
;;;   states from this list which are inconsistent
;;;-----------------------------------------------------------------------------

(defun filter-inconsistent-successors (state-list)
  (remove-if #'(lambda (state) (member 'inconsistent (state-status state)))
	     state-list))

;;;----------------------------------------------------------------------------
;;; GET-SUCCESSORS returns a list of successors, considering abstracted states
;;;----------------------------------------------------------------------------
;;;
;;;  Modified to handle aggregate-intervals.  If the current
;;;  state has an aggregate interval attatched to it and
;;;  *filter-occ-branch* is set, then it will return this
;;;  interval if it is "active" at the current level of
;;;  description.
;;;
;;;  15 May 1991 DJC

;; moved to the end of the file, PF 03 Jun 1991
;(defun get-successors (state &optional (level *abstraction-level*))
;  (let ((succs ()))
;    (cond ((typep state 'aggregate-interval)
;	   (list (equivalence-node-abstract-state (car (aggregate-interval-cur-level state)))))
;	  ((and *filter-occ-branching*
;		(state-aggregates state)
;		(aggregate-interval-cur-level (state-aggregates state)))
;	   (list (state-aggregates state)))
;	  (t  (dolist (succ (successor-states state)) 
;		(pushnew (abstract-state succ level) succs))
;	      (nreverse succs)))))


;;;;-----------------------------------------------------------------------------
;;;; GET-PREDECESSORS returns a list of predecessors, considering abstracted states
;;;;-----------------------------------------------------------------------------

;(defun get-predecessors (state &optional (level *abstraction-level*))
;  (let ((preds ()))
;    (dolist (pred (state-predecessors state)) 
;      (pushnew (abstract-state pred level) preds))
;    preds))



;;; The following three functions are not used anymore, PF 03 Jun 1991
;;;;-----------------------------------------------------------------------------
;;;; MOST-ABSTRACT-STATE returns a state that describes the current state at
;;;; the highest level of abstraction.
;;;;-----------------------------------------------------------------------------

;(defun most-abstract-state (state)
;  (let ((abstract-state (state-coarsening state)))
;    (if abstract-state
;	(most-abstract-state abstract-state)
;	state)))


;;;;-----------------------------------------------------------------------------
;;;; GET-STATE-LEVEL computes the level of abstraction of a state.
;;;;-----------------------------------------------------------------------------

;(defun get-state-level (state)
;  (or (state-level state)
;      (let* ((finer-states (state-refinings state))
;	     (levels (mapcar #'get-state-level finer-states))
;	     (level (when levels (1+ (apply #'max levels)))))
;	(or level 0))))


;(defun abstract-state (state level)
;  (let* ((current-level (get-state-level state))
;	 (coarser-state (state-coarsening state))
;	 (coarser-level (when coarser-state (get-state-level coarser-state))))
;    (cond ((eq level t) (most-abstract-state state))
;	  ((or (not current-level) (not level)
;	       (>= current-level level)
;	       (not coarser-state))
;	   state)
;	  ((= level coarser-level) coarser-state)
;	  ((> coarser-level level) state)
;	  (t (abstract-state coarser-state level)))))



;;;-----------------------------------------------------------------------------
;;;  AVERAGE-BRANCHING-FACTOR returns two values, given an initial state: the
;;;  average branching factor and the number of nodes of the tree whose root is
;;;  the initial state
;;;  Pfouche 11/21/89
;;;-----------------------------------------------------------------------------

(defun average-branching-factor (&optional (state *initial-state*))
  "Returns two values: the average branching factor and the number of nodes"
  (let* ((successors (successor-states state))
	 (number-of-nodes 1)
	 (number-of-branches (length successors)))
    (if (null successors)
	(values 0 0)
	(progn
	  (dolist (succ successors)
	    (multiple-value-bind
	      (succ-factor succ-nodes)
		(average-branching-factor succ)
	      (setq number-of-nodes (+ number-of-nodes succ-nodes))
	      (setq number-of-branches (+ number-of-branches
					  (* succ-factor succ-nodes)))))
	  (values (float (/ number-of-branches number-of-nodes))
		  number-of-nodes)))))


;;;-----------------------------------------------------------------------------
;;;  PRUNE-INCONSISTENT-STATE
;;;                                                       /---> Z1
;;;  Suppose we have the sequence of states:  X1 ---> Y1 <
;;;                                              \        \---> Z2
;;;                                               \--...
;;;
;;;  Now suppose state Z1 has been found to be inconsistent.
;;;  We then call prune-inconsistent-state to remove Z1 from Y1's list of 
;;;  successors.  Fine.
;;;  Now suppose that later Z2 is also found to be inconsistent.
;;;  When prune-inconsistent-state is called to remove Z2 from Y1's list of
;;;  successors, it sees that Y1 has no other successors.  In the normal case,
;;;  Y1 would also be pruned by recursively call.  However, if Y1 has any
;;;  attached "twin" states (as determined by filter-for-similarity), then
;;;  the twin's state values will overlay those of Y1 and simulation will
;;;  again proceed forward from Y1.
;;;
;;;  Do nothing if this state has no predecessor.  It always returns NIL.
;;;-----------------------------------------------------------------------------


;; prune-inconsistent-state now considers state-predecessors instead of the
;; justification slot. In an envisionment graph, a state may have several
;; predecessors.
;; PF 08/30/90

(defun prune-inconsistent-state (state reason)
  (let ((preds (state-predecessors state))
	(pred (state-predecessor state)))
    (if (and preds (not (member pred preds))) ;Dan B. 10/22/90.
	(progn (format *qsim-report*
		       "~% INCONSISTENT DATA: state-predecessor of ~a is ~a which is not in state-predecessors ~a.~%"
		       state pred preds)
	       (format *qsim-report* "~%Processing all of them.~%")))
    (if preds 
	(dolist (pred preds) 
	  (mark-inconsistent-state state pred reason))       ;  DJC  02/16/91
	(mark-inconsistent-state state (state-predecessor state) reason))))   ;  DJC  02/16/91
  

;(defun delete-inconsistent-state (state predecessor reason)
;  (let* ((justification (state-justification state))
;	 (successors (when (typep predecessor 'state)
;		       (state-successors predecessor))))
;    (pushnew `inconsistent (state-status state))
;    (pushnew `(inconsistent ,reason) (state-status state) :test #'equal)
;    (pruning-trace state justification successors predecessor)
;    (when *delete-inconsistent-branches*
;      (setf *states* (delete state *states*))
;      (ecase (car justification)
;	((unique-successor-of one-of-several-successors-of successor-of
;			      perturbation-of spliced-in separated-branch-of)
;	 (clear-pred state predecessor))
;	(one-of-several-completions-of
;	  (clear-initial state)
;	  (clear-pred state predecessor)
;	  (when (eq (car (state-justification predecessor)) 'transition-from)
;	    (clear-pred state (state-predecessor predecessor))))
;	;; If the state  is not a member of the successors list, it
;	;;  will be filtered and never installed.
;	(transition-from
;	  (when (member state successors)
;	    (clear-pred state predecessor) ))
;	((initialized-with root-of-separated-tree copy-of)
;	  (clear-initial state)))))) 

;;;  Used to be called DELETE-INCONSISTENT-STATE name changed to MARK-INCONSISTENT-STATE.
;;;     Modified to eliminate the *delete-inconsistent-branches* flag.  All inconsistent
;;;     branches are maintained in the tree and *show-inconsistent-successors* determines
;;;     if they are visible to the modeler.  The visibility is controled via the 
;;;     SUCCESSOR-STATES function.  This function no longer deletes the inconsistent
;;;     states.  DJC 01/30/91


(defun mark-inconsistent-state (state predecessor reason)
  (let* ((justification (state-justification state))
	 (successors (when (typep predecessor 'state)
		       ;; modified to call filtered-successor-slot-contents instead 
		       ;;  of state-successor   02/13/91   DJC
		       (filtered-successor-slot-contents predecessor)))
	 (cross-edge-preds (state-cross-edge-predecessors state)))
    (pushnew `inconsistent (state-status state))
    (pushnew `(inconsistent ,reason) (state-status state) :test #'equal)
    (pruning-trace state justification successors predecessor)
    (ecase (car justification)
      ((unique-successor-of one-of-several-successors-of successor-of
			    perturbation-of spliced-in separated-branch-of)
       (clear-pred predecessor))
      (one-of-several-completions-of
       (clear-initial state)
       (clear-pred predecessor)
       (when (eq (car (state-justification predecessor)) 'transition-from)
	 (clear-pred (state-predecessor predecessor))))
      ;; If the state  is not a member of the successors list, it
      ;;  will be filtered and never installed.
      (transition-from
       (when (member state successors)
	 (clear-pred predecessor) ))
      ((initialized-with root-of-separated-tree copy-of)
       (clear-initial state)))
    (mapcar #'(lambda (state)
		(mark-inconsistent-state state (state-predecessor state)
					 "cross edge state pruned"))
	    cross-edge-preds)))

;; Code which deals with the deletion of states has been eliminated.  As a result the STATE
;;   paramter has also been eliminated.  02/09/91   DJC

(defun clear-pred (predecessor)
  (when predecessor
    ;; Because the CAR of the successors list is the atom SUCCESSORS, the deletion will 
    ;;  never return a NULL list.  If there are other successors, take no further action.
    (cond  ((consistent-state-in-list? (cdr (state-successors predecessor))) nil)
	   ((state-twins predecessor)
	    (substitute-twin-state predecessor))
	   ;; Otherwise prune the predecessor state since it has no remaining successors ...
	   (t (prune-inconsistent-state predecessor "last successor removed")))))



(defun pruning-trace (state justifications successors predecessor)
  (case (car justifications)
    ((unique-successor-of one-of-several-successors-of)
     (unless (eql (car successors) 'successors)
       (error "State ~a has bad successor pointer:  ~a." state successors))
     (when *trace-pruning*
       (format *QSIM-Trace* "~%Removing inconsistent state ~a from its predecessor ~a."
	       (state-name state) (state-name predecessor))))
    (one-of-several-completions-of
      (unless (eql (car successors) 'completions)
	(error "State ~s has bad completion pointer in ~s:  ~s."
	       state predecessor successors))
      (when *trace-pruning*
	(format *QSIM-Trace* "~%Removing inconsistent completion ~a from incomplete ~a."
		(state-name state) (state-name predecessor))))))


;;; Modified to update state's justification when predecessor has only one successor left
;;; PF 06/16/90

;(defun clear-pred (state predecessor)
;  (when predecessor
;    (delete state (state-successors predecessor) :count 1)
;    ;; Because the CAR of the successors list is the atom SUCCESSORS, the deletion will 
;    ;;  never return a NULL list.  If there are other successors, take no further action.
;    (cond  ((cddr (state-successors predecessor))
;	    ;; two remaining successors, nothing else to do
;	    nil)
;	   ((cdr (state-successors predecessor))
;	    ;; one successor left -> change the justification of the successor
;	    (if (eql (car (state-justification (cadr (state-successors predecessor))))
;		     'one-of-several-successors-of)
;		(setf (state-justification (cadr (state-successors predecessor)))
;		      `(unique-successor-of ,predecessor))))
;	   ((state-twins predecessor)
;	    (substitute-twin-state predecessor))
;	   ;; Otherwise prune the predecessor state since it has no remaining successors ...
;	   (t (prune-inconsistent-state predecessor "last successor removed")))))


;;;-----------------------------------------------------------------------------
;;; CONSISTENT-STATE-IN-LIST? receives a list of states and it will return
;;;   t if one of the states is consistent.  Called by CLEAR-PRED.  01/30/91  DJC
;;;-----------------------------------------------------------------------------


(defun consistent-state-in-list? (state-list)
  (cond ((null state-list) nil)
	((member 'inconsistent (state-status (car state-list)))
	 (consistent-state-in-list? (cdr state-list)))
	(t t)))



(defun clear-initial (state)
  ;  (setf *states* (delete state *states*))  ;;  commented out due to the elimination of the
                                              ;;  deletion of inconsistent states   02/16/91   DJC
  (when (and (listp *initial-state*)
	     (cdr *initial-state*)
	     (member state *initial-state*))
    (setf *initial-state* (delete state *initial-state*))))


(defun substitute-twin-state (state)
  "Overlay this state with one of its twin states."
  (let ((twin (pop (state-twins state)))
	(sim (state-sim state)))
    ;;(format *qsim-report* "~%Twin ~a overlaying ~a" twin state)
    (setf (state-qvalues state)    (state-qvalues twin)
	  (state-qspaces state)    (state-qspaces twin)
	  (state-cvalues state)    (state-cvalues twin)
	  (state-status  state)    (state-status  twin)
	  (state-successors state) (state-successors twin)
	  (state-text state)       (state-text twin)
	  (state-time-label state) (state-time-label twin))
    ;; Put twin state on agenda to be simulated.
    (push state (sim-agenda sim))))


;;; REINIT-FOR-STATE caches the "--" slots for a state.  The code has been
;;; modified from constraint-net-for-state because time has a qspace but no
;;; qvalue in an incomplete state (so the mapping fn there was off by one.)

(defun reinit-for-state (nstate)
  (install-corresponding-values nstate)
  ;; Install qspaces and qvals in constraint network. 
  (loop for (varname . qspace) in  (state-qspaces nstate)
	with qvals = (state-qvalues nstate)
	for qval = (alookup varname qvals)
	for var = (qval-variable qval)
	do (setf (variable--qspace var) qspace
		 (variable--qval   var) qval))
  ;; Activate/deactivate constraints within modes. 
  (activate-moded-constraints (state-qde nstate)))



;;;*****************************************************************************
;;; Functions that deal with abstraction level
;;;*****************************************************************************

;;;-----------------------------------------------------------------------------
;;;MAKE-BASIC-LEVEL defines the basic abstraction level.  Every state created
;;;during simulation belongs to that level.  It is called by
;;;INITIALIZE-DISPLAY-BLOCK.
;;;GET-BASIC-LEVEL returns the current basic level from a state or a display
;;;bloc.
;;;(BASIC-LEVEL-P <level>) is T if <level> is a basic level (i.e. its position
;;;is 0).
;;;-----------------------------------------------------------------------------

(defun make-basic-level (state &aux (name (genname 'level)))
  (set name (make-level :name name
                        :description (format nil "~a(0):Basic level" name)
                        :position 0
                        :states (list state))))

(defun get-basic-level (state-or-display-block)
  (etypecase state-or-display-block
    (state (display-block-basic-level (state-display-block state-or-display-block)))
    (display-block   (display-block-basic-level state-or-display-block))))

(defun basic-level-p (level)
  ;;(if level (zerop (level-position level)) t))
  (zerop (level-position level)))


;;;-----------------------------------------------------------------------------
;;; (MAKE-NEW-LEVEL <lower-level> <reason>) creates a new level
;;; on top of <lower-level>.
;;;-----------------------------------------------------------------------------

(defun make-new-level (lower-level reason)
  (let* ((name (genname 'level))
         (position (1+ (level-position lower-level)))
         (description  (format nil "~a(~a):~a/~a"
			       name position reason (level-name lower-level))))
    (set name (make-level :name name
                          :description description
                          :position
			  position))))


;;;-----------------------------------------------------------------------------
;;; Utilities to show and set levels
;;;-----------------------------------------------------------------------------

(defun show-levels (&optional (display-block *current-display-block*))
  (display-block-levels display-block))

(defun set-level (n)
  (setf *abstraction-level* (nth n (show-levels))))


;;;--------------------------------------------------------------------------------
;;;(CHECK-LEVEL <state> <level>) returns T if <state> belongs to <level> and 
;;;produces an error otherwise.
;;;--------------------------------------------------------------------------------

; moved to qdefs.lisp
;(defparameter *check-level* t)

(defun check-level (state &optional (level *abstraction-level*))
  ;;(setq level (or level (get-basic-level state)))
  (when (and *check-level* (not (member level (state-levels state))))
    (error "State ~a not in level ~a" state level))
  level)


;;;================================================================================
;;;Some functions to access successors, predecessors, coarsening, refinings
;;;slots safely.  Should the syntax of those slots change, only those functions
;;;have to be redefined.
;;;================================================================================

;;;-----------------------------------------------------------------------------
;;; (GET-SUCCESSORS <state> <level>) returns the successors of <state> in the
;;; level <level>. It first checks that <state> belongs to <level>. Similar
;;; definition for GET-PREDECESSORS.
;;;-----------------------------------------------------------------------------
 
(defun get-successors (state &optional (level *abstraction-level*))
  (cond ((typep state 'aggregate-interval)
	 (list (equivalence-node-abstract-state (car (aggregate-interval-cur-level state)))))
	((and *filter-occ-branching*
	      (state-aggregates state)
	      (aggregate-interval-cur-level (state-aggregates state)))
	 (list (state-aggregates state)))
	(t 
	 ;;(setq level (check-level state level))
	 (check-level state level)
	 (if (basic-level-p level)
	     (successor-states state)
	     (alookup level (state-new-successors state))))))

(defun get-predecessors (state &optional (level *abstraction-level*))
  ;;(setq level (check-level state level))
  (check-level state level)
  (if (basic-level-p level)
      (state-predecessors state)
      (alookup level (state-new-predecessors state))))


(defun get-ancestors (state)
  (cond ((null state) nil)
	(t (let ((preds (get-predecessors state)))
	     (append (mapcan #'get-ancestors preds) preds)))))

(defun get-coarsening (state &optional (level *abstraction-level*))
  (alookup level (state-coarsening state)))

(defun get-refinings (state)
  (state-refinings state))


;;;================================================================================
;;;Some functions to set successors, predecessors, coarsening, refinings
;;;slots safely.  Should the syntax of those slots change, only those functions
;;;have to be redefined.
;;;================================================================================

(defmacro set-state-level-slot (slot state value level)
  (let ((accessor (read-from-string (format nil "state-~a" slot))))
    `(let ((elts (assoc ,level (,accessor ,state))))
       (if elts (setf (cdr elts) ,value) 
	     (push (cons ,level ,value) (,accessor ,state))))))

(defun set-successors (state successors &optional (level *abstraction-level*))
  (if (basic-level-p level)
      (setf (state-successors state) (cons 'successors successors))
      (set-state-level-slot new-successors state successors level)))

(defun set-predecessors (state predecessors &optional (level *abstraction-level*))
  (if (basic-level-p level)
      (setf (state-predecessors state) predecessors)
      (set-state-level-slot new-predecessors state predecessors level)))

(defun set-coarsening (state coarsening &optional (level *abstraction-level*))
  (set-state-level-slot coarsening state coarsening level))

(defun set-refinings (state refinings)
  (setf (state-refinings state) refinings))

(defun set-state-neighbours (state preds succs &optional (level *abstraction-level*))
  (set-successors state succs level)
  (set-predecessors state preds level))


