;;; -*- Mode: LISP; Syntax: Common-lisp; Package: QSIM; Base: 10 -*-
;;;  $Id: qrules.lisp,v 1.1 1991/03/26 21:38:06 clancy Exp $

;;; Author: Adam Farquhar, October 1989.
;;;
;;; Code to take a set of rules and create compiled lisp code to
;;; efficiently execute them.
;;;
;;;======================================================================
;;; Contains three parts:
;;; 1. A list-encoding discrimination net constructor.  
;;;
;;; 2. A dnet translator.  It takes a dnet and produces a lisp expression
;;; which has the same meaning.  
;;;
;;; 3. A rule compiler, which takes a set of rules, builds a dnet out of
;;; the rule-heads, translates this into lisp code, builds a function
;;; out of the code, and compiles it with the lisp compiler.  Actually
;;; the compilation functions are more examples than anything else. They
;;; show how you can customize for your rule syntax, semantics, and
;;; desired results.
;;;
;;;
;;; NOTES:
;;; A variable is any atom begining with "?" or "_".  E.g. ?x or _1.
;;;
;;; "Rules" are only allowed to have a single antecedent, although it
;;; would be possible to modify the code slightly to allow for more
;;; complex formats.
;;;
;;;======================================================================
;;;

(defun variable? (pat)
  (and (symbolp pat)
       (let ((char (schar (string pat) 0)))
	 (or (char= char #\?)
	     (char= char #\_)))))

;;;======================================================================
;;; 1. DNETS
;;;
;;; I strongly recommend seeing Charniak & McDermott's AI programing for
;;; a nice discussion of DNETS.
;;;
;;; A simple dnet using list-encoding. I.e. a list (a ... z) is encoded
;;; as "*start* a ... z *end*".  This does not allow for last-segment
;;; matching, but can easily be modified by using a final *dot*
;;; notation.  It gives a much more compact notation than *cons*
;;; notation, and now that I've written it, it doesn't seem to be much
;;; more complicated.
;;;
;;; I use a flag, FINALP.  This is T when we are working on the top
;;; level of a pattern. I.e. if we reach the end of the pattern, we
;;; should either return some contents, or add an item to the contents.
;;; This allows for recursive calls for installing sub-parts of a
;;; pattern.
;;;
;;; DESIGN DECISION: there is a question as to how one should deal with
;;; variables.  There are two extremes: treat every variable as the
;;; SAME, or treat every distinct variable as different. C&McD take the
;;; first route, using a *VAR* link, and then leave disambiguation to
;;; a simple matcher after the small subset of potentially matching
;;; objects have been found.  I am tempted to fork whenever I hit
;;; multiple possibilities for matching an old var, as well as for
;;; hitting a new var.  This way I can expand all of the tests as I go
;;; along, rather than tacking on a separate set for var matching at the
;;; end.  
;;; (?x ?y ?z) ; general
;;; (?x ?y ?y) ; specialization
;;; (?x ?y ?x) ; variant of the specialization
;;;

(defstruct (link (:type list))
  key						
  links
  plist)


(defmacro link-var-type (l)
  `(getf (link-plist ,l) :var-type))

;; Leaves have some contents, a list of items which were indexed under
;; them. McDermott points out that one can use the "links" slot, as the
;; leaves have no links, but I find it a little clearer to be redundant.
;;
(defmacro link-contents (l)
  `(getf (link-plist ,l) :contents))

(defun leaf-p (l)
  (null (link-links l)))

;; This could be optimized somewhat by using three slots:
;; <list> <var> . <atoms>.  That would save looking for the common
;; cases.  But it's not as crucial as with *cons* encoding.
;;
(defun get-link (key link)
  "Return the first sublink on link which matches the KEY."
  (assoc key (link-links link)))

(defun ensure-link (item key link finalp)
  "If finalp is T, return the contents of link's sublink which matches
   key,  else just return the sublink.  Create the link if it is not
   there."
  (let ((next (get-link key link)))
    (when (null next)
      ;; add it now
      (push (setq next (make-link :key key))
	    (link-links link)))
    (when finalp
      (push item (link-contents next)))
    next))

;;; NOTE: the same pattern with the same item will result in multiple
;;; copies of the item in the link.contents.  Just change the push in
;;; ensure-link to pushnew to change this.

(defun INDEX (item pat &optional (dnet (make-link :key '*top*)))
  "Add an item to the dnet, indexed under a pattern.  A fetch of an
  object will return all items which were indexed under it. Returns the
  top node of the net."
  (let ((*pattern-vars*))
    (declare (special *pattern-vars*))
    (index-1 item pat dnet t)
    dnet))

;; Insert pat into net.  CALL FROM INDEX, which binds *pattern-vars*.
;;
(defun index-1 (item pat link finalp)
  (declare (special *pattern-vars*))
  (cond ((variable? pat)
	 (let ((l (ensure-link item pat
			       (ensure-link item '*VAR* link nil)
			       finalp)))
	   (cond ((member pat *pattern-vars*)
		  (setf (link-var-type l) 'OLD))
		 (T (setf (link-var-type l) 'NEW)
		    (push pat *pattern-vars*)))
	   l))
	((atom pat)
	 (ensure-link item pat
		      (ensure-link item '*ATOM* link  nil)
		      finalp))
	((consp pat)
	 (index-rest item pat
		     (ensure-link nil '*START* link  nil)
		     finalp))
	(T (error "~a is not a variable, atom or list."))))

;;(setq x (index 3 '(a ?x ?y ?x) (index 2 '(a ?x ?y ?y) (index 1 '(a ?x ?y ?z)))))
;;
;; Index a list.  We have already inserted the *start*, so we just
;; continue along the cdr of pat.
;;
(defun index-rest (item pat link finalp)
  (cond ((null pat)
	 (ensure-Link item '*END* link finalp))
	(T  (index-rest item (cdr pat)
			(index-1 item (car pat) link nil)
			finalp))))

		
;;;======================================================================
;;;
;;;  TRANSLATE-NET  (dnet  initially   -return- &key (conjunction 'or))
;;;
;;;    dnet      - a list coded discrimination net generated by INDEX.
;;;
;;;    initially - a var which (at run time) will be bound to the datum.
;;;
;;;    -return-  - a function (link pattern-vars) which should return
;;;                <code>.  The <code> will be executed when the datum
;;;                matches a full path through the dnet.  It will be
;;;                executed in a lexical environment in which all of the
;;;                pattern-vars are bound to the corresponding parts of
;;;                the datum. See translate-rules-fire-one for an example.
;;;
;;;    conjunction If there are multiple possibilities for following a
;;;                link, then the code for them is joined by
;;;                conjunction.  The default, OR, will select the first
;;;                match.  PROGN will execute all matches. 
;;;
;;;  Take a dnet and turn it into a lisp form which, if embedded in a
;;;  lambda, could be funcalled on a datum to determine if it matches
;;;  any of the patterns in the dnet.  The datum must be ground!  E.g.
;;;  if the dnet contains (a) (b), then we might translate this into:
;;;  (and (listp datum)
;;;       (<conjunction>
;;;           (and (eql (car datum) 'a) (null (cdr datum)) <result1>)
;;;           (and (eql (car datum) 'b) (null (cdr datum)) <result2>)))
;;;
;;;  The functions compile-xxx wrap a lambda around this and, perhaps,
;;;  invoke the lisp compiler on the result.
;;; 
;;;  The generated code is pretty good, it has alot of unnecessary ANDs,
;;;  but a decent lisp compiler should optimize these away.
;;;
;;;  NOTES on translate-net:
;;;  1. COMP returns the compilation for a single possibility.
;;;  2. COMPS returs the compilation for what might be a choice.
;;;  3. There are two sets of lexical variables which need to be kept
;;;  track of:
;;;     A. The pattern variables.  At any point in matching against the
;;;     datum, the lexical environment will include all of the currently
;;;     bound pattern variables.
;;;     B. As each new list is encountered, a new lexical variable is
;;;     introduced which will be set to this list.  This allows list
;;;     elements to be referenced with a single offset. E.g. 
;;;     (a (b (c d)) e)
;;;      X1 X2 X3
;;;     a = (nth 0 x1), b = (nth 0 X2), x3 = (nth 2 x1), e = (nth 2 e)
;;;  VAR is the current Xi, I is the current offset, VARS is an alist of
;;;  (var idx) which is used as a stack.  The code is much simpler than
;;;  the description!

(defun translate-net (dnet initially -return- &key (conjunction 'or))
  (labels
    ((COMP (link accessor var i vars pattern-vars )
       (let ((key (link-key link))
	     (acc (accessor accessor var i)))
	 (case key
	   (*VAR*
	     (comps link 'nth var  i vars pattern-vars))

	   (*START*
	     ;; The start of a list
	     (let ((newvar (gensym "X")))
	       `(let ((,newvar ,acc))
		  (and (consp ,newvar)
		       ,(comps link 'nth newvar 0 (acons var i vars) pattern-vars)))))

	   (*END*
	     ;; The end of a list.  We need to pop our stack of vars.
	     `(and (null ,acc)
		   ,(comps link  'nth (caar vars)
			   (1+ (cdar vars)) (cdr vars) pattern-vars)))

	   (*ATOM*
	     `(and (atom ,acc)
		   ,(comps link 'nth var  i vars pattern-vars)))
	   (OTHERWISE
	     (case (link-var-type link)
	       (NEW 
		 `(let ((,key ,acc))
		    ,(comps link 'nth var (1+ i) vars (cons key
							    pattern-vars))))
	       (OLD 
		 `(and (equalp ,acc ,key)
		       ,(comps link 'nth var (1+ i) vars pattern-vars )))
	       (OTHERWISE 
		 `(and (eql ,acc ',key)
		       ,(comps link 'nth var (1+ i) vars pattern-vars)))))
	   )))

     (COMPS (link acc var i vars pattern-vars )
       (cond ((leaf-p link)
	      (funcall -return-  link pattern-vars))
	     (T (let ((code (mapcar #'(lambda (sub)
					(comp sub acc var i vars pattern-vars ))
				    (link-links link))))
		  (if (= (length code) 1)
		      (car code)
		      `(,conjunction ,@code)))))))

    (comps dnet initially initially 0 () ())
    ))

(defun accessor (type var i)
  (if (eq type 'nth)
      `(nth ,i ,var)
      type))



;;;======================================================================
;;;
;;; LINKING TO RULES
;;;
;;; We can use TRANSLATE-NET to generate a program to find a single
;;; match, using OR as the :conjunction, or we can find all of the
;;; matches using PROGN.  It would also be easy to find the first N, or
;;; whatever, just by writing a custom conjunction.
;;;
;;; We define
;;; TRANSLATE-RULES-FIRE-ONE (rules) 
;;;     which returns the result of the first match, or FAILED if no
;;;     pattern matched. 
;;;
;;; TRANSLATE-RULES-FIRE-ALL (rules)
;;;    which returns a list of (bindings rule-contents) with one member
;;;    for every pattern which matched.  
;;;
;;; These two functions provide obvious templates for different rule
;;; syntaxes, and different sorts of actions upon matching.
;;;
;;; They return a lambda, so the COMPILE should not be taken too
;;; seriously.  The results should be compiled by the lisp compiler, as
;;; in the function:
;;; 
;;; LISP-COMPILE-RULES (rules)
;;;    which compiles "fire-one" rules.

(defun LISP-COMPILE-RULES (rules)
  (compile nil (translate-rules-fire-one rules)))

(defun TRANSLATE-RULES-FIRE-ONE (rules)
  (labels ((return-bindings (link pattern-vars)
	     `(return-from RULES
		(values T
			,(binding-list pattern-vars)
			',(link-contents link)))))
    `(lambda (datum)
       (block RULES
	 (multiple-value-bind (success bindings body)
	     ,(translate-net  (dnet rules) 'datum  #'return-bindings)
	   (if success
	       bindings
	       'failed))))))

(defun TRANSLATE-RULES-FIRE-ALL (rules)
  (labels ((return-all (link pattern-vars)
	     `(push (list ,(binding-list pattern-vars)
			  ',(link-contents link))
		    results)))
    `(lambda (datum)
       (let ((results))
	 ,(translate-net (dnet rules) 'datum #'return-all :conjunction 'progn)
	 results))))


;;;----------------------------------------------------------------------
;;;
;; Turn a set of rules into a discrimination net.  Hack this for your
;; private rule syntax.
;;
(defun DNET (rules)
  (let ((dnet (make-link :key '*top*))
	(vars nil))
    (dolist (rule rules)
      (setq vars (ensure-same-var-order vars (car rule) rule))
      (index rule (car rule) dnet))
    dnet))

(defun ensure-same-var-order (vars head rule)
  "Ensure that the variables in HEAD are in the same order as the
   variables in VARS. The list of all vars mentioned in order."
  (let ((rule-vars (expression-vars head)))
    (cond ((prefix rule-vars vars) vars)
	  ((prefix vars rule-vars) rule-vars)
	  (T (error "When compiling the rules into a discrimination net.~@
                     The rule heads must introduce variables in the same order.~@
                     I.e. (a ?x (b ?y)) and (foo ?x ?y). ~@
                     The rule: ~a ~@
                     does not match the order of variables in the previous~@
                     rules: ~a.  Use (order-pattern-vars rule) on all your rules~@
                     to automatically replace ?x ?y ... with ?1 ?2 ..."
		    rule vars)))))

(defun prefix (pre list)
  (cond ((null pre) T)
	((eql (car pre) (car list))
	 (prefix (cdr pre) (cdr list)))))

(defun expression-vars (exp)
  "Return a list of all of the variables in EXP."
  (labels
    ((exp-vars (exp vars)
       (cond ((null exp) vars)
	     ((variable? exp)
	      (pushnew exp vars))
	     ((atom exp) vars)
	     (T (exp-vars  (cdr exp)
			   (exp-vars (car exp) vars))))))
    (nreverse (exp-vars exp ()))))

(defun rename-vars-in-rules (rules)
  (mapcar #'ORDER-PATTERN-VARS rules))

(defun ORDER-PATTERN-VARS (exp)
  "Rename the variables in EXP so that they are called ?1 ?2..."
  (let ((alist nil)
	(i 0))
    (labels
      ((rename (exp)
	 (cond ((null exp) nil)
	       ((variable? exp)
		(or (cdr (assoc exp alist))
		    (let ((sym (intern
				 (concatenate 'string "?"
					      (write-to-string (incf i))))))
		      (setq alist (acons exp sym alist))
		      sym)))
	       ((atom exp) exp)
	       (T (cons (rename (car exp))
			(rename (cdr exp)))))))
      (rename exp))))

;; A helpful function. Given a list of pattern vars, (?x ?y ?z), return
;; code to create an alist of bindings in the lexical environment where
;; the vars are bound. I.e.
;; (list (cons '?x ?x) (cons '?y ?y) (cons '?z ?z))
;;
(defun BINDING-LIST (pattern-vars)
  `(list ,@(mapcar #'(lambda (var)
	      `(cons ',var ,var))
	  pattern-vars)))
