;;; -*- Mode:Common-Lisp; Package:QSIM; Syntax:COMMON-LISP; Base:10;  -*-

;;; $Id: nsim-interval.lisp,v 1.5 1992/07/08 12:55:52 clancy Exp $

(in-package 'qsim)

(defparameter *RK-RELERR* 1.e-9)
(defparameter *RK-ABSERR* 0.0)

(defother nsim-status sVector2)

;;;----------------------------------------------------------------------
;;; Take an nsim-struct and simulate it out until new-time.
;;; Inputs:  struct      - An nsim-struct.
;;;          new-time    - The time to simulate until.
;;;          start       - A starting time (need only on first call).
;;;          simStep     - A simulation step (first call only).
;;;          q2Intersect - T if state vector results are to be
;;;                        intersected with Q2 (first call only).
;;; Returns: #1: sVector (or sVector2 when defined)
;;;          #2: oVector
;;; Notes:  This needs something to detect when the simulation is
;;;         no longer valid.
;;;         This assumes that we are using rk45, so stepsize chopping
;;;         is unnecessary.
;;;----------------------------------------------------------------------
;;;
(defun nsim-until (struct new-time 
		   &key (start 0) (simStep .1) (q2Intersect NIL))
  (let* ((status   (if (nsim-struct-status struct)
		       (nsim-struct-status struct)
		       ;; Create an nsim-status struct to control the simulation 
		       ;; if one doesn't already exist.
		       (nsim-status-init struct :start start :simStep simStep
					 :q2Intersect q2Intersect)))
	 (sVector  (nsim-status-sVector status))
	 (sVector2 (nsim-status-sVector2 status))
	 (oVector  (nsim-status-oVector status)))

    (when (> new-time (nsim-status-time status))
      (let ((cVector        (nsim-status-cVector status))
	    (rk             (nsim-status-rk status))
	    (other-function (nsim-status-other-function status)))

	;; Compute the new system values.
	(rk45-run sVector rk new-time)

	;; sVector2 will contain the intersected predictions between Q2
	;; and NSIM.  It is certainly safe to use the intersection for
	;; ovar computations, but I'm preventing them from being used
	;; in svar computations because I haven't shown that it is safe
	;; to do so yet.
	(when sVector2
	  (intersect-q2-and-nsim-predictions sVector sVector2))

	;; Compute the new other variables
	(nsim-compute-other-vars new-time (if sVector2 sVector2 sVector)
				 oVector cVector other-function)

	(setf (nsim-status-time status) new-time)))
    (values (if sVector2 sVector2 sVector) oVector)))

      

;;; Intersect the values in *boundVector* with the values in sVector and
;;; return the results in sVector2.
;;; Returns: sVector2.
;;;
(defun intersect-q2-and-nsim-predictions (sVector sVector2)
  (declare (special *boundVector*))
  (loop for i from 0 upto (1- (array-dimension sVector 0))
	do (setf (svref sVector2 i)
		 (if
		  (or (and (evenp i)
			   (< (svref sVector i) (svref *boundVector* i)))
		      (and (oddp i)
			   (> (svref sVector i) (svref *boundVector* i))))
		  (svref *boundVector* i)
		  (svref sVector i))))
  sVector2)


;;;----------------------------------------------------------------------
;;; Creates an nsim-status struct, initializes it, links it to its parent
;;; nsim-struct, and returns the status structure.
;;;----------------------------------------------------------------------
;;;
(defun nsim-status-init (struct 
			 &key (start 0) (simStep .1) (q2Intersect NIL))
  (let* ((num-svars       (length (nsim-struct-state-vars struct)))
	 (num-ovars       (length (nsim-struct-other-vars struct)))
	 (state-function  (nsim-struct-state-function struct))
	 (other-function  (nsim-struct-other-function struct))
	 (sVector         (nsim-struct-sVector struct))
	 (sVector2        (when q2Intersect
			    (make-array (list (* 2 num-svars))
;					:element-type 'float
					:initial-contents
					(loop for i from 0 upto (1- (array-dimension sVector 0))
					      collect (svref sVector i)))))
	 (oVector         (make-array (list (* 2 num-ovars))
;				      :element-type 'float
				      :initial-element 0.0))
	 (cVector         (nsim-struct-cVector struct))
	 (rk	          (rk45-init state-function (* 2 num-svars)
				     *rk-relerr* *rk-abserr*
				     start 1 cVector))
	 (status          (make-nsim-status
			   :nsim-struct    struct
			   :num-svars      num-svars
			   :sVector        sVector
			   :oVector        oVector
			   :cVector        cVector
			   :rk             rk
			   :other-function other-function
			   :time           start)
			  ))
    ;; sVector2 is stored in the other slot.
    (setf (nsim-status-sVector2 status) sVector2)

    ;; These shouldn't be here as they don't set things in the status struct.
    (setf (nsim-struct-status struct) status)
    (setf (nsim-struct-start     struct) start)
    (setf (nsim-struct-simStep  struct) simStep)

    ;; Compute the initial values for the other variables based on
    ;; the state variables.
    (nsim-compute-other-vars start sVector oVector cVector other-function)

    status))


;;;----------------------------------------------------------------------------
;;; Do a numerical simulation on a set of variables.
;;; Inputs:  struct-or-state - an nsim-struct (obsolete) or a state.
;;;          start           - the starting time.
;;;          stop            - the stop time.
;;;          simstep         - the simulation step.
;;;          q2Intersect     - T if the results should be intersected with Q2.
;;; Returns: none.
;;; Updates: the nsim-struct.
;;;----------------------------------------------------------------------------

(defun nsim (struct-or-state
	     &key (start 0) (stop 15) (simStep .1) (q2Intersect NIL))
  (let ((struct (if (state-p struct-or-state) (state-nsim-struct struct-or-state)
		                              struct-or-state)))
    (setf (nsim-struct-start     struct) start)
    (setf (nsim-struct-stop      struct) stop)
    (setf (nsim-struct-simstep  struct) simStep)

    ;; Clear any existing status struct by defining a new one
    (nsim-status-init struct :start start :simStep simStep :q2Intersect q2Intersect)

    ;; Set up a place to store the simulation results
    (setf (nsim-struct-results struct) (create-nsim-result-store struct))

    (do ((time start (+ time simStep)))
	((>= time stop))
      
      ;; compute the new values
      (multiple-value-bind (sVector oVector)
	  (nsim-until struct time)

	;; Store the results in the result-matrix and update the range vector.
	(store-nsim-record time sVector oVector NIL struct))
      )))


;;;----------------------------------------------------------------------------
;;; Compute the vals for the other-vars. 
;;; Inputs:  time      - The current time.
;;;          sVector   - The state vector for the extremal system.
;;;          oVector   - The other vector for the system.
;;;          cVector   - The constant vector.
;;;          fctn      - The lisp function that does the computation.
;;; Returns: oVector or NIL if no function is defined.
;;;----------------------------------------------------------------------------
;;;
(defun nsim-compute-other-vars (time sVector oVector cVector fctn)
  (when fctn
    (funcall fctn time sVector oVector cVector)
    oVector))



;;;--------------------------------------------------------------------
;;; Result storage mechanism.
;;;
;;; Results are conceptually stored as records of the form
;;; <time sVector oVector cVector>.  The following functions are defined :
;;; create-nsim-result-store (nsim-struct)
;;; store-nsim-record (time sVector oVector cVector nsim-struct)
;;; create-rs-pointer (time var nsim-struct)
;;; retrieve-nsim-record-var (time var bound nsim-struct)
;;; 

;;; Implementation
;;;
;;; The result-store is implemented as a vector with a fill pointer.
;;; The vector is partitioned into lengths of 2*numSvars+2*numOvars.
;;; The cVector is not stored, nor is the time.


;;;--------------------------------------------------------------------
;;; Create a result store for the given nsim-struct.
;;; Inputs:  ns - An nsim-struct.
;;; Returns: A result store.
;;;--------------------------------------------------------------------
;;;
(defun create-nsim-result-store (ns)
  (let ((num-svars (* 2 (length (nsim-struct-state-vars ns))))
	(num-ovars (* 2 (length (nsim-struct-other-vars ns))))
	(start     (nsim-struct-start ns))
	(stop      (nsim-struct-stop ns))
	(simStep   (nsim-struct-simStep ns)))
    (make-array (list (ceiling (* (/ (- stop start) simStep)
				  (+ num-svars num-ovars))))
		:element-type 'float
		:fill-pointer 0)))


;;;--------------------------------------------------------------------
;;; Store a record in the result-store for ns.
;;; Inputs:  time    - The time (ignored).
;;;          sVector - The state vector.
;;;          oVector - The other vector.
;;;          cVector - The constant vector (ignored).
;;;          ns      - The nsim-struct.
;;; Returns: nothing.
;;;--------------------------------------------------------------------
;;;
(defun store-nsim-record (time sVector oVector cVector ns)
  (declare (ignore time cVector))
  (let ((results (nsim-struct-results ns)))

    ;; Bump up the array size if we are out of space.
    ;; This happens when simulation is restarted after the initial
    ;; nsim call.
    (when (< (array-dimension results 0) (length results))
      (adjust-array results
		    (list (ceiling
			   (* (/ (- (nsim-struct-stop ns)
				    (nsim-struct-start ns))
				 (nsim-struct-simStep ns))
			      (+ (length (nsim-struct-state-vars ns))
				 (length (nsim-struct-other-vars ns))))))))

    (loop for i from 0 upto (1- (length sVector))
	  do (vector-push (aref sVector i) results))
    (loop for i from 0 upto (1- (length oVector))
	  do (vector-push (aref oVector i) results))
    ))


;;;--------------------------------------------------------------------
;;; Position an rspointer within a nsim result-store.
;;; See the file NQ:NQ;structures.lisp for the definition of an rspointer.
;;; Inputs:  var   - The variable name to find.
;;;          time  - The time (this is rounded).
;;;          ns    - The nsim-struct.
;;;          rs    - The rspointer.
;;; Returns: rs.
;;;--------------------------------------------------------------------
;;;
(defun position-rspointer (var time ns &key (rs (make-rspointer)))
  (assert (nsim-struct-p ns))
  (let* ((vartype (cond
		   ((member var (nsim-struct-state-vars ns))
		    'svar)
		   ((member var (nsim-struct-other-vars ns))
		    'ovar)
		   ((member var (nsim-struct-constant-vars ns))
		    'cvar)
		   (T
		    (error "~a is an unknown variable" var))))
	 (num-svars  (* 2 (length (nsim-struct-state-vars ns))))
	 (num-ovars  (* 2 (length (nsim-struct-other-vars ns))))
	 (total-vars (+ num-svars num-ovars))
	 (location (case vartype
		     (svar (position-in-vector var 'LB
					       (nsim-struct-state-vars ns)))
		     (ovar (+ num-svars
			      (position-in-vector var 'LB
					       (nsim-struct-other-vars ns))))
		     (cvar (position-in-vector var 'LB
					       (nsim-struct-constant-vars ns)))))
	 (results    (nsim-struct-results ns))
	 (simStep    (nsim-struct-simStep ns))
	 (start      (nsim-struct-start ns))
	 (stop       (nsim-struct-stop ns)))
    (if (or (< time start) (> time stop))
	(error "Selected time ~,3f is outside of the stored time indices ~
               (~,3f ~,3f)" time start stop)
	       
	(let ((index (round (/ (- time start) simStep))))
	  (setf (rspointer-type rs)           vartype)
	  (setf (rspointer-recordIndex rs)    index)
	  (setf (rspointer-recordVarIndex rs) location)
	  (setf (rspointer-recordLen rs)      total-vars)
	  (setf (rspointer-maxlen rs)         (length (nsim-struct-results ns)))
	  (setf (rspointer-results rs)        (if (eq vartype 'cvar)
						  (nsim-status-cVector
						   (nsim-struct-status ns))
						  results))
	  rs))))


;;;--------------------------------------------------------------------
;;; Retrieve the value for a given var and a given bound at a given time.
;;; Inputs:  rs     - An rspointer.
;;;          bound  - LB, UB, or (LB UB).
;;;          ns     - The nsim-struct.
;;; Returns: A list of bounds (in the order implied by bound) or NIL
;;;          if no such record exists.
;;;--------------------------------------------------------------------
;;;
(defun retrieve-nsim-record-var (rs bound &key (increment NIL))
  (assert (rspointer-p rs))
  (let ((type     (rspointer-type rs))
	(results  (rspointer-results rs))
	(index    (rspointer-recordIndex rs))
	(varIndex (rspointer-recordVarIndex rs))
	(recLen   (rspointer-recordLen rs))
	(maxLen   (rspointer-maxLen rs))
	(entry    NIL))
    (cond
     ((>= index maxLen)
      NIL)
     (T
      (when (atom bound)
	(setf bound (list bound)))
      (setf entry
	    (mapcar #'(lambda (b)
			(aref results
			      (+ (if (eq type 'cvar) 0 index)
				 varIndex
				 (if (eq b 'LB) 0 1))))
		    bound))
      (when increment
	(setf (rspointer-recordIndex rs) (+ index recLen)))
      entry))))



;;; These are various functions of use to applications that use nsim-until.
;;;


;;; This one was inspired by Dan Dvorak
;;;----------------------------------------------------------------------------
;;;  Function:  get-nsim-range
;;;
;;;  Given:     -- name of a state variable;
;;;             -- the nsim-struct
;;;
;;;  Returns:   a 2-element list (LB UB), where LB is the numeric lower-bound
;;;             and UB is the numeric upper-bound.
;;;----------------------------------------------------------------------------
;;;
(defun get-nsim-range (varname struct)
  (assert (nsim-struct-p struct))
  (multiple-value-bind (type vector pos)
      (nsim-var-type varname struct)
    (declare (ignore type))
    (list (aref vector pos) (aref vector (1+ pos)))))


;;;----------------------------------------------------------------------------
;;; Get the type info and location of a variable.
;;; Inputs:  varname - The name of the variable.
;;;          ns      - An nsim-struct.
;;; Returns: #1: The type - SVAR,CVAR,OVAR.
;;;          #2: The vector containing the value.
;;;          #3: The position of the LB element for the variable.
;;;----------------------------------------------------------------------------
;;;
(defun nsim-var-type (varname ns)
  (assert (nsim-struct-p ns))
  (let ((type (cond
		((member varname (nsim-struct-state-vars ns)) 'svar)
		((member varname (nsim-struct-other-vars ns)) 'ovar)
		((member varname (nsim-struct-constant-vars ns)) 'cvar))))
    (when (null type)
      (error "Variable ~a is of unknown type in ~a" varname ns))
    (values type
	    (case type
	      (svar (nsim-status-sVector (nsim-struct-status ns)))
	      (ovar (nsim-status-oVector (nsim-struct-status ns)))
	      (cvar (nsim-status-cVector (nsim-struct-status ns))))
	    (position-in-vector varname 'LB
				(case type
				  (svar (nsim-struct-state-vars ns))
				  (ovar (nsim-struct-other-vars ns))
				  (cvar (nsim-struct-constant-vars ns))))
	    )))


;;;--------------------------------------------------------------------------
;;; Reset an nsim-status structure with new data.
;;; Inputs:  ns          - an nsim-status structure.
;;;          sVector     - a new vector of [LB1 UB1 LB2 UB2 ...] state var bounds.
;;;          sVector2    - ditto
;;;          cVector     - ditto, for constants.
;;;          new-svals   - a LIST of values for each element of sVector.
;;;                        a NIL indicates that an old value should be kept.
;;;          start-time  - a new start time.
;;; Returns: nothing, but resets slots in ns.
;;;--------------------------------------------------------------------------
;;;
(defun reset-nsim-status (ns
			  &key sVector sVector2 cVector start-time new-svals)
  (assert (nsim-status-p ns))

  ;; Note that by changing sVector and cVector only in the nsim-status struct,
  ;; they are no longer the same vectors as are stored in the nsim-struct.
  ;; This should not be a problem as their values in the nsim-struct are
  ;; really only used to pass info from nsim-initialize to nsim-until.
  (when sVector
    (setf (nsim-status-sVector ns) (copy-vector sVector)))
  (when sVector2
    (setf (nsim-status-sVector2 ns) (copy-vector sVector2)))
  (when cVector
    (setf (nsim-status-cVector ns) (copy-vector cVector)))
  (when start-time
    (setf (nsim-status-time ns) start-time))
  (when new-svals
    (replace-array-elements (nsim-status-sVector ns) new-svals))
  (rk45-reset (nsim-status-rk ns))
  )


;;;----------------------------------------------------------------------------
;;; Reset an rk structure.
;;; This is better than building a new one 'cuz it doesn't generate
;;; throwaway structures and it requires no knowledge of parameters.
;;; Inputs:  rk - a rk structure.
;;; Returns: none.
;;;----------------------------------------------------------------------------

(defun rk45-reset (rk)
  (dotimes (n (rk-neqn rk))
    (setf (aref (rk-yp rk) n) 0.0)
    (setf (aref (rk-f1 rk) n) 0.0)
    (setf (aref (rk-f2 rk) n) 0.0)
    (setf (aref (rk-f3 rk) n) 0.0)
    (setf (aref (rk-f4 rk) n) 0.0)
    (setf (aref (rk-f5 rk) n) 0.0))
  (setf (rk-h rk) 0.0)
  (setf (rk-savre rk) 0.0)
  (setf (rk-savae rk) 0.0)
  (setf (rk-nfe rk) 0)
  (setf (rk-kop rk) 0)
  (setf (rk-init rk) 0)
  (setf (rk-jflag rk) 0)
  (setf (rk-kflag rk) 0)
  (setf (rk-iflag rk) 1))


;;;-------------------------------------------------------------
;;; Replace the elements in an array.
;;; If a replacing element is NIL, then don't replace it.
;;; Inputs:  array        - an array of numbers.
;;;          element-list - a list of replacement elements.
;;; Returns: none.
;;;-------------------------------------------------------------

(defun replace-array-elements (array element-list)
  (do ((count 0            (incf count))
       (el    element-list (cdr el)))
      ((null el) nil)
    (when (car el)
      (setf (aref array count) (car el)))))



;;;-------------------------------------------------------------
;;; Return a copy of vec1
;;;-------------------------------------------------------------
;;;
(defun copy-vector (vec1)
  (let* ((len   (array-dimension vec1 0))
	 (vec2  (make-array (list len))))
    (dotimes (i len)
      (setf (svref vec2 i) (svref vec1 i)))
    vec2))
