;;; -*- Mode:Common-Lisp; Package:QSIM; Syntax:COMMON-LISP; Base:10;  -*-

;;; $Id: nsim-init.lisp,v 1.4 1992/07/17 13:21:50 bert Exp $

(in-package 'qsim)

;;; This file contains the initialization and equation-building functions.
;;;
;;; The main function NSIM-INITIALIZE sets up the equations and does a 
;;; syntax check to
;;; be sure that everything needed to do a simulation has been supplied.
;;; It creates an NSIM-STRUCT and attaches it to each state in the list.

;;; Initialization of the equations requires that the initial ranges for at 
;;; least the state variables are given.  There are two approaches to dealing
;;; with this :
;;;
;;;  1) Wait until Q2 has decided what the initial values of the state vars
;;;     are.  This approach is very general in that unanticipated initial
;;;     states are handled OK.  The problem is that if the computed initial
;;;     range is very large, it may be unusable.  Also, this initial guess 
;;;     could be further shrunk by the end of the simulation if it was not
;;;     an initial range.
;;;
;;;  2) Require that all state vars have an initial-ranges statement.  This 
;;;     means that if a non-range specified initial state is specified, it
;;;     will not be acceptable.  This leads to a problem with transition 
;;;     regions as the new qde state may want to inherit the qmags from the
;;;     previous qde and these values will not be in the initial-ranges. 
;;;     [I need to figure out what happens in detail in this case.  Maybe
;;;      the assert-ranges option will save me.]
;;;
;;; Right now, I'm using method 1.
;;; 

;;; SEE NQ;STRUCTURES.LISP FOR A DESCRIPTION OF THE NSIM STRUCTURES.

;;; Result storage :
;;;
;;; Variables are divided into two classes : state and other.
;;; State vars need a diffeq solver to compute next values.  Other vars
;;; can be computed from the state vars.
;;; The state-function slot holds a compiled lisp function that computes
;;; the derivatives of the extremal system which consists of the minimal
;;; and maximal equations for each state variable.
;;; The state vector is of length 2*n where n is the order of the QDE.
;;; the vector contains LB UB versions of each variable in the state-vars
;;; slot in the same order as they occur in that slot.
;;; The other-function also consists of a compiled lisp function
;;; It takes as input a vector of state vars.  Its output vector also contains
;;; a LB and UB for each variable in the other-vars slot.


(defvar *use-dynamic-envelopes* nil)
(defvar *trace-nsim* nil)

;;;--------------------------------------------------------------------------
;;; Create an initial nsim-struct for the state.
;;; Inputs:  state-or-states - a state or states (possibly incomplete).
;;;          simvars         - A list of vars to simulate (or :all).
;;; Returns: none.
;;; Updates: the state nsim-struct slot.
;;; Notes:   - Only get initial vals for state-vars.  Other-vars ranges will
;;;            be computed automatically.
;;;          - All state vars are always simulated, even if not specified
;;;            in simvars.
;;;--------------------------------------------------------------------------
;;;
(defun nsim-initialize (state-or-states &key (simvars :all))
  (when *use-dynamic-envelopes*
    (if (listp state-or-states)
	(dolist (state state-or-states)
	  (nsim-init-completions state simvars))
	(nsim-init-completions state-or-states simvars))))


;;;--------------------------------------------------------------------------
;;; Given a (possibly incomplete) state, generate an nsim-struct for it.
;;; Inputs:  state     - A qsim state.
;;;          simvars   - A list of vars to simulate (or :all).
;;; Returns: Nothing.
;;; Notes:  This should be replaced so that the function forms are computed
;;;         only once but the constants are computed for each initialization.
;;;         The above comment is wrong, since the actual form of the equations
;;;         will be determined by the initial conditions in the case of QDEs
;;;         with mult constraints since they are nonmonotonic.
;;; Implementation idea: Additionally, to determine the initial signs of the
;;;                state vars, simulation through T1 may be necessary.
;;;--------------------------------------------------------------------------
;;;
(defun nsim-init-completions (state simvars)
  (if (eq (first (state-successors state)) 'Completions)
      (dolist (s (cdr (state-successors state)))
	(nsim-init-completions s simvars))
      (nsim-init state simvars)))


;;; Note : I'm assuming that a constraint name is exactly the list shown
;;;        in the QDE for the constraint.
;;;
(defun nsim-init (state simvars)
  (let ((*qde*          (state-qde state))
	(*state*        state)
	cons
	state-function
	other-function
	exprTable
	*svars* *cvars* *ovars*
	*sVector* *cVector*)
    ;; These lisp vars are made dynamic since are set once in this function
    ;; and are used in a lot of the calling functions, but I don't
    ;; want to pass them around all the time.
    (declare (special *state* *qde* *svars* *cvars* *ovars* *sVector* *cVector*))

    ;; First ensure that Q2 has been run on the state.
    (quantitative-range-reasoning *state*)

    ;; Partition the variable space of the qde and get the active constraints.
    (multiple-value-setq (*svars* *ovars* *cvars* cons)
	(partition-qde *qde*))
      
    ;; Check the simvars list to make sure that it contains only ovars
    ;; right now, rather than waiting until later.
    (setf simvars
	  (if (eq simvars :all)
	      *ovars*
	      (mapcan #'(lambda (s)
			  (cond
			   ((member s *svars*) NIL)
			   ((member s *cvars*) NIL)
			   ((member s *ovars*) (list s))
			   (T
			    (error "~a in :simvars is not a QDE variable"
				   s))))
		      simvars)))

    ;; Create expressions for every variable in the QDE in terms of state
    ;; variables and constants.
    (setf exprTable (create-exprTable cons *svars* *ovars* *cvars*))
    (create-expressions exprTable)

    ;; Normalize the expressions (this means expanding sum-zero right now).
;    (normalize-expressions exprTable)
    (when *trace-exprtable-gen*
      (let ((*detailed-printing* T))
	(format t "~%Normalized Exprtable:~%~A" exprTable)))

    ;; Build the sVector and cVector.
    (setf *sVector*
	  (make-array (list (* 2 (length *svars*)))
		      ;; can't make a svrefable vector with an element type other
		      ;; than T.
;		      :element-type 'float   
		      :initial-contents
		      (mapcar #'float
			      (build-boundlist-from-state *svars*))))
    (setf *cVector*
	  (make-array (list (* 2 (length *cvars*)))
;		      :element-type 'float
		      :initial-contents
		      (mapcar #'float
			      (build-boundlist-from-state *cvars*))))
  
    ;; Define lisp functions that plug into the numerical simulator.
    (setf state-function (build-extremal-system exprTable))

    ;; Compile a lisp function to compute the other vars from the
    ;; svars.
    (setf other-function
	  (if simvars
	      (build-ovars-equations simvars exprTable)
	      nil))
      
    (setf (state-nsim-struct state)
	  (make-nsim-struct :qde               *qde*
			    :state-vars        *svars*
			    :other-vars        simvars
			    :constant-vars     *cvars*
			    :state-function    state-function
			    :other-function    other-function
			    :sVector           *sVector*
			    :cVector           *cVector*
			    ))))

;;;-------------------------------------------------------------------
;;; Given a set of variables, return a vector of LB UB values.
;;; Inputs:  vars - A list of variable names.
;;; Returns: A list of (LB1 UB1 LB2 UB2 ...) for a total of 2*(length vars)
;;;          entries.
;;; Notes:   The call to get-current-range assumes that *state* is
;;;          a special variable.
;;;-------------------------------------------------------------------
;;;
(defun build-boundlist-from-state (vars)
  (mapcan #'(lambda (var)
	      (let ((bound (get-current-range var)))
		(if (and (numberp (first bound))
			 (numberp (second bound)))
		    (list (first bound) (second bound))
		    (error "Can't map infinities for bound ~a on variable ~a"
			   bound var))))
	  vars))




;### take this out!
;;;-------------------------------------------------------------------
;;; Transform a set of general expressions into normal expressions.
;;; Right now, this means replacing any operators other than
;;; +, -, /, *, FUNCTION, or INVERSE with combinations of these operators.
;;; Inputs:  exprTable - the exprTable for the QDE.
;;; Returns: The exprtable subexpr slots are updated to a
;;;          normalized form.
;;;-------------------------------------------------------------------
;;;
(defun normalize-expressions (exprTable)
  (dolist (entry exprTable)
    (setf (exprTableEntry-subexpr entry)
	  (normalize-subexpr (exprTableEntry-subexpr entry))))
  exprTable)


;;; The only operator that needs changing right now is sum-zero.
;;; Perhaps this should be done in the code in numeric-eqn-gen.lisp?
;;;
(defun normalize-subexpr (subexpr)
  (cond
   ((null subexpr) NIL)
   ((atom subexpr) subexpr)
   (T
    (case (first subexpr)
      ((* / + -)
       `(,(first subexpr) ,.(mapcar #'normalize-subexpr (cdr subexpr))))
      ((FUNCTION INVERSE)
       `(,(first subexpr) ,(second subexpr) ,(normalize-subexpr (third subexpr))))
      (SUM-ZERO
       ;; This expands into (sum-zero a b c) into (- (+ a b c))
       `(- (+ ,.(mapcar #'normalize-subexpr (cdr subexpr)))))
      (T
       (error "Don't know how to normalize ~A" subexpr))))))


;;;-------------------------------------------------------------------
;;; Get the type of a normalized expression.
;;;-------------------------------------------------------------------
;;;
(defun expr-type (expr)
  (declare (special *svars* *ovars* *cvars*))
  (cond
    ((atom expr)
     (cond
       ((member expr *svars*)  'svar)
       ((member expr *cvars*)  'cvar)
       ((member expr *ovars*)  'ovar)
       (T                      (error "Unknown variable named ~a" expr))
       ))
    ((member (first expr) '(FUNCTION INVERSE))
     'function)
    ((and (eq (first expr) '-) (= (length expr) 2))
     'minus)
    (T
     (case (first expr)
       (+ 'addop)
       (- 'subop)
       (* 'mulop)
       (/ 'divop)
       (sum-zero 'sumop)
       (T (error "Unknown expression type ~a" expr))))))

						  

;;;-------------------------------------------------------------------
;;; Compile a lisp function for the given exprTable.
;;; The form of this function is
;;;      (lambda (tin y yp c)
;;;        (declare (ignore tin))
;;;        (setf (svref yp <pos>) <expression>)
;;;         ... )
;;; Inputs:  exprTable       - The exprTable for the QDE.
;;; Returns: A compiled lisp function.
;;;-------------------------------------------------------------------
;;;
(defun build-extremal-system (exprTable)
  (declare (special *svars* *cvars*))
  (let ((lisp-exprs
	 ;; Compute the extremal expressions
	 (loop for svar in *svars*
	       for v = (find svar exprTable :key #'exprTableEntry-varname)
	       nconc (loop for bound in '(LB UB)
			   collect (generate-extremal-expr (exprTableEntry-subexpr v)
							   bound
							   svar bound exprTable))))
	(ign-decl (if (null *cvars*)
		      '(declare (ignore tin c))
		      '(declare (ignore tin))))
	(fname (gentemp "F"))
	fctn)

    ;; Now build the lisp function and compile it.
    (setf fctn 
	  `(lambda (tin y yp c)
	     ,ign-decl
	     ,.(loop for expr in lisp-exprs
		     for i from 0 upto (1- (length lisp-exprs))
		     collect `(setf (svref yp ,i) ,expr))))

    (when *trace-function-gen*
      (format t "~%Creating function ~a" fname)
      (pprint fctn))
    (compile fname fctn)))


;;;-------------------------------------------------------------------
;;; Compile a lisp function for the given ovar entry in exprTable.
;;; The form of this function is
;;;      (lambda (tin y o c)
;;;        (declare (ignore tin))
;;;        (setf (svref o <pos>) <expression>)
;;;         ... )
;;; Inputs:  ovars     - The set of other vars to compute expresions for.
;;;          exprTable - The exprTable for the QDE.
;;; Returns: A compiled lisp function.
;;;-------------------------------------------------------------------
;;;
(defun build-ovars-equations (ovars exprTable)
  (declare (special *cvars*))
  (let ((lisp-exprs
	 ;; Compute the extremal expressions
	 (loop for ovar in ovars
;	       for v = (find ovar exprTable :key #'exprTableEntry-varname)
	       nconc (loop for bound in '(LB UB)
			   collect (generate-ovar-expr ovar
						       bound exprTable))))
	(ign-decl (if (null *cvars*)
		      '(declare (ignore tin c))
		      '(declare (ignore tin))))
	(fname (gentemp "F"))
	fctn)

    ;; Now build the lisp function and compile it.
    (setf fctn 
	  `(lambda (tin y o c)
	     ,ign-decl
	     ,.(loop for expr in lisp-exprs
		     for i from 0 upto (1- (length lisp-exprs))
		     collect `(setf (svref o ,i) ,expr))))
    (when *trace-function-gen*
      (format t "~%Creating function ~a" fname)
      (pprint fctn))
    (compile fname fctn)))



;;;-------------------------------------------------------------------
;;; Using the transformation table, take a "top-level" derivative
;;; expression and generate the extremal expression according to bound.
;;; Inputs:  dTvar       - the derivative ovar.
;;;          ebound      - LB or UB.
;;;          svar        - the state variable that the expression is for.
;;;          sbound      - the bound for svar.
;;;          exprTable   - the exprTable for the SQDE.
;;; Returns:  An evaluable lisp form that returns the value of svar under
;;;           the given bound.  Variables and constants are represented
;;;           as vector references.
;;;-------------------------------------------------------------------
;;;
(defun generate-extremal-expr (dTvar ebound svar sbound exprTable)
  (let (result-dTvar
	(*subexprs* NIL))
    (declare (special *subexprs*))
    (setf result-dTvar (extremize dTvar ebound svar sbound exprTable))
    (when *trace-nsim*
      (format *Qsim-Report* "~&~a extremal expr dTvar table for ~a :"
	      sbound svar)
      (pprint *subexprs*)
      (format *Qsim-Report* "~&expr is :")
      (pprint result-dTvar *Qsim-Report*)
      (format *Qsim-Report* "~%~%"))
    (instantiate-extremal-expr result-dTvar *subexprs* exprTable)))


;;;-------------------------------------------------------------------
;;; Using the transformation table, take the expression and generate
;;; an ovar equation based on bound.  This differs from generate-extremal-expr
;;; in that there is no svar that is not bounded according to the
;;; transformation table.
;;; Inputs:  ovar        - an ovar.
;;;          ebound      - LB or UB.
;;;          exprTable   - the exprTable for the SQDE.
;;; Returns: An evaluable lisp form that returns the value of svar under
;;;          the given bound.  Variables and constants are represented
;;;          as vector references.
;;;-------------------------------------------------------------------
;;;
(defun generate-ovar-expr (ovar ebound exprTable)
  (let (result-ovar
	(*subexprs* NIL))
    (declare (special *subexprs*))
    ;; '(NIL) is guaranteed not to match any svar.
    (setf result-ovar (extremize ovar ebound '(NIL) NIL exprTable))
    (when *trace-function-gen*
      (format *Qsim-Report* "~&~a ovar expr for ~a is ~a"
	      ebound ovar result-ovar))
    (instantiate-extremal-expr result-ovar *subexprs* exprTable)))


;;;-------------------------------------------------------------------
;;; Generate an extremal expression in a form where variables reference
;;; in the form (B var) and monotonic functions are in the form
;;; (B (FUNCTION/INVERSE name (B arg))).  This form is useful for
;;; debugging.
;;; Inputs:  subexpr     - a subexpression (possibly containing ovars).
;;;          ebound      - LB or UB.
;;;          svar        - a state variable.
;;;          sbound      - the bound for svar.
;;;          exprTable   - the exprTable for the system.
;;; Returns: A bound expression.
;;; Updates: The variable *subexprs* to include bounds for ovars.
;;;-------------------------------------------------------------------
;;;
(defun extremize (subexpr ebound svar sbound exprTable)
  (case (expr-type subexpr)
    
    (svar (if (eq subexpr svar) 
	      ;; The svar for the expression returns the bound expr for f_i
	      `(,sbound ,subexpr)

	      ;; Any other state var returns the proper bound
	      `(,ebound ,subexpr)))

    ;; Constants return their proper bound
    (cvar `(,ebound ,subexpr))

    ;; Ovars get looked up in *subexprs*
    (ovar (get-ovar-entry subexpr ebound svar sbound exprTable))
    
    (function (case (first (second subexpr))
	
		;; (FUNCTION/INVERSE (M+ X Y) arg) maps to
		;; (B (FUNCTION/INVERSE (M+ X Y) (B arg)))
		(M+  `(,ebound (,(first subexpr)
				,(second subexpr)
				,(extremize (third subexpr) ebound svar sbound exprTable))))

		;; (FUNCTION/INVERSE (M- X Y) arg) maps to
		;; (B (FUNCTION/INVERSE (M+ X Y) (Bbar arg)))
		(M-  `(,ebound (,(first subexpr)
				,(second subexpr)
				,(extremize (third subexpr) (bbar ebound) svar sbound exprTable))))

		(T (error "Don't know about function type in ~a" subexpr))))

    ;; (+ A B ... ) maps to (+ (B A) (B B) ...)
    (addop  `(+ ,.(mapcar #'(lambda (arg)
			      (extremize arg ebound svar sbound exprTable))
		   (cdr subexpr))))

    ;; (- A) maps to (- (Bbar A))
    (minus  `(- ,(extremize (second subexpr) (bbar ebound) svar sbound exprTable)))

    ;; (- A B ...) maps to (- (B A) (Bbar B) ...)
    (subop  (let ((first T))
	      `(- ,.(mapcar #'(lambda (arg)
				(extremize arg (cond
						 (first (setf first NIL)
							ebound)
						 (T     (bbar ebound)))
					   svar sbound exprTable))
		            (cdr subexpr)))))

    ;; (sum-zero A B ...) maps to (sum-zero (Bbar A) (Bbar B) ...)
    (sumop   `(sum-zero ,.(mapcar #'(lambda (arg)
				      (extremize arg (bbar ebound)
						 svar sbound exprTable))
			          (cdr subexpr))))
			 
    ;; (*/ A B) maps to (*/ LB/UB (LB A) (UB A) (LB B) (UB B))
    ((mulop divop)  (compute-muldiv-expr subexpr ebound svar sbound exprTable))

    (T              (error "Don't know how to extremize ~a" subexpr))
    ))


;;;--------------------------------------------------------------------------
;;; See if there is an entry for the ovar in *subexprs* and return (bound ovar)
;;; if it is.  Otherwise, make an entry and return (bound ovar).
;;; Inputs:  ovar      - an ovar.
;;;          bound     - the desired bound.
;;;          svar      - the state var for the current expression.
;;;          sbound    - the state var's bound.
;;;          exprTable - the exprTable for the system.
;;; Note: The form of *subexprs* is a list of (var LBexpr UBexpr)
;;;--------------------------------------------------------------------------
;;;
(defun get-ovar-entry (ovar bound svar sbound exprTable)
  (declare (special *subexprs*))
  (let ((subexpr-entry NIL))
    (cond
     ((setf subexpr-entry (assoc ovar *subexprs*)))
     (T
      (setf subexpr-entry (list ovar NIL NIL))
      (push subexpr-entry *subexprs*)))
    (when (null (nth (if (eq bound 'LB) 1 2) subexpr-entry))
      (setf (nth (if (eq bound 'LB) 1 2) subexpr-entry)
	    (extremize (exprTableEntry-subexpr (find ovar exprTable
						     :key #'exprTableEntry-varname))
		       bound svar sbound exprTable)))
    `(,bound ,ovar)))


(defun compute-muldiv-expr (expr ebound svar sbound exprTable)
  (let* ((alo-expr (extremize (second expr) 'LB svar sbound exprTable))
	 (ahi-expr (extremize (second expr) 'UB svar sbound exprTable))
	 (blo-expr (extremize (third expr) 'LB svar sbound exprTable))
	 (bhi-expr (extremize (third expr) 'UB svar sbound exprTable)))
    `(,(first expr) ,ebound ,alo-expr ,ahi-expr ,blo-expr ,bhi-expr)))


;;;--------------------------------------------------------------------------
;;; Multiply or divide two bounded expressions (A and B) using the bound 
;;; endpoints that give the most conservative bound.
;;; Inputs:  op       - The operator (LB or UB).
;;;          bound    - The desired bound (LB or UB).
;;;          lba, uba - The bounds on A.
;;;          lbb, ubb - The bounds on B.
;;; Returns: The requested bound on (* A B)
;;; Note:  This function is called inside the generated code.
;;;--------------------------------------------------------------------------
;;;
(defun nsim-muldiv (op bound lba uba lbb ubb)
  (declare (special *muldiv-bound-table*))
  (let ((Asign (interval-sign lba uba))
	(Bsign (interval-sign lbb ubb))
	(table (if (eq op '*) (first *muldiv-bound-table*)
		              (second *muldiv-bound-table*)))
	boundset entry)
    (when (and (eq op '/) 
	       (or (eq Bsign 'S)
		   (= lbb 0)
		   (= ubb 0)))
      (error "Interval for divisor in [~a ~a] contains 0." lbb ubb))
    (setf entry (alookup (list Asign Bsign) table :test #'equal))
    (cond
      (entry
       (setf boundset (if (eq bound 'LB) (first entry) (second entry)))
       (funcall op
		(if (eq (first boundset) 'LB) lba uba)
		(if (eq (second boundset) 'LB) lbb ubb)))
      (T
       ;; Ensure that this is (S S) and then figure it out.
       (when (not (and (eq Asign 'S) (eq Bsign 'S)))
	 (error "Don't know what the bound symbol ~a or ~a means" Asign Bsign))
       ;; I'm punting here for now
       (error "Haven't implemented multiplication for (S S) and this expression requires~
               ~%it since its range is [~a ~a]" `(,lba ,uba) `(,lbb ,ubb))))
    ))


;;; Table of bounds.  The format is (Mulbnds Divbnds)
;;; where Mulbnds | Divbnds ::== ((Asign Bsign) Lbnd Ubnd)
;;;       Lbnd | Ubnd       ::== (LB/UB LB/UB)
;;; Note that this doesn't take into account the min/max needed for (S S).
;;;
(defparameter *muldiv-bound-table*
  `(;; Mul is first
    (((+ +) (LB LB) (UB UB))
     ((+ -) (UB LB) (LB UB))
     ((+ s) (UB LB) (UB UB))
     ((- +) (LB UB) (UB LB))
     ((- -) (UB UB) (LB LB))
     ((- s) (LB UB) (LB LB))
     ((s +) (LB UB) (UB UB))
     ((s -) (UB LB) (LB LB)))
    ;; Div is next  (Fixed 17Jul92)
    (((+ +) (LB UB) (UB LB))
     ((+ -) (UB UB) (LB LB))
     ((+ s) (UB LB) (UB UB))
     ((- +) (LB LB) (UB UB))
     ((- -) (LB UB) (UB LB))
     ((- s) (LB UB) (LB LB))
     ((s +) (LB LB) (UB LB))
     ((s -) (UB LB) (LB LB)))))
    


;;;--------------------------------------------------------------------------
;;; Compute where the interval [lo hi] lies.
;;; Inputs:  lo, hi - the interval [lo hi].
;;; Returns: + if the interval is above 0 (including 0)
;;;          - if the interval is below 0 (including 0)
;;;          S if the interval straddles 0.
;;;--------------------------------------------------------------------------
;;;
(defun interval-sign (lo hi)
  (cond
    ((>= lo 0) '+)
    ((<= hi 0) '-)
    (T         'S)))


;;; Return the opposite bound to "bound".
;;;
(defun bbar (bound)
  (if (eq bound 'UB) 'LB 'UB))




;;;--------------------------------------------------------------------------
;;; Construct a lisp function to calculate the given bound for the given
;;; extremal expression with bound labels.
;;; Inputs:  bvar      - A bounded var (i.e., (LB/UB var)
;;;          subexprs  - A list of (ovar LB UB) entries.
;;;          exprTable - the exprTable for the system.
;;; Returns: An evaluable lisp form with vector references and envelope functions.
;;;          The form of this is :
;;;          (let* ((ovar1LB <expr>)
;;;                 (ovar1UB <expr>)
;;;                 ...)
;;;            <expr>)
;;;--------------------------------------------------------------------------
;;;
(defun instantiate-extremal-expr (bvar subexprs exprTable)
  (let ((ordered-subexprs (order-subexprs (second bvar) subexprs exprTable)))
    `(let* ,(instantiate-let-expr ordered-subexprs)
      ,(instantiate-expr bvar))))


;;;--------------------------------------------------------------------------
;;; Order a set of subexprs by using the "parse tree" for expr contained
;;; implicitly in exprTable.  This is needed so that the let* clause
;;; is guaranteed to initialize variables before they are needed in other
;;; initializations.
;;; Inputs:  var       - a var.
;;;          subexprs  - A list of (ovar LB UB) entries.
;;;          exprTable - the exprTable for the system.
;;; Returns: subexprs ordered properly for use by instantate-let-expr.
;;;--------------------------------------------------------------------------
;;;
(defun order-subexprs (var subexprs exprTable)
  (let ((included NIL))
    (loop with vars = (extract-ovars-for-var-expr var exprTable)
	  until (null vars)
	  do

	  ;; For each ovar in vars, see if its subexpr can be expressed solely
	  ;; in terms of vars in included (or svars and cvars).  If so, push it
	  ;; onto included.
	  (loop for v in vars
		for subexpr = (exprTableEntry-subexpr
			       (find v exprTable
				     :key #'exprTableEntry-varname))
		when (all-vars-included subexpr included)
		do
		(setf vars (remove v vars))
		(push v included)))

    ;; Included now has a reverse ordered total ordering for the vars
    (loop for v in (nreverse included)
	  collect (find v subexprs :key #'first))))


(defun extract-ovars-for-var-expr (var exprTable)
  (remove-duplicates (extract-ovars-for-var-expr-1 var exprTable)))


(defun extract-ovars-for-var-expr-1 (var exprTable)
  (let ((ee (find var exprTable :key #'exprTableEntry-varname)))
    (case (exprTableEntry-vtype ee)
      ((state constant) NIL)
      (T
       (let ((subexpr (exprTableEntry-subexpr ee)))
	 (cons var
	       (case (expr-type subexpr)
		 (function (extract-ovars-for-var-expr-1 (third subexpr)
							 exprTable))
		 (T        (mapcan #'(lambda (x)
				       (extract-ovars-for-var-expr-1 x
								     exprTable))
				      (cdr subexpr))))))))))

;;;--------------------------------------------------------------------------
;;; Given a subexpr, return the variables from it.
;;; Inputs:  subexpr - a subexpr (guaranteed not to have nested subexprs).
;;; Returns: A list of the variables in the expression.
;;;--------------------------------------------------------------------------
;;;
(defun extract-vars-from-subexpr (subexpr)
  (remove-duplicates (extract-vars-from-subexpr-1 subexpr)))


(defun extract-vars-from-subexpr-1 (subexpr)
  (case (expr-type subexpr)
    ((svar cvar ovar) (list subexpr))
    (function         (list (third subexpr)))
    (minus            (list (second subexpr)))
    (T                (cdr subexpr))))


;;;--------------------------------------------------------------------------
;;; See if all the vars in subexpr are also in included.
;;; Inputs:  subexpr  - a subexpr (guaranteed not to have nested subexprs).
;;;          included - a list of vars.
;;;--------------------------------------------------------------------------
;;;
(defun all-vars-included (subexpr included)
  (every #'(lambda (x) (or (member (expr-type x) '(svar cvar))
			   (member x included)))
	 ;; Since there are only ovars in included, pull out any var that
	 ;; isn't.
	 (remove-if-not #'(lambda (x) (eq (expr-type x) 'ovar))
			(extract-vars-from-subexpr subexpr))))
	 

;;;--------------------------------------------------------------------------
;;; Generate the subexpressions of a lisp let* clause based on the subexprs
;;; Inputs:  subexprs - The ordered subexpr list.
;;; Returns: A list of the form ((ovar1 <expr1>) ...)
;;;--------------------------------------------------------------------------
;;;
(defun instantiate-let-expr (subexprs)
  (loop for subexpr in subexprs
	nconc (loop for boundexpr in (cdr subexpr)
		    for bound in '(LB UB)
		    when boundexpr
		      collect `(,(gen-var-name (first subexpr) bound)
				,(instantiate-expr boundexpr)))))


;;;--------------------------------------------------------------------------
;;; Instantiate an expression.
;;; Inputs:  expr - a bounded expression.
;;; Returns: An evaluable lisp form that calculates expr.
;;;--------------------------------------------------------------------------
;;;
(defun instantiate-expr (expr)
  (declare (special *state* *svars* *cvars*))
  (cond 
    ;; nils return themselves
    ((null expr) nil)
    
    ;; numbers map to themselves
    ((numberp expr) expr)

    ;; bounded expressions work as follows :
    ((member (car expr) '(LB UB))
     (let ((bound (car expr))
	   (var-or-fctn (cadr expr)))
       (cond
	 ;; constant map to their location in the constant vector.
	 ((member var-or-fctn *cvars*)
	  `(svref c ,(position-in-vector var-or-fctn bound *cvars*)))

	 ;; state-vars map to their location in the state vector.
	 ((member var-or-fctn *svars*)
	  `(svref y ,(position-in-vector var-or-fctn bound *svars*)))

	 ;; other-vars return references to let vars in the let*
	 ((atom var-or-fctn) 
	  (gen-var-name var-or-fctn (car expr)))

	 ;; functions map to their envelope function
	 ((and (listp var-or-fctn)
	       (member (first var-or-fctn) '(FUNCTION INVERSE)))
	  (return-function-bound (first var-or-fctn) (second var-or-fctn)
				 (third var-or-fctn) bound))

	 (T
	  (error "Unknown expression ~a" expr)))))
	      
    ;; complex expressions recurse.
    ((listp expr)
     (if (member (car expr) '(* /))
	 (instantiate-mul-expr expr)
	 `(,(car expr) ,.(mapcar #'(lambda (arg)
				     (instantiate-expr arg))
			         (cdr expr)))))
    (T
     (error "Unknown expression ~a" expr))))


(defun gen-var-name (var bound)
  (intern (concatenate 'string (symbol-name var) (symbol-name bound))))


;;;--------------------------------------------------------------------------
;;; Take a bounded multiplication expression and return an evaluable lisp
;;; function.  The function will look like this :
;;;     (nsim-muldiv */ bound alo ahi blo bhi))
;;; Inputs:  expr - (*/ LB/UB (LB A) (UB A) (LB B) (UB B))
;;; Returns: An evaluable lisp expression.
;;;--------------------------------------------------------------------------
;;;
(defun instantiate-mul-expr (expr)
  (let ((alo (instantiate-expr (third expr)))
	(ahi (instantiate-expr (fourth expr)))
	(blo (instantiate-expr (fifth expr)))
	(bhi (instantiate-expr (sixth expr))))
    `(nsim-muldiv (quote ,(first expr)) (quote ,(second expr)) ,alo ,ahi ,blo ,bhi)))


;;;--------------------------------------------------------------------------
;;; Return the bound on a QSIM function constraint.
;;; Inputs:  type     - FUNCTION or INVERSE.
;;;          fctn     - A form like (M+ X Y)
;;;          arg      - the argument (in label form).
;;;          bound    - UB or LB.
;;; Returns: an evaluable expression for the function constraint.
;;;--------------------------------------------------------------------------
;;;
(defun return-function-bound (type fctn arg bound)
  (declare (special *state*))
  (let* ((accessor (cond
		    ((eq type 'FUNCTION)
		     (if (eq bound 'UB)
			 'upper-envelope
		         'lower-envelope))
		    ((eq type 'INVERSE)
		     (if (eq bound 'UB)
			 'upper-inverse
		         'lower-inverse))))
	 (m-envelopes (qde-m-envelopes (state-qde *state*)))
	 (env-set     (if m-envelopes (lookup-set fctn m-envelopes
						  :test #'equal)))
	 (env         (lookup accessor env-set)))
;    (format t "~%M-envelopes = ~a" m-envelopes)
;    (format t "~%env-set     = ~a" env-set)
;    (format t "~%env         = ~a" env)
    (when (null env)
      (error "No envelope for ~a" fctn))
    `(funcall (function ,env) 
	      ,(instantiate-expr arg))))
	    

;;;--------------------------------------------------------------------------
;;; Find the position that the bound of var would have in a vector ordered
;;; by varlist.
;;; Inputs:  var     - a variable name.
;;;          bound   - LB or UB.
;;;          varlist - a list of varnames.
;;; Returns: The location of the bound of var in a vector of the form
;;;          [v1LB v1UB v2LB v2UB ...] where varlist = (v1 v2 ...)
;;;--------------------------------------------------------------------------
;;;
(defun position-in-vector (var bound varlist)
  (let ((pos (position var varlist)))
    (when pos
      (+ (* 2 (position var varlist)) (if (eq bound 'UB) 1 0)))))


;;; This is using method 2 as described above.
;;; Look up the landmark name in the initial values list and grab the
;;; associated range.  Complain bitterly if it is not there.
;;; An exception is made if the lmark is 0, in which case (0 0) is
;;; returned.  [Q2 gets mad if you try to put 0 as an initial-range
;;; lmark.]
;;;
(defun get-current-range-old (var)
  (declare (special *state*))
  (let* ((initial-ranges (qde-initial-ranges (state-qde *state*)))
	 (cur-qval (alookup var (state-qvalues *state*)))
	 (cur-qmag (if cur-qval (qval-qmag cur-qval))))
    (when (null initial-ranges)
      (error "No INITIAL-RANGES clause for qde ~a" (state-qde *state*)))
    (cond
      ((eq cur-qmag *zero-lmark*)
       '(0 0))
      ((lmark-p cur-qmag)
       (let ((range (lookup (list var (lmark-name cur-qmag)) initial-ranges
			    :test #'equal)))
	 (when (null range)
	   (error "No INITIAL-RANGE associated with ~a = ~a" var cur-qval))
	 range))
      (T
       (error "GET-CURRENT-RANGE expects to find a lmark for ~a.  It found ~a"
	      var cur-qval)))))

;;; This is using method 1 as described above.
;;; Assume that Q2 has already been run on the initial state so that
;;; its values are correct.
;;; Look up the landmark in the state-bindings and grab the
;;; associated range.  Complain bitterly if it is not a number.
;;; This incorporates Dan Dvorak's change to check for (AT S-x) specs
;;; (are they guaranteed to be smaller than the enclosing lmark ranges?
;;;
(defun get-current-range (var)
  (declare (special *state*))
  (let ((*bindings* (state-bindings *state*))
	(var-struct (find var (state-variables *state*) :key #'variable-name))
	(lm         (qmag (qval var *state*)))
	(range      NIL))
    ;; get-range-binding needs *bindings*
    (declare (special *bindings*))
    (cond
      ;; If qmag is at a landmark value, return range of landmark.
      ((atom lm)
       (setf range (car (get-range-binding var-struct lm))))

      ;; See if bindings contain a value for '(AT S-x)
      ((lookup `(AT ,*state*) (alookup var *bindings*) :test #'equal))

      ;; Otherwise, return the range from the bounding landmarks.
      (T
       (let ((r1 (car (get-range-binding var-struct (car  lm))))
	     (r2 (car (get-range-binding var-struct (cadr lm)))))
	 (setf range (list (emin (lo r1) (lo r2))
			   (emax (hi r1) (hi r2)))))))
    (when (or (not (numberp (lo range)))
	      (not (numberp (hi range))))
      (error "NSIM requires that the range for ~a be non-infinite.~
              The stored range is ~a" var range))
    (when (and (< (lo range) 0) (> (hi range) 0))
      (error "NSIM requires that the range for ~a not straddle 0.~
              The stored range is ~a" var range))
    range))
      


