;;; -*- Mode:Common-Lisp; Package:QSIM; Base:10 -*-
;;;  $Id: energy-gf.lisp,v 1.5 1991/09/27 13:43:12 clancy Exp $
(in-package :qsim)

;;;=============================================================================
;;;
;;;       T H E   E N E R G Y - B A S E D   G L O B A L   F I L T E R
;;;
;;;=============================================================================


;;; by Pierre Fouche <fouche@cs.utexas.edu>, October 1989
;;;
;;; Now including: automatic derivation of the energy constraint clause,
;;; Pierre fouche, August 1990


;;;-----------------------------------------------------------------------------
;;; This is an implementation of the energy-constraint described in the
;;; Tec-Report AI-TR-90-134.
;;;
;;; It is implemented as a global state filter called: energy-global-filter.
;;;
;;; In this program the variables x, v, -v-, represents the position, speed,
;;; absolute value (absolute magnitude maybe is a more appropriate term) of the
;;; speed of some mechanical device modeled by the QDE, c and nc the
;;; conservative and not conservative forces acting on that device. It actually
;;; works if the QDE doesn't model a mechanical device, such as a PI controler
;;; (in this case, v is the error and x the integral of the error).
;;; s1 and s2 represents the states between which energies are computed.
;;;-----------------------------------------------------------------------------


;;;-----------------------------------------------------------------------------
;;; If *trace-energy-filter* is t then behaviors violating the energy
;;; conservation law are not deleted, but only labeled as inconsistent.
;;;
;;; If you want to see a more detailed trace of the global filter, run a
;;; simulation, switch on *trace-energy-detailed*, select a final state and evaluate
;;;  (energy-global-filter a-final-state)
;;;-----------------------------------------------------------------------------

;(defparameter *trace-energy-detailed* nil)
;(defparameter *trace-energy-filter* nil)
;(defparameter *check-energy-constraint* t)
;; moved to >nq>qdefs


;;;-----------------------------------------------------------------------------
;;; The energy-constraint dynamic slot contains the names of the variables used
;;; to compute energy-related terms.
;;;-----------------------------------------------------------------------------

;;; This definition has been moved to DEFOTHERS.LISP
;;; (defother qde energy-constraint)


;;;-----------------------------------------------------------------------------
;;; If energy-global-filter is not defined yet, then define it.
;;;-----------------------------------------------------------------------------

(unless (member 'energy-global-filter *global-state-filters*)
  (setq *global-state-filters*
        (nconc *global-state-filters* `(energy-global-filter))))



;;;=============================================================================
;;;                EVALUATION OF EXPRESSIONS CONTAINING SIGNS
;;;=============================================================================

;;;-----------------------------------------------------------------------------
;;; This kind of code is certainly already implemented somewhere in Qsim...
;;;
;;; The following functions implement arithmetic sign rules
;;;-----------------------------------------------------------------------------


(defmacro bad-op (op)
  `(error "~% ~a is not a valid operator !" ,op))

(defun multiplication-sign (sign1 sign2)
  "Sign multiplicatiom law"
  (cond ((or (eql sign1 0) (eql sign2 0)) 0)
        ((not sign1) nil)
        ((eq sign1 '+) sign2)
        ((eq sign1 '-) (opp-sign sign2))
        (t (bad-sign sign1))))

(defun addition-sign (sign1 sign2)
  "Sign addition law"
  (cond ((not sign1) nil)
        ((eql sign1 0) sign2)
        ((eql sign2 0) sign1)
        ((eq sign1 '-) (if (eq sign2 '-) '- nil))
        ((eq sign1 '+) (if (eq sign2 '+) '+ nil))
        (t (bad-sign sign1))))

(defun substraction-sign (sign1 sign2)
  "Sign substraction law"
  (addition-sign sign1 (opp-sign sign2)))


;;;-----------------------------------------------------------------------------
;;; Eval-sign-expression takes as input:
;;;   - an arithmetic expression containing operators (+, -, *), signs (+, 0, -
;;;     or nil (as unknown sign)) and variable names;
;;;   - a state from which to pick up the qmags of the variables
;;;
;;; ex: (eval-sign-expression '(+ (+ x y) (- u v w) (* a b)))
;;;
;;; "+" and "-" are interpreted as operators after a "(" and as signs otherwise.
;;;-----------------------------------------------------------------------------

(defun eval-sign-exp (exp &optional state)
  "Evaluates the sign of an arithmetic expression"
  (if (atom exp)
      ;; exp is a sign or a variable name
      (cond ((not exp) nil)
            ((eql exp 0) 0)
            ((eq exp '+) '+)
            ((eq exp '-) '-)
            (t (sign (qmag (qval exp state)) (qspace exp state))))

      ;; exp is an expression to reduce
      (let ((op (first exp))
            (arg1 (second exp))
            (other-args (cddr exp)))
        (if (null other-args)
            (cond ((eq op '+) (eval-sign-exp arg1 state))
                  ((eq op '-) (opp-sign (eval-sign-exp arg1 state)))
                  ((eq op '*) (eval-sign-exp arg1 state))
                  (t (bad-op op)))
            (cond ((eq op '+)
                   (eval-sign-exp
                     `(+ ,(addition-sign (eval-sign-exp arg1 state)
                                         (eval-sign-exp (first other-args) state))
                         ,@(rest other-args))
                     state))
                  ((eq op '-)
                   (substraction-sign (eval-sign-exp arg1 state)
                                      (eval-sign-exp
                                        `(+ ,@other-args) state)))
                  ((eq op '*)
                   (eval-sign-exp
                     `(* ,(multiplication-sign (eval-sign-exp arg1 state)
                                               (eval-sign-exp (first other-args) state))
                         ,@(rest other-args))
                     state))
                  (t (bad-op op)))))))


;;;-----------------------------------------------------------------------------
;;; (same-sign-p x y) is true if:
;;;   - x and y represent the same sign
;;;   OR
;;;   - if either x or y is zero.
;;;-----------------------------------------------------------------------------

(defun same-sign-p (sign1 sign2)
  (or (eql sign1 0)
      (eql sign2 0)
      (and (eql sign1 '+) (eql sign2 '+))
      (and (eql sign1 '-) (eql sign2 '-))))


;;;-----------------------------------------------------------------------------
;;; (my-qmag-order qmag1 qmag2 qspace) acts as qmag-order, except that it
;;; works also if qmag2 is an interval. It returns nil if the two qmags can't
;;; be compared (i.e. overlaping intervals).
;;;
;;; Qmag-order rewritten to accept any qmag; Pierre Fouche 03/12/90
;;;-----------------------------------------------------------------------------

;(defun qmag-order (qmag1 qmag2 qspace)
;  (cond ((or (not qmag1) (not qmag2)) nil)
;       ((and (qmag-point-p qmag1)
;             (qmag-point-p qmag2)
;             (eq qmag1 qmag2))
;        0)
;       ((qmag-point-p qmag2) (qmag-order-1 qmag1 qmag2 qspace))
;       ((qmag-point-p qmag1)
;        (opp-sign (qmag-order-1 qmag2 qmag1 qspace)))
;       ((or (eq (first qmag1) (first qmag2))
;            (eq (second qmag1) (second qmag2)))
;        nil)
;       (t (qmag-order-1 qmag1 (first qmag2) qspace))))


;;;=============================================================================
;;;                   EVALUATION OF ENERGY-RELATED TERMS:
;;;
;;;  - variation of "kinetic energy",
;;;  - "work" of conservative "forces",
;;;  - "work" of non-conservative forces.
;;;=============================================================================

;;;-----------------------------------------------------------------------------
;;; (ke-variation -v- s1 s2) returns the sign of the variation of the kinetic
;;; energy of a system between states s1 and s2. -v- is the name of a variable
;;; representing the absolute value of the speed variable.
;;;-----------------------------------------------------------------------------

(defun ke-variation (v -v- s1 s2)
  (let* (ke-var)
    (cond  (-v-
            ;; if -v- exists then the kinetic energy variation is easy to
            ;; compute
            (setf ke-var (qmag-order (qmag (qval -v- s2))
                                            (qmag (qval -v- s1))
                                            (qspace -v- s2))))
           (t
            ;; otherwise it is possible to compute it only if V(t1) and V(t2)
            ;; have the same sign.
            (let* ((v1 (qmag (qval v s1)))
                     (v2 (qmag (qval v s2)))
                     (v-qspace (qspace v s2))
                     (s1 (sign v1 v-qspace))
                     (s2 (sign v2 v-qspace)))
                (when (same-sign-p s1 s2)
                  (if (or (eq s1 '+) (eq s2 '+))
                      (setf ke-var (qmag-order v2 v1 v-qspace))
                      (setf ke-var (eval-sign-exp
                                     `(* - ,(qmag-order v2 v1 v-qspace)))))))))
    (when (eql *trace-energy* :detailed)
      (format *qsim-trace*
              "~% KE variation between ~a and ~a: ~a"
              s1 s2 ke-var))
    ke-var))

;;;-----------------------------------------------------------------------------
;;; (c-work c x s1 s2) returns the sign of the work of the conservative term c
;;; between s1 and s2. c is a function of x. If c keeps the same sign s-c
;;; between s1 and s2, the sign of the work only depends on the variation of x
;;; and on s-c.
;;;-----------------------------------------------------------------------------

(defun c-work (c x s1 s2)
  (when (eql c 0) (return-from c-work 0))       ;if no conservative term -> 0
  (let* ((x1-x2-order (qmag-order (qmag (qval x s2))
                                     (qmag (qval x s1))
                                     (qspace x s2)))
         (sign-of-c1 (eval-sign-exp c s1))
         (sign-of-c2 (eval-sign-exp c s2))
         (sign-of-c (when (same-sign-p sign-of-c1 sign-of-c2)
                        (if (eql sign-of-c1 0) sign-of-c2 sign-of-c1)))
         (work (eval-sign-exp `(* ,sign-of-c ,x1-x2-order))))

    (when (eql *trace-energy* :detailed)
      (format *qsim-trace*
              "~% Work of ~a=f(~a) between ~a(~a)=~a and ~a(~a)=~a: ~a"
              c x c s1 (qmag (qval x s1)) c s2 (qmag (qval x s2)) work))
    work))


;;;-----------------------------------------------------------------------------
;;; (nc-work nc v s1 s2) returns the sign of the work of the not conservative
;;; term nc between s1 and s2, which is the sign of the product nc by v if that
;;; sign remains constant between s1 and s2
;;;-----------------------------------------------------------------------------

(defun nc-work (nc v s1 s2)
  (when (eql nc 0) (return-from nc-work 0))     ;if no conservative term: 0
  (do* ((state s2 (state-predecessor state))
        (sign (eval-sign-exp `(* ,nc ,v) s2))
        (previous-sign sign (eval-sign-exp `(* ,nc ,v) state)))

       ((eq state s1)
        (unless (same-sign-p sign previous-sign)
          (setq sign nil))
        (when (eql *trace-energy* :detailed)
          (format *qsim-trace*
                  "~% Work of ~a(t) betwen ~a and ~a: ~a"
                  nc s1 s2 sign))
        sign)

    (unless (same-sign-p sign previous-sign)
      (when (eql *trace-energy* :detailed)
        (format *qsim-trace*
                "~% Work of ~a(t) betwen ~a and ~a: nil"
                nc s1 s2 sign))
      (return nil))
    (if (eql sign 0)
        (setq sign previous-sign))))


;;;-----------------------------------------------------------------------------
;;; check-energy-conservation computes the three energy terms and check if they
;;; are compatible.
;;; Watch out !
;;;  -> Returns nil IF OK, a string explaining why the energy is not conserved
;;; otherwise.
;;;-----------------------------------------------------------------------------

(defun check-energy-conservation (x v -v- c nc s1 s2)
  (let ((ke-var (ke-variation v -v- s1 s2))
        (*detailed-printing* nil)
        (*print-case*  :upcase)
        c-work nc-work s)
    (unless ke-var (return-from check-energy-conservation nil))

    (setq c-work (c-work c x s1 s2))
    (unless c-work (return-from check-energy-conservation nil))

    (setq nc-work (nc-work nc v s1 s2))
    (setq s (eval-sign-exp `(- ,ke-var ,c-work ,nc-work)))
    (unless (or (not s) (eql s 0))
      (format nil
              "Between ~a and ~a: KE variation = ~a, C work = ~a, NC work = ~a"
              (qmag (state-time s1)) (qmag (state-time s2))
              ke-var c-work nc-work))))



;;;=============================================================================
;;;                       ENERGY-BASED FILTER STRATEGY
;;;=============================================================================

;;;-----------------------------------------------------------------------------
;;; (continuous-p var s1 s2) is a predicate that returns true if s is continuous
;;; between s1 and s2. Continuity must be checked when transition occurs. Time
;;; is supposed to be the same in s1 and s2.
;;;-----------------------------------------------------------------------------

(defun continuous-p (var s1 s2)
  "Check continuity of VAR between S1 and S2"
  (when (symbolp var)
    (equal (qmag (qval var s1))
           (qmag (qval var s2)))))


;;;-----------------------------------------------------------------------------
;;; In order to check that a behavior leading to a state s2 is energetically
;;; valid, one has to look for a previous state s1 to compute energy-related
;;; terms between s1 and s2.
;;; Care must be taken at operating region transitions: check for continuity
;;; and be sure that the energy-constraint clause in the second qde is present
;;; in the first.
;;;-----------------------------------------------------------------------------

(defun find-good-predecessor (current-state energy-constraint)
  (let ((pred (state-predecessor current-state))
        (just (first (state-justification current-state))))
    (cond
      ;;no more candidates
      ((null pred) nil)

      ;;current-state is a transition state
      ;;=> pred and current-state have the same time index
      ;;  -> if the qdes are different, check that the energy-constraint clause
      ;;     of the preceding qde includes the energy-constraint of the second.
      ;;  -> check for continuity of x -v- and c at transition
      ;;  -> if continuity ok, find state with previous time index.
      ((or (eq just 'transition-from)
           (eq just 'one-of-several-completions-of))
       (if (or (not (member energy-constraint
                            (get-energy-constraint (state-qde pred))
                            :test #'equal))
               (not (continuous-p (first energy-constraint) pred current-state))
               (not (continuous-p (third energy-constraint) pred current-state))
               ;;(not (continuous-p (fourth energy-constraint) pred current-state))
               ;; if x is continuous, then c is continuous (as a continuous
               ;; function of x)
               )
           nil
           pred))

      ;;pred must be a state with a time point
      ((qintervalp (state-time pred))
       (find-good-predecessor pred energy-constraint))
      (t pred))))


;;;-----------------------------------------------------------------------------
;;; (energy-global-filter state) searchs for an energy-constraint clause in the
;;; QDE.
;;; This clause looks like (energy-constraint ((x v c nc))) where x is the
;;; "position" variable, v the "speed" variable, c and nc the conservative and
;;; not conservative terms.
;;;
;;; For each list of energy-related terms, EGF searches a predecessor of state
;;; to check energy conservation between the two states.
;;;-----------------------------------------------------------------------------



(defun find-absolute-value (varname state)
  "Return the variable representing the absolute value of a variable, if it exists"
  (let* ((var (var varname state))
         (qde (state-qde state)))
    (dolist (con (qde-constraints qde))
      (when (and (eq (contype-name (constraint-type con))
                     'abs-value)
                 (member var
                         (constraint-variables con)))
        (return (variable-name (first (remove var
                                              (constraint-variables con)))))))))

(defun energy-global-filter (s2)
  "The Energy-Based Global Filter"
  (when (and *check-energy-constraint*
             (qpointp (state-time s2)))

    ;;iterate on every energy constraint clause
    (dolist (energy-constraint (get-energy-constraint (state-qde s2)) s2)
      (let* ((x (first   energy-constraint))
             (v (second  energy-constraint))
             (c (third   energy-constraint))
             (nc (fourth energy-constraint))
             -v-)

        ;;iterate on predecessors of current state
        (do ((s1 (find-good-predecessor s2 energy-constraint)
                 (find-good-predecessor s1 energy-constraint)))
            ((null s1) s2)                      ;no more pred -> ok
          (setq -v- (find-absolute-value v s1))
          (let ((energy-not-conserved
                  (check-energy-conservation x v -v- c nc s1 s2)))
            (when energy-not-conserved
              (prune-inconsistent-state s2 Energy-not-conserved)
	      (when *trace-energy*      ; added DJC 08/23/91 to limit the trace information
		(format *qsim-trace* "~&STATE ~a filtered due to energy constraint. ~
                        ~%   REASON:  ~a" s2 energy-not-conserved))
               ; new special variable added DJC  09/23/91
              (return-from energy-global-filter nil)))))))
  s2)


;;;----------------------------------------------------------------------------
;;; GET-ENERGY-CONSTRAINT
;;;  - analyzes the qde on the first call, sets the qde-derived-energy-constraint
;;;    slot and returns the value of that slot
;;;  - returns the value of  qde-derived-energy-constraint or nil if eq
;;;    to :no-energy-constraint on next calls.
;;;
;;; The value of qde-derived-energy-constraint is the union of the user-
;;; asserted energy constraint (stored in qde-energy-constraint) and of the
;;; automatically derived energy constraint if *perform-energy-analysis* is T.
;;;-----------------------------------------------------------------------------

(defun get-energy-constraint (qde)
  (let ((derived-energy-constraint (qde-derived-energy-constraint qde)))
    (cond ((eq derived-energy-constraint :no-energy-constraint)
           nil)
          ((consp derived-energy-constraint)
           derived-energy-constraint)
          ((null derived-energy-constraint)
           (when *perform-energy-analysis*
             (setq derived-energy-constraint
                   (perform-energy-analysis qde)))
           (setq derived-energy-constraint
                 (union (qde-energy-constraint qde)
                        derived-energy-constraint
                        :test #'equal))
           (setf (qde-derived-energy-constraint qde)
                 (or derived-energy-constraint
                     :no-energy-constraint))
           derived-energy-constraint)
          (t (error "Bad derived-energy-constraint clause in qde ~a: ~a"
                    qde derived-energy-constraint)))))


;;;=============================================================================
;;; Automatic derivation of the energy constraint clause
;;;=============================================================================

;; define the data base first
(setq *default-db* 'ec)
(reset-db)

;;;-----------------------------------------------------------------------------
;;; CREATE-CONSTRAINT-PREDICATE creates a form:
;;;
;;; (defpredicate constraint
;;;    ((constraint <constraint>))*)
;;;
;;; where <constraint> is a constraint in a qde, and evaluate it.
;;;-----------------------------------------------------------------------------

(defun create-constraint-predicates (qde)
  (let (;(qde-name (qde-name qde))
        (vars (qde-variables qde))
        (traced-p #+ti (and (find-predicate 'constraint)        ; References to package
                            (traced-p 'constraint))             ; Prolog: deleted.
                  #-ti nil)
        constraint-clauses)
    (setq constraint-clauses
          (mapcar #'(lambda (con)
                      (let* ((con-type (contype-name (constraint-type con)))
                             (varnames (mapcar #'variable-name
                                               (constraint-variables con))))
                        `((constraint (,con-type .,varnames)))))
                  (qde-constraints qde)))
    ;; special for constant variables (not declared as ((constant <var>)))
    (dolist (var vars)
      (when (variable-independent-p var)
        (pushnew `((constraint (constant ,(variable-name var))))
                 constraint-clauses :test #'equal)))
    (eval `(defpredicate constraint .,constraint-clauses))
    (when traced-p (trace-predicate constraint))))


;;;-----------------------------------------------------------------------------
;;; EC takes a QDE, builds constraints predicate and tries to find an energy
;;; constraint.
;;;-----------------------------------------------------------------------------

(defun ec (qde)
  (setq *default-db* 'ec)
  (create-constraint-predicates qde)
  (eval `(? (ec ?x ?v ?c ?nc))))


;;;-----------------------------------------------------------------------------
;;; PERFORM-ENERGY-ANALYSIS analyzes the QDE, and sets the energy constraint
;;; clause if derivation is successful
;;;-----------------------------------------------------------------------------

(defun perform-energy-analysis (qde)
  (setq *default-db* 'ec)
  (create-constraint-predicates qde)
  ;;(trace-predicate constraint)
  (let ((d/dt-chains (eval `(? (derivative-chain ?x ?v ?a))))
        derived-energy-constraint)
    ;; If a derivative chain is found...
    (when d/dt-chains
      (when *trace-decomposition*
        (format *qsim-trace* "~% Performing energy analysis on QDE: ~a" qde))
      ;; then try to decompose the higher-order derivative.
      (setq derived-energy-constraint
            (mapcan #'(lambda (a-chain)
                        (let* ((x-v-a (first a-chain))
                               (x (second  x-v-a))
                               (v (third x-v-a))
                               (a (fourth  x-v-a))
                               c-nc-pairs constraints)
                          (when *trace-decomposition*
                            (format *qsim-trace* "~% Found a derivative chain:~a -> ~a -> ~a"
                                    x v a))
                          (setq c-nc-pairs (decompose-acceleration x a))
                          (when c-nc-pairs (setq constraints (mapcar #'(lambda(c-nc-pair)
                                                                         `(,x,v .,c-nc-pair))
								     c-nc-pairs)))
                          constraints))
                    d/dt-chains))
      (when *trace-decomposition*
        (format *qsim-trace* "~% Done."))
       derived-energy-constraint)))


;;;-----------------------------------------------------------------------------
;;; DECOMPOSE-ACCELERATION calls the prolog interpreter to find a decomposition
;;; of A into conservative and non-conservative terms.
;;;-----------------------------------------------------------------------------

(defun decompose-acceleration (x a)
  (let* ((decompositions (eval `(? (decomposition ,x ,a ?c ?nc))))
         (c-nc-pairs (mapcar #'cdddar decompositions)))
    (when *trace-decomposition*
      (if c-nc-pairs
          (dolist (c-nc-pair c-nc-pairs)
            (format *qsim-trace* "~%  Found the decomposition: ~a = ~a + ~a"
                    a (first c-nc-pair) (second c-nc-pair)))
          (format *qsim-trace* "~%  No decomposition found")))
    c-nc-pairs))


;;;=============================================================================
;;; Rules to derive the energy constraint
;;;=============================================================================

(defpredicate not
  ((not ?x) <- ?x ! (fail))
  ((not ?x)))

(defpredicate member
  ((member ?x nil) <- (fail))
  ((member ?x (?x . ?y)))
  ((member ?x (?y . ?list))  <- (member ?x ?list)))

;;;-----------------------------------------------------------------------------
;;; Predicates that handle properties of constraints, such as commutativity
;;;-----------------------------------------------------------------------------

(defpredicate M+
  ((M+ ?x ?y) <- (constraint (M+ ?x ?y)))
  ((M+ ?x ?y) <- (constraint (M+ ?y ?x))))

(defpredicate M-
  ((M- ?x ?y) <- (constraint (M- ?x ?y)))
  ((M- ?x ?y) <- (constraint (M- ?y ?x))))

(defpredicate Minus
  ((Minus ?x ?y) <- (constraint (Minus ?x ?y)))
  ((Minus ?x ?y) <- (constraint (Minus ?y ?x))))

(defpredicate U+
  ((U+ ?x ?y) <- (constraint (U+ ?x ?y))))

(defpredicate ADD
  ((ADD ?x ?y ?z) <- (constraint (ADD ?x ?y ?z)))
  ((ADD ?x ?y ?Z) <- (constraint (ADD ?y ?x ?z))))

(defpredicate MULT
  ((mult ?x ?y ?z) <- (constraint (mult ?x ?y ?z)))
  ((mult ?x ?y ?z) <- (constraint (mult ?y ?x ?z))))

(defpredicate d/dt
  ((d/dt ?x ?y) <- (constraint (d/dt ?x ?y)))
  ((d/dt ?x ?y) <- (add ?x1 ?x2 ?x)
                   (constant ?x1 ?x)
                   (constraint (d/dt ?x2 ?y))))

(defpredicate CONSTANT
  ((constant ?x) <- (constraint (constant ?x)))
  ((constant ?x) <- (add ?x1 ?x2)
                    (constant ?x1)
                    (constant ?x2))
  ((constant ?x) <- (mult ?x1 ?x2)
                    (constant ?x1)
                    (constant ?x2)))


;;;-----------------------------------------------------------------------------
;;; To determine the energy constraint, first find out a derivative chain, then
;;; decompose the HOD.
;;;-----------------------------------------------------------------------------

(defpredicate ec
  ((ec ?x ?v ?c ?nc)
   <-
   (derivative-chain ?x ?v ?a)
   (decomposition ?x ?a ?c ?nc)))


(defpredicate derivative-chain
  ((derivative-chain ?x ?v ?a) <- (d/dt ?x ?v)
                                  (d/dt ?v ?a)))


;;;-----------------------------------------------------------------------------
;;; Decomposition:
;;; - check whether the system is conservative,
;;; - find an appropriate decomposition otherwise.
;;;
;;; Right now, if the system is not conservative, the code below succeeds iff
;;; the HOD is already decomposed into two terms by an ADD constraint.
;;; Must be improved...
;;;-----------------------------------------------------------------------------

(defpredicate decomposition
  ((decomposition ?x ?a ?a 0)   <- (depends-only-on ?a ?x nil) !)
  ;; cut (!) is used because that predicate is deterministic.
  ((decomposition ?x ?a ?c ?nc) <- (add ?c ?nc ?a)
                                   (depends-only-on ?c ?x ((add ?c ?nc ?a))) !)
  ((decomposition ?x ?a ?c ?nc) <- (add ?a ?-nc ?c)
                                   (minus ?nc ?-nc)
                                   (depends-only-on ?c ?x ((add ?a ?-nc ?c) (minus ?nc ?-nc))) !)
  ((decomposition ?x ?a ?c ?nc) <- (add ?a ?-c ?nc)
                                   (minus ?c ?-c)
                                   (depends-only-on ?-c ?x ((add ?a ?-c ?nc) (minus ?c ?-c))) !))


;;;-----------------------------------------------------------------------------
;;; (depends-only-on ?y ?x ?constraints) succeeds if ?y depends only on
;;; ?x. ?constraints is the list of constraints that have been used to derive
;;; the dependency. It is used to prevent infinite loop.
;;;-----------------------------------------------------------------------------

(defpredicate depends-only-on
  ((depends-only-on ?x ?x ?used-constraints) <- !)

  ((depends-only-on ?c ?x ?used-constraints)
   <- (constraint (constant ?c)) !)

  ((depends-only-on ?c ?x ?used-constraints)
   <-
   (Minus ?c ?c1)
   (depends-only-on2 ?c1 ?x (minus ?c ?c1) ?used-constraints))

  ((depends-only-on ?c ?x ?used-constraints)
   <-
   (U+ ?c1 ?c)
   (depends-only-on2 ?c1 ?x (minus ?c ?c1) ?used-constraints))

  ((depends-only-on ?c ?x ?used-constraints)
   <-
   (M+ ?c ?c1)
   (depends-only-on2 ?c1 ?x (M+ ?c ?c1) ?used-constraints))

  ((depends-only-on ?c ?x ?used-constraints)
   <-
   (M- ?c ?c1)
   (depends-only-on2 ?c1 ?x (M- ?c ?c1) ?used-constraints))

  ((depends-only-on ?c ?x ?used-constraints)
   <-
   (add ?c1 ?c2 ?c)
   (depends-only-on2 ?c1 ?x (add ?c1 ?c2 ?c) ?used-constraints)
   (depends-only-on2 ?c2 ?x (add ?c1 ?c2 ?c) ?used-constraints))

  ((depends-only-on ?c ?x ?used-constraints)
   <-
   (mult ?c1 ?c2 ?c)
   (depends-only-on2 ?c1 ?x (mult ?c1 ?c2 ?c) ?used-constraints)
   (depends-only-on2 ?c2 ?x (mult ?c1 ?c2 ?c) ?used-constraints))

  ((depends-only-on ?c ?x ?used-constraints)
   <-
   (add ?c ?c1 ?c2)
   (depends-only-on2 ?c1 ?x (add ?c1 ?c2 ?c) ?used-constraints)
   (depends-only-on2 ?c2 ?x (add ?c1 ?c2 ?c) ?used-constraints))

  ((depends-only-on ?c ?x ?used-constraints)
   <-
   (mult ?c ?c1 ?c2)
   (depends-only-on2 ?c1 ?x (mult ?c1 ?c2 ?c) ?used-constraints)
   (depends-only-on2 ?c2 ?x (mult ?c1 ?c2 ?c) ?used-constraints)))


(defpredicate depends-only-on2
  ((depends-only-on2 ?c ?x ?constraint ?other-constraints)
   <-
   (not (member ?constraint ?other-constraints))
   (depends-only-on ?c ?x (?constraint . ?other-constraints))))



