;;; -*- Mode: LISP; Syntax: Common-lisp; Package: USER; Base: 10 -*-
;;;  $Id: dnet.lisp,v 1.1 1991/03/26 21:37:50 clancy Exp $
;;;
;;; Forward chaining rules.  Set up so that it is easy to apply all
;;; rules which match a particular datum.
;;;
;;; We use a simple matcher along with a discrimination net.
;;;

;;;======================================================================
;;;
; Recursive MATCH, based on the Abelson & Sussman matcher, with
; prolog-like variables matching a single subexpression.
; It returns an alist of variable bindings.
;  - explicit FAILED returned on failure; NIL means success with no bindings.

(defun MATCH (pat dat &optional alist)
  (cond ((eql alist 'failed) 'failed)
	((variable? pat) (check+update-binding pat dat alist))
	((atom pat) (cond ((eql pat dat) alist)
			  (t 'failed)))
	((atom dat) 'failed)
	(t (match (cdr pat)
		  (cdr dat)
		  (match (car pat) (car dat) alist)))))

; Variables are atoms with first character "_" or "?".

(defun variable? (pat)
  (and (symbolp pat)
       (let ((char (schar (string pat) 0)))
	 (or (char= char #\_)
	     (char= char #\?)))))

; Add a binding (<variable> <value>) to an alist.
;  If <variable> already has a binding, it must be the same.

(defun check+update-binding (pat dat alist)
  (let ((binding (assoc pat alist)))
    (cond ((null binding)
	   (acons pat dat alist))
	  (t (match dat (cdr binding) alist)))))

;;; In my situation, I know that pat almost matches dat, but not if the
;;; vars can be bound consistently.  So we want to zoom through doing as
;;; little work as possible.
;;;
(defun bind-vars (pat dat &optional alist)
  (cond ((eql alist 'failed) 'failed)
	((variable? pat) (check+update-binding pat dat alist))
	((atom pat) alist)
	(t (bind-vars (cdr pat)
		      (cdr dat)
		      (bind-vars (car pat) (car dat) alist)))))

; Use the bindings in <alist> to substitute values for variables in <exp>.

(defun substitute-bindings (exp alist)
  (if (isground! alist)
      ;; if all of the bindings are grounded, then we can do a cheap substitution.
      ;; symbolics manages to avoid alot of consing with sublis.
      (cond ((variable? exp)
	     (cdr (assoc exp alist)))
	    ((atom exp) exp)
	    ((sublis  alist exp)))
      (full-substitute-bindings alist exp)))

(defun full-substitute-bindings (exp alist)
  (cond ((variable? exp) (let ((binding (assoc exp alist)))
			   (cond ((null binding) exp)
				 (t (substitute-bindings (cdr binding) alist)))))
	((listp exp) (mapcar #'(lambda (term)
				 (substitute-bindings term alist))
			     exp))
	(t exp))
  )

(defun varfree (x)
  (cond ((variable? x) nil)
	((atom x) t)
	(t (and (varfree (car x)) (varfree (cdr x))))))

(defun isground! (alist)
  (dolist (b alist T)
    (when (not (varfree (cdr b)))
      (return NIL))))

; Since an alist may have chains of variables bound to other variables,
; keep looking until we get a value that's not a variable.
;  =>  obsolete?
;;; >>> slightly bugged, unless there is some restriction on the valid
;;; form of the bindings.  E.g. (lookup* '?x (match '(?x ?y) '(?y ?x))))
;;; will loop forever.

(defun lookup* (key alist)
  (let ((value (cdr (assoc key alist))))
    (cond ((null value) key)
	  ((variable? value) (lookup* value alist))
	  (t value))))


;;;======================================================================
;;;
;;; Discrimination net with variables from Charniak & McDermott.
;;;
;;; This is strange yet wonderful code.  It works with continuations,
;;; which are lexical closures.  The main function is TRAVERSE-LINK,
;;; which traverses a single link in the dnet.  If the node is a leaf
;;; (the next-link is nil), then it calls  the -fail- continuation,
;;; otherwise it calls the -succeed- one.  I'm not at all sure how
;;; efficiently this stuff is compiled on the symbolics.
;;;
;;;
;;; The interface is through the functions:
;;; INDEX item key link
;;;     Index ITEM under KEY in the dnet starting from LINK.  LINK would
;;;     normally be the top of the tree. 
;;;
;;; FETCH item link 
;;;     Return the contents of all links which match item.
;;;

;;; On the symbolics, the assoc is much faster than find, so we will
;;; make link a structure of type list, and define link-p (which is
;;; normally defined by defstruct).  If we ever had ALOT of rules, then
;;; it might pay out to use a hash-table for link-contents.
;;;

(defstruct (link
	     (:type list))
  key
  contents)

(defun link-p (l) (consp l))

(defun traverse-link (item link -succeed- -fail-)
  (declare (sys:downward-funarg -succeed- -fail-))
  (let ((L (next-link item link)))
    (if (link-p l)
	(funcall -succeed- l)
	(funcall -fail- item link -succeed-))))

(defun next-link (key link)
  (assoc key (link-contents link)))

;;  (find key (link-contents link) :key #'link-key))


;;;----------------------------------------------------------------------
;;;
;;;  FETCH and INDEX work with variables.
;;;

(defun fetch (item link)
  (let ((results '()))
    (traverse-links 
      item link
      #'(lambda (link)
	  (setq results
		(append (link-contents link) results)))
      #'(lambda (item link succeed)
	  (declare (ignore item link succeed))
	  NIL))
    results))

(defun traverse-links (item link -succeed- -fail-)
  (declare (sys:downward-funarg -succeed- -fail-))
  (cond ((variable? item)
	 (skip-exp link -succeed-))
	(T
	 (if (atom item)
	     (traverse-link item link -succeed- -fail-)
	     (labels ((traverse-cons (link)
			(traverse-links '*CONS* link
					 #'traverse-car -fail-))
		      (traverse-car (link)
			(traverse-links (car item) link
					 #'traverse-cdr
					 -fail-))
		      (traverse-cdr (link)
			(traverse-links (cdr item) link
					 -succeed-
					 -fail-)))
	       (traverse-cons link)))
	 (traverse-link '*VAR* link -succeed- -fail-))))

(defun skip-exp (link -succeed-)
  (declare (sys:downward-funarg -succeed-))
  (dolist (sublink (link-contents link))
    (cond ((not (eq (link-key sublink) '*CONS*))
	   (funcall -succeed- sublink))
	  (T (skip-exp sublink
		       #'(lambda (link)
			   (skip-exp link -succeed-)))))))

(defun index (item key link)
  (establish-links
    key link
    #'(lambda (link)
	(pushnew item (link-contents link)))
    #'(lambda (key link -succeed-)
	(let ((l (make-link :key key)))
	  (push l (link-contents link))
	  (funcall -succeed- l)))))

(defun establish-links (item link -succeed- -fail-)
  (declare (sys:downward-funarg -succeed- -fail-))
  (cond ((variable? item)
	 (traverse-link '*var* link -succeed- -fail-))
	((atom item)
	 (traverse-link item link -succeed- -fail-))
	(T
	 (labels ((establish-cons (link)
		    (traverse-link '*CONS*
				   link #'establish-car -fail-))
		  (establish-car (link)
		    (establish-links (car item)
				       link #'establish-cdr -fail-))
		  (establish-cdr (link)
		    (establish-links (cdr item)
				       link -succeed- -fail-)))
	   (establish-cons link)))))

;;;----------------------------------------------------------------------
;;;
;;; FETCH-1 does not allow for variables.
;;; It is included just for the heck of it.
;;; The corresponding index function would call traverse-links-1, and
;;; create links on failure.
;;;

(defun fetch-1 (item link)
  (let ((results '()))
    (traverse-links-1 
      item link
      #'(lambda (link)
	  (setq results
		(append (link-contents link) results)))
      #'(lambda (item link succeed)
	  (declare (ignore item link succeed))
	  NIL))
    results))

(defun traverse-links-1 (item link -succeed- -fail-)
  (if (atom item)
      (traverse-link item link -succeed- -fail-)
      (labels ((traverse-cons (link)
		 (traverse-links-1 '*CONS* link
				  #'traverse-car -fail-))
	       (traverse-car (link)
		 (traverse-links-1 (car item) link
				  #'traverse-cdr
				  -fail-))
	       (traverse-cdr (link)
		 (traverse-links-1 (cdr item) link
				  -succeed-
				  -fail-)))
	(traverse-cons link))))


#|

(setq tree (make-link :key '*top*))
(index 999 '(son jack alice) tree)
(index 888 '(son jack bill) tree)

(fetch '(son ?x alice) tree)

(index 1 '(foo a) tree)
(index 2 '(foo b) tree)

(setq pat '(plan talk-plan
		  (inform ?person1 ?person2 ?fact)
		  (achieve (at ?person1 ?person2?))
		  (say ?person1 ?fact)))
(index pat (caddr pat) tree)

|#


;;;======================================================================
;;; THE TRANSFORMATION RULES
;;;
;;; A rule is of the form:
;;; <pattern> <match-clause> -> <transformed-pattern>
;;;


;;; because our indexing scheme ignores bindings and problems due to
;;; them not working as it does retrieval, we need to check to see that
;;; the variables in the pattern can be consistently bound, and also
;;; bind them.  Because we are matching against pure data, we can just
;;; set the var to the corresponding part of the datum.

;;;
;;; We index the match-clause, a binding-test, and the transformationp
;;; under the pattern.
;;;

;;;----------------------------------------------------------------------
;;; APPLYING RULES
;;;

(defun APPLY-ALL-RULES (datum rulebase)
  (let ((results '()))
    (traverse-links 
      datum rulebase
      #'(lambda (link)
	  (let (bindings)
	    (dolist (rule (link-contents link))
	      (setq bindings (bind-vars (first rule) datum))
	      (unless (eq bindings 'failed)
		(setq bindings (check-condition
				 datum (second rule) bindings))
		(unless (eq bindings 'failed)
		  (push  (substitute-bindings (third rule) bindings)
			 results))))))
      #'false)
    results))

(defun APPLY-A-RULE (datum rulebase)
  (traverse-links
    datum rulebase
    #'(lambda (link)
	;; link.contents containts a list of rules which might match
	(let (bindings)
	  (dolist (rule (link-contents link))
	    (setq bindings (bind-vars (first rule) datum))
	    (unless (eq bindings 'failed)
	      (setq bindings (check-condition
			       datum (second rule) bindings))
	      (unless (eq bindings 'failed)
		;; Everything is cool, so return the result with
		;; bindings substituted.  No rule should ever result in
		;; the datum being returned unchanged!!!
		(RETURN-FROM APPLY-A-RULE
		  (substitute-bindings (third rule) bindings)))))))
    #'false))



(defun check-condition (datum condition alist)
  ;; Ben uses this to add bindings from the constraints (mostly), as
  ;; well as predicates such as independent or constant.
  ;; Stubb it out with a query to the user (for the moment).
  (cond ((null condition) alist)
	(T
	 (let ((reply (prompt-and-read 
			'listp
			"Checking condition: ~a.~%On datum: ~a~@
                 Input any additional bindings: " condition datum)))
	   (append reply alist)))))

;;;----------------------------------------------------------------------
;;;
;;; DEFINING RULES
;;;

(defun INDEX-RULES (rules &key rulebase (rulebase-name '*rulebase*))
  "Add rules to rulebase, and returns the rulebase."
  (or rulebase
      (setq rulebase (make-link :key rulebase-name)))
  (dolist (rule rules)
    (index-rule rule rulebase))
  rulebase)

(defun INDEX-RULE (rule  rulebase )
  (unless (or (and (= (length rule) 3) (eql (second rule) '->))
	      (and  (= (length rule) 4) (eql (third rule) '->)))
    (error "Illegal syntax in rule: ~s" rule))
  (index (list (car rule)
	       (if (eql (second rule) '->) nil (second rule))
	       (car (last rule)))
	 (car rule)
	 rulebase)
  rulebase)


;;;======================================================================
;;;
;;; USING RULES TO SIMPLIFY EXPRESSIONS
;;;


(defvar *transformation-rulebase* ())
(defvar *identity-rulebase* ())

(defun qa-simplify (exp)
  (when (listp exp)
    ;; Try to simplify the whole expression.
    (loop for result = (apply-a-rule exp *identity-rulebase*)
	  while result
	  do (setq exp result))
    ;; Now try to simplify its parts
    (when (listp exp)
      (setq exp (mapcar #'qa-simplify exp))
      ;; and do the whole thing again. I think that this should
      ;; probably be recursive.
      (loop for result = (apply-a-rule exp *identity-rulebase*)
	    while result
	    do (setq exp result))))
  exp)

(defun test ()
  (declare (special t1 t2 t3 t3 t4))
  (setq t1 '(+ 0 (+ 0 (+ (* 2 3) (^ (* 88 23) 0)))))
  (setq t2 '(+ (- 0 (- 4 (- 4 (^ 1 1)))) 1))
  (setq t3 '(+
	      (+ (^
		   (- 9 (+ 9 0))
		   (* 1 (* 1 (* 1 90))))
		 (- 0 1))
	      (+ 0 (+ (* 2 3) (* (- 4 (- 4 (^ 99 1))) (^ (* 88 23) 0) )))))
  (setq t4 `(+ ,t3 (* ,t1 ,t2)))
  (setq t5 `(+ (- 0 (+ ,t4 ,t1)) (+ ,t1 ,t4)))

  (time (qa-simplify t1))
  (time (qsim::qa-simplify t1))
  (time (qa-simplify t2))
  (time (qsim::qa-simplify t2))
  (time (qa-simplify t3))
  (time (qsim::qa-simplify t3))
  (time (qa-simplify t4))
  (time (qsim::qa-simplify t4))
  (time (qa-simplify t5)) (time (qsim::qa-simplify t5)))

(defvar *rules* ())
(defun tst ()
  (setq *rules* (make-link :key '*top*))
  (index-rule '((+ 0 ?x) -> ?x) *rules*)
  (index-rule '((+ ?x 0) -> ?x) *rules*))

(defun setup-hod-rules ()
  (setq *transformation-rulebase*
	(index-rules *transformation-rules* :rulebase-name
		     '*transformation-rules*))
  (setq *identity-rulebase*
	(index-rules *identity-rules*
		     :rulebase-name
		     '*identity-rules*))
  ;; get cdr-coded trees?
  (setq *transformation-rulebase* (copy-tree *transformation-rulebase*))
  (setq *identity-rulebase* (copy-tree *identity-rulebase*))
  nil)

  
; These are the transformation rules for deriving Higher-Order Derivative constraints
; (sometimes called Curvature-at-Steady Constraints).
;   - The first clause in the rule is matched against the sd2 expression.
;   - Additional clauses before "->" are matched against QDE constraints, after substitutions.
;   - The clause after the "->" has bindings substituted, and is returned.

(defparameter *transformation-rules*
	      '(((sd2 ?x) (M+ ?x ?y) -> (sd2 ?y))
		((sd2 ?y) (M+ ?x ?y) -> (sd2 ?x))
		;
		((sd2 ?x) (M- ?x ?y) -> (- 0 (sd2 ?y)))
		((sd2 ?y) (M- ?x ?y) -> (- 0 (sd2 ?x)))
		;
		((sd2 ?z) (add ?x ?y ?z) -> (+ (sd2 ?x) (sd2 ?y)))
		((sd2 ?x) (add ?x ?y ?z) -> (- (sd2 ?z) (sd2 ?y)))
		((sd2 ?y) (add ?x ?y ?z) -> (- (sd2 ?z) (sd2 ?x)))
		;
		((sd2 ?z) (mult ?x ?y ?z) -> (+ (* ?y (sd2 ?x))
						(+ (* ?x (sd2 ?y))
						   (* 2 (* (sd1 ?x) (sd1 ?y))))))
		((sd2 ?x) (mult ?x ?y ?z) -> (- (/ (sd2 ?z) ?y)
						(- (* 2 (* (sd1 ?z)
							   (/ (sd1 ?y) (^ ?y 2))))
						   (- (* 2 (* ?z (/ (^ (sd1 ?y) 2)
								    (^ ?y 3))))
						      (* ?z (/ (sd2 ?z) (^ ?y 2)))))))
		((sd2 ?y) (mult ?x ?y ?z) -> (- (/ (sd2 ?z) ?x)
						(- (* 2 (* (sd1 ?z)
							   (/ (sd1 ?x) (^ ?x 2))))
						   (- (* 2 (* ?z (/ (^ (sd1 ?x) 2)
								    (^ ?x 3))))
						      (* ?z (/ (sd2 ?z) (^ ?x 2)))))))
		;
		((sd2 ?x) (minus ?x ?y) -> (- 0 (sd2 ?y)))
		((sd2 ?y) (minus ?x ?y) -> (- 0 (sd2 ?x)))
		;
		((sd2 ?x) (d/dt ?x ?y) -> (sd1 ?y))
		))
; => extend to make these state-dependent, and handle S+/S-.

(defparameter *identity-rules*
	      '(((+ 0 ?x) -> ?x)		; additive and multiplicative identities
		((+ ?x 0) -> ?x)
		((- ?x 0) -> ?x)
		((* 0 ?x) -> 0)
		((* ?x 0) -> 0)
		((/ 0 ?x) -> 0)
		((* 1 ?x) -> ?x)
		((* ?x 1) -> ?x)
		((/ ?x 1) -> ?x)
		((^ ?x 0) -> 1)
		((^ 0 ?x) -> 0)
		((^ ?x 1) -> ?x)

		((- 0 (- 0 ?x)) -> ?x)		; strict simplifications
		((- 0 (- ?x ?y)) -> (- ?y ?x))
		((- 0 (+ (- 0 ?x) ?y)) -> (- ?x ?y))
		((- ?x (- 0 ?y)) -> (+ ?x ?y))
		((- ?x (- ?x ?y)) -> ?y)
		((- ?x (+ ?x ?y)) -> (- 0 ?y))
		((- ?x (+ ?y ?x)) -> (- 0 ?y))
		((- (+ ?x ?y) ?x) -> ?y)
		((- (+ ?y ?x) ?x) -> ?y)
		((+ ?x (- ?y ?x)) -> ?y)
		((+ (- ?y ?x) ?x) -> ?y)
		((- (- ?x ?y) ?x) -> (- 0 ?y))

		((+ ?x ?x) -> (* 2 ?x))		; preferable form
		((+ (+ ?x ?y) ?z) -> (+ ?x (+ ?y ?z)))
		((- (- ?x ?y) ?z) -> (- ?x (+ ?y ?z)))
		((* (* ?x ?y) ?z) -> (* ?x (* ?y ?z)))
		;
		((sd2 ?x) (constant ?x) -> 0)
		((sd2 ?x) (independent ?x) -> 0)	; final simplifications
		((sd1 ?x) (chatter ?x) -> 0)
		))

(defvar *r nil)
(defun tst1 ()
  (setq *r (make-link :key 'top))
  (rindex 1 '(a) *r)
  (rindex 2 '(?x) *r))

(defun tst2 ()
  (setq *r (make-link :key 'top))
  (rindex 1 '(a b) *r)
  (rindex 2 '(a ?x) *r)
  (rindex 3 '(a (c d) e) *r)
  (rindex 4 '(a ?x e) *r))


(defun tst3 ()
  (setq *rules* (make-link :key 'top))
  (rindex 1 '(a b) *rules*)
  (rindex 2 '(a b c) *rules*)
  (rindex 3 '(a d) *rules*)
  (rindex 4 '(a (b d) d) *rules*))

(defun rindex (ignore pat rs)
  (index pat pat rs))
