;;;  -*- Mode:Common-Lisp; Package:QSIM; Syntax:COMMON-LISP; Base:10 -*-
;;;  $Id: cfilter.lisp,v 1.5 1992/07/24 15:06:20 clancy Exp $
;;;  Dan Dvorak, 1989.
;;;
;;;  This is an experimental version of the QSIM cfilter, designed to
;;;  process large QDE models in less time and with less cons'ing.

(in-package 'QSIM)


;;;-----------------------------------------------------------------------------
;;;  Function:  (cfilter-init  qde)
;;;
;;;  Purpose:   Given the qde, this function performs all initialization on the
;;;             VARIABLE and CONSTRAINT structure instances, in preparation for
;;;             cfilter.
;;;
;;;  Returns:   the list of constraints that must be filtered.
;;;-----------------------------------------------------------------------------

(defun cfilter-init (qde)
  (let ((active-constraints nil))

    ;; Initialize some working values.
    (dolist (var (qde-variables qde))
      (setf (variable--npvals  var) (length (variable--pvals var))
	    (variable--onpvals var) nil)
      ;; Is this needed any more?
      (when (or (variable-independent-p var)
		(variable-discrete-p var))
	(setf (variable--qval var) (car (variable--pvals var)))))

    ;; Install the time value.  If "time" was known to be a variable in some
    ;; constraint, then this would not be necessary, but "time" is rarely in
    ;; a constraint.
    (let ((time-var (time-variable qde)))
      (setf (variable--qval time-var) (first (variable--pvals time-var))))

    (activate-moded-constraints qde)

    ;; Reset all tuple lists and collect all active constraints.
    (dolist (con (qde-constraints qde))
      ;; Contents of slot tested in Incremental-Waltz-Filter
      (setf (constraint--tuples con) nil)
      (if (constraint-active-p con)
	  (push con active-constraints)))

    active-constraints))

;;; Redefined by DClancy 20July92 to pass a state to eval-mode-expr

(defun activate-moded-constraints (qde &optional (state nil))
  ;; For each constraint that is conditional on a mode value, get the
  ;; mode-expressions for the constraint and see if any of them eval true.
  (let ((constraints-within-modes (qde-constraints-within-modes qde)))
    (when constraints-within-modes
      (dolist (con constraints-within-modes)
	(setf (constraint-active-p con)
	      (if (some #'(lambda (expr) (eval-mode-expr expr state))
			(constraint-mode-expressions con))
		  T
		  NIL))
	))))


;;; OLD DEFINITION
;(defun activate-moded-constraints (qde)
;  ;; For each constraint that is conditional on a mode value, get the
;  ;; current mode value to determine if the constraint should be marked
;  ;; active or not.
;  (let ((constraints-within-modes (qde-constraints-within-modes qde)))
;    (when constraints-within-modes
;      (dolist (con constraints-within-modes)
;	(setf (constraint-active-p con) nil))
;      (let (modes mode-qmag)
;	(dolist (var (qde-variables qde))
;	  (when (setq modes (variable-modes var))
;	    ;; Note: mode variables can have only one pval.
;	    (setq mode-qmag (qmag (variable--qval var)))
;	    (dolist (con (alookup mode-qmag modes))
;	      (setf (constraint-active-p con) t))))))))

;;; Added by BKay 9Sept91
;;; Evaluate a mode expression.  Return NIL if it fails.
;;;
;;; Modified to take an optional state argument DJC 20 July 92

(defun eval-mode-expr (expr &optional (state nil))
  (cond
    ((mode-designator-struct expr)
     (eval-mode-designator expr state))
    ((eq (car expr) 'NOT)
     (not (eval-mode-expr (second expr) state)))
    ((eq (car expr) 'AND)
     (every #'(lambda (arg) (eval-mode-expr arg state)) (cdr expr)))
    ((eq (car expr) 'OR)
     (some #'(lambda (arg) (eval-mode-expr arg state))  (cdr expr)))
    (t
     (error "Unknown mode expression"))))


;;; Added by BKay 9Sept91
;;; Modified to take an optional state argument DJC 20 July 92
(defun eval-mode-designator (md &optional (state nil))
  (let* ((var       (first md))
	 (mode-qmag (second md))
	 (qmag      (qmag (if state
			      (get-qval (variable-name var) state)
			      (variable--qval var)))))
    (equal qmag mode-qmag))) ; should this be eq?

;;; Added by BKay 9Sept91
(defun mode-designator-struct (expr)
  (variable-p (car expr)))

;;;-----------------------------------------------------------------------------
;;;  Function:  (cfilter  qde  constraint-check-function)
;;;
;;;  Given:     -- qde, the qde structure where the full set of variables and
;;;                   constraints are anchored.
;;;             -- constraint-check-function, such as #'check-qsim-constraint;
;;;             -- It is required that the caller have already stored the
;;;                possible values for each variable in its variable--pvals slot.
;;;
;;;  Returns:   -- value-completions, a list of lists of qvals, where each list
;;;                  of qvals contains one qval for each variable of the qde.
;;;                  Each list of qvals is required to be in the same order as
;;;                  the list of variables in the qde (in order to facilitate
;;;                  efficient global filtering).
;;;
;;;  Design:    Generate possible values for all variables.
;;;             While constraints remain
;;;             {
;;;                Get the "most restrictive" remaining constraint.
;;;                Delete this constraint from the list of remaining constraints.
;;;                Generate all tuples for this constraint.
;;;                Filter the tuples through the constraint.
;;;                Prune the pvals for each variable of this constraint 
;;;                  (based on the vals appearing in the surviving tuples).
;;;                Waltz-filter the partial set of tuples.
;;;             }
;;;             Form all states from the surviving tuples, done so as to
;;;               minimize backtracking.
;;;
;;;  Note:      All callers of cfilter are required to bind the special
;;;             variable Current-State so that cfilter's subordinates
;;;             (such as constraint check functions) can access the current
;;;             state.  This is provided for use in some experimental 
;;;             constraints; it's not used by any "standard" constraints.
;;;-----------------------------------------------------------------------------


(defun cfilter (*current-qde* constraint-check-function)
  (declare (special *current-qde*))
  (start-timer 'cfilter)
  (let* ((states        nil)
	 (myconstraints (cfilter-init *current-qde*))
	 con tuples1 tuples2)

    ;; While any constraints remain unprocessed ...
    (do ()
	((null myconstraints))
      ;; Get most restrictive remaining constraint
      (setq con (most-restrictive-constraint myconstraints))

      ;; Delete this constraint from the list of unprocessed constraints.
      (setq myconstraints (delete con myconstraints :count 1))
      
      ;; Generate and filter the tuples for this constraint.
      (setq tuples1 (product-space (mapcar #'variable--pvals (constraint-variables con))))
      (trace-current-tuples con tuples1 "before constraint filtering")
      (setq tuples2 (filter-one-tuple-set con tuples1 constraint-check-function))
      (trace-current-tuples con tuples2 "after constraint filtering")
      (unless tuples2
	;; Mark this constraint so that print-completions can inform the
	;; user where the problem was first discovered.
	(setf (constraint--tuples con) 'no-tuples-survived)
	(stop-timer 'cfilter)
	(return-from cfilter nil))
      (setf (constraint--tuples con) tuples2)

      ;; Do Waltz filtering on the partial constraint network.
      (unless (incremental-Waltz-filter con nil)
	(return-from cfilter nil)))

    (start-timer 'form-all-states)
    (setq states (form-all-states *current-qde*))
    (stop-timer 'form-all-states)
    (stop-timer 'cfilter)

    states))
  

;;;  Trace function for tuples.

(defun trace-current-tuples (con tuples when)
  (cond (*trace-tuples*
	 (format *qsim-report* "~%Number of tuples is ~a" (length tuples))
	 (let ((con-name (constraint-name con)))
	   (format *qsim-report* "~%Set of tuples for ~a ~a:" con-name when)
	   (format *qsim-report* "~%~20t~a" tuples)))))
    


;;;-----------------------------------------------------------------------------
;;;  Function:  (most-restrictive-constraint  constraints)
;;;
;;;  Given:     a list of constraint instances
;;;
;;;  Returns:   the constraint that is estimated to produce the fewest tuples.
;;;
;;;  Design:    This function estimates the number of tuples that a constraint
;;;             will produce by simply computing the product of the number of
;;;             possible values of its variables.  In effect, this heuristic
;;;             favors constraints having few variables and few possible values
;;;             for those variables.
;;;-----------------------------------------------------------------------------

(defun most-restrictive-constraint (constraints)
  (let ((minproduct most-positive-fixnum)
	minconstraint)
    (dolist (con constraints minconstraint)
      (let ((product (variable--npvals (first (constraint-variables con)))))
	(dolist (variable (rest (constraint-variables con)))
	  (setq product (* product (variable--npvals variable))))
	(if (= 1 product) (return con))		; return early if product = 1
	(if (< product minproduct)
	    (setq minproduct product
		  minconstraint con))))))



;;;-----------------------------------------------------------------------------
;;;  Function:  (update-pvals  con  notvariable)
;;;
;;;  Given:     -- con, a constraint
;;;             -- notvariable, a variable of this constraint whose pvals need not
;;;                be updated (since the caller had just updated that variable's
;;;                pvals [and then subsequently deleted one or more tuples
;;;                from this constraint]).
;;;
;;;  Purpose:   Updates the possible values for each variable of the constraint.
;;;
;;;  Returns:   T if each variable has at least one pval, else nil.
;;;
;;;  Design:    For each variable of this constraint (except the variable that the
;;;             caller "came from"), delete from the variable's list of pvals any
;;;             pval that does not appear in at least one of the constraint's
;;;             tuples.
;;;-----------------------------------------------------------------------------
	
(defun update-pvals (con notvariable)
  (let ((tuple-set  (constraint--tuples con)))

    ;; For each variable of this constraint ...
    (do ((variables (constraint-variables con) (cdr variables))
	 (n 0 (1+ n)))
	((endp variables) t)

      (let ((variable (car variables)))
	(if (not (eq variable notvariable))
	    (let ((pruned-pvals (delete-if-not
				  #'(lambda (pval)
				      (member-if #'(lambda (tuple)
						     (qval-equal pval (nth n tuple)))
						 tuple-set)) 
				  (variable--pvals variable))))
	      ;; If all pvals eliminated, return.
	      (when (null pruned-pvals)
		;; Record where the problem occurred so that print-completions
		;; can inform the user.
		(setf (variable--pvals variable) 'no-pvals-survived)
		(return-from update-pvals nil))
	      (setf (variable--pvals variable)   pruned-pvals
		    (variable--npvals variable)  (length pruned-pvals))
	      (if (null (variable--onpvals variable))
		  (setf (variable--onpvals variable) (variable--npvals variable)))))))))



;;;-----------------------------------------------------------------------------
;;;  Function:  (incremental-Waltz-filter  con  notvariable)
;;;
;;;  Given:     -- con, a constraint to start from
;;;             -- notvariable, a variable that the caller knows does not need to
;;;                have its pvals updated.
;;;
;;;  Purpose:   Deletes all tuples that are not pair-wise consistent with all
;;;             neighboring tuples.
;;;
;;;  Returns:   T if each constraint has at least one surviving tuple, else NIL.
;;;
;;;  Design:    This is a recursive function.  Starting from the given
;;;             constraint, it first updates the pvals of that constraint.
;;;             Then, for those variables of the constraint whose pvals were
;;;             actually reduced, it checks the tuples of each attached
;;;             constraint.  If any tuples contain values that are no longer
;;;             covered in the pvals, then those tuples are deleted and a
;;;             recursive call is made on the affected constraint.
;;;-----------------------------------------------------------------------------

(defun INCREMENTAL-WALTZ-FILTER (con notvariable)

  ;; Return immediately if some variable has no more pvals.
  (if (null (update-pvals con notvariable))
      (return-from incremental-Waltz-filter nil))

  ;; For each variable of the constraint, check its constraints only if its
  ;; number of pvals has decreased.
  (dolist (var (constraint-variables con) T)
    (if (< (variable--npvals var) (variable--onpvals var))
	(let ((pvals (variable--pvals var)))
	  (setf (variable--onpvals var) (variable--npvals var))

	  ;; For each constraint attached to this variable ...
	  (dolist (ncon (alookup (qde-name *current-qde*)
				 (variable-constraints var)))
	    (if (and (not (eq con ncon))
		     (constraint-active-p ncon))
		(let* ((index (position var (constraint-variables ncon)))
		       (changed nil)
		       (tuples (constraint--tuples ncon)))
		  ;; If the tuples have not been generated yet for this constraint,
		  ;; [(constraint--tuples ncon) is NIL] then skip it for now.
		  (when tuples
		    (setq tuples
			  (delete-if-not
			    #'(lambda (tuple)
				(cond ((member (nth index tuple) pvals :test #'qval-equal))
				      (t (waltz-tracer tuple index pvals)
					 (setq changed t)
					 nil)))
			    tuples))

		    (setf (constraint--tuples ncon) tuples)
		    ;; If no more tuples, return.
		    (when (null tuples)
		      ;; Record where the problem occurred so that print-completions
		      ;; can inform the user.
		      (setf (constraint--tuples ncon) 'no-tuples-survived)
		      (return-from incremental-Waltz-filter nil))
		    (if changed
			(incremental-Waltz-filter ncon var))))))))))


(defun WALTZ-TRACER (tuple index pvals)
  (if *trace-tuples*
      (format *Qsim-Report* "~%Waltz: Deleting tuple ~a~%    because ~a not in pvals ~a"
	      tuple (nth index tuple) pvals)))

;;;-----------------------------------------------------------------------------
;;;  Function:  (form-all-states  qde)
;;;
;;;  Given:     the qde
;;;
;;;  Returns:   a list of consistent value assignments for each variable.
;;;
;;;  Purpose:   This function, which is run after Waltz filtering, takes the
;;;             surviving tuples of each constraint and forms all possible
;;;             consistent value completions.
;;;
;;;  Design:    The basic idea in this algorithm is that we pick some constraint,
;;;             bind its variables using the next tuple of the constraint, and
;;;             then move on to another unprocessed constraint that is connected
;;;             to one of the processed constraints.  Of the candidate constraints
;;;             on this "boundary", we always take the one with the fewest tuples.
;;;             By moving in a "connected order" from constraints having few
;;;             tuples to constraints having many tuples, value completions are
;;;             formed recursively with a minimum amount of backtracking.
;;;             Here are the basic steps:
;;;
;;;             1.  For every constraint having only one tuple, bind its
;;;                 variables "permanently" since they must be the same for all
;;;                 completions.
;;;             2.  From the list of constraints processed in step 1, form the
;;;                 initial boundary consisting of directly-connected constraints
;;;                 that were not processed in step 1.
;;;             3.  Find the constraint having the fewest tuples in the boundary
;;;                 and delete it from the boundary.  Watch out for two special
;;;                 cases here -- the boundary could be nil either because there
;;;                 were no 1-tuple constraints or because ALL constraints had
;;;                 only one tuple.  In the latter case we're all done.
;;;             4.  Call (recursive-form-states constraint boundary).  The
;;;                 remaining steps are inside this recursive function.
;;;             5.  Temporarily bind any unbound variables of this constraint.
;;;                 Add to the boundary the unprocessed constraints that are
;;;                 directly connected to these newly bound variables.
;;;             6.  If boundary is empty, then we're all done.  Otherwise, find
;;;                 the constraint having the fewest tuples in the boundary and
;;;                 remove it.  Call (recursive-form-states constraint boundary).
;;;
;;;  Note:      This function could have been written in a very simple recursive
;;;             way to bind the variables of the first constraint, then do the
;;;             second constraint, etc, backtracking whenever an inconsistency
;;;             occurred.  This would probably be fine for small constraint
;;;             networks, but the amount of backtracking grows exponentially
;;;             with the number of constraints.
;;;-----------------------------------------------------------------------------

(defun form-all-states (qde)
  (let* ((variables    (qde-variables qde))
	 (constraints  (qde-constraints qde))
	 (*all-states* nil)
	 (starting-constraints nil)
	 (num-remaining-constraints (length constraints)))
    (declare (special variables *all-states*))

    ;; Step 0 -- Prepare 'variable' structures (leaving "time" variable alone).
    ;;           If a variable has only one possible value, then bind it now.
    ;;           This ensures that discrete variables are bound since they
    ;;           don't necessarily participate in constraints.
    (dolist (var (cdr variables))
      (setf (variable--qval var) (if (= 1 (variable--npvals var))
				     (car (variable--pvals var))
				     nil)))

    ;; Step 1 -- For all 1-tuple constraints, permanently bind the associated
    ;;           variables.
    (dolist (con constraints)
      (if (not (constraint-active-p con))
	  (progn
	    (setf (constraint-done con) t)	; want to skip inactive constraints
	    (decf num-remaining-constraints))
	  (progn
	    (setf (constraint-done con) nil)
	    (let ((tuples (constraint--tuples con)))
	      (when (singleton-p tuples)
		(setf (constraint-done con) t)
		(push con starting-constraints)
		(decf num-remaining-constraints)
		;; Bind variable values from this tuple.
		(mapc #'(lambda (var tval)
			  (let ((qval (variable--qval var)))
			    (if (null qval)
				(setf (variable--qval var) tval)
				(if (not (qval-equal qval tval))
				    (error "Inconsistent values.")))))
		      (constraint-variables con)
		      (first tuples)))))))
	
    ;; If all constraints were processed above (rare, but possible),
    ;; then form the single state and return.
    (when (= 0 num-remaining-constraints)
      (recursive-form-states nil)
      (return-from form-all-states *all-states*))

    ;; Step 2 -- From the list of constraints processed in step 1, form the
    ;;           initial boundary consisting of directly-connected constraints
    ;;           that were not processed in step 1.
    (let ((ordered-constraints nil)
	  (boundary nil)
	  nextcon)

      ;; Build initial boundary.
      (dolist (con starting-constraints)
	(dolist (ncon (constraint-neighbors con))
	  (if (not (constraint-done ncon))
	      (pushnew ncon boundary))))

      ;; Loop over remaining constraints to build an ordered list of
      ;; constraints for recursive formation of states.
      (dotimes (n num-remaining-constraints)

	;; If a boundary (still) exists ...
	(if boundary

	    ;; then find the constraint in the boundary having the fewest tuples
	    ;; and update the boundary accordingly.
	    (multiple-value-setq (nextcon boundary) (find-min-constraint boundary))

	    ;; else either there was no initial boundary (because there was no
	    ;; 1-tuple constraint) or else the constraint network consists of
	    ;; two or more unconnected networks, and we must now jump over to
	    ;; another network.
	    (multiple-value-setq (nextcon boundary) (find-min-constraint-2 constraints)))

	(push nextcon ordered-constraints))

      ;; Go form the states.
      (recursive-form-states (nreverse ordered-constraints))

      ;; Return the states, if any, created in the recursive function.
      *all-states*)))




;;;-----------------------------------------------------------------------------
;;;  Function:  (recursive-form-states  remaining-constraints)
;;;
;;;  Given:     -- a list of constraints remaining to be checked in forming
;;;                a state.
;;;
;;;  Returns:   nothing of interest, but the last (deepest) recursive call
;;;               forms a list of value completions and pushes that list onto
;;;               *all-states*.
;;;
;;;  Design:    If no more constraints, push value completions onto *all-states*
;;;             and return.  Otherwise, take the first remaining constraint,
;;;             and check/bind its variables.  If there is a contradiction
;;;             such that one of its variables cannot be bound, then return.
;;;             Otherwise, do a recursive call with the tail of the list of
;;;             constraints.
;;;-----------------------------------------------------------------------------

(defun recursive-form-states (remaining-constraints)
  (declare (special variables *all-states*))
  (declare (optimize speed))

  ;; If no more constraints ...
  (when (null remaining-constraints)
    ;;(format *qsim-report* "~%RFS: state = ~a" (mapcar #'variable--qval variables))

    ;; then push the state values onto *all-states* and return.
    (push (mapcar #'(lambda (var)
		      (cons (variable-name var) (variable--qval var)))
		  variables)
	  *all-states*)
    (return-from recursive-form-states (values)))

  ;; For each tuple of this constraint ...
  (let ((con (car remaining-constraints))
	val)
    (dolist (tuple (constraint--tuples con))
      (let ((assigned-variables nil))

	(if (do ((vars (constraint-variables con) (cdr vars))
		 (tvals tuple (cdr tvals)))
		((endp vars) t)
	      (if (setq val (variable--qval (car vars)))
		  (unless (qval-equal val (car tvals))
		    ;;(format *qsim-report* "~%Conflict: ~a = ~a, attempted ~a"
		    ;;	    (variable-name (car vars)) (variable--qval (car vars)) (car tvals))
		    (return nil))
		  (let ((var (car vars)))
		    ;;(format *qsim-report* "~%Assigning ~a = ~a" (variable-name var) (car tvals))
		    (setf (variable--qval var) (car tvals))
		    (push var assigned-variables))))

	    ;; We've accepted this tuple of this constraint, so now we can
	    ;; proceed to the next constraint.
	    (recursive-form-states (cdr remaining-constraints)))
	      
	;; Unassign the variable values that were tentatively assigned.
	(dolist (var assigned-variables)
	  (setf (variable--qval var) nil))))))




;;; Returns 2 values:  constraint having fewest tuples in boundary,
;;; and updated boundary.
(defun find-min-constraint (boundary)
  (let ((min-num-tuples most-positive-fixnum)
	(min-constraint nil)
	n)
    (dolist (con boundary)
      (when (< (setq n (length (constraint--tuples con))) min-num-tuples)
	(setq min-num-tuples n
	      min-constraint con)))
    (setq boundary (delete min-constraint boundary :count 1))
    (setf (constraint-done min-constraint) t)
    (dolist (con (constraint-neighbors min-constraint))
      (if (not (constraint-done con))
	  (pushnew con boundary)))
    (values min-constraint boundary)))


;;; Of the unprocessed constraints, return the one having the fewest tuples.
(defun find-min-constraint-2 (constraints)
  (let ((min-num-tuples most-positive-fixnum)
	(min-constraint nil)
	(boundary nil)
	n)
    (dolist (con constraints)
      (when (and (not (constraint-done con))
		 (< (setq n (length (constraint--tuples con))) min-num-tuples))
	(setq min-num-tuples n
	      min-constraint con)))
    (setf (constraint-done min-constraint) t)
    (dolist (con (constraint-neighbors min-constraint))
      (if (not (constraint-done con))
	  (pushnew con boundary)))
    (values min-constraint boundary)))


    


; PRODUCT-SPACE enumerates the points generated by a set of axes,
; each of which is a list of values.  It is used to make the tuples.

(defun product-space (axes)
  (cond ((null axes) (list nil))
	(t (add-axis (car axes) (product-space (cdr axes))))))

(defun add-axis (axis subspace)
  (mapcan #'(lambda (point) (add-point point subspace))
	  axis))

(defun add-point (point subspace)
  (mapcar #'(lambda (vector) (cons point vector))
	  subspace))




; This scans down the set of tuples, deleting the ones that fail to satisfy
; the constraint.

(defun filter-one-tuple-set (con tuples constraint-check-function)
  (let ((*detailed-printing* nil))
    (if *trace-count*
	(format *Qsim-Trace* "~%Constraint filter ~s: (~a) "
		con (length tuples))))
  (setq tuples (delete-if
		 #'(lambda (tuple)
		     (unless (funcall constraint-check-function tuple con)
		       (if *trace-constraint-filter*
			   (format *Qsim-Trace* "~%Constraint ~a filtering out tuple ~a."
				   (constraint-name con) tuple))
		       t))	; Return t so this one is deleted.
		 tuples))
  (if *trace-count* (format *Qsim-Trace* "-> (~a)" (length tuples)))
  tuples)

