;;; -*- Mode:Common-Lisp; Package:QSIM; Base:10 -*-
;;;  $Id: arithmetic.lisp,v 1.12 1992/07/17 15:51:48 bert Exp $

(in-package 'QSIM)

; arithmetic.lisp.
; Copyright (c) 1990 Benjamin Kuipers

; Modified 3Sept91 by BKay to make vlps (<var> <lmarkexp>) rather than
; (varname lmarkname) [This is actually a nop for this file as vlps
; aren't mentioned in here.]
; Modified 9Sept91 by BKay to use multiplicative interval rounding a la Moore.
; This method multiplies positive upper bounds an negative lower bounds of
; intervals by (1.0 + *rounding-epsilon*).  Rounding is performed by the
; function round-interval.  It will only round floating-point values.  This
; function is called whenever a new interval is created (by int).  As long
; as every arithmetic operation calls int before returning an interval, this
; is the only place where round need be dealt with.
; Modified 27May92 to fold round-interval into the int call to save consing.
;;; Modified 5May92 by BKay to include multivariate M constraints
;;; (see equations.lisp for details).



;  The Q2 interval arithmetic package, that supports reasoning
; with incomplete quantitative knowledge (in the form of intervals) about 
; qualitative behaviors (by propagation across constraints).
;
; RANGER* is the principal access function.
;
; Uses globals:
;   *bindings*     -  assocations  <vlp> -> <range>  (used within get-range & update-range)
;   *current-QDE*  -  for finding monotonic function envelopes
;
; Calls external functions
;   VLP            -  test for a vlp
;   GET-RANGE      -  retrieve <vlp> -> <range>
;   UPDATE-RANGE   -  set      <vlp> -> <range>
;   GET-ENVELOPES

; The code consists of several chapters:
;   1.  RANGER* --- the quantitative range interpreter
;   2.  RANGER  --- a simpler interval-arithmetic interpreter
;   3.  Interval-arithmetic routines.
;   4.  Extended scalar arithmetic, over [-inf,+inf].
;   5.  Retrieve monotonic function envelopes.

; Parameters.

; The rounding is to make sure that interval arithmetic is sound.
; This is bigger than CommonLisp single-float-epsilon [Steele, 2e, p.370].

;;; Rounding and propagation epsilons.
;;; 
;;; The variable *rounding-epsilon* controls the correctness of interval
;;; computations by rounding the interval up and down by (1 +/- *rounding-epsilon*)
;;; for positive intervals (and similarly for negative intervals).  See the
;;; function int.
;;;
;;; The variable *epsilon* is used to determine if a two ranges are essentially
;;; identical so that the Q2 interval refinement process reaches a fixpoint.
;;; These two variables do interact however, and setting them to reasonable
;;; values is vital for correct Q2 performance.  It is used by the function
;;; insignificant-difference.
;;;
;;; First off, *rounding-epsilon* <= *epsilon* or else every interval rounding will
;;; generate a significant difference that will cascade outwards producing 
;;; ever-bigger intervals.  Secondly, *rounding-epsilon* >= double-float-epsilon
;;; or else the rounding will have no effect.
;;;
;;; A good compromise is to have *epsilon* = (1.e+4) * (*rounding-epsilon*).
;;; This means that interval rounding will not normally cause a significant 
;;; difference event.  Ideally, I thought that this should mean that 10000
;;; int operations could be performed before their effect would be seen
;;; by insignificant-difference.  Tests on the performance of Q2, however,
;;; seem to suggest that this hypothesis is incorrect since factors as large
;;; as 1.e+6 still seem to reduce the amount of work done by Q2.
;;;
(defparameter *rounding-epsilon* 1.0D-10)	
(defparameter *epsilon* 1.0D-6)			; minimum change to propagate

(defparameter *trace-ranger* nil)		; trace the range interpreter
	

#| This has been folded into int.
;;; Round an interval.
;;; Inputs:  an interval
;;; Returns: The same interval, with endpoints possibly rounded according to
;;;          the rule "positive right endpoints round up, negative left
;;;          endpoints round down."
;;; Notes:   Only floats are rounded.
;;;          This function added by BKay 9Sept91.
;;;
(defun round-interval (int)
  (let ((lo (lo int))
	(hi (hi int)))
    (setf (lo int)
	  (etypecase lo
	    (symbol  lo)
	    (integer lo)
	    (ratio   lo)
	    (float   (if (< lo 0) (* lo (+ 1.0 *rounding-epsilon*)) lo))))
    (setf (hi int)
	  (etypecase hi
	    (symbol  hi)
	    (integer hi)
	    (ratio   hi)
	    (float   (if (> hi 0) (* hi (+ 1.0 *rounding-epsilon*)) hi))))
    int))
|#

;;;  Placed the macros for hi and lo in qutils   DJC 06/4/91

;   1.  RANGER* --- the quantitative range interpreter
;
; This range-expression interpreter evaluates expressions and updates ranges.
; It differs from RANGER in that:
;   - it must look up values indexed under <vl> in a two-level alist;
;   - it must handle (d <l> <l>) expressions.  
;   - it handles >, >=, <, <=, treating the strict and non-strict inequalities identically.

(defun ranger* (exp)
  (let ((value
	  (cond ((numberp exp) (int exp exp))
		((atom exp) (error "Bad expression:  ~a."  exp))
		((vlp exp) (get-range exp))
		(t (ecase (car exp)
		     (=         (update-range (cadr exp)
					      (ranger* (caddr exp))))
		     ((<= <)    (update-range (cadr exp)
					      (int '-inf (hi (ranger* (caddr exp))))))
		     ((>= >)    (update-range (cadr exp)
					      (int (lo (ranger* (caddr exp))) '+inf)))
		     (+         (plus-handle (cdr exp)))
		     (-         (neg-handle (cdr exp)))
		     (*         (mult-handle (cdr exp)))
		     (/         (idiv (ranger* (cadr exp))
				      (ranger* (caddr exp))))
		     (abs       (iabs (ranger* (cadr exp))))
		     (sum       (isum (mapcar #'ranger* (cdr exp))))
		     (span-of   (span-of (ranger* (cadr exp))
				         (ranger* (caddr exp))))
		     (square    (isquare (ranger* (second exp))))
		     (sqrt      (isquart (ranger* (second exp))
				         (ranger* (third exp))))
		     ;; Added BKay 28Apr92
		     ;; The form of an m-function is
		     ;;  (M-function name .  vl-tuple-excluding-y)
		     (M-function (apply-MvM-fn (cadr exp) 
					       (mapcar #'ranger* (cddr exp))))
		     (function  (apply-fn (cadr exp)
					  (ranger* (caddr exp))))
		     (inverse   (apply-finv (cadr exp)
				 	    (ranger* (caddr exp))))
		     (slope     (explicit-bounds (car exp) (cadr exp)))
		     (curvature (explicit-bounds (car exp) (cadr exp))))))))
    (trace-ranger exp value)
    value
    ))



(defun trace-ranger (exp value)
  (if *trace-ranger*
      (format *QSIM-Trace* "~%Ranger:  ~a  ->  ~a" exp value)))

; RANGER* is the only interface between the range propagator and the incomplete
; quantitative knowledge representation and arithmetic package.  It can be replaced
; by either of two strategies:
;  1. Redefine:  iintersect, iplus, idiff, itimes, idiv, iabs, span-of
;                (??) apply-fn, apply-finv
;  2. Redefine RANGER* to call an alternate representation and access functions.
;     (This seems the more sensible.)


;;; Give "*" arity of 2 or more.

(defun mult-handle (args)
  (itimes (ranger* (car args))
	  (if (cddr args)
	      (mult-handle (cdr args))
	      (ranger* (cadr args)))))

;;; Give "+" arity of 2 or more.

(defun plus-handle (args)
  (iplus (ranger* (car args))
	  (if (cddr args)
	      (plus-handle (cdr args))
	      (ranger* (cadr args)))))

;;; Give "-" unfixed arity - 1 arg is negation, 2 args are subtraction,
;;; >2 args substracts the sum of the trailing args from the first arg.

(defun neg-handle (args) 
  (cond ((cddr args)(idiff (ranger* (first args))
			   (plus-handle (cdr args))))
	((cdr args)(iplus (ranger* (first args))
			  (iminus (ranger* (second args)))))
	(t (iminus (ranger* (first args))))))

;   2.  RANGER  --- a simple interval-arithmetic interpreter
;
; Here we have a range interpreter for algebraic expressions.
;   <bindings>  ::=  alist of (<variable> <interval>)

(defun range-interpreter (exp bindings)
  (let ((*bindings* bindings))
    (declare (special *bindings*))
    (ranger exp)))

(defun ranger (exp)
  (declare (special *bindings*))		; expects bindings on global variable
  (let ((value
	  (cond ((symbolp exp) (or (cadr (assoc exp *bindings*))
				   (error "No binding for ~a in ~a." exp *bindings*)))
		((numberp exp) (int exp exp))
		(t (ecase (car exp)
		     (+ (iplus (ranger (cadr exp))
			       (ranger (caddr exp))))
		     (- (idiff (ranger (cadr exp))
			       (ranger (caddr exp))))
		     (* (itimes (ranger (cadr exp))
				(ranger (caddr exp))))
		     (/ (idiv (ranger (cadr exp))
			      (ranger (caddr exp))))
		     (abs (iabs (ranger (cadr exp))))
		     (span-of (span-of (ranger (cadr exp))
				       (ranger (caddr exp))))
		     (function (apply-fn (cadr exp)
					 (ranger (caddr exp))))
		     (inverse  (apply-finv (cadr exp)
					   (ranger (caddr exp)))))))))
    (trace-ranger exp value)
    value))


; 3.  Interval-arithmetic routines.
;
; An interval is defined as a pair (lo hi), where lo <= hi, describing a real value
; that must lie within the interval.
;   Intervals are defined over the extended real number line, [-inf,+inf],
;   but must include finite values, so +inf cannot be a lower bound, nor -inf an upper bound.

; This makes the set of intervals closed under division, branching on the cases:
;       - zero is in the interior of the interval
;       - zero is an upper or lower bound
;       - zero is outside the interval.

; The rounding is to make sure that interval arithmetic is sound.

(defun int (lo hi)
  (when (or (not (or (numberp lo) (eql lo '-inf)))
	    (not (or (numberp hi) (eql hi '+inf)))
	    (and (numberp lo) (numberp hi) 
		 ;; The precision under lucid is SO GOOD that sometimes
		 ;; the lower and upper bounds will differ (with lo > up)
		 ;; in the last few digits (like around the 16th digit of
		 ;; precision) when upper and lower bounds for the same
		 ;; value are calculated.  This test makes sure that the
		 ;; difference is significant before (> lo hi) throws
		 ;; somethng out.
		 ;; Note that since round-interval will be called anyway,
		 ;; the final interval will obey (>= lo hi).
		 (and (> (rel-error hi lo) *rounding-epsilon*)
		      (> lo hi))))
    (error "Illegal interval bounds:  [~a, ~a]." lo hi))
  
  ;; Unlike Moore, we expand the lower and upper bounds of the interval
  ;; because the floating point calculation automatically rounds up, so there
  ;; is no guarantee that a lower positive endpoint is <= the true value
  ;; (see [Moore, 79] for more info).
  (list
   (if (floatp lo)
       (if (< lo 0.0)
	   (* (the float lo) (+ 1.0 (the float *rounding-epsilon*)))
	   (* (the float lo) (- 1.0 (the float *rounding-epsilon*))))
       lo)
   (if (floatp hi)
       (if (> hi 0.0)
	   (* (the float hi) (+ 1.0 (the float *rounding-epsilon*)))
	   (* (the float hi) (- 1.0 (the float *rounding-epsilon*))))
       hi)))


; An update is insignificant if the change falls below some threshold.
; This is necessary to avoid infinite convergent sequences.

;;; This function has been replaced by the one below it.
;;; The problem is that if the new binding has a wider range than the
;;; old one, then the difference should be considered insignificant from
;;; the point of view of the update-range function.
;;; Change made by BKay 3Sept91.
;(defun insignificant-difference (xrange yrange)
;  (and (if (and (numberp (lo xrange)) (numberp (lo yrange)))
;	   (< (abs (- (lo xrange) (lo yrange))) *epsilon*)
;	   (equal (lo xrange) (lo yrange)))
;       (if (and (numberp (hi xrange)) (numberp (hi yrange)))
;	   (< (abs (- (hi xrange) (hi yrange))) *epsilon*)
;	   (equal (hi xrange) (hi yrange)))))

;;; New and improved version.
;;; Note that old-range is the binding to be modified.
;;; Note also that relative error is used to determine insignificance.
;;;
(defun insignificant-difference (new-range old-range) 
  (and (cond
	 ((eql (lo new-range) '-INF)) ; This must be insig.
	 ((and (numberp (lo new-range)) (numberp (lo old-range)))
	  ;; New range is bigger than old OR diff in ranges is too small to
	  ;; matter.
	  (or (< (lo new-range) (lo old-range))  
	      (< (rel-error (lo new-range) (lo old-range)) *epsilon*)))
	 ((equal (lo new-range) (lo old-range))))
       (cond
	 ((eql (hi new-range) '+INF))
	 ((and (numberp (hi new-range)) (numberp (hi old-range)))
	  (or (> (hi new-range) (hi old-range))  ; new range is bigger
	      (< (rel-error (hi new-range) (hi old-range)) *epsilon*)))
	 ((equal (hi new-range) (hi old-range))))))


;;; Compute the relative error abs((x-y)/x) or abs((x-y)/y).
;;;
(defun rel-error (x y)
  (if (= x y)
      0
      (abs (/ (- x y) (if (zerop x) y x)))))

; An update to a value is insignificant if the change falls below some threshold.
; This function is used exclusively in zip-up and zip-down to prevent infinite
; convergent sequences.  I use this function rather than dummying up a range and using
; insignificant-difference because I felt that that was uglier.  [I also didn't 
; want to make any assumptions about how range structures are formed.]
; This function added by BKay 7Mar91
; Modified by BKay 2Aug91 to check for relative error.
(defun insignificant-value-difference (x y)
  (if (and (numberp x) (numberp y))
      (< (rel-error x y) *epsilon*)
      (equal x y)))

;;; Old version
;(defun insignificant-value-difference (x y)
;  (if (and (numberp x) (numberp y))
;      (< (abs (- x y)) *epsilon*)
;      (equal x y)))


; Testing sign

(defun ipos (range)
  (and (numberp (lo range))
       (> (lo range) 0)))

(defun ineg (range)
  (and (numberp (hi range))
       (< (hi range) 0)))

(defun iabs (range)
  (cond ((ipos range) range)
	((ineg range) (iminus range))
	(t					; (<= (lo range) 0 (hi range))
	 (int 0 (cond ((eql (lo range) '-inf) '+inf)
		      ((eql (hi range) '+inf) '+inf)
		      (t (max (abs (lo range)) (abs (hi range)))))))))

; Addition, subtraction, negation, and absolute-value are straight-forward.

(defun iplus (xrange yrange)
  (int (if (and (numberp (lo xrange)) (numberp (lo yrange)))
	   (+ (lo xrange) (lo yrange))
	   '-inf)
       (if (and (numberp (hi xrange)) (numberp (hi yrange)))
	   (+ (hi xrange) (hi yrange))
	   '+inf)))

(defun iminus (range)
  (int (if (eql (hi range) '+inf)  '-inf  (- (hi range)))
       (if (eql (lo range) '-inf)  '+inf  (- (lo range)))))

(defun idiff (xrange yrange)
  (iplus xrange (iminus yrange)))

;;; Modified by BKay 27May92 based on code from af.
;;; This should save consing.
;;;
(defun isum (L)
  (loop with lb = 0 with minf? = nil
	with ub = 0 with inf? = nil
	for (l u) in L
	do
	(if (eq l '-inf)
	    (setq minf? `-inf)
	    (incf lb l))
	(if (eq u '+inf)
	    (setq inf? `+inf)
	    (incf ub u))
	finally (return (int (or minf? lb) (or inf? ub)))))

#| original code
(defun isum (L)
  (let* ((lbs (mapcar #'car L))
	 (ubs (mapcar #'cadr L))
	 (lb (if (member '-inf lbs)  '-inf  (apply #'+ lbs)))
	 (ub (if (member '+inf ubs)  '+inf  (apply #'+ ubs))))
    (int lb ub)))
|#

; Check whether one interval includes another.

(defun span>= (xrange yrange)				; range xrange spans range yrange
  (and (e<= (lo xrange) (lo yrange))
       (e<= (hi yrange) (hi xrange))))

(defun span-of (xrange yrange)				; smallest interval containing both xrange and y.
  (int (emin (lo xrange) (lo yrange))
       (emax (hi xrange) (hi yrange))))

; Intersecting two intervals gives a smaller interval, or NIL if they are disjoint.

(defun iintersect (xrange yrange)
  (let ((lb (emax (lo xrange) (lo yrange)))
	(ub (emin (hi xrange) (hi yrange))))
    (cond ((and (numberp lb) (numberp ub) (> lb ub)) nil)	; empty intersection!
	  (t (int lb ub)))))


; Multiplication and division.
;   The code is defined in terms of e*, emin, and emax, "extended" functions
;   that can cope with the values +inf and -inf.
; => Check for incorrect handling of 0 * inf (undefined, so should be avoided).

(defun itimes (xrange yrange)
  (cond ((and (ipos xrange) (ipos yrange))
	 (int (e* (lo xrange) (lo yrange)) (e* (hi xrange) (hi yrange))))
	((and (ineg xrange) (ineg yrange))
	 (int (e* (hi xrange) (hi yrange)) (e* (lo xrange) (lo yrange))))
	((and (ipos xrange) (ineg yrange))
	 (int (e* (hi xrange) (lo yrange)) (e* (lo xrange) (hi yrange))))
	((and (ineg xrange) (ipos yrange))
	 (int (e* (lo xrange) (hi yrange)) (e* (hi xrange) (lo yrange))))
	(t (int (emin (e* (lo xrange) (hi yrange))
		      (e* (hi xrange) (lo yrange)))
		(emax (e* (lo xrange) (lo yrange))
		      (e* (hi xrange) (hi yrange)))))))

; Decompose x/y = x*(1/y) unless y = [0,0] or lb(y) < 0 < ub(y).

(defun idiv (xrange yrange)
  (cond ((and (numberp (lo yrange))		;   - the most common case
	      (numberp (hi yrange))
	      (or (< 0 (lo yrange)) (< (hi yrange) 0)))
	 (itimes xrange (int (/ 1 (hi yrange)) (/ (lo yrange)))))
	((equal yrange '(0 0))
	 (int '-inf '+inf))
	((and (or (eql (hi yrange) '+inf) (> (hi yrange) 0))
	      (or (eql (lo yrange) '-inf) (< (lo yrange) 0)))
	 (int '-inf '+inf))
	(t (itimes xrange
		   (int (cond ((eql (hi yrange) '+inf) 0)
			      ((= (hi yrange) 0) '-inf)
			      (t (/ 1 (hi yrange))))
			(cond ((eql (lo yrange) '-inf) 0)
			      ((= (lo yrange) 0) '+inf)
			      (t (/ 1 (lo yrange)))))))))

(defun iinv (range)				; 1/x
  (cond ((and (numberp (lo range))		;   - the most common case
	      (numberp (hi range))
	      (or (< 0 (lo range)) (< (hi range) 0)))
         (int (/ 1 (hi range)) (/ (lo range))))
	((equal range '(0 0))
	 (int '-inf '+inf))
	((and (or (eql (hi range) '+inf) (> (hi range) 0))
	      (or (eql (lo range) '-inf) (< (lo range) 0)))
	 (int '-inf '+inf))
	(t (int (cond ((eql (hi range) '+inf) 0)
		      ((= (hi range) 0) '-inf)
		      (t (/ 1 (hi range))))
		(cond ((eql (lo range) '-inf) 0)
		      ((= (lo range) 0) '+inf)
		      (t (/ 1 (lo range))))))))

;;; ISQUARE does not check for malformed RANGE.

(defun isquare (range)
  (let ((lo (lo range))
	(hi (hi range)))
    (cond ((and (numberp lo)(>= lo 0))		; completely nonnegative interval
	   (int (* lo lo) (e* hi hi)))
	  ((and (numberp hi)(<= hi 0))		; completely nonpositive interval
	   (int (* hi hi )(e* lo lo)))	
	  (t (int 0 (emax (e* hi hi)		; interval crosses 0.
			  (e* lo lo)))))))

;;; Interval square root.

;;; In ISQUART, currange is the currently known range of an expression
;;; E1, and E1 = +/- sqrt(E2).  The central problem for
;;; combining square roots across ranges is that the sqrt of a range has
;;; two disconnected intervals.  Suppose we know that e1 = sqrt(e2),
;;; and we already know (from other constraints) that e1 = [-10 3.5]
;;; and e2 = [9 16].  The sqrt relationship tells us that we can
;;; refine exp1 to [-4 -3] OR [3 3.5].  However, we have no way of
;;; representing disjunt intervals.  In this case, we set e1's new
;;; range to [-4 3.5].
;;; Fortunately, in most cases, e1 will have an unambiguous sign, and
;;; we can select the correct root.

;;; The SQRT relation in Q2 has a unique syntax, (sqrt e1 e2).  The
;;; interpretation is "The sqrt(e2), using e1, if possible, to
;;; determine the correct root."

(defun isquart (currange sqrange)
  (let ((lo (lo currange))
	(hi (hi currange))
	(slo (lo sqrange))
	(shi (hi sqrange)))
    (when (or (and (numberp shi)(minusp shi))
	      (eq shi '-inf))
      (error "~s can't be the square root of ~s" currange sqrange))
    (cond
      ((and (numberp lo)
	    (e<= (e-inverse (esqrt (emax slo 0))) lo))	; ranges overlap only on + side.
       (int (esqrt (emax slo 0)) (esqrt shi)))
      ((and (numberp hi)
	    (e<= hi  (esqrt (emax slo 0))))	        ; ranges overlap only on - side.
       (int (e-inverse (esqrt shi))
	    (e-inverse (esqrt (emax slo 0)))))	
      (t (let ((shisqrt (esqrt shi)))	; ranges overlap on both sides.
	   (int (emax lo (e-inverse shisqrt))
		(emin hi shisqrt)))))))


;   4.  Extended scalar arithmetic, over [-inf,+inf].


; E* is an extended scalar multiplication.

(defun e* (a b)
  (cond ((and (numberp a) (numberp b)) (* a b))	;   - most common case
	((eql a '+inf) (cond ((eql b '+inf) '+inf)
			     ((eql b '-inf) '-inf)
			     ((> b 0) '+inf)
			     ((= b 0) 0)
			     ((< b 0) '-inf)))
	((eql a '-inf) (cond ((eql b '+inf) '-inf)
			     ((eql b '-inf) '+inf)
			     ((> b 0) '-inf)
			     ((= b 0) 0)
			     ((< b 0) '+inf)))
	((> a 0) (cond ((eql b '+inf) '+inf)
		       ((eql b '-inf) '-inf)))
	((= a 0) 0)
	((< a 0) (cond ((eql b '+inf) '-inf)
		       ((eql b '-inf) '+inf)))))

; Bug:  0 * inf should be undefined, not zero.  [Thomas, Calculus].

(defun emax (a b)
  (cond ((and (numberp a) (numberp b)) (max a b))
	((eql a '+inf) '+inf)
	((eql b '+inf) '+inf)
	((eql a '-inf) b)
	((eql b '-inf) a)))

(defun emin (a b)
  (cond ((and (numberp a) (numberp b)) (min a b))
	((eql a '-inf) '-inf)
	((eql b '-inf) '-inf)
	((eql a '+inf) b)
	((eql b '+inf) a)))

(defun e<= (a b)
  (cond ((and (numberp a) (numberp b)) (<= a b))
	((eql a '-inf) t)
	((eql b '+inf) t)))

;;; ESQRT returns the positive, extended sqrt of a value.

(defun esqrt (val)
  (cond ((and (numberp val)
	      (>= val 0))
	 (sqrt val))
	((eq val '+inf) '+inf)
	(t (error "Can't take square root of ~s" val))))

(defun e-inverse (value)
  (cond ((numberp value)
	 (- value))
	((eq value '-inf) '+inf)
	((eq value '+inf) '-inf)))


;   5.  Retrieve monotonic function envelopes.

; Monotonic functions are interpreted according to their computable envelopes.
; The computable envelopes are guaranteed to be called with numerical arguments.
; They must return numerical values, or NIL where they are undefined.

(defun apply-fn (f range)
  (multiple-value-bind (function upper-env lower-env)
      (get-envelopes f)
    (case function 
      ((M+ S+ m0+)
       (int (or (and (numberp (lo range)) lower-env (funcall lower-env (lo range))) '-inf)
	    (or (and (numberp (hi range)) upper-env (funcall upper-env (hi range))) '+inf)))
      ((M- S- m0-)
       (int (or (and (numberp (hi range)) lower-env (funcall lower-env (hi range))) '-inf)
	    (or (and (numberp (lo range)) upper-env (funcall upper-env (lo range))) '+inf)))
      (t (error "Unknown Function ~S" function)))))

(defun apply-finv (f y)
  (multiple-value-bind (function upper-inv lower-inv)
      (get-envelopes f :inverse t)
    (case function 
      ((M+ S+ m0+)
       (int (or (and (numberp (lo y)) upper-inv (funcall upper-inv (lo y))) '-inf)
	    (or (and (numberp (hi y)) lower-inv (funcall lower-inv (hi y))) '+inf)))
      ((M- S- m0-)
       (int (or (and (numberp (hi y)) lower-inv (funcall lower-inv (hi y))) '-inf)
	    (or (and (numberp (lo y)) upper-inv (funcall upper-inv (lo y))) '+inf)))
      (t (error  "Unknown Function ~S" function)))))


; Get the envelopes from the current state.

(defun get-envelopes (fspec &key (inverse))
  (declare (special *current-qde*))
  (let ((up (if inverse 'upper-inverse 'upper-envelope))
	(down (if inverse 'lower-inverse 'lower-envelope)))
    (cond ((and (symbolp fspec)
		(q-function-p fspec))
	   (if inverse
	       (values (get-function-type fspec)
		       (get-function-upper-inverse fspec)
		       (get-function-lower-inverse fspec))
	       (values (get-function-type fspec)
		       (get-function-upper-envelope fspec)
		       (get-function-lower-envelope fspec))))
	  ((and (consp fspec)
		;; BKay 3Sept91 Added s+ s- to the list
		(member (car fspec) '(m+ m0+ m- m0- s+ s-)))
	   (values (car fspec)
		   (cadr (assoc up
				(cdr (assoc fspec (qde-M-envelopes *current-qde*)
					    :test #'equal))))
		   (cadr (assoc down
				(cdr (assoc fspec (qde-M-envelopes *current-qde*)
					    :test #'equal))))))
	  (t (error "Don't know this function spec ~s" fspec)))))

;;; Multivariate M constraint function application (BKay 5May92)

;;; Apply a multivariate M constraint.
;;; Inputs:  f      - a list of the form ((M s1 ... sn) v1 ... vn y).
;;;          ranges - the list of ranges for v1...vn.
;;; Returns: an interval.
;;; Added BKay 28Apr92
;;;
(defun apply-MvM-fn (f ranges)
  (let (upper-env lower-env upper-argset lower-argset)
    (multiple-value-setq (upper-env lower-env)
	(get-MvM-envelopes f))
    
    ;; Don't go through the bother if there are no envelopes.
    (when (or upper-env lower-env)
      (multiple-value-setq (lower-argset upper-argset)
	  (select-args-for-MvM-function f ranges)))

    ;; Modified BKay 29May92 to allow the result of the apply
    ;; to be NIL.
    (int (if (and lower-env (every #'numberp lower-argset))
	     (or (apply lower-env lower-argset) '-inf)
	     '-inf)
	 (if (and upper-env (every #'numberp upper-argset))
	     (or (apply upper-env upper-argset) '+inf)
	     '+inf))))

	
;;; Return the envelope functions for a multivariate M constraint.
;;; Inputs:  f      - a list of the form ((M s1 ... sn) v1 ... vn y)
;;; Returns: #1: A pointer to the upper envelope (or NIL)
;;;          #2: A pointer to the lower envelope (or NIL)
;;; Added BKay 28Apr92
;;;
(defun get-MvM-envelopes (f)
  (declare (special *current-qde*))
  (let ((envelope-clause (alookup f (qde-m-envelopes *current-qde*)
				  :test #'equal)))
    (values (if envelope-clause
		(lookup 'upper-envelope envelope-clause)
		NIL)
	    (if envelope-clause
		(lookup 'lower-envelope envelope-clause)
		NIL))))


;;; Select the arguments needed for a Multivariate M constraint.
;;; Inputs:  f      - a list of the form ((M s1 ... sn) v1 ... vn y)
;;;          ranges - a list of ranges v1...vn
;;; Returns: #1: A list of values for the lower envelope function
;;;          #2: A list of values for the upper envelope function
;;; Added BKay 28Apr92
;;;
(defun select-args-for-MvM-function (f ranges)
  (loop with lower-args = nil
	with upper-args = nil
	for range in (reverse ranges)
	for sign  in (reverse (cdr (first f)))
	do  (case sign
	      (+  (push (lo range) lower-args)
		  (push (hi range) upper-args))

	      ;; This will be ignored by the function
	      (0  (push 0 lower-args)
		  (push 0 upper-args))

	      (-  (push (hi range) lower-args)
		  (push (lo range) upper-args)))
	finally
	    (return (values lower-args upper-args))))



;;; Functions for finding information about qde-functions - named
;;; monotonic functions.  - drt 

(defun get-function-type (fnname)
  (declare (special *current-qde*))
  (or (getf (alookup fnname (qde-functions *current-qde*)) :type)
      (error "Couldn't find ~s" fnname)))

(defun get-function-upper-envelope (fnname)
  (declare (special *current-qde*))
  (or (getf (alookup fnname (qde-functions *current-qde*)) :upper-envelope)
      (error "Couldn't find ~s" fnname)))

(defun get-function-lower-envelope (fnname)
  (declare (special *current-qde*))
  (or (getf (alookup fnname (qde-functions *current-qde*)) :lower-envelope)
      (error "Couldn't find ~s" fnname)))

(defun get-function-upper-inverse (fnname)
  (declare (special *current-qde*))
  (or (getf (alookup fnname (qde-functions *current-qde*)) :upper-inverse)
      (error "Couldn't find ~s" fnname)))

(defun get-function-lower-inverse (fnname)
  (declare (special *current-qde*))
  (or (getf (alookup fnname (qde-functions *current-qde*)) :lower-inverse)
      (error "Couldn't find ~s" fnname)))

(defun get-function-signmaps (fnname)
  (declare (special *current-qde*))
  (or (getf (alookup fnname (qde-functions *current-qde*)) :signmaps)
      (error "Couldn't find ~s" fnname)))

(defun q-function-p (fnname)
  (declare (special *current-qde*))
  (or (member fnname (qde-functions *current-qde*) :key #'car)
	    (eq fnname 'function)))

; Generalization to S+/S- is not straight-forward (or for that matter to M+/M- with
; finite asymptotes) since it is unclear how to handle a bounded range for the
; envelope function or its inverse.

; Various explicitly-asserted bounds for f may be retrieved:
;    (slope (M+ x y))   or   (curvature (M+ x y))

(defun explicit-bounds (property f)
  (declare (special *current-qde*))
  (let ((bounds (lookup property
			(cdr (assoc f (qde-M-envelopes *current-qde*) :test #'equal)))))
    (cond ((null bounds) (int '-inf '+inf))
	  (t bounds))))

