;;; -*- Mode:Common-Lisp; Package:QSIM; Default-character-style:(FIX BOLD NORMAL); Base:10 -*-
;;; Copyright 1990, David Throop, University of Texas at Austin
(in-package :qsim)

#+symbolics
(progn
  (zwei:defindentation (vbind 0 3 1 2))
  (zwei:defindentation (versiplot 0 3 1 3))
  (zwei:defindentation (vplot-one-parameter 0 3 1 3 2 3 3 3 4 3))
  (zwei:defindentation (vplot-tree 0 3 1 3 2 3 3 3 4 3))
  (zwei:defindentation (vplot-phase-plane 0 3 1 3 2 3 3 3 4 3))
  (zwei:defindentation (vplot-text 0 3 1 3 2 3 3 3 4 3))
  (zwei:defindentation (vplot-parameters 0 3 1 3 2 3 3 3 4 3))
  (zwei:defindentation (vplot-many-phase-planes 0 3 1 3 2 3 3 3 4 3)))

(defparameter *share-for-line* .2)
(defparameter *inset-for-dotted-line* .03)

;;; This is the top level macro for versatile plotting.  Row-shares
;;; should be a list of numbers.  The Nth number determines what share
;;; of the total width will be allocated to the Nth row (counting is 1
;;; based); the lenght of Row-Shares determines the number of rows.
;;; Column-shares is similar.  Horzlines is a list of row-numbers
;;; (integers); a dotted line will be drawn below each row mentioned.
;;; Verlines is similar; a dashed line will be drawn to the right of
;;; each column mentioned.
;;;   Allox is a list of ((left-edge width)(left-edge width)...)
;;; specifying the left-edge and width of each column (in pixels.)
;;; Alloy is similar the top-edge and height of rows.  Alloxline is a
;;; list of x values for the vertical dashed lines; Alloyline is similar
;;; for the horizontal lines.


(defmacro versiplot (row-shares column-shares horzlines verlines
		     &rest body)
  `(bboxbind-no-margin
     (let ((verrows ,(length row-shares))	; Used by vplot-parameters macro
	   (vercols ,(length column-shares)))
       (declare (special vercols verrows))	; so we don't get an warn if this isn't used.
       (multiple-value-bind (allox alloxline)
	   (plotalloc ',column-shares ',verlines xscreen)
	 (multiple-value-bind (alloy alloyline)
	     (plotalloc ',row-shares ',horzlines yscreen)
	   (loop for x in alloxline		; Draw the dashed lines between plots
		 with yup = (* *inset-for-dotted-line* yscreen)
		 with ydn = (* (- 1 *inset-for-dotted-line*) yscreen)
		 do (qplot-line x yup x ydn :dashed t))
	   (loop with xleft = (* *inset-for-dotted-line* xscreen)
		 with xright = (* (- 1 *inset-for-dotted-line*) xscreen)
		 for y in alloyline
		 do (qplot-line xleft y xright y :dashed t))
	   ,@body)))))

;;; The V-Place macro allows the location and dimensions for a plot to
;;; be passed as single argument.  It takes a row and a column number as
;;; args, and produces, (by default) the location and dimensions of a
;;; plot one column wide and 1 row tall at that location.  The
;;; COLUMNS-WIDE and ROWS-TALL keyword args allow the plot's width and
;;; height to be adjusted.  The COLUMN-OFFSET and ROW-OFFSET allow the
;;; plot's top and left edge to be adjusted.  
;;;  The V-place macro returns an object := ((left-edge width)(top-edge height)).
;;; This is unpacked by the the VBIND macro.

(defmacro v-place (row column &key (columns-wide 1)(rows-tall 1)
		   (column-offset 0)(row-offset 0))
  `(list (list (+ (car (nth (1- ,column) allox))
		  (* ,column-offset (cadr (nth (1- ,column) allox))))
	       (* ,columns-wide (cadr (nth (1- ,column) allox))))
	 (list (+ (car (nth (1- ,row) alloy))
		  (* ,row-offset (cadr (nth (1- ,row) alloy))))
	       (* ,rows-tall (cadr (nth (1- ,row) alloy))))))

;;; The VBIND macro takes an object of form ((left-edge width)(top-edge
;;; height)) and a margin and binds XPOS, YPOS, XSIZE and YSIZE to plot
;;; specifications corrsponding to the given specification trimmed by
;;; the margin.
(defmacro vbind (v-place &rest body)
  `(let* ((vpl ,v-place)
	  (xpos (+ (caar vpl) margin))
	  (xsize (- (cadar vpl) (* 2 margin)))
	  (ypos (+ (caadr vpl) margin))
	  (ysize (- (cadadr vpl)(* 2 margin ))))
     ,@body))

;;; Given a list of shares for rows (or columns), a list of lines, and a
;;; width (or height), this calculates edges and widths for each row (or
;;; column) and the postions of the lines.

(defun plotalloc (shares lines screen)
  (let ((totshare (+ (apply #'+ shares)
		     (* *share-for-line* (length lines))))
	(runshare 0))
    (loop for share in shares
	  for count from 1
	  collect (list (* (/ runshare totshare) screen)
			(* (/ share totshare) screen))
	    into plotallo
	  do (incf runshare share)
	  when (member count lines :test #'=)
	    collect (* screen (/ (+ runshare (* .5  *share-for-line*)) totshare)) into lineallo
	  when (member count lines :test #'=)
	    do (incf runshare  *share-for-line*)
	  finally (return (values  plotallo  lineallo)))))

;;; This draws the behavior tree in the plot specified by v-place.  It's
;;; given a large margin, in part because the plot-state-tree actually
;;; draws a little bit outside the box it's given.  (the circles for the
;;; first behavior have their centers on the top edge of the box.)

(defun vplot-tree (v-place &key (bnum 0) (istate *initial-state*)(margin 10))
  (vbind v-place
     (let* ((initials (get-list-of-initial-states istate :complete-partial-state nil))
	    (behaviors (apply #'append (mapcar #'get-behaviors initials)))
	    (btotal (length behaviors))
	    (bmax (apply #'max (mapcar #'length behaviors)))
	    (uxsize (min xsize (+ 10 (* bmax  tree-xstep-max)))))
       (if (zerop btotal)
	   (warn "Can't plot state tree - there are no behaviors for ~a" istate)
	   (plot-state-tree initials  bmax btotal
			    xpos ypos uxsize ysize bnum)))))

;;; Draw a phase plane in the plot specified by v-place.  Params is a
;;; 2-list of parameters, bnum is the number of the behavior from which
;;; to draw.

(defun vplot-phase-plane (v-place params bnum
			  &key (istate *initial-state*)(margin 4)
			  (label (format nil "Behavior ~a  -  ~{[~a vs ~a.]~}" bnum params)))
  (vbind v-place
     (let* ((initials (get-list-of-initial-states istate :complete-partial-state nil))
	    (behaviors (apply #'append (mapcar #'get-behaviors initials)))
	    (beh (nth (1- bnum) behaviors)))
       (plot-one-phase-diagram params beh xpos ypos xsize ysize label))))


;;; Draw the time-plot of a single parameter in the plot specified by
;;; v-place.  Bnum is the number of the behavior from which to draw.

(defun vplot-one-parameter (v-place param bnum &key (istate *initial-state*)(margin 4)
			(label (format nil "~a: Behavior ~a." param bnum))
			(refpoints))
  (vbind v-place
     (let* ((initials (get-list-of-initial-states istate :complete-partial-state nil))
	    (behaviors (apply #'append (mapcar #'get-behaviors initials)))
	    (beh (nth (1- bnum) behaviors))
	    (*xsize* xsize)
	    (xalloc (allocate-h-axis refpoints beh xsize)))
       (plot-parameter param refpoints beh 
		       xpos ypos xsize ysize xalloc
		       :label label))))

;;; Draw a plain box in the plot specified by v-place.  This i mostly
;;; for quickly looking at layouts.

(defun vplot-box (v-place &key (label)(margin 4))
  (vbind v-place
     (qplot-box xpos ypos xsize ysize)
     (when label
       (qplot-string label (+ xpos 10)(+ ypos 20)))))

;;; Plot some text in the plot specified by v-place.  The extra offset
;;; for the first line is because the y coordintate to QPLOT-LINE is for
;;; the bottom of the string, but we want the top of the string to lie
;;; within the plot.

(defun vplot-text (v-place textlist font &key (margin 4)(max-spacing 20))
  (vbind v-place
     (declare (ignore xsize))
     (loop for text in textlist
	   for y from (+ ypos margin
			 #+symbolics (zl:font-char-height	
				       (scl:with-character-style
					 (font)
					 (send *qplot-output* :current-font)))
			 #-symbolics 10)
		 by (min max-spacing (/ ysize (length textlist)))
	   do (qplot-string text xpos y :font font))))

;;; Plot a layout of a set of parameters between specified rows and
;;; between specied columns.

(defmacro vplot-parameters (layout bnum begin-row end-row begin-column end-column
			    &key (margin 4)(istate)(refpoints))
  `(progn (error-check-vplot-parameters ,layout ,begin-row ,end-row
					,begin-column ,end-column verrows vercols)
	  (loop for row from ,begin-row to ,end-row
		for lrow in layout
		do (loop for column from ,begin-column to ,end-column
			 for params on lrow
			 for param = (car params)
			 when param
			   do (vplot-one-parameter (v-place row column) param ,bnum
				  :margin ',margin :label param
				  :refpoints ,refpoints
				  :istate (or ,istate *initial-state*))))))

;;; Do error checking on the input to the VPLOT-PARAMETERS macro. 

(defun error-check-vplot-parameters
       (layout begin-row end-row begin-column end-column  verrows vercols)
  (error-check-rows-and-columns begin-row end-row
				begin-column end-column  verrows vercols)
  (when (> (loop for lrow in layout
		 maximize (length lrow))
	   (1+ (- end-column begin-column)))
    (error "Cannot fit the layout ~a between columns ~a and ~a." layout begin-column end-column))
  (when (> (length layout) (1+ (- end-row begin-row)))
    (error "Cannot fit the layout ~a between rows ~a and ~a." layout begin-row end-row)))

(defun error-check-rows-and-columns
       (begin-row end-row begin-column end-column  verrows vercols)
   (when (or (not (numberp begin-row))
	    (< begin-row 1))
    (error "Begin-row must be at least 1 but is ~a" begin-row))
  (when (or (not (numberp begin-column))
	    (< begin-column 1))
    (error "Begin-column must be at least 1 but is ~a" begin-column))
  (when (or (not (numberp end-row))
	    (> end-row verrows))
    (error "End-row is ~a number of rows is ~a" end-row verrows))
  (when (or (not (numberp end-column))
	    (> end-column vercols))
    (error "End-column is ~a number of columns is ~a" end-column vercols)))

(defmacro vplot-many-phase-planes (bnum-layout params begin-row end-row begin-column end-column
			    &key (margin 4)(istate))
  `(progn (error-check-rows-and-columns ,begin-row ,end-row
				,begin-column ,end-column  verrows vercols)
	  (loop with *phase-slice-cols*  = vercols
		with *phase-slice-rows* = verrows
		for row from ,begin-row to ,end-row
		for lrow in ,bnum-layout  
		do (loop for column from ,begin-column to ,end-column
			 for bnum in lrow
			 when bnum
			   do (vplot-phase-plane (v-place row column) ,params bnum
				  :margin ',margin
				  :label (format nil "Behavior: ~a" bnum)
				   :istate (or ,istate
					       *initial-state*))))))