;;; -*- Mode:Common-Lisp; Package:Qsim; Syntax:Common-Lisp; Base:10 -*-

(in-package 'Qsim)


#| Functions for re-arranging and creating customized displays and plots,
   typically for customized printed output, as for an article.
|#


;;; Binding form for global variables for BoundingBox output, needed by
;;; psfig macros.  If printed normally (outside psfig macros) figure
;;; will be *Text-format-text-width* inches (6") wide.
;;; NOTICE: This is a special set of bindings to work with DISPLAY-BEHAVIOR.
;;; It assumes that lmargin, rmargin and bmargin are empty.

(defmacro BBOXBIND (&rest args)
  `(let* ((*x-translation*
	    (/ (- 8.5 *Text-format-text-width*) 2))	; Position on paper of 
	  (*y-translation* 1)				; lowerleft corner, in inches
	  (*rotation* 0)
	  (*bounding-box* (list lmargin (- yscreen bmargin) (- xscreen rmargin) 0))
	  (*postscript-style* :bounded)
	  (*x-scale* (/ (* 72.0 *Text-format-text-width*) xscreen))
	  (*y-scale* *x-scale*)) 
     (device-interface
       (qplot-new-behavior)
       ,@args
       (qplot-end-display (- yscreen 25)))))


(defmacro BBOXBIND-NO-MARGIN (&rest args)
  `(let* ((*x-translation*
	    (/ (- 8.5 *Text-format-text-width*) 2))	; Position on paper of 
	  (*y-translation* 1)				; lowerleft corner, in inches
	  (*rotation* 0)
	  (*bounding-box* (list 0 yscreen xscreen 0))
	  (*postscript-style* :bounded)
	  (*x-scale* (/ (* 72.0 *Text-format-text-width*) xscreen))
	  (*y-scale* *x-scale*)) 
     (device-interface
       (qplot-new-behavior)
       ,@args
       (qplot-end-display (+ yscreen 50)))))


;;; COMPARATIVE-BEHAVIOR-DISPLAYS - for displaying different behaviors
;;; from a behavior tree.  Initials is the list of intial states.
;;; Allbehaviors is the set of behaviors.  Behnumbers is a list of the
;;; numbers of the behavior to plot (1 based).  Bmax is the length of
;;; the longest tree branch.  Btotal is the number of branches in the
;;; tree.  Lmargin and Rmargin are bound lexically.

(defun COMPARATIVE-BEHAVIOR-DISPLAYS
       (initials allbehaviors behnumbers bmax btotal trees reference-states layout)
  (bboxbind 
    (%comparative-behavior-displays
      initials allbehaviors behnumbers bmax btotal trees reference-states layout)))


(defun %COMPARATIVE-BEHAVIOR-DISPLAYS
       (initials allbehaviors behnumbers bmax btotal trees reference-states layout)
  (loop with plotwide = (/ (- xscreen lmargin rmargin)	;width of single plot
			   (length behnumbers))
	with ulayout = (or layout
			   (find-some-layout allbehaviors reference-states))
	for bnum in behnumbers
	for count from 0
	for beh = (nth (1- bnum) allbehaviors )	; zero based indexing
	for rmargin from (- (+ xscreen (* 2 xsep))	; size of right margin of single plot
			    plotwide lmargin) by (- plotwide)
	for lmargin from lmargin by plotwide	; position of left margin of single plot
	for xtree from (+ lmargin (* .35 plotwide))	; x position of left edge of tree
		  by plotwide
	do (onecompbehplot bnum count beh xtree ulayout
			   trees initials bmax btotal reference-states)))


;;; ONECOMPBEHPLOT -- One complete behavior plot.  Bnum is the number of
;;; the behavior (1 based).  Count is 0 based count of displayed
;;; behaviors, with 0 as leftmost.  Beh is the behavior (a list of
;;; states).  Xtree is the position of the left edge of the behavior
;;; tree.   Trees is boolean - whether to plot the statetree.  Initials
;;; is a list of initial-states (used for plotting statetree.  Bmax is
;;; the length of the longest tree branch.  Btotal is the number of
;;; branches in the tree.

(defun ONECOMPBEHPLOT (bnum count beh xtree layout trees
		       initials bmax btotal reference-states)
  (compute-layout layout)			; May change each time
  (unless (zerop count)				; Vertical dashed dividing line 
    (qplot-line (- lmargin xsep) tmargin	;  between plots
		(- lmargin xsep) (- yscreen bmargin)
		:dashed t :thickness 2))
  (when trees
    (plot-state-tree initials bmax btotal xtree ybloc xbsize ybsize bnum))
  (display-behavior				; Normal plotting routine - called
    reference-states beh layout nil))		;  with global var lexically bound.


;;; CHOOSE-COMPARATIVE is an easy interface for setting up multiple
;;; behavior plots from a *single* behavior tree.  It displays behaviors
;;; from the tree rooted in *intial-state*.  *Reference-states*
;;; and *layout* are assumed to be bound to their appropriate values by
;;; the simulation.  Givenbehs, when supplied, should be a list of
;;; behavior numbers. [ (1 3 7) would select the 1st, 3rd \& 7th
;;; behaviors of the tree rooted in *initial-state*.]  If no behavior
;;; numbers are supplied the user is prompted.}

(defun CHOOSE-COMPARATIVE (&optional (givenbehs)(vlshare))
  (let* ((initials (get-list-of-initial-states *initial-state* :complete-partial-state nil)) ; keyword added DJC 08/20/91
	 (behaviors (apply #'append (mapcar #'get-behaviors initials)))
	 (bmax (apply #'max (mapcar #'length behaviors)))
	 (btotal (length behaviors))
	 (behnumbers #-symbolics givenbehs	; The ability to choose behaviors using the
		     #+symbolics (or givenbehs	; mouse has not been implemented for other architectures.
			 (reverse (select-behs-from-beh-tree
				    initials bmax btotal t))))
	 (vlabelshare (or vlshare		; keep from squeezing lmark names
			  (length behnumbers))))
  (comparative-behavior-displays
    initials behaviors behnumbers bmax btotal t *reference-states* *layout*)))
    

(defun SELECTED-USUSAL-TIMEPLOT (istate bnum &key refpoints layout box (top-label t))
  (let* ((initials (get-list-of-initial-states istate :complete-partial-state nil))  ; keyword added DJC 08/20/91
	 (behaviors (apply #'append (mapcar #'get-behaviors initials)))
	 (bmax (apply #'max (mapcar #'length behaviors)))
	 (btotal (length behaviors))
	 (behavior (nth (1- bnum) behaviors)))
    (unless layout
      (setf layout (find-some-layout
			       behaviors refpoints)))
    (compute-layout layout)
    (when box (qplot-box lmargin 0 (- xscreen rmargin lmargin)
			 (- yscreen  bmargin)))
    (when top-label 
      (qplot-label behavior bnum btotal))
    (display-behavior refpoints behavior layout nil)
    (when (<= btotal 20)
      (plot-state-tree initials bmax btotal (- xscreen rmargin xbsize)
		       ybloc xbsize ybsize bnum))
    ))


;;; PLOT-BOUNDED-TREE plots a single behavior tree, suitable for psfig
;;; inclusion.  It plots a rectangle on the screen which is not included
;;; in the .ps output; this rectangle shows the extent of the bounding
;;; box for the figure.  If the :bnum figure is included, an arrow is
;;; printed to the right of the bum'th figure.  


(defun PLOT-BOUNDED-TREE (&key (bnum) (istate *initial-state*))
  (let* ((initials (get-list-of-initial-states istate :complete-partial-state nil))  ; keyword added DJC 08/20/91
	 (behaviors (apply #'append (mapcar #'get-behaviors initials)))
	 (btotal (length behaviors))
	 (bmax (apply #'max (mapcar #'length behaviors)))
	 (tree-xstep-max 20)
	 (tree-ystep-max 20)
	 (arrowmargin 10)
	 (xscreen (+ arrowmargin (* (1+ bmax) tree-xstep-max)))
	 (yscreen (* (1+ btotal) tree-ystep-max))
	 (xloc tree-xstep-max )
	 (yloc tree-ystep-max))
    (if (zerop btotal)
	(warn "Can't plot state tree - there are no behaviors for ~a" istate)
	(bboxbind-no-margin
	  (with-plotting-to-postscript-inhibited
	    (qplot-box 0 0 xscreen yscreen))
	  (plot-trees initials xloc yloc
		      tree-xstep-max tree-ystep-max nil nil)
	  (when bnum
	    (qplot-symbol (* (1+ bmax) tree-xstep-max)
			  (+ yloc (* (- bnum 1) tree-ystep-max)) 'left))))))


(defun SELECTED-USUSAL-VARSLICE (istate parm option &optional box bnum-list)
  (let* ((initials (get-list-of-initial-states istate :complete-partial-state nil))
	 (behaviors (apply #'append (mapcar #'get-behaviors initials)))
	 (beh-lists (mapcar #'get-behaviors initials))
	 (lengths (mapcar #'length beh-lists))
	 (tmargin 30)
	 (btotal (length behaviors)))
    (declare (special viewer-rows viewer-cols))
    (compute-rectangular-layout viewer-rows viewer-cols)
    (qplot-string (format nil "~a shown for different behaviors" parm)
		  (* .15 xscreen)(* .8 tmargin) :font label-font)
    (when box (qplot-box lmargin 0 (- xscreen rmargin lmargin)
			 (- yscreen  bmargin)))
    (view-behs option initials parm behaviors btotal lengths bnum-list)
    ))


;;; PLOT-COMPARITIVE-FROM-MULTIPLE-SOURCES draws a series of behavior
;;; plots.  It is given 4 lists, defining a series for 4-tuples.  Each
;;; tuple has
;;;  > a root - the initial-state for this plot (possibly a list of
;;;    states, as long as it's acceptable to GET-LIST-OF-INITIAL-STATES.)
;;;  > a ref - to be used as the form passed as :reference-states arg to 
;;;    qsim-display; should have the form := ((descatom <state>)*) where 
;;;    descatom is a descriptively named atom, like NORMAL.
;;;  > a bnum - then number of the behavior to plot.  1 based counting.
;;;  > a layout to use in this plot.

(defun PLOT-COMPARITIVE-FROM-MULTIPLE-SOURCES
       (roots refs bnums layouts &key (trees t))
  (bboxbind
    (loop with plotwide = (/ (- xscreen lmargin rmargin) (length roots))
	  with vlabelshare = (* .5 (length roots))
	  for root in roots
	  for refs in refs
	  for bnum in bnums
	  for layout in layouts
	  for initials = (get-list-of-initial-states root :complete-partial-state nil)
	  for behaviors = (get-behaviors root)
	  for bmax = (apply #'max (mapcar #'length behaviors))
	  for btotal = (length behaviors)
	  for beh = (nth (1- bnum) behaviors)
	  for count from 0
	  for rmargin from (- (+ xscreen (* 2 xsep))
			      plotwide lmargin) by (- plotwide)
	  for lmargin from lmargin by plotwide
	  for xtree from (+ lmargin (* .35 plotwide)) by plotwide
	  do (onecompbehplot bnum count beh xtree layout
		       trees initials bmax btotal refs))))


;;; Plot a box on the screen showing the borders of the figure (and thus
;;; the bounding box) but don't print it into the .ps file.

(defun INV-BOX ()
  (with-plotting-to-postscript-inhibited
    (qplot-box 0 0 xscreen yscreen)))

