;;; -*- Mode:Common-Lisp; Package:Qsim; Base:10 -*-
;;;     Copyright 1987, 1990, 1991 Benjamin Kuipers

(in-package 'Qsim)


#| Functions for examining and displaying useful information about states, behavior trees,
   QDEs, and other data structures, typically for debugging.  This file is intended to be
   a "scratch pad" for useful functions of this kind, to be shared among users.

   Please contribute your favorites, and PLEASE DOCUMENT THEM.
|#


#| -----------------------------------------------------------
   EXAMINE INCONSISTENT BEHAVIORS
   -----------------------------------------------------------
|#

(defun TREE-STATUS (state)
  "Print the reasons for inconsistency of all behaviors starting in STATE.
   Simulation must have been run with
    (make-sim ... :prune-inconsistent-states nil ...)."
  (setq state (get-state state))
  (format t "~%Beh State  Status")
  (loop
    for beh in (get-behaviors state)
    for last-state = (first (last beh))
    for i from 1
    do (format t "~%~3D ~6A ~A" i last-state (state-status last-state))))



#| -----------------------------------------------------------
   EXAMINE DIFFERENCES AMONG BEHAVIORS
   -----------------------------------------------------------

   From Schepps:throop.nasa;tools.lisp
|#


(defun REPORT-DIFFERENCES (&optional (state *initial-state*))

  "For each pair of consecutive behaviors which flow from state (1 and 2, 2 and 3, etc.),
   report differences in variable values at the time when the two behaviors first diverge
   (that is, just after the branch point, at the first two distinct states in the two
   behaviors)."

  (loop ;; beh1 and beh2 set to successive pairs of adjacent behaviors
        for (beh1 beh2) on (apply #'append (mapcar #'get-behaviors
						   (get-list-of-initial-states state :complete-partial-state nil)))
	;; div1 and div2 set to first two distinct states in beh1 and beh2
	for (div1 div2) = (loop for state1 in beh1
				for state2 in beh2
				unless (eq state1 state2)
				    return (list state1 state2))
	for behcount from 1
	with *detailed-printing* = ()
	when beh2
	  do (format t "~%~%Between Behaviors ~d and ~d [States ~s  and ~s]"
		     behcount (1+ behcount) div1 div2)
	  (loop for qspace in (state-qspaces div1)
		for param = (car qspace)
		for val1 = (qval param div1)
		   for val2 = (qval param div2)
		   unless (and val1 val2)
		     do (error "Missing Qvals")
		   ;; Print only distinct variable values
		   unless (or (equal val1 val2)
			      (and (atom (qmag val1)) (atom (qmag val2))
				   (equal (lmark-where-defined (qmag val1))
					  (lmark-where-defined (qmag val2)))
				   (equal (qdir val1) (qdir val2))))
		     do (format t "~%  ~a~25t~a~40t~a" param val1 val2))))


#+Symbolics
(cp:define-command (COM-REPORT-DIFFERENCES
		     :provide-output-destination-keyword nil
		     :command-table "Global")
		   ()
  (report-differences))



#| -----------------------------------------------------------
   EXAMINE AN INCONSISTENT STATE
   -----------------------------------------------------------
|#

; (defun GRIPE-CONSTRAINT (state)
;   "Print the constraints, if any, that inconsistent STATE violates."
;   (let ((constraints (state-constraints state)))  
;     ;;  The test for active constraints is wrong.
;     ;;  It ought to examine the modes of state!
;     (reinit-for-state state)
;     (loop for con in  constraints
;	   with *detailed-printing* = nil
;	   for tuple = (mapcar #'(lambda (var) (qval (variable-name var) state))
;			       (constraint-variables con))
;	   do (format *Qsim-report* "~%Checking constraint ~s" con)
;	   when (and (constraint-active-p con)
;		     (not (member nil tuple))	; Incomplete state
;		     (not (check-qsim-constraint tuple con)))
;	   do  (format *Qsim-Report*
;		       "~%  Variable values ~S~%    fail to check.  Corresponding values are ~S"
;		       tuple (alookup con (state-cvalues state))))))


#| GRIPE-CONSTRAINT
   Print the constraints violated by an inconsistent state.

   The test for active constraints below is wrong.  It ought to
   examine the modes of nstate!   [Whatever that means.]

   Revised from nq:nq;states.lisp to:
   1.  Notify of inactive constraints not checked.
   2.  Notify of incomplete tuples preventing checking.
   3.  Usually omit printing constraints which checked ok.
|#

(defun GRIPE-CONSTRAINT (state &optional print-all)

  "Print the constraints, if any, that STATE violates.
   If PRINT-ALL is T, print each constraint checked, even if it checks ok."
  (setq state (get-state state))
  (reinit-for-state state)
  (let ((ok t))
    (loop for con in  (state-constraints state)	; Constraints
	  with *detailed-printing* = nil
	  for tuple = (mapcar #'(lambda (var) (qval (variable-name var) state))
			      (constraint-variables con))
	  do (cond ((not (constraint-active-p con))
		    (setq ok nil)
		    (format *Qsim-report*
			    "~%Constraint ~s is inactive and not checked."
			    con))
		   ((member nil tuple)
		    (setq ok nil)
		    (format *Qsim-report*
			    "~%Constraint ~S can't be checked:~
			     ~%     Tuple ~S is incomplete."
			    con tuple))
		   ((not (check-qsim-constraint tuple con))
		    (setq ok nil)
		    (format *Qsim-Report*
			    "~%Constraint ~S complains:~
			     ~%     Tuple ~S fails to check.~
			     ~%     Corresponding values are ~S."
			    con tuple (alookup con (state-cvalues state))))
		   (print-all
		    (format *Qsim-Report*
			    "~%Constraint ~S ckecks ok." con))
		   (t nil)))
    (if ok (format *Qsim-Report* "~%All constraints check ok."))))


(defmacro SUBSTITUTE-QVAL (one two)
  `(if (and (eq (,one qmag-org) (,one qmag-rtn))
	    (not (eq (,two qmag-org) (,two qmag-rtn)))
	    (lmark-where-defined (,two qmag-rtn)))
       (setf (rest val-rtn)  (copy-qval (rest val-rtn))
	     (qval-qdir (rest val-rtn)) qdir-org
	     (qval-qmag (rest val-rtn)) ,(if (eq one 'first)
					     `(list (,two qmag-rtn) (,two qmag-org))
					     `(list (,two qmag-org) (,two qmag-rtn))))))

;;; Typically for use with gripe-constraint:

(defun SUBSTITUTE-QVALS (state-new state-org state-name)

  "Given a state with newly created landmarks (state-new) and an identical state without
  such landmarks (state-org), return a copy of state-new with some new qvalues.  For each
  var with qvalues of the form --
     in state-new:  ((orig-lm-1 new-lm)    qdir)
     in state-org:  ((orig-lm-1 orig-lm-2) opposite-qdir)
  and substitute in the copied state a new qvalue --
		    ((new-lm orig-lm-2)    opposite-qdir)
  and similarly for variables with qvalues of the form --
     in state-new:  ((new-lm orig-lm-2)    qdir).
  Set the value of the variable named in state-name to the returned state.
  Do enough copying so that state-new is unaltered."

  (declare (special first second))
  (let ((state-rtn (copy-state state-new)))
    (setf (state-qvalues state-rtn) (copy-alist (state-qvalues state-rtn))
	  (state-name state-rtn) state-name
	  (state-text state-rtn)
	    (concatenate 'string
			 (state-text state-rtn)
			 "  From state "  (string (state-name state-new))
			 ", qvalues revised re " (string (state-name state-org))))
    (loop
      for val-rtn in (state-qvalues state-rtn)
      for val-org in (state-qvalues state-org)
      for qmag-rtn = (qval-qmag (rest val-rtn))
      for qdir-rtn = (qval-qdir (rest val-rtn))
      for qmag-org = (qval-qmag (rest val-org))
      for qdir-org = (qval-qdir (rest val-org))
      when (and (listp qmag-rtn)
		(listp qmag-org)
		(opposite-qdirs qdir-rtn qdir-org))
      do (or (substitute-qval first second)
	     (substitute-qval second first)))
    (reset-state state-rtn)
    (set state-name state-rtn)))


(defun RESET-STATE (state)
  "Reset state so that it can be simulated from again."
  (setf (state-status state) nil
	(state-successors state) nil))




#| -----------------------------------------------------------
   EXAMINE AN INCOMPLETE STATE
   -----------------------------------------------------------
|#

(defun CORT (&optional (state *initial-state*))
  "Print, for each constraint, how many qmags and qdirs are null in
   incomplete state STATE."
  (setq state (get-state state))
  (format t "~&Null   Null~%Qmags  Qdirs  Constraint~%")
  (reinit-for-state state)
  (loop for constraint in (state-constraints state)
	for (mags dirs) = (%cort constraint)
	do (format t "~%~3d~7d    ~s" mags dirs constraint))
  (values))


(defun %CORT (constraint)
  (loop for var in (constraint-variables constraint)
	when (null (qmag (variable--qval var)))
	  sum 1 into mags
	when (null (qdir (variable--qval var)))
	  sum 1 into dirs
	finally (return (list mags dirs))))


#+Symbolics
(cp:define-command (COM-UNFILLED-PARAMETERS
		     :provide-output-destination-keyword nil
		     :command-table "Global"
		     :values t)
    ((state '((cl:symbol)) :default *initial-state*))
  (cort (get-state state)))



#| -----------------------------------------------------------
   EXAMINE A STATE
   -----------------------------------------------------------
|#

;;; Print tabular form of state values, and some other information.
;;; Qvalue printing moved to separate function PV.  RSM 22 Feb 91.

(defun PS (state)
  "Print a state structure nicely."
  (when (listp state)
    (format *Qsim-Trace* "~%Please use function PV for printing Qvalues.~%")
    (return-from ps (values)))
  (setq state (get-state state))
  (let* ((*detailed-printing* nil))
    (format *Qsim-Trace* "~%State ~a (~a):~40tin ~s~%Qvalues:"
	    (state-name state)
	    (state-time state)
	    (state-qde  state))
    (pv (state-qvalues state))
    (case (car (state-justification state))
      ((transition-from one-of-several-completions-of
			unique-successor-of one-of-several-successors-of)
       (format *Qsim-Trace* "~%Justification: ~19t~s~&"
	       (state-justification state)))
      (t (format *Qsim-Trace* "~%Justification: ~19t~s~&"
	       (state-justification state))))
    (format *Qsim-Trace* "~&Status: ~19t~s"
	    (state-status  state))
    (format *Qsim-Trace* "~&Successors:  ~19t~s~&"
	    ;;  Eliminated direct access of state.successor slot 02/14/91  DJC
	    (filtered-successor-slot-contents state)))
  (values))


;;; Extended to print in two columns, in alphabetical order. -drt 3may90

(defun PV (qvalues)
  "Print a list of qvalues nicely."
  (let* ((qvals (sort (copy-list qvalues) #'string-lessp :key #'first))
	 (*detailed-printing* nil))
    (loop with halfpoint = (ceiling (/ (length qvals) 2))
	  for (varname1 . qval1) in (subseq qvals 0 halfpoint)
	  for (varname2 . qval2) in (append (subseq qvals halfpoint) '(()))
	  do (format *Qsim-Trace* "~%  ~a~18t~a" varname1 qval1)
	  when varname2
	  do (format *Qsim-Trace* "~50t~a~66t~a" varname2 qval2)))
  (values))


(defun GET-STATE (state)
  "Get a state structure from a state, list, number, or symbol [in any package]."
  (etypecase state
    (state state)
    (cons (get-state (first state)))
    (fixnum (symbol-value (find-symbol (format nil "S-~a" state) :qsim)))
    (null (error "Null state"))
    (symbol  (if (state-p (symbol-value state))
		 (symbol-value state)
		 (if (eq (symbol-package state)
			 (find-package "QSIM"))
		     (format *qsim-report* "There is no state with name ~a" state)
		     (get-state (intern (string state) :qsim)))))))

(defun PRINT-STATE (state &optional detailed)
  "Print every slot of a state structure."
  (setq state (get-state state))
  (let ((*detailed-printing* detailed))
    (format *Qsim-Trace* "~@{~%~A: ~A~}"
      "Name"	      (state-name state)
      "Qde"	      (state-qde state)
      "Text"	      (state-text state)
      "Time-label"    (state-time-label state)    
      ;; Eliminated direct access to state.successor slot  02/14/91 DJC
      "Successors"    (filtered-successor-slot-contents state)
      "Other"	      (state-other state)
      "Misc"	      (state-misc state))
    (format *Qsim-Trace* "~@{~%~A:~% ~A~}"
      "Status"	      (state-status state)
      "Justification" (state-justification state))
    (format *Qsim-Trace* "~@{~%~A:~{~% ~A~^~50T~A~}~}"
      "Qvalues"       (state-qvalues state)
      "Qspaces"       (state-qspaces state))
    (format *Qsim-Trace* "~@{~%~A:~{~% ~A~}~}"
      "Cvalues"       (state-cvalues state)))
  (values))



#| -----------------------------------------------------------
   EXAMINE SUCCESSORS OF A STATE
   -----------------------------------------------------------
|#

;;; Modified by DJC to handle *show-inconsistent-successors* and tree keywords after
;;; discussions with Jack Vinson at Upenn.  07/11/91

(defun ALL-SUCCESSORS (&optional (state *initial-state*) &key tree
		       (*show-inconsistent-successors* *show-inconsistent-successors*))
  "Return a list or tree of STATE and all its successors and their successors,
   etc., to the end of the behavior tree headed by state.  A tree is returned
   if TREE is non-nil.  The list or tree includes inconsistent states if
   *SHOW-INCONSISTENT-SUCCESSORS* is non-nil."
  (cons state (funcall (if tree #'mapcar #'mapcan)
		       #'(lambda (cur-state) (all-successors cur-state :tree tree))
		       (successor-states state))))

(defun SORT-SUCCESSORS (state)
  "Sort (all-successors state) by state name."
  (sort (all-successors state)
	#'string-lessp
	:key #'state-name))


#| -----------------------------------------------------------
   EXAMINE A CONSTRAINT
   -----------------------------------------------------------
|#

(defun PRINT-CONSTRAINT (constr &key (source *current-qde*) detailed)
  "Print every slot of a constraint structure.  CONSTR and SOURCE
   must be appropriate arguments for function Get-Constraint (q.v.)."
  (declare (special *current-qde*))
  (let ((constraint (get-constraint constr source))
	(*detailed-printing* detailed))
    (format *Qsim-Trace* "~@{~%~A: ~A~}"
      "Name"	    (constraint-name	    constraint)
      "Type"	    (constraint-type	    constraint)
      "Variables"   (constraint-variables   constraint)
      "Bend-points" (constraint-bend-points constraint)
      "Neighbors"   (constraint-neighbors   constraint)
      "Other"	    (constraint-other	    constraint)
      "-Cvals"      (constraint--cvals	    constraint))
    (format *Qsim-Trace* "~@{~%~A:~{~% ~A~}~}"
      "-Tuples"     (constraint--tuples     constraint))
    (format *Qsim-Trace* "~%Flags:~:[~; Done~]~:[~; Active~]~%~%"
	    (constraint-done     constraint)
	    (constraint-active-p constraint))
    constraint))


(defun GET-CONSTRAINT (constr &optional (source *current-qde*))
  "Get a constraint structure.
   If CONSTR is a constraint structure, it is returned and SOURCE is ignored.
   Otherwise, SOURCE must be a state or qde, where CONSTR will be looked up,
   and CONSTR may be --
   a constraint name (a list of symbols such as '(m+ a b)) or
   an integer (meaning the n-th constraint in SOURCE, starting with 1)."
  (declare (special *current-qde*))
  (let ((constraint-list (etypecase source
			   (qde    (qde-constraints   source))
			   (state  (state-constraints source)))))
    (values	; Return just 1 value
      (etypecase constr
	(constraint constr)
	(fixnum (nth (1+ constr) constraint-list))
	(cons (find constr constraint-list
		    :key #'constraint-name
		    :test #'equal))))))



#| -----------------------------------------------------------
   EXAMINE CONSTRAINTS ON A VARIABLE
   -----------------------------------------------------------
|#

(defun CONSTRAINTS-ON-VARIABLE (variable &optional (source *current-qde*))
  "Print the constraints from SOURCE (a state or qde) that contain VARIABLE."
  (declare (special *current-qde*))
  ;; Use (variable-constraints variable) instead?
  (loop for const in (etypecase source
		       (qde    (qde-constraints   source))
		       (state  (state-constraints source)))
	when (member variable (constraint-varnames const))
	do (format t "~&  ~s" const))
  (values))


#+Symbolics
(cp:define-command (COM-CONSTRAINTS-ON-VARIABLE
		     :provide-output-destination-keyword nil
		     :command-table "Global")
    ((variable '((cl:symbol)))
     (source '((cl:atom)) :default *current-qde*))
    (constraints-on-variable variable source))



#| -----------------------------------------------------------
   EXAMINE A VARIABLE
   -----------------------------------------------------------
|#

(defun GET-VARIABLE (var &optional (source *current-qde*))
  "Get a variable structure.
   If VAR is a variable structure, it is returned and SOURCE is ignored.
   Otherwise, SOURCE must be a state or qde, where VAR will be looked up,
   and VAR must be a variable name (a symbol such as 'pressure)."
  (let ((variable-list (etypecase source
			 (qde    (qde-variables   source))
			 (state  (state-variables source)))))
    (values	; Return just 1 value
      (etypecase var
	(variable var)
	(symbol (find var variable-list
		      :key #'variable-name
		      :test #'eq))))))



#| -----------------------------------------------------------
   EXAMINE A QDE
   -----------------------------------------------------------
|#

; (defun PRINT-QDE (qde-name)
;   "Show the text of the qde definition for QDE-NAME.  Argument should be
;    the first argument given to (define-qde ...)."
;   (format t "~{~{~% (~S~@{~%  ~S~})~}~}" (get qde-name 'qde-definition)))


(defun PRINT-QDE (qde-name file)
  "Print the text of the qde definition for QDE-NAME, intelligibly indented, on FILE.
  Argument should be the first argument given to (define-qde ...)."
  (with-open-file (stream file :direction :output
			  :if-exists :new-version :if-does-not-exist :create)
    (write-qde stream qde-name)))



#+TI
(defun INSERT-QDE (qde-name)
  "Insert the text of the qde definition for QDE-NAME, intelligibly indented, into the
   current buffer.  QDE-NAME should be the first argument given to (define-qde ...)."
  (let ((stream (zwei:open-editor-stream
		; :buffer-name (zwei:buffer-name (first zwei::*zmacs-buffer-list*))
		  :interval zwei:*interval*	; Current buffer
		  :undo-saving t)))
    (write-qde stream qde-name)
    ;; Commented out code works but is more cumbersome than code used.
;   (let ((self zwei:*window*))			; Current window
;     (zwei:tell-editor-to-redisplay zwei:dis-all))
    (zwei:must-redisplay-buffer zwei:*interval* zwei:dis-all)))


(defun WRITE-QDE (stream qde-name)
  "Write the text of the qde definition for QDE-NAME, intelligibly indented, to STREAM."
  (format stream "~%(Define-QDE  ~:@(~A~)" qde-name)
  (dolist (elt (get qde-name 'qde-definition))
    (format stream (case (first elt)
		     ((constraints other) "~% (~S~{~{~%  (~S~@{~%   ~S~})~}~})")
		     (text		  "~% (~S~{ ~S~})")
		     (t			  "~% (~S~{~%  ~S~})"))
	    (first elt) (rest elt)))
  (format stream "~% )~%"))



#| -----------------------------------------------------------
   COMPARE TWO QDES
   -----------------------------------------------------------
|#

#+(or ti symbolics)
(defun COMPARE-TWO-QDES (q1 q2 &optional (type 'qde))
  (loop for slot-and-accessor in (slots-and-accessors type)
	for accessor = (cadr slot-and-accessor)
	for f1 = (funcall accessor q1)
	for f2 = (funcall accessor q2)
	unless (equal f1 f2)
	  do (format t "~&~a is different:" accessor)
	     (cond ((not (and (listp f1) (listp f2)))
		    (format t " ~s    ~s~&" f1 f2))
		   ((equal (sort (copy-list f1) 'string<)
			   (sort (copy-list f2) 'string<))
		    (format t "  - But only in their order~&"))

		   (t (let ((e1 (set-difference f1 f2 :test 'equal))
			    (e2 (set-difference f2 f1 :test 'equal)))
			(when e1
			  (format t "~&   for ~S~{~&~8t~s~^~50t~s~}~&" (qde-name q1) e1))
			(when e2
			  (format t "~&   for ~S~{~&~8t~s~^~50t~s~}~&" (qde-name q2) e2)))))))


; (slots-and-accessors 'qde) currently returns:
;   ((qspaces qde-qspaces) (constraints qde-constraints)
;    (independent qde-independent) (history qde-history)
;    (transitions qde-transitions) (print-names qde-print-names)
;    (text qde-text) (name qde-name) (layout qde-layout)
;    (other qde-other)) 

#+(or ti symbolics)
(defun SLOTS-AND-ACCESSORS (type)
  (loop for manifest in (fourth (get type #+Symbolics 'Si:Defstruct-Description
				     #+TI      'sys::Defstruct-Description)
	collect (list (first manifest)
		      (seventh manifest)))))



#| -----------------------------------------------------------
   CONDITIONAL BREAK AND RESUME
   -----------------------------------------------------------

   Better than Tracing in appropriate cases because it interrupts only when
   the specified condition is true, permits free examination of the machine
   state, and then permits execution to be resumed or killed.
|#

(defmacro CATCH-ON (form &optional (errormsg "Caught it! "))
  "Break when FORM evaluates to non-nil.  Resume execution with <Resume> key."
  `(#+Symbolics scl:catch-error-restart
    #+TI       ticl:catch-error-restart  (error "So What? ")
    (when ,form
      (error ,errormsg))
    nil))


;;; Some usefull functions from Jack Vinson and Cathy Catino of the
;;; University of Pennsylvania.  Utilities for QSIM


;;; NUMBER-BEHAVIOR finds the number of the first behavior associated with
;;; the state
(defun number-behavior (state)
  (loop for beh in (get-behaviors)
	for i from 1 to 10000
	when (member state beh) do (return i)))

;;; FIND-INIT  finds the initializing state of state if I want the
;;; current simulation, just look at the qsim variable *initial-state*
(defun find-init (state)
  (let ((pred (state-predecessor state)))
    (if (null pred) state
      (find-init pred))))

;;; FIND-QUIESCENT finds the quiescent states in the list of all states
;;; By looking at the tail of each behavior - it's either quiescent, not done
;;; or transition.  Inconsistent states are filetered out by get-behaviors
(defun find-quiescent (&optional (state *initial-state*))
  (remove nil (mapcar #'(lambda (s) 
			  (if (quiescent-p (car (last s)))
			      (car (last s)) nil))
		      (get-behaviors state))))

;;; FIND-TRANSITIONS this one will find states which are transitions to
;;; other regions.
(defun find-transitions (&optional (state *initial-state*))
  (remove-duplicates
  (apply #'append
  (remove nil (mapcar #'(lambda (series) 
			  (remove nil (mapcar #'(lambda (s)
						  (if (transition-p s) s))
					      series)))
		      (get-behaviors state))))))

;;; pretty obvious, but not in structures.lisp
(defun transition-p (state)
  (or (member 'transition (state-status state))
      (member 'transition-identity (state-successors state))))
	      
;;; FIND-INCONSISTENT will find all inconsistent states from a given
;;; intialization 
(defun find-inconsistent (&optional (state *initial-state*))
  (remove-duplicates
  (apply #'append
  (remove nil (mapcar #'(lambda (series)
			  (remove nil (mapcar #'(lambda (s)
						  (if (inconsistent-p s) s))
					      series)))
		      (unfiltered-get-behaviors state))))))

;;; UNFILTERED-GET-BEHAVIORS, doesn't filter the successors through
;;; filtered-successor-slot-contents
(defun unfiltered-get-behaviors (&optional (initial-state *initial-state*)
		      &key (final-state nil)
		      (successor-function #'get-successors))
  (let ((*show-inconsistent-successors* t))
    (get-behaviors initial-state :*final-state* final-state
		   :successor-function successor-function)))



;  (if (listp initial-state)
;      (if (and (null *final-state*) (eql successor-function #'successors)) 
;	  (mapcan #'unfiltered-get-behaviors initial-state))
;      (let ((incomplete-p (incomplete-p initial-state)))
;	(mapcan #'(lambda (states)
;		    (let* ((new-states (if *final-state*
;					   (if (eq *final-state* (car states))
;					       states nil) states))
;			   (beh (reverse new-states))
;			   (new-beh (if incomplete-p (cdr beh) beh)))
;		      (when new-beh (list new-beh))))			  
;		(mapgraph
;		  #'(lambda (x) (values (list x) (not (eq x *final-state*))))
;		  (list initial-state)
;		  :apply-when-cycle t
;		  :successor-function successor-function))))

#| -----------------------------------------------------------
   EXAMINE SUCCESSORS OF A STATE
   -----------------------------------------------------------
Combined with ALL-SUCCESSORS, these functions examine the successors of a
state by returning a list or a nested list or a pretty printed list.
|#

(defun NESTED-SUCCESSORS (&optional (state *initial-state*))
  "Return a nested list of state and all its successors and their successors,
   etc., to the end of the behavior tree headed by state."
  (all-successors state :tree t))

(defun PP-NESTED (&optional (state *initial-state*))
  "Return a pretty list of state and all its successors and their successors,
   etc., to the end of the behavior tree headed by state."
  (pprint (nested-successors state)))

;;; These are the same as above, except that inconsistent states are shown
;;; as well.
(defun i-ALL-SUCCESSORS (&optional (state *initial-state*))
  (all-successors state :*show-inconsistent-successors* t))

(defun i-NESTED-SUCCESSORS (&optional (state *initial-state*))
  (all-successors state :tree t
		  :*show-inconsistent-successors* t))

(defun i-PP-NESTED (&optional (state *initial-state*))
  (pprint (i-nested-successors state)))


;;; This is similar to REPORT-DIFFERENCES (above), but it just takes two
;;; states as the arguments, reporting the differences between the qvalues
;;; of corresponding variables.
(defun STATE-DIFFERENCES (state1 state2)
  "For the pair of behaviors report differences in variable values "
  (format t "~%~%Between States ~s  and ~s" state1 state2)
  (loop for state-qspace in (state-qspaces state1)
	for param = (car state-qspace)
	for val1 = (qval param state1)
	for val2 = (qval param state2)
	unless (and val1 val2)
	do (error "Missing Qvals")
	;; Print only distinct variable values
	unless (or (equal val1 val2)
		   ;; JMV 92.7.15, if qmag is an interval
		   (and (equal (qmag val1) (qmag val2))
			(equal (qdir val1) (qdir val2)))
		   (and (atom (qmag val1)) (atom (qmag val2))
			(equal (lmark-where-defined (qmag val1))
			       (lmark-where-defined (qmag val2)))
			(equal (qdir val1) (qdir val2))))
	do (format t "~%  ~a~25t~a~40t~a" param val1 val2)))

;;; abbreviation
(defun state-diff (s1 s2) (state-differences s1 s2))


;;; LIST-DIFF runs STATE-DIFFERENCES on the list of STATES
(defun list-diff (states)
  (mapcar #'(lambda (a b) (state-differences a b))
	  states (cdr states)))
	  
;;; SUCCESSOR-DIFFERENCES uses STATE-DIFFERENCES to look at the differences
;;; between the successors of STATE
(defun successor-differences (state)
  (list-diff (get-successors state)))


;;; EXAMINE A STATE OR SERIES OF STATES using something that looks like PS
;;; (above), but only on certain variables.

;;; SHORT-PS get the qvalues of a list of VARS from STATE
;;; i.e.  (short-ps '(amount temperature) s-5)
(defun short-ps (vars state)
  (format t "~%State ~a (~a):~25t in ~s~%Qvalues:"
	  (state-name state) (state-time state) (state-qde  state))
  (pv (mapcar #'(lambda (var) (assoc var (state-qvalues state))) vars)))

;;; MULTI-PS gets the qvalues of a list of vars for a list of states
;;; i.e.  (multi-ps '(amount temperature) (list s-1 s-2 s-5 s-10))
(defun multi-ps (vars states)
  (mapcar #'(lambda (s) (short-ps vars s)) states) t)

;;; GET-QVAL gets the qvalue of VAR in STATE
(defun get-qval (var state)
  (cdr (assoc var (State-qvalues state))))

;;; LIST-QVALS prints the qvals of a particular variable over all the
;;; STATES.  If STATES is not provided list-qvals assumes it is all the
;;; states of the current simulation.
(defun list-qvals (var &optional (states (all-successors)))
  "Print the qvalues of var for every state."
  (format t "~%State     Qvalue~30tState     Qvalue~%")
  (loop with halfpoint = (ceiling (/ (length states) 2))
	for state1 in (subseq states 0 halfpoint)
	for state2 in (append (subseq states halfpoint) '(()))
	do (format t "~% ~23@<~A~;~A~>" state1 (get-qval var state1))
	when state2
	do (format t "~30t~23@<~A~;~A~>" state2 (get-qval var state2))
	))

;;; GET-QSPACE gets the qspace of VAR in STATE
;;; added 5-28-91 by Cathy Catino
(defun get-qspace (var state)
  (cdr (assoc var (state-qspaces state))))

;;; LIST-QSPACES prints the qspaces of a particular variable over the
;;; range of states in STATES.  If STATES is not provided list-qspaces
;;; assumes it is all the states of the curren simulation.
;;; added 5-28-91 by Cathy Catino
(defun list-qspaces (var &optional (states (all-successors)))
  "Print the qspaces of var for every state."
  (format t "~%State       Qspace~40tState       Qspace~%")
  (loop with halfpoint = (ceiling (/ (length states) 2))
	for state1 in (subseq states 0 halfpoint)
	for state2 in (append (subseq states halfpoint) '(()))
	do (format t "~% ~35@<~A~;~A~>" state1 (get-qspace var state1))
	when state2
	do (format t "~40t~35@<~A~;~A~>" state2 (get-qspace var state2))
	))


;;;
;;; STATE-DIFFERENCES-CHART
;;;
;;; This set of functions will report the differeces between a list of states in a
;;; chart format.

(defun successor-differences-chart (state)
  (state-differences-chart (successor-states state) state))

(defun state-differences-chart (states &optional (pred nil))
  (let* ((vars (compare-states-for-diff-vars states))
	 (num-vars (length vars))
	 (num-cols 5))
    (loop as cur-var-num from 0 to (1- num-vars) by num-cols
	  as end-loc = (if (< (+ cur-var-num num-cols) num-vars)
			   (+ cur-var-num num-cols)
			   num-vars)
	  do (print-vars-in-states states (subseq vars cur-var-num end-loc) pred)
	  do (format *Qsim-Report* "~%"))))

(defun print-vars-in-states (states vars &optional (pred nil))
  "This function will print a chart for the VARS in STATE using a single line for
each state"
  (let ((col-size 17)
	(col-start 6)
	(diff-type-list (mapcar #'(lambda (var)
				    (type-of-difference states var))
				vars)))
    (format *Qsim-Report* "~%")
    (loop for var in vars
	  for i from col-start by col-size
	  do (format *Qsim-Report* "~VT~a" i var))
    (format *Qsim-Report* "~%")
    (loop for j from 1 to (length vars)
	  for i from col-start by col-size
	  do (format *Qsim-Report* "~VT----------" i))
    (when pred
      (format *Qsim-Report* "~%Prev ~%Value:")
      (loop for var in vars
	    for i from col-start by col-size
	    do (format *Qsim-Report* "~VT~a" i (get-qval var pred))))
    (format *Qsim-Report* "~%~%")
    (loop for var in vars
	  for diff-type in diff-type-list
	  for i from col-start by col-size
	  do (format *Qsim-Report* "~VT~a" i
		     (cond ((equal diff-type :qdir)
			    (qmag (get-qval var (car states))))
			   ((equal diff-type :qmag)
			    (format nil "        ~(~a~)"
				    (qdir (get-qval var (car states)))))
			   ((null diff-type)
			    (get-qval var (car states)))
			   (t ""))))
    (loop for state in states
	  do (format *Qsim-Report* "~%~a" state)
	  do (loop for var in vars
		   for diff-type in diff-type-list
		   for i from col-start by col-size
		   do (let ((qval (get-qval var state)))
			(format *Qsim-Report* "~VT~a" i
			      (case diff-type
				((:qval) qval)
				((:qdir) (format nil "        ~(~a~)" (qdir qval)))
				((:qmag) (qmag qval)))))))
    (format *Qsim-Report* "~%")))


(defun type-of-difference (states var)
  "This function determines what is the difference between the states with respect
to VAR.  It returns one of the following :QDIR :QMAG :QVAL nil depending upon
what varies between the states."
  (let* ((ref-qdir (qdir (get-qval var (car states))))
	 (ref-qmag (qmag (get-qval var (car states))))
	 qdir-diff qmag-diff)
    (loop for state in (cdr states)
	  never (and qdir-diff qmag-diff)
	  do (let ((qdir (qdir (get-qval var state)))
		   (qmag (qmag (get-qval var state))))
	       (unless (eql qdir ref-qdir)
		 (setf qdir-diff t))
	       (unless (qmag-equivalent qmag ref-qmag)
		 (setf qmag-diff t))))
    (cond ((and qdir-diff qmag-diff)
	   :qval)
	  (qdir-diff :qdir)
	  (qmag-diff :qmag)
	  (t nil))))


