;;; -*- Mode:Common-Lisp; Package:QSIM; Default-character-style:(FIX BOLD NORMAL); Syntax:COMMON-LISP; Base:10 -*-
;;; $Id: comparison.lisp,v 1.12 1992/07/07 11:56:52 clancy Exp $

(in-package :qsim)

#+symbolics (import '(scl:condition-case sys:Inhibit-Fdefine-Warnings) :qsim)
#+ti (import '(ticl:condition-case ticl:Inhibit-Fdefine-Warnings) :qsim)

(defparameter *test-file* "q:q;test-record.lisp")

(defparameter *check-catalogs*
	      '(demos tanks medical tsa chatter1 chatter2
		      chatter3 chatter4 extras q-features))

(defparameter *qsim-test-output* t)  ; this will be a stream that directs the output

;;; This command is the normal entrance to the comparison test facility.
#+symbolics
(cp:define-command (com-QSIM-Compare-Examples
		     :provide-output-destination-keyword nil
		     :command-table "Global")
    (&key (mode '((cl:member :build :compare nil))
		:default :compare
		:documentation ":COMPARE to existing record, :BUILD new record, or neither.")
	  (test-file '((fs:pathname))
		     :default (parse-namestring "schepps:>q>examples>test-record.lisp")
		     :documentation
		     "If building, where to store new record.  If Comparing, location of existing test record.")
	  (menu 'si:boolean :default nil :mentioned-default t :documentation 
		"Whether or not to prompt with a menu of catalogs to be considered")
	  (report 'si:boolean :default t :documentation 
		  "1 line per initialization running properly.  Errors, warnings always reported."))
   (when menu (setf  *check-catalogs* (catalog-menu)))
   (scl:send-if-handles *standard-output* :clear-history)
   (let ((*test-file* test-file))
     (test-structure :mode mode :report report)))

(defparameter *warn-string* #+:lispm (merge-font-info label-font "WARNING:")
                            #-:lispm "WARNING:")

(defparameter *error-string* #+:lispm (merge-font-info label-font "ERROR:")
                             #-:lispm "ERROR")

;;; Modified so that the output can be directed to a file, the screen or to both.
;;; DJC  09/15/91

(defun qsim-test (&key (mode :compare)    ; changed to keyword paramters  DJC
		  (test-file (parse-namestring (format nil "~a:examples;test-record.lisp" *system*)))
		  (report t)
		  (output :screen)  ; can be :screen :file or :both
		  (output-file-name nil))
  (let* ((*test-file* test-file)
	 (output-file-stream nil)
	 (*qsim-test-output* (case output
			       (:screen *standard-output*)
			       (:file (setq output-file-stream (get-output-stream output-file-name)))
			       (:both (make-broadcast-stream *standard-output*
							     (setq output-file-stream
								   (get-output-stream output-file-name))))))
         #+:ccl(user::*warn-if-redefine* nil)   ;  added DJC to eliminate compiler warnings 26 Nov 91
         #+:ccl(user::*compiler-warnings* nil))
    (unwind-protect
      (test-structure :mode mode :report report)
      (when output-file-stream
        (close output-file-stream)))))


;;; It will open a stream which directs output to the output-file-name.  If 
;;; a file-name is not provided, then it prompts the user for a file-name.
;;; DJC  09/15/91

(defun get-output-stream (output-file-name)
  (let ((default-file-name "nq:nq;test-output.text"))
    (open
      (merge-pathnames 
	(cond ((null output-file-name)
	       (format t "~%Direct output to what file? [Default: ~a] " default-file-name)
	       (clear-input t)
	       (read-line t))
	      (t output-file-name))
	default-file-name)
      :direction :output :if-exists :new-version)))

    
#+:lispm
(defun catalog-menu ()
  (let ((choices (loop for cat in *available-catalogs*
		       for gen = (gensym)
		       collect (list gen (format nil "~a~10t" (symbol-name cat)) :boolean)
		       do (set gen (if (member cat  *check-catalogs*) t nil)))))
    (tv:choose-variable-values
      choices)
    (loop for (gen . tail) in choices
	  for cat in  *available-catalogs*
	  when (symbol-value gen)
	    collect cat)))

;;; *TEST* is a three level alist of the form
;;; ((catalog (structure (initialization n)(initialization n)...)
;;;           (structure (initialization n) ...)...)
;;;  (catalog (structure (initialization n) ...)...)
;;;  ...)

;;; Where catalog is on of the catalogs in *check-catalogs*, structure is one of the QDEs in
;;; that catalog, Initialization is one of the intialization for that QDE.  N is originally
;;; set to NIL.  Then it is set to the number of behaviors that intialization showed.  This
;;; structure is written out to *test-File* when mode is build and read in from *test-file*
;;; when mode is compare.

(defparameter *test* nil)



;;; Values for all these switches are bound so that test results are
;;; independent of the global settings.  Note if adding new variable:
;;; right hand sides should not be quoted, as they are not EVALed.


(defparameter *globals*
	      '((*query-user*      nil)		; Turn off questions
		(*state-limit* 20)		; Variables found in Set-Control-Variables menu.
		(*check-for-cycles* T)
		(*analytic-functions-only* NIL)
		(*check-for-unreachable-values* T)
		(*new-landmarks-on-transition* T)
		(*fire-transitions* :all)
		(*new-landmarks-across-M-constraints* NIL)
		(*new-landmarks-at-initial-state* T)
		(*ask-about-multiple-completions* NIL)
		(*show-inconsistent-successors* nil)   ; added DJC
		(*trace-count* NIL)
		(*trace-tuples* NIL)
		(*trace-constraint-filter* NIL)
		(*trace-mult-constraint* NIL)
		(trace-propagation NIL)
		(*Qsim-trace* nil)
		(*show-completion-steps* NIL)
		(*trace-global-filters* NIL)
		(*print-timing* nil)

		(*auto-sd3-evaluation*  NIL)	; variables in Set-Acc-control-variables menu
		(*derive-curvature-constraints* t)
		(*perform-acc-analysis* NIL)
		(*perform-acc-analysis* nil)
		(*search-state-limit* 30)
		(*trace-acc* nil)
		(*trace-acc-filter-application* nil)
		(*trace-aonode-creation* nil)
		(*trace-auto-sd3-evaluation* nil)
		(*trace-chatter-vars* nil)
		(*trace-children* nil)
		(*trace-sd2-derivation* nil)
		(*trace-successful-rules* nil)
		
		(*ask-before-each-tsa-step* nil)	; Undocumented variables from TIME-SCALE
		(*trace-tsa-initial-values* nil)
		(*trace-landmark-translation* nil)
		(*show-behaviors-during-tsa* nil)
		(*check-abstracted-constraints* t)
		
		(*allow-plotting* nil)		; Newly added to inhibit plotting
						; Zetalisp system variable -
		(*QSIM-Report* nil)

		(*trace-equation-propagation* nil)
		(Inhibit-Fdefine-Warnings t)	; see document examiner.
		(*check-quantitative-ranges* NIL)

		(*trace-Q2-check* NIL)		; Q2 control variables from
		(*trace-eqn-indexing* NIL)	;  Q2-CONTROL-VARIABLES 
		(*trace-ranger* NIL)
		(*trace-range-update* NIL)
		(*trace-single-char* nil)))


(defparameter *known-descriptors*
	      '((brief-description)(describe-nic)(describe-region-trans)
		(describe-ignore-qdirs)(Describe-Ign-Qval)(5box)
		(describe-curvature-constraints)(describe-egf)
		(describe-analytic-function)(describe-unreachable-landmarks))
  "These are functions in the catalog which do not actually run any example")


;;; TEST-STRUCTURE is the workhorse function of the testing program.  It goes through all of
;;; the catalogs and gets all of the QDEs.  If we are :BUILDing a new standard file, It
;;; invokes the building of a bare *TEST* structure - one with NILs for the number of
;;; behaviors for each initialization of each QDE.  The same code, mostly, is used both for
;;; building and comparing.  The difference is controlled by MODE. Depending on its value,
;;; either COMPARE-RESULT or RECORD-RESULT (or nothing) gets invoked.

;;; The PROGV binds all of the global variables within *Global*. I use a Progv (instead of
;;; LET) here so that I can also print the bindings out to the recording file.  The binding
;;; allows us to make certain that the examples are always run under the same conditions.

;;; The BLOCK form is used to handle hard errors - LISP errors and files failing to load
;;; properly.  If we know that a file failed to load, we don't bother to run the QDE.  Also,
;;; if the running the initializing evalform associated with a QDE causes a LISP error, we
;;; needn't run the different problem initializations.

;;; We use *check-catalogs* instead of *available-catalogs* because things may have been
;;; added to *Available-Catalogs* that we don't expect to be stable.

;;; Added a read-time conditional so that handler-case is called when not on a lisp
;;; machine.  This function is specified in the latest Common Lisp Spec so it may not be 
;;; included in all compilers.
;;; DJC  09/14/91

(defun test-structure (&key (mode :compare) (Report t)
		       &aux (action (case mode
				      (:compare #'compare-results)
				      ((:build nil) #'record-results))))
  (progv (mapcar #'first *globals*)(mapcar #'second *globals*)	; Bind the globals
    (case mode
      (:compare (load  *test-file*))
      (:build (build-bare-record-structure)))
    (loop for catalog in  *check-catalogs*
	  with canon = (format nil #+symbolics "~a:Examples;"
			       #+ti "~a:Examples;*.lisp"
			       #-(or symbolics ti) "~a:examples;"
			       *system*)
	  do (loop for (structure filelist evalform alist) in (symbol-value catalog)
		   do (terpri)
		      (block one-qde		
			(loop for filestring in filelist
			      for path = (merge-pathnames filestring canon)
			      do #+:lispm (condition-case ()
					      (load path :verbose nil)
					    (error (format *qsim-test-output* "~&~a Error when loading ~a~&"
							   *error-string* path)
						   (return-from one-qde nil)))
                                 #+:ccl (multiple-value-bind (val errorp)
                                                             (user::catch-error-quietly
                                                               (load path :verbose nil))
                                          (when errorp
                                            (format *qsim-test-output* "~&~a Error when loading ~a~&"
                                                    *error-string* path)
                                            (return-from one-qde nil))
                                          val)
			         #-(or :lispm :ccl)
                                    (user::handler-case                    ; Lucid uses handler case to provide
                                       (load path :verbose nil)           ; the same functionality as
                                       (error ()                          ; condition-case.
                                           (progn                         ; added DJC  09/14/91
						     (format *qsim-test-output* "~&~a Error when loading ~a~&"
							     *error-string* path)
						     (return-from one-qde nil)))))
			#+:lispm (condition-case ()
				     (eval evalform)	; Evalform is NIL for most examples.
				   (error (format *qsim-test-output* "~&~a in ~a ~a encountered a LISP error while ~
				                  EVALing initializing form ~a~&"  
						  *error-string* catalog structure 
						  evalform)
					  (return-from one-qde nil)))
                        #+:ccl (multiple-value-bind (val errorp)
                                                    (user::catch-error-quietly
                                                     (eval evalform))
                                 (when errorp
                                   (format *qsim-test-output* "~&~a in ~a ~a encountered a LISP error while ~
                                                               EVALing initializing form ~a~&"  
                                           *error-string* catalog structure 
                                           evalform)
                                   (return-from one-qde nil))
                                 val)
			#-(or :lispm :ccl)
                             (user::handler-case    ; added DJC  09/14/91
				   (eval evalform)	; Evalform is NIL for most examples.
				   (error ()
					  (progn (format *qsim-test-output* "~&~a in ~a ~a encountered a LISP error while ~
				                         EVALing initializing form ~a~&"  
							 *error-string* catalog structure 
							 evalform)
						 (return-from one-qde nil))))
			(run-all-initializations catalog structure mode
						 action alist report filelist))))
    (when (eq mode :build)
      (record-a-test))))


;;; The structure and reasoning for RUN-ALL-INITIALIZATIONS follows that of TEST-STRUCTURE.
;;; Each Initialization is an atom, and is the choice seen in the initialization menu.
;;; Evalform2 is the form (as defined in the q:catalogs;examples file) that actually runs
;;; the problem.  It is normally a function call that in turn invokes QSIM, and that resets
;;; *initial-state*.  This code depends on *initial-state* being bound to a state or list of
;;; states after the example is run.

;;; The value of ACTION depends on whether we are in a Build or Compare mode.  

;;; When loading files, use LOAD and not LOAD-FILE-UNLESS-ALREADY-LOADED because
;;; several examples use the same Function names and Variable names.  If we run the
;;; comparison more than once, the definitions will get repeatedly overloaded.  We
;;; PREFER this to having them overloaded once, and sticking with the last
;;; definition.

;;; Added a read-time conditional so that handler-case is called when not on a lisp
;;; machine.  This function is specified in the latest Common Lisp Spec so it may not be 
;;; included in all compilers.
;;; DJC  09/14/91

(defun run-all-initializations (catalog structure mode action alist report filelist)
  (loop for (initialization filestring2 evalform2) in alist
	with canon = (format nil #+symbolics "~a:Examples;"
			     #+ti "~a:Examples;*.lisp" *system*)
	for path2 = (and filestring2 (merge-pathnames  filestring2 canon))
	do (block one-initial-state
	     (when path2
	       #+:lispm (condition-case ()
			    (load path2 :verbose nil)
			  (error (format *qsim-test-output* "~&~a Error when loading ~a~&"
					 *error-string* path2)
				 (return-from one-initial-state nil)))
               #+:ccl (multiple-value-bind (val errorp)
                                           (user::catch-error-quietly
                                             (load path2 :verbose nil))
                        (when errorp
                          (format *qsim-test-output* "~&~a Error when loading ~a~&"
                                  *error-string* path2)
                          (return-from one-initial-state nil)))
	       #-(or :lispm :ccl)
                   (user::handler-case    ; added DJC  09/14/91 
			  (load path2 :verbose nil)
			  (error ()
				 (progn 
				   (format *qsim-test-output* "~&~a Error when loading ~a~&"
					   *error-string* path2)
				   (return-from one-initial-state nil)))))
	     (setf *initial-state* nil)
	     #+:lispm (condition-case ()			
			  (eval evalform2)		; Run the initialization here.
			(error (format *qsim-test-output* "~&~a in ~a~18t~a ~40t~a ~58tencountered a LISP error~&  ~
				       while EVALing execution form ~a~&  ~
				       defined in ~{~a.lisp~}" 
				       *error-string* catalog structure initialization 
				       evalform2 filelist)
			       (return-from one-initial-state nil)))
             #+:ccl (multiple-value-bind (val errorp)
                                         (user::catch-error-quietly
                                           (eval evalform2))
                      (when errorp
                        (format *qsim-test-output* "~&~a in ~a~18t~a ~40t~a ~58tencountered a LISP error~&  ~
                                                    while EVALing execution form ~a~&  ~
                                                    defined in ~{~a.lisp~}" 
                                *error-string* catalog structure initialization 
                                evalform2 filelist)
                        (return-from one-initial-state nil))
                      val)
	     #-(or :lispm :ccl)
                 (user::handler-case               ; added DJC  09/14/91			
			(eval evalform2)		; Run the initialization here.
			(error ()
			       (progn
				 (format *qsim-test-output* "~&~a in ~a~18t~a ~40t~a ~58tencountered a LISP error~&  ~
				         while EVALing execution form ~a~&  ~
				         defined in ~{~a.lisp~}" 
					 *error-string* catalog structure initialization 
					 evalform2 filelist)
				 (return-from one-initial-state nil))))
	     (unless *initial-state*		
	       (unless (member evalform2		; Some catalogs have this as an entry
			      *known-descriptors* :test 'equal)	; which doesn't run any example.
		 (format *qsim-test-output* "~&~a Within ~a ~a ~a~&    the form ~a did not ~
			   reset *initial-state*~&" *error-string* catalog structure
			   initialization evalform2))
	       (return-from one-initial-state nil))
	     (when (typecase *initial-state*
		     (cons (some #'inconsistent-p *initial-state*))
		     (state (inconsistent-p *initial-state*)))
	       (format *qsim-test-output* "~&~a in ~a~18t~a ~40t~a~58tInconsistency in *Initial-State*~&"
		       *warn-string* catalog structure initialization))
	     (funcall action catalog structure initialization mode :report report))))

(defun count-behaviors ()
  (apply #'+ (mapcar #'length
		     (mapcar #'get-behaviors
			     (get-list-of-initial-states *initial-state*)))))

(defun compare-results (catalog structure initialization ignore &key (report t)(warn-missing t))
  (declare (ignore ignore))
  (let ((standard (second (record-for catalog structure initialization)))
	(found (count-behaviors)))
    (cond ((and (null standard)  warn-missing)
	   (format *qsim-test-output* "~&~a  In ~a catalog -  Couldn't find ~a ~a in Test record.~&"
		   *warn-string* catalog structure initialization))
	  ((or (not (numberp found))(zerop found))
	   (format *qsim-test-output* "~&~a  In ~a catalog -  Found ~s bevaviors for ~a ~a.~&"
		   *warn-string* catalog found structure initialization))
	  ((not  warn-missing)
	   (format *qsim-test-output* "~&Not cataloged: ~s~18t~s ~40t~s ~73twith ~d behavior~:p.~&"
		    catalog structure initialization standard))
	  ((equal standard found)
	   (when report
	     (format *qsim-test-output* "~&Good: ~a ~18t~a ~40t~a ~73twith ~d behavior~:p.~&"
		     catalog structure initialization standard)))
	  (t  (format *qsim-test-output* "~&~a in ~a ~a ~a ~&  -found ~d behaviors, expected ~d for *initial-state* ~s"
		      *error-string* catalog structure initialization found standard *initial-state*)))))


(defun build-bare-record-structure ()
  (setf *test*
	(loop for catalog in *check-catalogs*
	      collect `(,catalog
			,@(loop for (structure filelist evalform alist)
				    in (symbol-value catalog)
				collect `(,structure
					  ,@(loop for (initialization . rest) in alist
						  collect (list initialization nil))))))))




;;; When RECORD-RESULTS is invoked, a bared record structure is already sitting there as the
;;; symbol-value of *test*.  RECORD-RESULTS uses Record-For to get the place for this
;;; catalog/structure/initialization.  It returns a list of length two.  The car is the
;;; initialization.  The cadr will be nil.  RECORD-RESULTS stores the number of behaviors
;;; into the cadr.  It (optionally) issues a report.

(defun record-results (catalog structure initialization mode &key (report t))
  (let ((found (count-behaviors)))
    (when (or (null found)(zerop found))
      (format *qsim-test-output* "~&~a  In ~a catalog -  Found ~s bevaviors for ~a ~a.~&"
	      *warn-string* catalog structure initialization))
    (when mode
      (setf (second (record-for catalog structure initialization))
	  found))
    (when report
      (format *qsim-test-output* "~%For ~a ~18t~a ~40t~a ~73t~d behavior~:p."
	      catalog structure initialization found))))

(defun record-for (catalog structure initialization)
  (assoc initialization
	 (cdr (assoc structure
		     (cdr (assoc catalog (symbol-value '*test*)))))))

;;; RECORD-A-TEST first writes a fileheader.  Then it records the values for all of the
;;; global variable (from *globals*) in a commented section.  Then it writes out the value
;;; for *test* - ie, the record showing how many behaviors for each
;;; catalog/structure/initialization .

(defun record-a-test ()
  (let ((path (parse-namestring *test-file*)))
    (with-open-file (*standard-output* path
				       :direction :output)
      (format t ";;; -*- Package: QSIM; Syntax: Common-lisp; Mode: Common-Lisp; -*-~2&~
	(in-package :qsim)~2&~
	;;; With global variables bound as~&")
      (format t "~{~&;;;   ~s~}" *globals*)
      (format t "~2&(setf *test* '~&")
      
      (pprint *test*)
      (format t ")~&"))
    (format t "~&Printed test record to file ~a~&" path)))

;;; Bound-Q allow the simulation to be run under the same bindings as
;;; *globals*, except that *qsim-report* and *allow-plotting* are T.
;;; Useful when debugging why the utility is recording an error.


(defun bound-q (&optional input)
  (progv (mapcar #'first *globals*)(mapcar #'second *globals*)	; Bind the globals
    (let ((*allow-plotting* t)
	  (*QSIM-Report* t))
      (q input))))
