;;; -*- Mode: LISP; Syntax: Common-Lisp; Base: 10; Package: (ACOMMAND) -*-

;;; COCO.
;;; Copyright: Adam Farquhar, 1990.
;;;

(in-package :acommand :use '(:lisp))

(export '(def-com
	   command-top-level
	   show-matching-commands
	   exececute-command
	   safe-eval
	   read-with-prompt
	   read-in
	   *echo-command*)
	:acommand)

;;;
;;; TODO: be robust when there is extra white space!
;;;

;;; This file implements a simple command loop.  It should be fine for
;;; command tables with up to 30 or so commands.
;;;
;;; Commands are stored in a command-table, which is currently a list of
;;; commands.
;;;
;;; Commands are defined via the macro def-com.  It places a command
;;; structure on the command table, and defines a function
;;; com-<command-name>.
;;;
;;;A command name is a list of words (strings).
;;;
;;; A line of input is read from the user, and the input is broken into
;;; words.  We then see how many commands match the words.  A command
;;; matches a list of words if every word in the list is a prefix of the
;;; corresponding word in the command name.  E.g.
;;;
;;; commands = (("show" "all" "nodes") ("show current node") ("show all
;;; frogs")) if the input were "s a", the words are ("s" "a"), and the
;;; matching commands would be ("show" "all" "nodes") and ("show all
;;; frogs").  The interpreter would indicate that there were two
;;; possible completions for the given input. "s a f" or "s al" would
;;; have been enough to disambiguate.
;;;
;;; A  "help" function  may find (SHOW-MATCHING-COMMANDS NIL *command-table*)
;;; useful.
;;;

(defparameter *count-threshold* 10)
(defparameter *echo-command* t
  "Echo the full command name before executing it.")

(defstruct (command
	     (:print-function print-command))
  name						; a list of strings
  function					; function to execute
  )

(defun print-command (command stream depth)
  (declare (ignore depth))
  (format stream "#<Com~{ ~a~}>" (command-name command)))

(defmacro def-com (name command-table &body body)
  "Define a command.  Body can be either a symbol - the name of a function
   which executes the command, or code which will be used to define a function
   named com-<name> of no arguments in the :COCO package."
  ;; 1. Define a command execution function
  ;; 2. Add it to the list of defined commands.
    (let ((command-function
	    (if (symbolp (car body))
		(car body)
		(INTERN (concatenate 'string
				     "COM-" (substitute #\- #\space (string-upcase name)))
			(symbol-package command-table)))))
      (setq name (split-string name '(#\Space #\-)))
      `(progn
	 ,@(unless (symbolp (car body))
	     `((defun ,command-function ()
	       ,@body)))
	 (let ((command (find ',name ,command-table :test #'equalp :key #'command-name)))
	   (if command
	       (setf (command-function command) ',command-function)
	       (add-command (make-command :name ',name
					  :function ',command-function) ,command-table))))))


(defmacro add-command (command command-table)
  `(setf ,command-table
	 (merge 'list (list ,command) ,command-table 'command<=)))

(defun sort-commands (table)
  (sort table 'command<=))

(defun command<= (c1 c2)
  (sentence<= (command-name c1) (command-name c2)))

(defun sentence<= (s1 s2)
  (cond ((null s1) s2)
	((null s2) NIL)
	((string> (car s1) (car s2)) NIL)
	((string<= (car s1) (car s2)) T)
	(T (sentence<= (cdr s1) (cdr s2)))))

(defun split-string (string separators &key (start 0))
  "Separate String into words which were separated by the Separators.
   Return a list of the words."
  (let ((next-good-char (position-if #'(lambda (char)
					 (not (member char separators)))
				     string :start start)))
    (when next-good-char
      (let ((next-separator 
	      (position-if #'(lambda (char)
			       (member char separators))
			   string :start next-good-char)))
	(if next-separator
	    (cons (subseq string next-good-char next-separator)
		  (split-string string  separators :start (1+ next-separator)))
	    (list (subseq string next-good-char)))))))

(defun prefix-match (partial-words words)
  "True if every partial word matches a prefix of a word."
  (cond ((null partial-words) T)
	((> (length (car partial-words))
	    (length (car words)))
	 NIL)
	((string-equal (car partial-words) (car words)
		       :end2  (length (car partial-words)))
	 (prefix-match (cdr partial-words) (cdr words)))
	(T NIL)))

(defun command-matches-p (partial-words command)
  (prefix-match partial-words (command-name command)))

(defun map-matching-commands (function partial-words command-table)
  "Partial-words is a list of strings, each of which corresponds to a word in a command name.
   Command-table is a list of commands. Apply Function to each matching command."
  (dolist (com command-table)
    (when (command-matches-p partial-words com)
      (funcall function com))))

(defun matching-commands (partial-words command-table)
  "Partial-words is a list of strings, each of which corresponds to a word in a command name.
   Command-table is a list of commands.  Return a list of the
   commands in Command-table which match the partial words."
  (remove-if-not #'(lambda (command)
		     (command-matches-p partial-words command))
		 command-table))

(defun handle-prompt (prompt)
  (cond ((stringp prompt)
	 (fresh-line)(princ prompt))
	((and (symbolp prompt)
	      (boundp prompt))
	 (fresh-line) (princ (eval prompt)))
	((functionp prompt)
	 (funcall prompt))))

(defun command-top-level (prompt command-table &key (stream t))
  "Loops forever. Print Prompt, if there is a single matching command in Command-table,
   then execute it.  Otherwise provide some feed back and continue.
   If the input begins with a ( or , then simply read, eval, and print it."
  (let (input-string
	partial-words
	count
	matching
	strong-match)
    (loop
      (handle-prompt prompt)
      (cond
	((equal (peek-char) #\( )
	 (print (safe-eval (read))))
	((equal (peek-char) #\, )
	 (read-char)
	 (print (safe-eval (read))))
	(T
	 (setq input-string (read-line))
	 (setq partial-words (split-string input-string '(#\Space)))
	 (setq matching (matching-commands partial-words command-table))
	 (setq count (length matching))
	 (cond 
	   ((zerop count) (format stream "~&There are no commands matching ~a."
				  input-string))
	   ((= count 1)
	    (execute-command (car matching)))
	   ((setq strong-match
		  (only-one-strong-match  partial-words matching))
	    (execute-command strong-match))
	   ((<= count *count-threshold*)
	    (format stream "~&There are ~a commands matching your input:" count)
	    (show-matching-commands partial-words command-table))
	   (t (if (y-or-n-p "~&There are ~a possible commands.  Show them?" count)
		  (show-matching-commands partial-words command-table)))))))))

(defun only-one-strong-match (words matching-commands)
  "If there is a command in matching with only (length words) words in its name, then
   we have a prefered match.  This allows us to prefer Foo to Foo Bar if the in put were F."
  (let ((found nil)
	(n (length words)))
    (dolist (com matching-commands found)
      (when (= (length (command-name com)) n)
	(if found
	    (return-from only-one-strong-match NIL)
	    (setq found com))))))

(defun execute-command (com)
  (when *echo-command*
    (format t "~&~{~a ~}~%" (command-name com)))
  (funcall (command-function com)))

(defun show-matching-commands (partial-words command-table &optional (stream t))
  "Print out the commands in Command-table which match Partial-words.  If Partial-words.
   is NIL, then all commands match."
  (map-matching-commands #'(lambda (com)
			     (format stream "~&    ~{ ~a~}~%" (command-name com)))
			 partial-words command-table))


(defun safe-eval (form)
  "Try to determine if form can be safely evalued before doing it.
   Warn if it cannot be."
  (cond ((numberp form) form)
	((symbolp form)
	 (if (boundp form)
	     (eval form)
	     (coco-warn "The symbol ~a is not bound." form)))
	((listp form)
	 (if (and (symbolp (car form))
		  (fboundp (car form)))
	     (eval form)
	     (coco-warn "The function ~a is not defined." (car form))))
	(t (coco-warn "I do not know that ~a can be safely evaled." form))))

(defun read-with-prompt (prompt &optional (stream t))
  (format stream "~&~a: " prompt)
  (read stream))

(defun read-in (&key prompt (type t) (stream t))
  (format stream "~&~a: " prompt)
  (let ((input (read stream)))
    (cond ((typep input type)
	   input)
	  (t
	   (format stream "~&Your input, ~a, did not satisfy the type specifier ~a."
		   input type)
	   (read-in :prompt prompt :type type :stream stream)))))

(defun print-punctuated-list (list &key (key #'identity))
  "Print the elements of list separated by commas, ending with a period."
  (do ((rest list (cdr rest))
       (len  (length list) (1- len)))
      ((null rest))
    (princ (funcall key (car rest)))
    (case len
      (1 (princ "."))
      (2 (princ ", and "))
      (otherwise (princ ", "))))
  (terpri))

(defparameter *list-delimiters* '(nil))

(defun read-list (&key (stream t) (delimiters *line-delimiters*))
  (loop  for in = (read stream)
	 if (member in delimiters)
	   do (return list)
	 else collect in into list))

