;;;  -*- Mode:Common-Lisp; Package:QSIM; Syntax:COMMON-LISP; Base:10 -*-
;;;  Dan Dvorak, 1989.
;;;
;;;  This is an experimental version of the QSIM cfilter, designed to
;;;  process large QDE models in less time and with less cons'ing.

(in-package 'QSIM)


;;;-----------------------------------------------------------------------------
;;;  Function:  (cfilter-init  qde)
;;;
;;;  Purpose:   Given the qde, this function performs all initialization on the
;;;             VARIABLE and CONSTRAINT structure instances, in preparation for
;;;             cfilter.
;;;
;;;  Returns:   the list of constraints that must be filtered.
;;;-----------------------------------------------------------------------------

(defun cfilter-init (qde)
  (let ((active-constraints nil))

    ;; Initialize some working values.
    (dolist (var (qde-variables qde))
      (let* ((pvals  (variable--pvals var))
	     (npvals (length pvals)))
	(setf (variable--npvals  var) npvals
	      (variable--onpvals var) nil)
	;; Variables having only 1 possible value (such as discrete,
	;; independent, and "time" variables) have their values set
	;; permanently here.
	(if (= 1 npvals)
	    (setf (variable--qval var) (car pvals)))))

    ;; Of the constraints within modes, activate the appropriate ones.
    (activate-moded-constraints qde)

    ;; Reset all tuple lists and collect all active constraints.
    (dolist (con (qde-constraints qde))
      (setf (constraint--tuples con) nil)
      (if (constraint-active-p con)
	  (push con active-constraints)))

    active-constraints))


(defun activate-moded-constraints (qde)
  ;; For each constraint that is conditional on a mode value, get the
  ;; current mode value to determine if the constraint should be marked
  ;; active or not.
  (let ((constraints-within-modes (qde-constraints-within-modes qde)))
    (when constraints-within-modes
      (dolist (con constraints-within-modes)
	(setf (constraint-active-p con) nil))
      (let (modes mode-qmag)
	(dolist (var (qde-variables qde))
	  (when (setq modes (variable-modes var))
	    ;; Note: mode variables can have only one pval.
	    (setq mode-qmag (qmag (variable--qval var)))
	    (dolist (con (alookup mode-qmag modes))
	      (setf (constraint-active-p con) t))))))))

;;;-----------------------------------------------------------------------------
;;;  Function:  (cfilter  qde  constraint-check-function)
;;;
;;;  Given:     -- qde, the qde structure where the full set of variables and
;;;                   constraints are anchored.
;;;             -- constraint-check-function, such as #'check-qsim-constraint;
;;;             -- It is required that the caller have already stored the
;;;                possible values for each variable in its variable--pvals slot.
;;;
;;;  Returns:   -- value-completions, a list of lists of qvals, where each list
;;;                  of qvals contains one qval for each variable of the qde.
;;;                  Each list of qvals is required to be in the same order as
;;;                  the list of variables in the qde (in order to facilitate
;;;                  efficient global filtering).
;;;
;;;  Design:    Generate possible values for all variables.
;;;             While constraints remain
;;;             {
;;;                Get the "most restrictive" remaining constraint.
;;;                Delete this constraint from the list of remaining constraints.
;;;                Generate all tuples for this constraint.
;;;                Filter the tuples through the constraint.
;;;                Prune the pvals for each variable of this constraint 
;;;                  (based on the vals appearing in the surviving tuples).
;;;                Waltz-filter the partial set of tuples.
;;;             }
;;;             Form all states from the surviving tuples, done so as to
;;;               minimize backtracking.
;;;
;;;  Note:      All callers of cfilter are required to bind the special
;;;             variable Current-State so that cfilter's subordinates
;;;             (such as constraint check functions) can access the current
;;;             state.  This is provided for use in some experimental 
;;;             constraints; it's not used by any "standard" constraints.
;;;-----------------------------------------------------------------------------


(defun cfilter (qde constraint-check-function)
  ;(start-timer 'cfilter)
  (let* ((completions   nil)
	 (myconstraints (cfilter-init qde)))

    ;; While any constraints remain unprocessed ...
    (do ()
	((null myconstraints))
      (let (con tuples)

	;; Get most restrictive remaining constraint
	(setq con (most-restrictive-constraint myconstraints))

	;; Delete this constraint from the list of unprocessed constraints.
	(setq myconstraints (delete con myconstraints :count 1))
      
	;; Generate and filter the tuples for this constraint.
	(setq tuples (product-space (mapcar #'variable--pvals (constraint-variables con))))
	(trace-current-tuples con tuples "before constraint filtering")
	(setq tuples (filter-one-tuple-set con tuples constraint-check-function))
	(trace-current-tuples con tuples "after constraint filtering")
	(unless tuples
	  ;; Mark this constraint so that print-completions can inform the
	  ;; user where the problem was first discovered.
	  (setf (constraint--tuples con) 'no-tuples-survived)
	  ;(stop-timer 'cfilter)
	  (return-from cfilter nil))
	(setf (constraint--tuples con) tuples)

	;; Do Waltz filtering on the partial constraint network.
        (unless (incremental-Waltz-filter con nil)
	  (return-from cfilter nil)))
	)

    ;(start-timer 'form-all-completions)
    (setq completions (find-completions qde))
    ;(stop-timer  'form-all-completions)
    ;(stop-timer  'cfilter)

    completions))
  

;;;  Trace function for tuples.

(defun trace-current-tuples (con tuples when)
  (cond (trace-tuples
	 (format t "~%Number of tuples is ~a" (length tuples))
	 (let ((con-name (constraint-name con)))
	   (format t "~%Set of tuples for ~a ~a:" con-name when)
	   (format t "~%~20t~a" tuples)))))
    


;;;-----------------------------------------------------------------------------
;;;  Function:  (most-restrictive-constraint  constraints)
;;;
;;;  Given:     a list of constraint instances
;;;
;;;  Returns:   the constraint that is estimated to produce the fewest tuples.
;;;
;;;  Design:    This function estimates the number of tuples that a constraint
;;;             will produce by simply computing the product of the number of
;;;             possible values of its variables.  In effect, this heuristic
;;;             favors constraints having few variables and few possible values
;;;             for those variables.
;;;-----------------------------------------------------------------------------

(defun most-restrictive-constraint (constraints)
  (let ((minproduct most-positive-fixnum)
	minconstraint)
    (dolist (con constraints minconstraint)
      (let ((product (variable--npvals (first (constraint-variables con)))))
	(dolist (variable (rest (constraint-variables con)))
	  (setq product (* product (variable--npvals variable))))
	(if (= 1 product) (return con))		; return early if product = 1
	(if (< product minproduct)
	    (setq minproduct product
		  minconstraint con))))))



;;;-----------------------------------------------------------------------------
;;;  Function:  (update-pvals  con  notvariable)
;;;
;;;  Given:     -- con, a constraint
;;;             -- notvariable, a variable of this constraint whose pvals need not
;;;                be updated (since the caller had just updated that variable's
;;;                pvals [and then subsequently deleted one or more tuples
;;;                from this constraint]).
;;;
;;;  Purpose:   Updates the possible values for each variable of the constraint.
;;;
;;;  Returns:   T if each variable has at least one pval, else nil.
;;;
;;;  Design:    For each variable of this constraint (except the variable that the
;;;             caller "came from"), delete from the variable's list of pvals any
;;;             pval that does not appear in at least one of the constraint's
;;;             tuples.
;;;-----------------------------------------------------------------------------
	
(defun update-pvals (con notvariable)
  (let ((tuple-set  (constraint--tuples con)))

    ;; For each variable of this constraint ...
    (do ((variables (constraint-variables con) (cdr variables))
	 (n 0 (1+ n)))
	((endp variables) t)

      (let ((variable (car variables)))
	(if (not (eq variable notvariable))
	    (let ((pruned-pvals (delete-if-not
				  #'(lambda (pval)
				      (member-if #'(lambda (tuple)
						     (qval-equal pval (nth n tuple)))
						 tuple-set)) 
				  (variable--pvals variable))))
	      ;; If all pvals eliminated, return.
	      (when (null pruned-pvals)
		;; Record where the problem occurred so that print-completions
		;; can inform the user.
		(setf (variable--pvals variable) 'no-pvals-survived)
		(return-from update-pvals nil))
	      (setf (variable--pvals variable)   pruned-pvals
		    (variable--npvals variable)  (length pruned-pvals))
	      (if (null (variable--onpvals variable))
		  (setf (variable--onpvals variable) (variable--npvals variable)))))))))



;;;-----------------------------------------------------------------------------
;;;  Function:  (incremental-Waltz-filter  con  notvariable)
;;;
;;;  Given:     -- con, a constraint to start from
;;;             -- notvariable, a variable that the caller knows does not need to
;;;                have its pvals updated.
;;;
;;;  Purpose:   Deletes all tuples that are not pair-wise consistent with all
;;;             neighboring tuples.
;;;
;;;  Returns:   T if each constraint has at least one surviving tuple, else NIL.
;;;
;;;  Design:    This is a recursive function.  Starting from the given
;;;             constraint, it first updates the pvals of that constraint.
;;;             Then, for those variables of the constraint whose pvals were
;;;             actually reduced, it checks the tuples of each attached
;;;             constraint.  If any tuples contain values that are no longer
;;;             covered in the pvals, then those tuples are deleted and a
;;;             recursive call is made on the affected constraint.
;;;-----------------------------------------------------------------------------

(defun incremental-Waltz-filter (con notvariable)

  ;; Return immediately if some variable has no more pvals.
  (if (null (update-pvals con notvariable))
      (return-from incremental-Waltz-filter nil))

  ;; For each variable of the constraint, check its constraints only if its
  ;; number of pvals has decreased.
  (dolist (var (constraint-variables con) T)
    (if (< (variable--npvals var) (variable--onpvals var))
	(let ((pvals (variable--pvals var)))
	  (setf (variable--onpvals var) (variable--npvals var))

	  ;; For each constraint attached to this variable ...
	  (dolist (ncon (variable-constraints var))
	    (if (and (not (eq con ncon))
		     (constraint-active-p ncon))
		(let* ((index (position var (constraint-variables ncon)))
		       (changed nil)
		       (tuples (constraint--tuples ncon)))
		  ;; If the tuples have not been generated yet for this constraint,
		  ;; then skip it for now.
		  (when tuples
		    (setq tuples
			  (delete-if-not
			    #'(lambda (tuple)
				(cond ((member (nth index tuple) pvals :test #'qval-equal))
				      (t (if trace-tuples
					     (format t "~%Waltz: deleting tuple ~a because ~a not in pvals ~a"
						 tuple
						 (nth index tuple)
						 pvals))
					 (setq changed t) nil)))
			    tuples))

		    (setf (constraint--tuples ncon) tuples)
		    ;; If no more tuples, return.
		    (when (null tuples)
		      ;; Record where the problem occurred so that print-completions
		      ;; can inform the user.
		      (setf (constraint--tuples ncon) 'no-tuples-survived)
		      (return-from incremental-Waltz-filter nil))
		    (if changed
			(incremental-Waltz-filter ncon var))))))))))



;;;  Adds backjump (intelligent backtracking) and temporary learning of
;;;  bad patterns to form-all-states, recursive-form-states.


;;;-----------------------------------------------------------------------------
;;;  Function:  (find-completions  qde)
;;;
;;;  Given:     the qde
;;;
;;;  Returns:   a list of consistent value completions of the qde.
;;;
;;;  Purpose:   This function, which is run after Waltz filtering, takes the
;;;             surviving tuples of each constraint and forms all possible
;;;             consistent value completions.
;;;
;;;  Design:    The basic idea in this algorithm is that we pick some constraint,
;;;             bind its variables using the next tuple of the constraint, and
;;;             then move on to another unprocessed constraint that is connected
;;;             to one of the processed constraints.  Of the candidate constraints
;;;             on this "boundary", we always take the one with the fewest tuples.
;;;             By moving in a "connected order" from constraints having few
;;;             tuples to constraints having many tuples, value completions are
;;;             formed recursively with a minimum amount of backtracking.
;;;             Here are the basic steps:
;;;
;;;             1.  For every constraint having only one tuple, bind its
;;;                 variables "permanently" since they must be the same for all
;;;                 completions.
;;;             2.  From the list of constraints processed in step 1, form the
;;;                 initial boundary consisting of directly-connected constraints
;;;                 that were not processed in step 1.
;;;             3.  Find the constraint having the fewest tuples in the boundary
;;;                 and delete it from the boundary.  Watch out for two special
;;;                 cases here -- the boundary could be nil either because there
;;;                 were no 1-tuple constraints or because ALL constraints had
;;;                 only one tuple.  In the latter case we're all done.
;;;             4.  Call (recursive-form-states constraint boundary).  The
;;;                 remaining steps are inside this recursive function.
;;;             5.  Temporarily bind any unbound variables of this constraint.
;;;                 Add to the boundary the unprocessed constraints that are
;;;                 directly connected to these newly bound variables.
;;;             6.  If boundary is empty, then we're all done.  Otherwise, find
;;;                 the constraint having the fewest tuples in the boundary and
;;;                 remove it.  Call (recursive-form-states constraint boundary).
;;;
;;;  Note:      This function could have been written in a very simple recursive
;;;             way to bind the variables of the first constraint, then do the
;;;             second constraint, etc, backtracking whenever an inconsistency
;;;             occurred.  This would probably be fine for small constraint
;;;             networks, but the amount of backtracking grows exponentially
;;;             with the number of constraints.
;
; In 1st pass, set all 1-tuple constraints and collect next-best set of constraints.
;    and compute and save ntuples and mark inactives done and 1-tuples done.
; If no more constraints (all are 1-tuples), then done.
; Form initial boundary from neighbors of 1-tuples.
; If no initial boundary (no 1-tuples), then look at set of next-best.
; If exactly one next-best constraint, use it as initial boundary. Naw,
; just take the first example of next-best and forget look-ahead.

; Normal boundary update: pick constraint on boundary having fewest tuples and,
;   in case of a tie, the one whose estimated number of actual tuples is fewest,
;   where estimated is: if a tuple var has N>1 pvals but is already set by a
;   previous constraint, then estimated tuples = #tuples/N.
;;;-----------------------------------------------------------------------------

;(defparameter *all-completions* nil)
;(defparameter *num-completions* 0)

;(defother qde      num-mode-variables)
;(defother qde      ordered-constraints)
(defother variable level-when-set)
(defother variable level-when-used)
(defother constraint -ntuples)			; compute once and save!


(defun find-completions (qde)
  (let ((*all-completions*  nil)
	(*num-completions*  0)
	(variables          (qde-variables qde)))
    (declare (special *all-completions* *num-completions* variables))

    ;; Process each constraint, saving all consistent value completions.
    (find-completions-2 (ordered-constraints qde) 1)

    ;; Return the completions.
    *all-completions*))


;;;-----------------------------------------------------------------------------
;;;  Function:  (ordered-constraints  qde)
;;;
;;;  Purpose:   This function does three things:
;;;             1.  It initializes the constraint network for constraint-
;;;                 satisfaction processing (as done by find-completions-2);
;;;             2.  It binds the variables of all constraints having only
;;;                 one tuple (since there is only one possible value); and
;;;             3.  Of the remaining constraints, it returns them in an order
;;;                 that minimizes backtracking during constraint satisfaction.
;;;
;;;  Note:      The ordering generated in step 3 is *extremely* important in
;;;             reducing the running time of doing constraint satisfaction
;;;             (which is exponentially complex in the number of constraints).
;;;             Briefly, the constraints are put in a connected order with
;;;             constraints having the fewest tuples appearing first and
;;;             constraints having the most tuples appearing last.
;;;-----------------------------------------------------------------------------
    
(defun ordered-constraints (qde)
  (let* ((variables    (qde-variables qde))
	 (constraints  (qde-constraints qde))
	 (num-remaining-constraints (length constraints))
	 (min-ntuples  most-positive-fixnum)
	 (max-nvars    0)
	 (best-constraint nil))

    ;; Step 0 -- Prepare 'variable' structures (leaving "time" variable alone).
    ;; Any variable which has only one possible value can have its qval set
    ;; permanently here.  This is in fact necessary for discrete variables
    ;; since they don't necessarily participate in constraints.
    (dolist (var (cdr variables))
      (setf (variable--qval var)           nil
	    (variable-level-when-set  var) nil
	    (variable-level-when-used var) nil
	    (variable-done-p var)          nil))

    ;; Step 1 -- Mark inactive constraints "done", bind variables of
    ;;           all 1-tuple constraints, and find best starting
    ;;           constraint (fewest tuples, most variables).
    (dolist (con constraints)
      (if (constraint-active-p con)

	  ;; Active constraint.
	  (let* ((tuples  (constraint--tuples con))
		 (ntuples (length tuples)))
	    (setf (constraint--ntuples con) ntuples)
	    (if (= 1 ntuples)
		;; Only 1 tuple.
		(progn
		  (setf (constraint-done con) t)
		  (decf num-remaining-constraints)
		  ;; Bind variable values from this tuple.
		  (unless (bind-vars (first tuples) 0 0)
		    (error "Inconsistent values.")))
		;; > 1 tuples.
		(let ((nvars (length (constraint-variables con))))
		  (setf (constraint-done con) nil)
		  (cond ((< ntuples min-ntuples)
			 (setq min-ntuples     ntuples
			       max-nvars       nvars
			       best-constraint con))
			((and (= ntuples min-ntuples)
			      (> nvars max-nvars))
			 (setq max-nvars nvars
			       best-constraint con))))))

	  ;; Inactive constraint.
	  (progn
	    (setf (constraint-done con) t)	; want to skip inactive constraints
	    (decf num-remaining-constraints))))
	
    ;; If all constraints were processed above (rare, but possible),
    ;; then form the single state and return.
    (when (= 0 num-remaining-constraints)
      (return-from ordered-constraints nil))
	
    ;; Step 2 -- From the list of constraints processed in step 1, form the
    ;;           initial boundary consisting of directly-connected constraints
    ;;           that were not processed in step 1.
    (let ((ordered-constraints (list best-constraint))
	  (boundary nil)
	  nextcon)

      ;; Build initial boundary.
      (dolist (ncon (constraint-neighbors best-constraint))
	(unless (constraint-done ncon)
	  (pushnew ncon boundary)))

      ;; Loop over remaining constraints to build an ordered list of
      ;; constraints for recursive formation of states.
      (dotimes (n num-remaining-constraints)
	(multiple-value-setq (nextcon boundary) (find-best-constraint boundary constraints))
	(push nextcon ordered-constraints))

      ;; For each variable, record the level at which it is set and
      ;; the level beyond where it is first used.
      (let ((ordered-constraints (nreverse ordered-constraints))
	    (level 0))
	;;(pprint ordered-constraints)
	(dolist (con ordered-constraints)
	  (incf level)
	  (dolist (var (constraint-variables con))
	    (cond
	      ;; If variable not previously set, then set it at this level.
	      ((null (variable-level-when-set var))
	       (setf (variable-level-when-set var) level))

	      ;; If variable not previously used, then this is the level of
	      ;; first usage (unless var appears twice in same constraint).
	      ((and (null (variable-level-when-used var))
		    (not (= level (variable-level-when-set var))))
	       (setf (variable-level-when-used var) level)))))

	;;(format t "~%level-when-set:  ~a" (mapcar #'variable-level-when-set variables))
	;;(format t "~%level-when-used: ~a" (mapcar #'variable-level-when-used variables))
	;(print (mapcar #'constraint--ntuples ordered-constraints))
	ordered-constraints))
    )
  )

;(defvar *invisible-qval* (make-qval :qmag (list *minf-lmark* *inf-lmark*) :qdir 'ign))

;;;-----------------------------------------------------------------------------
;;;  Function:  (recursive-form-states  remaining-constraints  my-level)
;;
;;;  Given:     -- a list of constraints remaining to be checked in forming
;;;                a state;
;;;             -- the current recursion depth.
;;;
;;;  Returns:   two values:
;;;             -- an integer equal to the depth that should be backjumped to,
;;;                else its own depth if no backjumping needed;
;;;             -- the smallest depth that this call or any deeper call
;;;                depended on
;;;
;;;  Side Effects:  The last (deepest) recursive call forms a list of value
;;;                 completions and pushes that list onto *all-completions*.
;;;
;;;  Design:    If no more constraints, push value completions onto *all-completions*
;;;             and return.  Otherwise, take the first remaining constraint,
;;;             and check/bind its variables.  If there is a contradiction
;;;             such that one of its variables cannot be bound, then return.
;;;             Otherwise, do a recursive call with the tail of the list of
;;;             constraints.  
;;;
;;;             3/27/89: Intelligent backtracking ("backjumping") added.
;;;             If none of the tuples of a given constraint are compatible with
;;;             the current bindings, then backjump to the nearest constraint
;;;             (in terms of recursion depth) that has bound a value that
;;;             conflicts with the current constraint.
;
;for each tuple of the constraint
;   if tuple is compatible with all previous settings
;   then
;      if tuple matches a temporary disallowed pattern, then skip this tuple.
;      if tuple matches a chain of disallowed patterns (either starting at
;         this level or being tracked from a previous level)
;         then
;            if this is end of chain,
;               then skip this tuple,
;               else include this chain for next deeper r-f-s call
;
;   then if tuple does not match any temporary disallowed pattern from a
;              previous backjump) or any permanent disallowed pattern in the
;              branches being tracked
;        then call r-f-s and save return-depth, dependent-depth
;        if return-depth < my-depth
;        then return with return-depth and updated dependent depth
;        if return-depth = my-depth
;        then (0) I have been backjumped to;
;             (1) save my current tuple as a temporary disallowed pattern,
;                 with nil for any vals not depended upon at time of backjump;
;                 For each var that I set, if its depended-on-level is <= level
;                 of backjumper, then the var's current value is used in the
;                 pattern, else NIL is used.  If no such var found, error.
;             (2) remember that a backjump occurred.
;( else tuple is either incompatible or matches a disallowed pattern, so skip it)
;
;
;if no tuple of this constraint produced a completion
;   then 
;      (2) backjump to closest conflicting dependency.
;   else
;      (1) make each temporary disallowed pattern into a chain running from
;          smallest-dependent-depth to my depth, making sure to use as much
;          of a common branch that starts at s-d-d as possible.
;;;-----------------------------------------------------------------------------

(defun find-completions-2 (remaining-constraints my-level)
  (declare (special variables *all-completions*))
  (declare (optimize speed))

  ;; If no more constraints ...
  (when (null remaining-constraints)
    (record-completion variables)
    (return-from find-completions-2 (values nil nil)))

  ;; Take the next constraint on the list.
  (let* ((con  (car remaining-constraints))
	 (old-num-completions *num-completions*)
	 (conflict-patterns   nil)
	 (my-conflict-bits    0)		; levels that I conflict with.
	 bound?)

    ;; For each tuple of this constraint ...
    (dolist (tuple (constraint--tuples con))

      (multiple-value-setq (bound? my-conflict-bits)
	(bind-vars tuple my-level my-conflict-bits))

      ;(when (and bound? (goodp tuple conflict-patterns))
      (when bound?
	(multiple-value-bind (abort-level returned-conflict-bits)
	    (find-completions-2 (cdr remaining-constraints) (1+ my-level))

	  (cond
	    ;; No backjump, so just continue on to next tuple.
	    ((null abort-level))

	    ;; Got a backjump to my level, so:
	    ;; (1) merge my conflicts with those of backjumper in case I backjump
	    ;;     (so that I don't backjump over a level that the current
	    ;;     backjumper had a conflict with);
	    ;; (2) make a conflict pattern (when appropriate); and
	    ;; (3) continue on to next tuple.
	    ((logbitp my-level returned-conflict-bits)
	     ;;(write-char #\.)
	     (setq my-conflict-bits (logior my-conflict-bits returned-conflict-bits))
;	     (let ((pattern (make-conflict-pattern con tuple my-level abort-level)))
;	       (when pattern
;		 (push pattern conflict-patterns)))
	     )

	    ;; This backjump is just "passing through", so continue returning.
	    (t (return-from find-completions-2
		 (values abort-level returned-conflict-bits))))

	  )					; end of multiple-value-bind
	)					; end of when and bound? ...
      )						; end of dolist tuple

      ;; All tuples have been tried.  If none of them yielded a completion,
      ;; then backjump to nearest level where a conflict occurred.
      (if (= *num-completions* old-num-completions)
	  ;; No completions, so backjump.
	  (values my-level my-conflict-bits)
	  ;; At least one completion, so normal return (no backjump).
	  (values nil nil))
      )
    )

;;;-----------------------------------------------------------------------------
;;;-----------------------------------------------------------------------------

  
(defvar *check-ignore-qvals* nil)

(defun record-completion (variables)

  ;; See if this completion matches any previous completion (ignoring
  ;; all invisible variables).
  (when *check-ignore-qvals*
    (dolist (completion *all-completions*)
      (when (do ((vars  (cdr variables)  (cdr vars))
		 (pairs (cdr completion) (cdr pairs)))
		((endp vars) t)
	      (unless (variable-invisible-p (car vars))
		(unless (eq (variable--qval (car vars)) (cdar pairs))
		  (return nil))))
	;; this completion matches a previous one, so don't save it.
	(return-from record-completion))))
    
  ;; This completion does not match any previous completion,
  ;; so push the state values onto *all-completions* and return.
  ;(write-char #\.)
  (push (mapcar #'(lambda (var)
		    (cons (variable-name var)
			  (variable--qval var)))
		variables)
	*all-completions*)

;    (when (= 0 (mod *num-completions* 20))
;      (multiple-value-bind (sec min hour)
;	  (get-decoded-time)
;	(declare (ignore sec))
;	(format t "~%~4d completions at ~2d:~2,'0d" *num-completions* hour min)))

  (incf *num-completions*))

;;;-----------------------------------------------------------------------------
;;;  Function:  (bind-vars  tuple  my-level  conflict-bits)
;;;
;;;  Purpose:   If each qval of the given tuple (1) is compatible with a
;;;             previous binding of its variable, or (2) is bound at this level,
;;;             then the new bindings are established and the function returns
;;;             T.  If the tuple is not compatible with any previous bindings,
;;;             then return NIL along with updated conflict bits.
;;;
;;;  Note:      Conflict-bits is an integer where bit j is set to 1 if a 
;;;             conflict has occurred with a variable that was bound at level j.
;;;-----------------------------------------------------------------------------

(defun bind-vars (tuple my-level conflict-bits)

  ;(format t "~%Trying tuple ~a" tuple)

  (let ((conflict nil))
    (dolist (tval tuple)
      (let* ((var (qval-variable tval))
	     (var-level-when-set (variable-level-when-set var)))

	(cond ((null var-level-when-set)
	       (setf (variable--qval var) tval
		     (variable-level-when-set var) my-level))

	      ((= var-level-when-set my-level)
	       ;; then set its value
	       (setf (variable--qval var) tval))

	      ((< var-level-when-set my-level)
	       ;; else it is already set,so check for compatible value.
	       (unless (qval-equal tval (variable--qval var))
		 ;(format t " --> conflicts with ~a = ~a" var (variable--qval var))
		 (setq conflict t)
		 (setf (logbitp var-level-when-set conflict-bits) t)))

	      (t (error "Attempt to bind a level ~d variable at level ~d"
			var-level-when-set my-level)))))

    (values (not conflict) conflict-bits)))

	

;;;-----------------------------------------------------------------------------
;;;  Function:  (goodp  tuple  conflict-patterns)
;;;
;;;  Purpose:   Returns NIL if the given tuple matches one of the conflict
;;;             patterns; otherwise returns non-nil.
;;;
;;;  Note:      A conflict pattern is like a partially-specified tuple where
;;;             a NIL matches anything.  A tuple matches a conflict pattern if
;;;             all their corresponding elements match.
;;;-----------------------------------------------------------------------------

(defun goodp (tuple conflict-patterns)
  (dolist (pattern conflict-patterns T)
    (when (do ((tvals   tuple    (cdr tvals))
	       (pvals   pattern  (cdr pvals)))
	      ((endp tvals) t)
	    (and (car pvals)
		 (not (eq (car tvals) (car pvals)))
		 (return nil)))
      ;; We've matched a conflict pattern, so skip this tuple.
      (return-from goodp NIL))))


;;;-----------------------------------------------------------------------------
;;;  Function:  (make-conflict-pattern  constraint  tuple  my-level  backjumper-level)
;;;
;;;  Purpose:   Decide if a conflict pattern should be created and, if so,
;;;             create and return it.
;;;
;;;  Design:    A conflict pattern is essentially a partially-specified
;;;             constraint tuple which, given the variable bindings up to this
;;;             point, can *never* result in a completion.
;;;-----------------------------------------------------------------------------

(defun make-conflict-pattern (con tuple my-level backjumper-level)
  (let ((cvars (constraint-variables con))
	(num-set-vars 0)
	(num-non-used-vars 0))

    ;; Count how many variables of this constraint are set at this level and
    ;; how many are not used between this level and the backjumper's level.
    (dolist (var cvars)
      (when (= my-level (variable-level-when-set var))
	(incf num-set-vars)
	(when (and (variable-level-when-used var)
		   (> (variable-level-when-used var) backjumper-level))
	  (incf num-non-used-vars))))

    ;; Don't bother to create a conflict pattern unless there are at least
    ;; two variables set at this level and at least one of them is "dangling"
    ;; (i.e., not used between here and the backjumper level).  All other
    ;; cases, although valid as conflict patterns, will not recur and
    ;; therefore don't need to be saved.
    (if (and (> num-set-vars 1)
	     (> num-non-used-vars 0))

	(progn
	  (write-char #\.)
	;; Construct the conflict pattern and save for this invocation.
	(mapcar #'(lambda (tval)
		    (let ((var (qval-variable tval)))
		      (cond ((not (= my-level (variable-level-when-set var)))    nil)
			    ((null (variable-level-when-used var))               nil)
			    ((> (variable-level-when-used var) backjumper-level) nil)
			    (t tval))))
		tuple)
	)
	;; else return nil
	nil)))




;;;-----------------------------------------------------------------------------
;;;  Function:  (update-boundary  boundary  con)
;;;
;;;  Purpose:   Given a constraint that has just been selected from the
;;;             boundary, this function deletes the constraint from the
;;;             boundary and then adds the constraint's unprocessed neighbors
;;;             to the boundary.
;;;
;;;  Returns:   The updated boundary (which is a list of constraints).
;;;-----------------------------------------------------------------------------

(defun update-boundary (boundary con)
  (setf (constraint-done con) t)
  (dolist (var (constraint-variables con))
    (setf (variable-done-p var) t))
  (when boundary
    (setq boundary (delete con boundary :count 1)))
  (dolist (ncon (constraint-neighbors con))
    (if (not (constraint-done ncon))
	(pushnew ncon boundary)))
  boundary)


;;;-----------------------------------------------------------------------------
;;;  Function:  (find-best-constraint  boundary &optional constraints)
;;;
;;;  Purpose:   This function is called repeatedly to find the next best
;;;             constraint to put on the list of ordered constraints.  The
;;;             "best" constraint is the one with the fewest estimated tuples.
;;;             The estimate can be lower than the actual number of tuples.
;;;             For example, if some variable of the constraint has 3 possible
;;;             values but the variable has already been set (according to the
;;;             ordering produced so far), then divide the actual number of
;;;             tuples by 3.
;;;
;;;  Returns:   Two values:
;;;             -- best constraint
;;;             -- updated boundary
;;;-----------------------------------------------------------------------------

(defun find-best-constraint (boundary &optional constraints)
  (let ((min-ntuples    most-positive-fixnum)
	(min-constraint nil)
	(my-constraints (if boundary boundary constraints)))

    (dolist (con my-constraints)
      (when (not (constraint-done con))
	(let* ((ntuples     (constraint--ntuples con))
	       (est-ntuples ntuples)
	       npvals)
;	  (dolist (var (constraint-variables con))
;	    (when (and (variable-done-p var)
;		       (> (setq npvals (variable--npvals var)) 1))
;	      (setq est-ntuples (/ est-ntuples npvals))
;	      (format t " ~d->~d" ntuples est-ntuples)
;	      ))
	  (when (<= est-ntuples 1)
	    (return-from find-best-constraint (values con (update-boundary boundary con))))
	  (when (< est-ntuples min-ntuples)
	    (setq min-ntuples est-ntuples
		  min-constraint con)))))

    ;(format t "~%~27a because it has fewest estimated TUPLES (~d)." min-constraint min-ntuples)
    (values min-constraint (update-boundary boundary min-constraint))))

    


; PRODUCT-SPACE enumerates the points generated by a set of axes,
; each of which is a list of values.  It is used to make the tuples.

(defun product-space (axes)
  (cond ((null axes) (list nil))
	(t (add-axis (car axes) (product-space (cdr axes))))))

(defun add-axis (axis subspace)
  (mapcan #'(lambda (point) (add-point point subspace))
	  axis))

(defun add-point (point subspace)
  (mapcar #'(lambda (vector) (cons point vector))
	  subspace))




; This scans down the set of tuples, deleting the ones that fail to satisfy
; the constraint.

(defun filter-one-tuple-set (con tuples constraint-check-function)
  (if trace-count
      (format *QSIM-Trace* "~%Constraint filter ~a: (~a) "
	      (constraint-name con) (length tuples)))
  (setq tuples (delete-if
		 #'(lambda (tuple)
		     (unless (funcall constraint-check-function tuple con)
		       (if trace-constraint-filter
			   (format *QSIM-Trace* "~%Constraint ~a filtering out tuple ~a."
				   (constraint-name con) tuple))
		       t  ; return t so this one is deleted.
		       ))
		 tuples))
  (if trace-count
      (format *QSIM-Trace* "-> (~a)" (length tuples)))
  tuples)

