;;; -*- Mode:Common-Lisp; Package:QSIM; Syntax:COMMON-LISP; Base:10 -*-

(in-package 'QSIM)

;       Copyright (c) 1987, Benjamin Kuipers.

; The home-heater is a tutorial example that may replace the bathtub example.

(define-QDE HOME-HEATER-PROP
  (text "Proportional control of home temperature by heating/cooling")
  (quantity-spaces
    (heat    (0 inf))
    (mass    (0 inf))
    (TempIn  (0 inf))
    (TempOut (0 inf))
    (TempSet (0 inf))
    (dTemp   (minf 0 inf))
    (error   (minf 0 inf))
    (R       (0 inf))
    (HFout   (minf 0 inf))
    (HFin    (MaxCool 0 MaxHeat))
    (netHF   (minf 0 inf)))
  (constraints
    ((mult TempIn Mass Heat))
    ((add TempOut dTemp TempIn))
    ((add TempIn error TempSet))
    ((mult R HFout dTemp))
    ((m+ error HFin)         (0 0)  (minf MaxCool) (inf MaxHeat))
    ((add HFout netHF HFin))
    ((d//dt Heat netHF)))
  (independent Mass TempOut TempSet R)
  (dependent Heat TempIn dTemp HFout error HFin netHF)
  (history TempIn)
  (layout (Heat Mass nil)
	  (TempOut TempIn TempSet)
	  (dTemp R error)
	  (HFout netHF HFin))
  (print-names
    (heat    "Heat content"   H)
    (mass    "Thermal mass"   M)
    (TempIn  "Temp(inside)"   Ti)
    (TempOut "Temp(outside)"  To)
    (TempSet "Temp(set)"      Ts)
    (dTemp   "dTemp(in,out)"  dT)
    (error   "dTemp(set,in)"  E)
    (R       "Heat flow resistance")
    (HFout   "Heat flow (out)" HFo)
    (HFin    "Heat flow (in)" HFi)
    (netHF   "net Heat Flow"  nHF))
  (other
    (define-normal ((HFin ((0 MaxHeat) std)) (Heat ((0 inf) std))
		    (Mass ((0 inf) std)) (TempOut ((0 inf) std)) (TempSet ((0 inf) std))
		    (R ((0 inf) std))))
    (normal-state nil)))

(defun outside-temperature-drop ()
  (let* ((normal (get-normal-state Home-Heater-Prop))
	 (new (make-modified-state
		normal
		`((TempOut  (,(lower-than-normal 'TempOut normal) std))
		  (TempSet  (,(normal-value 'TempSet normal) std))
		  (Mass     (,(normal-value 'Mass normal) std))
		  (R        (,(normal-value 'R normal) std))
		  (TempIn   (,(normal-value 'TempIn normal) nil)))
		"Decreased outside temperature."
		)))
    (simulate-across-time-scales new `((normal ,normal)))
    t))

; Need:
;  - extended named-state capability:  high and low normals


; The home-heater with thermostat illustrates the heater going on and off.
;  - branches according to possible too-cold temperature for heater.
;  - currently, doesn't heat up enough to turn heater off again!

(define-QDE HOME-HEATER-THERMO
  (text "Thermostatic control of home heater")
  (quantity-spaces
    (heat    (0 inf))
    (mass    (0 inf))
    (TempIn  (0 RoomTemp inf))
    (TempOut (0 Cold RoomTemp Hot inf))
    (TempSet (0 RoomTemp inf))
    (dTemp   (minf 0 inf))
    (error   (minf Lo 0 Hi inf))
    (R       (0 inf))
    (HFout   (minf 0 inf))
    (HFin    (0 On))
    (netHF   (minf 0 inf)))
  (constraints
    ((mult TempIn Mass Heat))
    ((add TempOut dTemp TempIn)      (RoomTemp 0 RoomTemp) )
    ((add TempSet error TempIn)      (RoomTemp 0 RoomTemp) )
    ((mult R HFout dTemp))
    ((add HFout netHF HFin))
    ((d//dt Heat netHF)))
   (transitions
     ((error (Lo dec)) turn-furnace-on)
     ((error (Hi inc)) turn-furnace-off))
  (independent HFin Mass TempOut TempSet R)
  (dependent Heat TempIn dTemp HFout error netHF)
  (history TempIn)
  (layout (Heat Mass nil)
	  (TempOut TempIn TempSet)
	  (dTemp R error)
	  (HFout netHF HFin))
  (print-names
    (heat    "Heat content"   H)
    (mass    "Thermal mass"   M)
    (TempIn  "Temp(inside)"   Ti)
    (TempOut "Temp(outside)"  To)
    (TempSet "Temp(set)"      Ts)
    (dTemp   "dTemp(in,out)"  dT)
    (error   "dTemp(set,in)"  E)
    (R       "Heat flow resistance")
    (HFout   "Heat flow (out)" HFo)
    (HFin    "Heat flow (in)" HFi)
    (netHF   "net Heat Flow"  nHF)))

(defun low-outside-temperature ()
  (let ((start (make-initial-state Home-Heater-Thermo
				   '((HFin (0 std)) (Heat ((0 inf) nil))
				     (TempIn (RoomTemp nil))
				     (Mass ((0 inf) std)) (TempOut (Cold std))
				     (TempSet (RoomTemp std)) (R ((0 inf) std)))
				   "Room temperature; furnace off.")))
    (qsim start)
    (qsim-display start)
    t))

(defun turn-furnace-on (heater-state)
  (let ((nstate (make-transition-result
		  heater-state
		  home-heater-thermo
		  (switch-values heater-state '((HFin (On std))
						(netHF ((0 inf) nil)))))))
    nstate))

(defun turn-furnace-off (heater-state)
  (let ((nstate (make-transition-result
		  heater-state
		  home-heater-thermo
		  (switch-values heater-state '((HFin (0 std))
						(netHF ((minf 0) nil)))))))
    nstate))

; Make an alist of values taken from the old state, changing the specified values,
; leaving the other independent variables alone, and preserving the values, but
; not directions of change, of the remaining variables.

(defun switch-values (old-state changed)
  (do ((L (state-values old-state) (cdr L))
       (independents (qde-independent (state-qde old-state)))
       (val nil)
       (new-values nil))
      ((null L) (nreverse new-values))
    (cond ((setq val (assoc (car (car L)) changed))
	   (setq new-values (cons val new-values)))
	  ((member (car (car L)) independents)
	   (setq new-values (cons (car L) new-values)))
	  (t (setq new-values
		   (cons (list (car (car L)) (list (car (cadr (car L))) nil))
			 new-values))))))

