;;;  -*- Mode:Common-Lisp; Package: QSIM; Syntax:COMMON-LISP; Base:10 -*-
;;;  Dan Dvorak, 1989.

(in-package 'QSIM)

;;;  Adds backjump (intelligent backtracking) to form-all-states.

(defother variable depth)
(defother variable disallowed-qval)

;;;-----------------------------------------------------------------------------
;;;  Function:  (form-all-states  qde)
;;;
;;;  Given:     the qde
;;;
;;;  Returns:   a list of consistent value assignments for each variable.
;;;
;;;  Purpose:   This function, which is run after Waltz filtering, takes the
;;;             surviving tuples of each constraint and forms all possible
;;;             consistent value completions.
;;;
;;;  Design:    The basic idea in this algorithm is that we pick some constraint,
;;;             bind its variables using the next tuple of the constraint, and
;;;             then move on to another unprocessed constraint that is connected
;;;             to one of the processed constraints.  Of the candidate constraints
;;;             on this "boundary", we always take the one with the fewest tuples.
;;;             By moving in a "connected order" from constraints having few
;;;             tuples to constraints having many tuples, value completions are
;;;             formed recursively with a minimum amount of backtracking.
;;;             Here are the basic steps:
;;;
;;;             1.  For every constraint having only one tuple, bind its
;;;                 variables "permanently" since they must be the same for all
;;;                 completions.
;;;             2.  From the list of constraints processed in step 1, form the
;;;                 initial boundary consisting of directly-connected constraints
;;;                 that were not processed in step 1.
;;;             3.  Find the constraint having the fewest tuples in the boundary
;;;                 and delete it from the boundary.  Watch out for two special
;;;                 cases here -- the boundary could be nil either because there
;;;                 were no 1-tuple constraints or because ALL constraints had
;;;                 only one tuple.  In the latter case we're all done.
;;;             4.  Call (recursive-form-states constraint boundary).  The
;;;                 remaining steps are inside this recursive function.
;;;             5.  Temporarily bind any unbound variables of this constraint.
;;;                 Add to the boundary the unprocessed constraints that are
;;;                 directly connected to these newly bound variables.
;;;             6.  If boundary is empty, then we're all done.  Otherwise, find
;;;                 the constraint having the fewest tuples in the boundary and
;;;                 remove it.  Call (recursive-form-states constraint boundary).
;;;
;;;  Note:      This function could have been written in a very simple recursive
;;;             way to bind the variables of the first constraint, then do the
;;;             second constraint, etc, backtracking whenever an inconsistency
;;;             occurred.  This would probably be fine for small constraint
;;;             networks, but the amount of backtracking grows exponentially
;;;             with the number of constraints.
;;;-----------------------------------------------------------------------------

(defparameter *all-states* nil)

(defun form-all-states (qde)
  (let* ((variables    (qde-variables qde))
	 (constraints  (qde-constraints qde))
	 (starting-constraints nil)
	 (num-remaining-constraints (length constraints)))
    (declare (special variables))

    (setq *all-states* nil)

    ;; Step 0 -- Prepare 'variable' structures (leaving "time" variable alone).
    ;; Any variable which has only one possible value can have its qval set
    ;; permanently here.  This is in fact necessary for discrete variables
    ;; since they don't necessarily participate in constraints.
    (dolist (var (cdr variables))
      (let ((pvals (variable--pvals var)))
	(setf (variable--qval var) nil
	      (variable-depth var) nil)
	(if (singleton-p pvals)
	    (setf (variable--qval var) (car pvals)
		  (variable-depth var) 0))))

    ;; Step 1 -- For all 1-tuple constraints, permanently bind the associated
    ;;           variables.
    (dolist (con constraints)
      (if (not (constraint-active-p con))
	  (progn
	    (setf (constraint-done con) t)	; want to skip inactive constraints
	    (decf num-remaining-constraints))
	  (progn
	    (setf (constraint-done con) nil)
	    (let ((tuples (constraint--tuples con)))
	      (when (singleton-p tuples)
		(setf (constraint-done con) t)
		(push con starting-constraints)
		(decf num-remaining-constraints)
		;; Bind variable values from this tuple.
		(mapc #'(lambda (var tval)
			  (let ((qval (variable--qval var)))
			    (if (null qval)
				(setf (variable--qval var) tval
				      (variable-depth var) 0)
				(if (not (qval-equal qval tval))
				    (error "Inconsistent values ~a and ~a for ~a."
					   qval tval (variable-name var))))))
		      (constraint-variables con)
		      (first tuples)))))))
	
    ;; If all constraints were processed above (rare, but possible),
    ;; then form the single state and return.
    (when (= 0 num-remaining-constraints)
      (recursive-form-states nil 1)
      (return-from form-all-states *all-states*))

    ;; Step 2 -- From the list of constraints processed in step 1, form the
    ;;           initial boundary consisting of directly-connected constraints
    ;;           that were not processed in step 1.
    (let ((ordered-constraints nil)
	  (boundary nil)
	  nextcon)

      ;; Build initial boundary.
      (dolist (con starting-constraints)
	(dolist (ncon (constraint-neighbors con))
	  (if (not (constraint-done ncon))
	      (pushnew ncon boundary))))

      ;; Loop over remaining constraints to build an ordered list of
      ;; constraints for recursive formation of states.
      (dotimes (n num-remaining-constraints)

	;; If a boundary (still) exists ...
	(if boundary

	    ;; then find the constraint in the boundary having the fewest tuples
	    ;; and update the boundary accordingly.
	    (multiple-value-setq (nextcon boundary) (find-min-constraint boundary))

	    ;; else either there was no initial boundary (because there was no
	    ;; 1-tuple constraint) or else the constraint network consists of
	    ;; two or more unconnected networks, and we must now jump over to
	    ;; another network.
	    (multiple-value-setq (nextcon boundary) (find-min-constraint-2 constraints)))

	(push nextcon ordered-constraints))

      ;; Go form the states.
      (let ((ordered-constraints (nreverse ordered-constraints))
	    (depth 0))
	(dolist (con ordered-constraints)
	  (incf depth)
	  (dolist (var (constraint-variables con))
	    (unless (variable-depth var)
	      (setf (variable-depth var) depth))))

	(catch 'no-more-completions
	  (recursive-form-states ordered-constraints 1)))

      ;; Return the states, if any, created in the recursive function.
      *all-states*)))




;;;-----------------------------------------------------------------------------
;;;  Function:  (recursive-form-states  remaining-constraints  depth)
;;
;;;  Given:     -- a list of constraints remaining to be checked in forming
;;;                a state;
;;;             -- the current recursion depth.
;;;
;;;  Returns:   two values:
;;;             -- an integer equal to the depth that should be backjumped to,
;;;                else its own depth if no backjumping needed;
;;;             -- the smallest depth that this call or any deeper call
;;;                depended on
;;;
;;;  Side Effects:  The last (deepest) recursive call forms a list of value
;;;                 completions and pushes that list onto *all-states*.
;;;
;;;  Design:    If no more constraints, push value completions onto *all-states*
;;;             and return.  Otherwise, take the first remaining constraint,
;;;             and check/bind its variables.  If there is a contradiction
;;;             such that one of its variables cannot be bound, then return.
;;;             Otherwise, do a recursive call with the tail of the list of
;;;             constraints.  
;;;
;;;             3/27/89: Intelligent backtracking ("backjumping") added.
;;;             If none of the tuples of a given constraint are compatible with
;;;             the current bindings, then backjump to the nearest constraint
;;;             (in terms of recursion depth) that has bound a value that
;;;             conflicts with the current constraint.
;
;for each tuple of the constraint
;   if tuple is compatible with all previous settings
;   then
;      if tuple matches a temporary disallowed pattern, then skip this tuple.
;      if tuple matches a chain of disallowed patterns (either starting at
;         this level or being tracked from a previous level)
;         then
;            if this is end of chain,
;               then skip this tuple,
;               else include this chain for next deeper r-f-s call
;
;   then if tuple does not match any temporary disallowed pattern from a
;              previous backjump) or any permanent disallowed pattern in the
;              branches being tracked
;        then call r-f-s and save return-depth, dependent-depth
;        if return-depth < my-depth
;        then return with return-depth and updated dependent depth
;        if return-depth = my-depth
;        then (0) I have been backjumped to;
;             (1) save my current tuple as a temporary disallowed pattern,
;                 with nil for any vals not depended upon at time of backjump;
;                 For each var that I set, if its depended-on-level is <= level
;                 of backjumper, then the var's current value is used in the
;                 pattern, else NIL is used.  If no such var found, error.
;             (2) remember that a backjump occurred.
;( else tuple is either incompatible or matches a disallowed pattern, so skip it)
;
;
;if no tuple of this constraint produced a completion
;   then 
;      (2) backjump to closest conflicting dependency.
;   else
;      (1) make each temporary disallowed pattern into a chain running from
;          smallest-dependent-depth to my depth, making sure to use as much
;          of a common branch that starts at s-d-d as possible.
;;;-----------------------------------------------------------------------------

(defparameter *num-completions* 0)

(defun recursive-form-states (remaining-constraints my-depth incoming-bad-branches)
  (declare (special variables *all-states*))
  (declare (optimize speed))

  ;; If no more constraints ...
  (when (null remaining-constraints)
    ;; then push the state values onto *all-states* and return.
    (push (mapcar #'(lambda (var)
		      (cons (variable-name var) (variable--qval var)))
		  variables)
	  *all-states*)
    (when (= 0 (mod *num-completions* 25))
      (multiple-value-bind (sec min hour)
	  (get-decoded-time)
	(format t "~%~4d completions at ~d:~d" *num-completions* hour min)))
    (incf *num-completions*)
    (return-from recursive-form-states my-depth))

  ;; Take the next constraint on the list.
  (let ((con (car remaining-constraints))
	(compatible-tuple-p nil)
	(backjump-occurred nil)
	(max-depth 0))

    ;; For each tuple of this constraint ...
    (dolist (tuple (constraint--tuples con))
      (block do-this-tuple

	;; For each qval of this tuple ...
	(dolist (tval tuple)
	  (let* ((var   (qval-variable tval))
		 (depth (variable-depth-when-bound var)))
	    ;; If we are at the depth where this variable should be set ...
	    (if (= depth my-depth)
		;; then set its value
		(setf (variable--qval var) tval)
		;; else it must have already been set at a lower depth,
		;; so we must check for compatible value.
		(unless (qval-equal tval (variable--qval var))
		  (if (> depth max-depth)
		      (setq max-depth depth))
		  (return-from do-this-tuple)))))

	;; The only way to get here is if the tuple is compatible with
	;; all prior settings.  Now, see if this tuple is covered by any
	;; known bad pattern.  First, we check the temporary bad patterns
	;; that have (just) been created while in this call.

	;; However, if an earlier backjump occurred
	;; to this depth, then we need to see if the tuple matches the
	;; disallowed pattern at the time of the backjump.  If so, then
	;; there skip this tuple since it will only result in another
	;; backjump.

	;; Check tuple against temporary/local bad patterns.
	(if (match-bad-pattern tuple bad-patterns)
	    (return-from skip-this-tuple))

	;; Check tuple against bad-pattern trees that start at this level.
	(if (setq node (match-bad-branch tuple (bad-patterns tree-node)))
	    (if (null (tree-next-node node))
		(return-from do-this-tuple)
		(push (tree-next-node node) follow-these-branches)))

	;; Check tuple against bad-pattern trees that we have been tracking.
	;; WHAT IF more than one node matches here?  Can it?
	(if (setq node (match-bad-branch tuple incoming-bad-nodes))
	    (if (null (tree-next-node node))
		(return-from do-this-tuple)
		(push (tree-next-node node) nodes-to-follow)))

	;; Tuple has survived all tests, so make recursive call.
	(multiple-value-bind (backjumper-level return-level dependent-level)
	    (recursive-form-states (cdr remaining-constraints)
				   (cdr level-info-list)
				   (1+ my-depth)
				   nodes-to-follow)

	  (cond
	    ;; If return-depth < my-depth, then a backjump is "passing through".
	    ((< return-level my-level)
	     (let ((min-level (level-min-dependent (car level-info-list))))
	       (if (< min-level dependent-level)
		   (setq dependent-level min-level))
	       (return-from recursive-form-states
		 (values backjumper-level return-level dependent-level))))

	    ;; If return-level = my-level, then this is target of backjump.
	    ((= return-level my-level)
	     ;; Save current tuple as a temporarily bad pattern.
	     (push (make-bad-pattern tuple backjumper-level level-info) bad-patterns)
	     (setq backjump-occurred t))

	    ;; Should never arrive here.
	    (t (error "Impossible levels: ~a, ~a" return-level my-level)))
	  )  ; end of multiple-value-bind
	)  ; end of block do-this-tuple
      )  ; end of dolist tuple

    ;; All tuples have been tried.  If none of them yielded a completion,
    ;; then backjump to my nearest dependent level.
    (if (= *num-completions* old-num-completions)
	;; No completions, so backjump.
	(return-from recursive-form-states
	  (values my-level (my-backjump-level me) (my-dependent-level me)))
	;; At least one completion.
	(push 

	(dolist (bad-pattern bad-patterns)
	  (when (do ((tuple-vals   tuple       (cdr tuple-vals))
		     (pattern-vals bad-pattern (cdr pattern-vals)))
		    ((endp tuple-vals) t)
		  (and (car tuple-vals)
		       (not (eq (car tuple-vals) (car pattern-vals)))
		       (return nil)))
	    ;; We've matched a bad pattern, so skip this tuple.
	    (return-from do-this-tuple)))

;	(when (and backjump-occurred
;		   (dolist (var (constraint-variables con) t)
;		     (unless (eq (variable--qval var) (variable-disallowed-qval var))
;		       (return nil))))
;	  (return-from do-this-tuple))

	;; Tuple is OK, so recur to next constraint.
	(setq compatible-tuple-p t)
	(let ((return-depth
		(recursive-form-states (cdr remaining-constraints) (1+ my-depth))))

	  ;; Check for backjump.  If return-depth <= my-depth, then a
	  ;; backjump occurred.  If return-depth < my-depth, then the
	  ;; backjump is just "passing through" this level, but if
	  ;; return-depth = my-depth, then it is backjumping to this
	  ;; level.  What this means for this instance of being at this
	  ;; level is that the values currently assigned at this level
	  ;; represent a pattern that is known to fail at a deeper level.
	  ;; Hence, for efficiency's sake, we should exclude any further
	  ;; tuples of this constraint that match the pattern.
	  (when (< return-depth my-depth)
	    (return-from recursive-form-states return-depth))
	  (when (= return-depth my-depth)
	    (setq backjump-occurred t)
	    (dolist (var (constraint-variables con))
	      (when (= my-depth (variable-depth var))
		(setf (variable-disallowed-qval var) (variable--qval var)))))
	  )
	)					; end of block do-this-tuple
      )						; end of dolist
	      
    (if compatible-tuple-p
	;; return normally, no backjump.
	(return-from recursive-form-states my-depth)
	;; No compatible tuple found, so backjump to nearest conflict.
	(progn
	  ;(format t "   Backjump ~d --> ~d" my-depth max-depth)
	  (return-from recursive-form-states max-depth)))
    
    ) ; end of let
  ) ; end of defun


	  ;; We've accepted this tuple of this constraint, so now we can
	  ;; proceed to the next constraint.
	  ;; IDEA!  At time of backjump, delete all tuples matching the
	  ;; disallowed pattern, but somehow preserve current position
	  ;; in the list of tuples.  NO!!!! Can't do this because backjump
	  ;; may have depended on value settings even less deep.



;;; Returns 2 values:  constraint having fewest tuples in boundary,
;;; and updated boundary.
(defun find-min-constraint (boundary)
  (let ((min-num-tuples most-positive-fixnum)
	(min-constraint nil)
	n)
    (dolist (con boundary)
      (when (< (setq n (length (constraint--tuples con))) min-num-tuples)
	(setq min-num-tuples n
	      min-constraint con)))
    (setq boundary (delete min-constraint boundary :count 1))
    (setf (constraint-done min-constraint) t)
    (dolist (con (constraint-neighbors min-constraint))
      (if (not (constraint-done con))
	  (pushnew con boundary)))
    (values min-constraint boundary)))


;;; Of the unprocessed constraints, return the one having the fewest tuples.
(defun find-min-constraint-2 (constraints)
  (let ((min-num-tuples most-positive-fixnum)
	(min-constraint nil)
	(boundary nil)
	n)
    (dolist (con constraints)
      (when (and (not (constraint-done con))
		 (< (setq n (length (constraint--tuples con))) min-num-tuples))
	(setq min-num-tuples n
	      min-constraint con)))
    (setf (constraint-done min-constraint) t)
    (dolist (con (constraint-neighbors min-constraint))
      (if (not (constraint-done con))
	  (pushnew con boundary)))
    (values min-constraint boundary)))


