;;; -*- Mode:Common-Lisp; Package:QSIM; Syntax:COMMON-LISP; Base:10 -*-

(in-package 'QSIM)

;;; Copyright 1987 David Franke
;;;
;;; -*- Mode: LISP; Syntax: Common-lisp; Package: QSIM; Base: 10 -*-

;;;
;;; D,#TD1PsT[Begin using 006 escapes](1 0 (NIL 0) (:SWISS :ITALIC NIL) "HL12I")*Component-Definitions* - The list of currently defined model component
(2 0 (NIL 0) (NIL NIL NIL) "CPTFONT");;; 1definitions.  New definitions are added via the macro Add-Component-Definition.
2;;; 

(defvar (3 0 (NIL 0) (NIL :BOLD NIL) "CPTFONTCB")*COMPONENT-DEFINITIONS* 2nil "List of model component definitions")

;;;
;;; 1*Domain-Specific-Variable-Types* - An alist containing mappings from the standard
2;;; 1symbols for effort, flow, resistance, and capacitance to the domain specific symbols.
2;;; 1The associated default variable names are also specified here.  The format of an item
2;;; 1in this list is (<standard symbol> <domain specific symbol> <default variable name>)
2;;; 1For example, in the electrical domain, *Domain-Specific-Variable-Types* contains
2;;; 1((effort voltage "V") (flow current "I") (resistance resistance "R") (capacitance
2;;; 1capacitance "C") (displacement charge "Q"))
2;;;

(defvar 3*DOMAIN-SPECIFIC-VARIABLE-TYPES*
2     (list (list 'effort 'effort "e")
	   (list 'flow 'flow "f")
	   (list 'resistance 'resistance "r")
	   (list 'capacitance 'capacitance "c")
	   (list 'displacement 'displacement "q"))
  "A-List of standard symbols to domain specific symbols.")

;;;
;;; 1The Add-Component-Definition macro places a component definition in the a-list
2;;; 1stored in the global variable *Component-Definitions*.  If a definition exists, it
2;;; 1is replaced by the new definition.
2;;;

(defmacro 3DEFINE-COMPONENT 2(component-name &rest component-specs)
  `(let ((current-definition-cons (assoc ',component-name *component-definitions* :test #'eq)))
     (if current-definition-cons
	 (setf (cdr current-definition-cons) ',component-specs)
	 (push (cons ',component-name ',component-specs) *component-definitions*))))



;;;
;;; 1Data Structure Declarations
2;;;

;;;
;;; 1Component Variable - A component variable structure is created for each variable defined
2;;; 1for a particular component.  For example, for a Battery, component variables are
2;;; 1generated for the voltage of each terminal (of which there are two), and the voltage
2;;; 1measured across the two terminals.
2;;;

(defstruct 3(COMPONENT-VARIABLE (:PRINT-FUNCTION PRINT-COMPONENT-VARIABLE))
2  (component-name nil)				; (4 0 (NIL 0) (:SWISS :ROMAN :SMALLER) "HL10")Name of the corresponding component
2  (name           nil)				; 4Name of the component variable
2  (type           nil)				; 4Type of the component variable
2  (independent    nil)				; 4Component variable independent?
2  (display        nil)				; 4Display corresponding model variable?
2  (quantity-space nil)				; 4Quantity space of the component variable
2  (initable       nil)				; 4Can initial values be assigned?
2  (model-variable nil))				; 4Corresponding model variable


2(defun 3PRINT-COMPONENT-VARIABLE 2(cv stream ignore)
  (format stream "<Component-Variable ~S of ~S, type ~S, independent ~S, initable ~S mv ~S>"
	  (component-variable-name cv) (component-variable-component-name cv)
	  (component-variable-type cv) (component-variable-independent cv)
	  (component-variable-initable cv)
	  (when (component-variable-model-variable cv)
	    (model-variable-name (component-variable-model-variable cv)))))


;;;
;;; 1Model Variable - A model variable is created for each unique variable in the model.  One
2;;; 1model variable may correspond to one or more component variables.  For example, on
2;;; 1model variable is created to represent the voltage at the point where a battery and a
2;;; 1resistor are connected.  This model variable would correspond to a component variable
2;;; 1of the battery and a component variable of the resistor.
2;;; 

(defstruct 3(MODEL-VARIABLE (:PRINT-FUNCTION PRINT-MODEL-VARIABLE))
2  (name           nil)				; 4Name of the model variable
2  (type           nil)				; 4Type of the model variable
2  (independent    nil)				; 4Model variable independent?
2  (display        nil)				; 4Display this model variable?
2  (quantity-space nil)				; 4Quantity space of the model variable
2  (initable       nil)				; 4Can initial values be assigned?
2  (component-variables nil))			; 4List of associated component variables


2(defun 3PRINT-MODEL-VARIABLE 2(mv stream ignore)
  (format stream "<Model-Variable ~S, type ~S, independent ~S, from ~S>"
	  (model-variable-name mv) (model-variable-type mv) (model-variable-independent mv)
	  (mapcar #'(lambda (cv) (list (component-variable-component-name cv) (component-variable-name cv)))
		  (model-variable-component-variables mv))))


;;;
;;; 1Trace variables
2;;; 
(defvar 3*TRACE-CONNECTION-PROCESSING* 2nil)

(defvar 3*TRACE-KCL-CONSTRAINT-APPLICATION* 2nil)

;;;
;;; 1Global variables manipulated by the Define-Model macro
2;;; 
(defvar 3*MODEL-VARIABLES* 2nil "List of model variables generated for the current QDE.")

(defvar 3*CONSTRAINTS* 2nil "List of constraints generated for the current QDE.")



(defun 3GENERATE-MODEL-VARIABLE-NAME 2(model-variable)
  "Generate a unique model variable name given the model variable type, e.g. Vn for type voltage."
  (let ((independent-variable-name
	  (do ((cv (model-variable-component-variables model-variable) (cdr cv)))
	      ((or (null cv)
		   (component-variable-independent (car cv)))
	       (when cv (concatenate 'string (string (component-variable-component-name (car cv)))
				     "." (string (component-variable-name (car cv)))))))))
    (setf (model-variable-name model-variable)
	  (if independent-variable-name
	      (intern independent-variable-name 'qsim)
	      (let* ((mv-type (model-variable-type model-variable))
		     (ch (third (find mv-type *domain-specific-variable-types*
				      :test #'eq :key #'first))))
		(do* ((i 1)
		      (new-model-variable-name (intern (concatenate 'string ch (format nil "~D" i)))
					       (intern (concatenate 'string ch (format nil "~D" i)))))
		     ((dolist (mv *model-variables* t)
			(when (eq new-model-variable-name (model-variable-name mv)) (return nil)))
		      new-model-variable-name)
		  (incf i)))))))




(defun 3DIGEST-CONSTRAINT-SPEC 2(component-type constraint-spec component-variable-alist)
  "Substitute symbolic variable references with the corresponding component variable structure instances."
  (mapl #'(lambda (variable-reference-list)
	    (let* ((variable-reference (car variable-reference-list))
		   (component-variable-structure-instance
		     (cdr (assoc variable-reference component-variable-alist :test #'eq))))
;	      (when *model-macro-trace-flag*
;		(format t "~&  Attempting to perform structure substitution for variable reference ~S"
;			variable-reference))
	      (if component-variable-structure-instance
		  (setf (car variable-reference-list) component-variable-structure-instance)
		  (format t "~&~% Unknown variable name ~S in constraint ~S of component-type ~S"
			  variable-reference constraint-spec component-type))))
	(cdar constraint-spec))
  constraint-spec)


(defun 3DIGEST-VARIABLE-SPEC 2(component-type component-instance-name variable-spec other-data)
  "Create a component variable instance from a variable spec."
  (let* ((new-component-variable (make-component-variable :component-name component-instance-name
							  :name (car variable-spec)))
	 (landmark-data (cadr (assoc (car variable-spec) (cdr (assoc 'landmarks other-data :test #'eq))
				     :test #'eq)))
	 (initable-variables (cdr (assoc 'initable other-data :test #'eq))))
    (dolist (variable-facet (cdr variable-spec))
      (cond ((listp variable-facet)
	     (case (car variable-facet)
	       (type
		 (let ((type-map (find (second variable-facet) *domain-specific-variable-types*
					    :test #'eq :key #'second)))
		   (if type-map
		       (setf (component-variable-type new-component-variable) (first type-map))
		       (format t "~&~% Unknown variable type ~S specified for variable ~S of component type ~S"
			       (second variable-facet) (car variable-spec) component-type))))
	       (quantity-space (setf (component-variable-quantity-space new-component-variable)
				     (cadr variable-facet)))
	       (t (format t "~&~% Unknown variable facet ~S specified for variable ~S of component type ~S"
			  (car variable-facet) (car variable-spec) component-type))))
	    ((symbolp variable-facet)
	     (case variable-facet
	       (independent (setf (component-variable-independent new-component-variable) t))
	       (dependent)
	       (display (setf (component-variable-display new-component-variable) t))
	       (t (format t "~&~% Unknown variable facet ~S specified for variable ~S of component type ~S"
			  variable-facet  (car variable-spec) component-type))))
	    (t (format t "~&~% Unknown variable facet ~S specified for variable ~S of component type ~S"
		       variable-facet (car variable-spec) component-type))))
    (when landmark-data
      (setf (component-variable-quantity-space new-component-variable)
	    (if (listp landmark-data)
		landmark-data
		(list '0 landmark-data 'inf))))
    (setf (component-variable-initable new-component-variable)
	  (member (car variable-spec) initable-variables :test #'eq))
    new-component-variable))

(defun 3INTERPRET-COMPONENT-REFERENCE 2(component-instance-name component-type other-data)
  "Given a component declaration (instance name, type, and landmark info), create the component-variables and constraints associated with the specified component type."
  (let ((component-definition (cdr (assoc component-type *component-definitions* :test #'eq))))
    (if (null component-definition)
	(progn (format t "~&~% Unknown component type ~S, declaration ignored." component-type)
	       (values nil nil nil nil))
	(let ((component-variable-alist nil)
	      (terminal-variable-alist))
	  (dolist (component-facet component-definition)
	    (case (car component-facet)
	      
	      (terminal-variables
		(dolist (terminal-spec (cdr component-facet))
		  (push (cons (car terminal-spec)
			      (mapcar #'(lambda (variable-spec)
					  (let ((new-component-variable
						  (digest-variable-spec component-type
									component-instance-name
									variable-spec other-data)))
					    (push (cons (car variable-spec) new-component-variable)
						  component-variable-alist)
					    new-component-variable))
				      (cdr terminal-spec)))
			terminal-variable-alist)))
	      
	      (component-variables
		(dolist (component-variable-spec (cdr component-facet))
		  (let ((new-component-variable
			  (digest-variable-spec component-type component-instance-name
						component-variable-spec other-data)))
		    (push (cons (car component-variable-spec) new-component-variable)
			  component-variable-alist)
		    (push (make-model-variable
			    :type (component-variable-type new-component-variable)
			    :independent (component-variable-independent new-component-variable)
			    :display (component-variable-display new-component-variable)
			    :quantity-space (component-variable-quantity-space new-component-variable)
			    :initable (component-variable-initable new-component-variable)
			    :component-variables (list new-component-variable))
			  *model-variables*)
		    (generate-model-variable-name (car *model-variables*))
		    (setf (component-variable-model-variable new-component-variable)
			  (car *model-variables*)))))
	      
	      (constraints
		(dolist (constraint-spec (cdr component-facet))
		  (push (digest-constraint-spec component-type (copy-tree constraint-spec)
						component-variable-alist)
			*constraints*)))
	      
	      (t (format t "~&~% Unknown component facet ~S specified for component type ~S"
			 component-facet component-type))))
	  (values (mapcar #'cdr component-variable-alist)
		  (cons component-instance-name terminal-variable-alist))
	  ))))

(defun 3SET-INITABLE-FOR-MODEL-VARIABLE 2(model-variable)
  "Given a model variable, determine if any of the corresponding component variables were initable.  If so, make the model variable initable."
  (setf (model-variable-initable model-variable)
	(dolist (cv (model-variable-component-variables model-variable) nil)
	  (when (component-variable-initable cv) (return t)))))


(defun 3GENERATE-COMPONENT-AND-VARIABLE-NAME-STRING 2(component-variable)
  (concatenate 'string (string (component-variable-component-name component-variable))
	       "." (string (component-variable-name component-variable))))


(defun 3GENERATE-MODEL-VARIABLE-TO-COMPONENT-VARIABLE-MAP-STRING 2(model-variable)
  "Given a model variable, generate the string describing the model to component variable map."
  (concatenate 'string (string (model-variable-name model-variable)) "("
	       (subseq (apply 'concatenate 'string
			      (mapcar #'(lambda (cv)
					  (concatenate 'string " "
						       (generate-component-and-variable-name-string cv)))
				      (model-variable-component-variables model-variable)))
		       1)
	       ")"))



(defun 3GENERATE-CONNECTION-MODEL-VARIABLE-FOR-EFFORT 2(effort-component-variables)
  "Given a list of component variables, create the model variable which represents effort at their connection."
  (let ((new-effort-model-variable (make-model-variable :type 'effort)))

    (push new-effort-model-variable *model-variables*)
    (setf (model-variable-component-variables new-effort-model-variable) effort-component-variables)
    (generate-model-variable-name new-effort-model-variable)
    (set-initable-for-model-variable new-effort-model-variable)
    (when *trace-connection-processing*
      (format t "~& Model variable ~A generated for effort component variables:"
	      (model-variable-name new-effort-model-variable)))
    
    (dolist (ccv effort-component-variables)
      (when *trace-connection-processing*
	(format t " ~A" (generate-component-and-variable-name-string ccv)))
      (setf (component-variable-model-variable ccv) new-effort-model-variable)
      (when (component-variable-independent ccv)
	(setf (model-variable-independent new-effort-model-variable) t)
	(setf (model-variable-quantity-space new-effort-model-variable)
	      (component-variable-quantity-space ccv)))
      (when (component-variable-display ccv)
	(setf (model-variable-display new-effort-model-variable) t)))

    new-effort-model-variable))

(defun 3MAKE-TEMP-VARIABLE-FOR-KCL-CONSTRAINT 2()
  "Create temporary model variable for representing conservation constraints, or KCL."
  (let ((new-model-variable (make-model-variable :type 'flow)))
    (generate-model-variable-name new-model-variable)
    (push new-model-variable *model-variables*)
    (when *trace-KCL-constraint-application*
      (format t "~&  Creating temporary variable ~A for KCL constraint" (model-variable-name new-model-variable)))
    (model-variable-name new-model-variable)))

(defun 3IN-QUANTITY-SPACE-P 2(value q-space)
  ;; 4Determine if a specified landmark or interval is in the given quantity space.  The quantity space is
2  ;; 4representedas a  list, with items specified in increasing (algebraic) order.
2  ;; 4For now, a simple membership test is performed, which assumes that the value is a landmark, and not an
2  ;; 4interval.  Quantity space of Nil is assumed to be (MINF 0 INF).
2  (or (null q-space)
      (member value q-space :test #'eq)))

(defun 3CORRESPONDENCE-IN-QUANTITY-SPACES-P 2(value-1 q-space-1 value-2 q-space-2)
  ;; 4Determine whether a correspondence should be included in a constraint by examining the landmarks of
2  ;; 4the correspondence and the quantity spaces of interest.  For example, the correspondences
2  ;; 4(INF MINF), (0 0) and (MINF INF) can be checked for the MINUS constraint.
2  (and (in-quantity-space-p value-1 q-space-1)
       (in-quantity-space-p value-2 q-space-2)))

(defun 3CORRESPONDENCES-FOR-FLOW-MINUS-CONSTRAINT 2(model-var-1 model-var-2)
  (let ((correspondences nil))
    (when (correspondence-in-quantity-spaces-p
		   'inf (model-variable-quantity-space model-var-1)
		   'minf (model-variable-quantity-space model-var-2))
      (push (list 'inf 'minf) correspondences))
    (when (correspondence-in-quantity-spaces-p
		   0 (model-variable-quantity-space model-var-1)
		   0 (model-variable-quantity-space model-var-2))
      (push (list 0 0) correspondences))
    (when (correspondence-in-quantity-spaces-p
		   'minf (model-variable-quantity-space model-var-1)
		   'inf (model-variable-quantity-space model-var-2))
      (push (list 'minf 'inf) correspondences))
    correspondences))

(defun 3GENERATE-CONNECTION-FLOW-MODEL-VARIABLES-AND-CONSTRAINTS 2(flow-component-variables)
  "Given a list of component variables associated with flow, generate the associated model variables and constraints."
  (dolist (ccv flow-component-variables)
    (setf (component-variable-model-variable ccv)
	  (make-model-variable :type 'flow :independent (component-variable-independent ccv)
			       :quantity-space (component-variable-quantity-space ccv)
			       :component-variables (list ccv)
			       :display (component-variable-display ccv)))
    (generate-model-variable-name (component-variable-model-variable ccv))
    (set-initable-for-model-variable (component-variable-model-variable ccv))
    (push (component-variable-model-variable ccv) *model-variables*))

  (when (or *trace-KCL-constraint-application* *trace-connection-processing*)
    (format t "~& Applying KCL constraint to variables:")
    (dolist (cv flow-component-variables)
      (format t " ~A" (generate-model-variable-to-component-variable-map-string
			(component-variable-model-variable cv)))))
  
  (cond ((= (length flow-component-variables) 2)
	 (push (nconc (list (list 'MINUS
				  (model-variable-name (component-variable-model-variable
							 (first flow-component-variables)))
				  (model-variable-name (component-variable-model-variable
							 (second flow-component-variables)))))
		      (correspondences-for-flow-minus-constraint
			(component-variable-model-variable (first flow-component-variables))
			(component-variable-model-variable (second flow-component-variables))))
	       *constraints*)
	 (when *trace-KCL-constraint-application*
	   (format t "~&  Constraint generated from KCL: ~S" (car *constraints*))))
	((> (length flow-component-variables) 2)
	 (let ((last-temp (make-temp-variable-for-KCL-constraint)))
	   (push (list (list 'ADD
			     (model-variable-name (component-variable-model-variable
						    (first flow-component-variables)))
			     (model-variable-name (component-variable-model-variable
						    (second flow-component-variables)))
			     last-temp)
		       (list 0 0 0))
		 *constraints*)
	   (when *trace-KCL-constraint-application*
	     (format t "~&  Constraint generated from KCL: ~S" (car *constraints*)))
	   (do ((ccv-list (cddr flow-component-variables) (cdr ccv-list))
		(next-temp nil))
	       ((= (length ccv-list) 1)
		(push (nconc (list (list 'MINUS
					 (model-variable-name
					   (component-variable-model-variable (car ccv-list)))
					 last-temp))
			     (correspondences-for-flow-minus-constraint
			       (component-variable-model-variable (car ccv-list))
			       (find last-temp *model-variables*
				     :test #'(lambda (x y) (eq x (model-variable-name y))))))
		      *constraints*)
		(when *trace-KCL-constraint-application*
		  (format t "~&  Constraint generated from KCL: ~S" (car *constraints*))))
	     (push (list (list 'ADD
			       (model-variable-name (component-variable-model-variable (car ccv-list)))
			       last-temp
			       (setq next-temp (make-temp-variable-for-KCL-constraint)))
			 (list 0 0 0))
		   *constraints*)
	     (when *trace-KCL-constraint-application*
	       (format t "~&  Constraint generated from KCL: ~S" (car *constraints*)))
	     (setq last-temp next-temp))))))


(defun 3INTERPRET-COMPONENT-CONNECTION 2(terminals component-terminal-alist)
  "Interpret a connection specification, creating the required model variables."
  (when *trace-connection-processing*  (format t "~&Interpreting connection between ~S" terminals))
  (let ((connected-component-variables nil))
    (dolist (terminal-spec terminals)
      (let* ((component-name (car terminal-spec))
	     (component-terminals-cons (assoc component-name component-terminal-alist :test #'eq))
	     (terminal-name (cadr terminal-spec))
	     (terminal-variables-cons (when component-terminals-cons
					(assoc terminal-name (cdr component-terminals-cons) :test #'eq))))
	(if terminal-variables-cons
	    (setq connected-component-variables (append (cdr terminal-variables-cons)
						    connected-component-variables))
	    (format t "Node spec ~S unknown, and excluded from the connection." terminal-spec))))
					
    ;; 4Identify and create Model-Variables.
2    ;; 4Model variables created here are for Effort and Flow.  Efforts are made equivalent by
2    ;; 4associating them with the same model variable.  Flow variables each have a model variable
2    ;; 4generated, and the conservation constraint is applied to the model variables.
2    (when connected-component-variables
      (let ((effort-component-variables nil)
	    (flow-component-variables nil))
	
	(dolist (ccv connected-component-variables)
	  (case (component-variable-type ccv)
	    (effort (push ccv effort-component-variables))
	    (flow (push ccv flow-component-variables))))

	(generate-connection-model-variable-for-effort effort-component-variables)
	
	(generate-connection-flow-model-variables-and-constraints flow-component-variables)))))

(defun 3DEFINE-MODEL-COLLECT-QUANTITY-SPACE-DECLARATIONS 2()
  (mapcar #'(lambda (mv) (list (model-variable-name mv)
			       (if (model-variable-quantity-space mv)
				   (model-variable-quantity-space mv)
				   (list 'minf 0 'inf))))
	  *model-variables*))


(defun 3DEFINE-MODEL-COLLECT-INDEPENDENT-VARIABLES 2()
  (let ((independent-variables nil))
    (dolist (mv *model-variables*) (when (model-variable-independent mv)
				     (push (model-variable-name mv) independent-variables)))
    independent-variables))


(defun 3DEFINE-MODEL-COLLECT-DEPENDENT-VARIABLES 2()
  (let ((dependent-variables nil))
    (dolist (mv *model-variables*) (unless (model-variable-independent mv)
				     (push (model-variable-name mv) dependent-variables)))
    dependent-variables))


(defun 3DEFINE-MODEL-COLLECT-CONSTRAINTS 2()
  (dolist (constraint *constraints*)
    (mapl #'(lambda (cv-list)
	      (when (component-variable-p (car cv-list))
		(setf (car cv-list)
		      (model-variable-name (component-variable-model-variable (car cv-list))))))
	  (cdr (first constraint))))
  *constraints*)


(defun 3DEFINE-MODEL-COLLECT-PRINT-NAMES 2()
  (let ((print-info-list nil))
    (dolist (mv *model-variables*)
      (when (model-variable-display mv)
	(push (list (model-variable-name mv)
		    (generate-model-variable-to-component-variable-map-string mv)
		    (intern (concatenate 'string (string (model-variable-name mv)) "-")))
	      print-info-list)))
    print-info-list))


(defun 3DEFINE-MODEL-COLLECT-LAYOUT-DESCRIPTION 2()
  (let ((print-list nil))
    (dolist (mv *model-variables*) (when (model-variable-display mv)
				     (push (model-variable-name mv) print-list)))
    (do ((temp print-list (nthcdr 3 temp))
	 (layout-rows nil))
	((null temp) (nreverse layout-rows))
      (if (> (length temp) 3)
	  (push (list (first temp) (second temp) (third temp)) layout-rows)
	  (push temp layout-rows)))))


(defmacro 3DEFINE-MODEL 2(model-name descriptive-text component-declarations &rest connections)
  ;; 4Model-Name - Symbol whose value will be the Q model
2  ;; 4Descriptive-Text - Model description (string) for pretty output
2  ;; 4Component-Declarations - List of the form (Components <component-declaration> ...)
2  ;; 4Connections - Lists of the form (Connect <node spec> ...)

2  (let ((*model-variables* nil)
	(*constraints* nil)
	(component-variables nil)
	(component-terminal-alist nil))
    (declare (special *model-variables* *constraints*))
    (dolist (component-spec (cdr component-declarations))
      (multiple-value-bind (cv ct)
	  (interpret-component-reference
	    (first component-spec) (second component-spec) (nthcdr 2 component-spec))
	(setq component-variables (nconc cv component-variables))
	(push ct component-terminal-alist)))
    
    ;; 4Identify and build model variables.
2    (dolist (connection-spec connections)
      (interpret-component-connection (cdr connection-spec) component-terminal-alist))

    (setq *model-variables* (nreverse *model-variables*))
    (setq *constraints* (nreverse *constraints*))
    ;; 4Build the define-QDE form
2    `(progn
       (format t "~&~% Initable variable mappings:")
       (mapc #'(lambda (initable-variable-description)
		 (format t "~&   ~A" initable-variable-description))
	     ',(let ((initable-model-variables nil))
		 (dolist (mv *model-variables*)
		   (when (model-variable-initable mv)
		     (push (generate-model-variable-to-component-variable-map-string mv)
			   initable-model-variables)))
		 initable-model-variables))
       
       (define-QDE ,model-name (text ,descriptive-text)
	 
	 ,(cons 'quantity-spaces (define-model-collect-quantity-space-declarations))
	 
	 ,(cons 'independent (define-model-collect-independent-variables))
	 
	 ,(cons 'dependent (define-model-collect-dependent-variables))
	 
	 ,(cons 'constraints (define-model-collect-constraints))
	 
	 ,(cons 'print-names (define-model-collect-print-names))
	 
	 ,(cons 'layout (define-model-collect-layout-description))
	 ))
    ))
