;;; -*- Mode:Common-Lisp; Package:QSIM; Syntax:COMMON-LISP; Base:10 -*-


;;;-----------------------------------------------------------------------------
;;;  Model:    WATER-HEATER
;;;
;;;  Intent:   This is a model of a thermostatically (on/off) controlled water
;;;            heater.  It has two heating elements controlled by a single
;;;            thermostatic switch.  Each heating element may be good or bad.
;;;            The power source may be on or off.  The thermostatic switch may
;;;            be operating normally or be stuck open or stuck closed.
;;;
;;;            This model demonstrates the use of "ignore-qvals" to ignore
;;;            irrelevant distinctions.  The initialization function
;;;            bad-h1h2-hot-hiflow generates 1 behavior with 15 states.
;;;            Without ignore-qvals, it creates 23 behaviors with 200 states.
;;;
;;;  Notes on Landmarks:
;;;    WaterFlow:
;;;      WF2 = highest water flow where TempWater does not go below Lo when
;;;            both heating elements are working.
;;;      WF1 = highest water flow where TempWater does not go below Lo when
;;;            only one heating element is working.
;;;
;;;    HFoutW:
;;;      W1  = heat loss from water flow when TempWater Lo, Waterflow Lo.
;;;      W2  = heat loss from water flow when TempWater Hi, Waterflow Lo.
;;;      W3  = heat loss from water flow when TempWater Lo, Waterflow Hi.
;;;      W4  = heat loss from water flow when TempWater Hi, Waterflow Hi.
;;;
;;;    HFout:
;;;      H1  = total heat loss when HFoutW  0, HFoutI Lo.
;;;      H2  = total heat loss when HFoutW  0, HFoutI Hi.
;;;      H3  = total heat loss when HFoutW W1, HFoutI Lo.
;;;      H4  = total heat loss when HFoutW W2, HFoutI Hi.
;;;      H5  = total heat loss when HFoutW W3, HFoutI Lo.
;;;      H6  = total heat loss when HFoutW W4, HFoutI Hi.
;;;-----------------------------------------------------------------------------


(define-QDE WATER-HEATER
  (text "Water heater with thermostatic control")
  (quantity-spaces
    (Heat       (0 Room Lo Hi inf))
    (TempWater  (0 Room Lo Hi inf))
    (TempRoom   (0 Room inf))
    (TempDiff   (0 Lo Hi inf))
    (HFoutI     (0 Lo Hi inf))			; Heat loss via insulation
    (HFoutW     (0 W1 W2 W3 W4 inf))		; Heat loss via water flow
    (HFout      (0 H1 H2 H3 H4 H5 H6 inf))	; Heat loss total.
    (netHF      (minf 0 inf))
    (H-power    (0 On))				; Heater power
    (S-power    (0 On))				; Source power
    (HFin1      (0 On))
    (HFin2      (0 On))
    (HFin       (0 On1 On2))
    (WaterFlow  (0 WF1 WF2 inf)))			; Water flow
  (discrete-variables
    (switch  (open closed stuck-open stuck-closed))
    (H1      (good bad))
    (H2      (good bad)))
  (constraints
    (mode (or (switch closed) (switch stuck-closed))
	  ((M+ S-power H-power) (0 0) (On On)))
    (mode (or (switch open) (switch stuck-open))
	  ((zero-std H-power)))
    (mode (or (H1 good))
	  ((M+ H-power HFin1) (0 0) (On On)))
    (mode (or (H1 bad))
	  ((zero-std HFin1)))
    (mode (or (H2 good))
	  ((M+ H-power HFin2) (0 0) (On On)))
    (mode (or (H2 bad))
	  ((zero-std HFin2)))
    ((ADD  HFin1 HFin2 HFin)            (0 0 0) (On 0 On1) (0 On On1) (On On On2))
    ((M+   Heat TempWater)              (0 0) (Room Room) (Lo Lo) (Hi Hi) (inf inf))
    ((ADD  TempDiff TempRoom TempWater) (0 Room Room) (0 inf inf) (Lo Room Lo) (Hi Room Hi))
    ((M+   TempDiff HFoutI)             (0 0) (Lo Lo) (Hi Hi) (inf inf))
    ((MULT WaterFlow TempDiff HFoutW)   (WF1 Lo W1) (WF1 Hi W2) (WF2 Lo W3) (WF2 Hi W4))
    ((ADD  HFoutI HFoutW HFout)         (Lo 0 H1) (Hi 0 H2) (Lo W1 H3) (Hi W2 H4) (Lo W3 H5) (Hi W4 H6))
    ((ADD  netHF HFout HFin)            (0 H3 On1) (0 H5 On2))
    ((D/DT Heat netHF)))
  (transitions
     ((and (switch (open   std)) (TempWater (Lo dec)))   turn-switch-on)
     ((and (switch (closed std)) (TempWater (Hi inc)))   turn-switch-off))
  (independent S-power WaterFlow TempRoom)
  (history TempWater)
  (layout (TempWater  WaterFlow    H-power)
	  (TempDiff   HFoutW       HFin1)
	  (nil        HFoutI       HFin2)
	  (netHF      HFout        HFin))
  (other
    (ignore-qvals HFoutW HFoutI HFout netHF)
    )
  )

;;; Normal cyclic on/off operation with no water flow, and heat loss
;;; only through the insulation.
(defun normal-hot-noflow ()
  (let ((start (make-initial-state water-heater
				   '(;; ----------------------- Discrete Variables.
				     (switch     (open nil))
				     (H1         (good nil))
				     (H2         (good nil))
				     ;; ----------------------- Independent Variables.
				     (S-power    (On std))
				     (WaterFlow  (0 std))
				     (TempRoom   (Room std))
				     ;; ----------------------- History Variables.
				     (TempWater  (Hi nil))
				     )
				   "Normal-hot-noflow")))
    (qsim start)
    (qsim-display start)
    t))

;;; Temp drops to low, heater on, temp std at low.
(defun normal-hot-hiflow ()
  (let ((start (make-initial-state water-heater
				   '(;; ----------------------- Discrete Variables.
				     (switch     (open nil))
				     (H1         (good nil))
				     (H2         (good nil))
				     ;; ----------------------- Independent Variables.
				     (S-power    (On std))
				     (WaterFlow  (WF2 std))
				     (TempRoom   (Room std))
				     ;; ----------------------- History Variables.
				     (TempWater  (Hi nil))
				     )
				   "Normal-hot-hiflow")))
    (qsim start)
    (qsim-display start)
    t))

;;; H1 bad, Temp drops to low, heater on, temp std at low.
(defun bad-H1-hot-loflow ()
  (let ((start (make-initial-state water-heater
				   '(;; ----------------------- Discrete Variables.
				     (switch     (open nil))
				     (H1         (bad  nil))
				     (H2         (good nil))
				     ;; ----------------------- Independent Variables.
				     (S-power    (On std))
				     (WaterFlow  (WF1 std))
				     (TempRoom   (Room std))
				     ;; ----------------------- History Variables.
				     (TempWater  (Hi nil))
				     )
				   "H1 bad, Normal-hot-loflow")))
    (qsim start)
    (qsim-display start)
    t))

;;; H1 & H2 both bad, Temp drops to low, heater switch turns on but no heat produced,
;;; so water temperature drops to room temperature.
(defun bad-H1H2-hot-hiflow ()
  (let ((*state-limit* 210)
	(start (make-initial-state water-heater
				   '(;; ----------------------- Discrete Variables.
				     (switch     (open nil))
				     (H1         (bad  nil))
				     (H2         (bad nil))
				     ;; ----------------------- Independent Variables.
				     (S-power    (On std))
				     (WaterFlow  (WF2 std))
				     (TempRoom   (Room std))
				     ;; ----------------------- History Variables.
				     (TempWater  (Hi nil))
				     )
				   "H1 & H2 bad, hot, hiflow")))
    (qsim start)
    (qsim-display start)
    t))


(defun turn-switch-on (water-heater-state)
  (create-transition-state :from-state   water-heater-state
			   :to-qde       water-heater
			   :assert       '((switch  (closed std)))
			   :inherit-qmag '(S-power WaterFlow TempRoom TempWater)
			   :inherit-qdir nil))


(defun turn-switch-off (water-heater-state)
  (create-transition-state :from-state   water-heater-state
			   :to-qde       water-heater
			   :assert       '((switch  (open std)))
			   :inherit-qmag '(S-power WaterFlow TempRoom TempWater)
			   :inherit-qdir nil))


