;;; -*- Package: QSIM; Syntax: Common-lisp -*-
(in-package :qsim)

; This is an attempt to build a sequence of linked mechanisms to explain the
; pop-up toaster example from Richard J. Doyle's JACK program.

; Start.

(defun toaster-scenario ()
  (let ((initial-state
	  (make-new-state :from-qde friction-dominated-spring
			  :assert-values '((lever-pressure  ((minf 0) std))
					   (net-pressure    ((minf 0) nil))
					   (spring-position (top nil)))
			  :text "Pushing down toaster")))
    (declare (special *state-limit* toaster-heat-flow-model))

    (format *QSIM-Report* 
"~2%  This is the QSIM model corresponding to the Toaster example developed
in Richard Doyle's JACK program [MIT-AI PhD 1988].  Eventually, it should be possible
to make a version of JACK that compiles automatically into QSIM.  The model consists
of four QDEs linked by operating region transitions:
  (1)  Pushing the carrying rack and toast down into the toaster, against spring pressure.
  (2)  Heat source heats up the interior of the toaster, including the bread.
       Bread approaches the burn point, latch approaches the release point.
  (3)  The carrying rack and toast accellerate upward, once the latch releases.
  (4)  After the rack stops, the toast continues upward, perhaps flying out of the toaster.
~2%")

    (qsim initial-state)
    (qsim-display initial-state :layout full-layout)
    ))

(defparameter full-layout '((lever-pressure T-air T-source T)
			    (spring-position dTA dTS T-bread bread-position)
			    (spring-pressure outflow inflow latch bread-velocity)
			    (net-pressure spring-velocity netflow gravity)))

(defun push-down-toaster ()
  (let ((*state-limit* 100)			; global QSIM control
	(toaster-heat-flow-model Double-Heat-Flow-toaster))
    (declare (special *state-limit* toaster-heat-flow-model))
    (toaster-scenario)
    ))

; This is the same initialization as before, except setting up the model to 
; use the BURN landmark in T-BREAD.  This increases branching, of course.

(defun push-down-toaster-w-burn ()
  (let ((*state-limit* 100)			; global QSIM control
	(toaster-heat-flow-model Double-Heat-Flow-w-Burn))
    (declare (special *state-limit* toaster-heat-flow-model))
    (toaster-scenario)
    ))

; (1) Friction-dominated spring:  press the carriage down until it clicks.

(define-QDE Friction-dominated-spring
  (text "Pushing the toaster down")
  (quantity-spaces
    (spring-position   (minf click top 0))
    (spring-pressure   (0 inf))
    (lever-pressure    (minf 0))
    (net-pressure      (minf 0 inf)))
  (constraints
    ((M- spring-position spring-pressure)    (0 0) (minf inf))
    ((ADD spring-pressure lever-pressure net-pressure))
    ((d/dt spring-position net-pressure))
    ((constant lever-pressure)))
  (transitions
    ((spring-position (click dec)) -> toaster-on))
  (layout (nil spring-position nil)
	  (lever-pressure spring-pressure nil)
	  (nil net-pressure))
)

; Turn on the toaster.

(defun toaster-on (ostate)
  (declare (special toaster-heat-flow-model))
  (create-transition-state :from-state ostate		; previous state
			   :to-qde toaster-heat-flow-model	; new mechanism
			   :assert '((T-air    (airtemp std))	; its initial state.
				     (T-source (sourcetemp std))
				     (T        (airtemp nil)))))


; (2) Heat flow from toaster into toast and ratchet release mechanism.
;     (based on Double Heat-Flow Example from AI Journal paper, 1984.)

(define-QDE Double-Heat-Flow-toaster
  (text "Toaster with heat from element heating bread and ratchet.")
  (quantity-spaces
   (T-air    (minf airtemp sourcetemp inf))
   (T        (minf airtemp sourcetemp inf))
   (T-source (minf airtemp sourcetemp inf))
   (dTA      (minf 0 inf))
   (dTS      (minf 0 inf))
   (outflow  (minf 0 inf))
   (inflow   (minf 0 inf))
   (netflow  (minf 0 inf))
   (T-bread  (minf airtemp inf))		; deleted BURN from (airtemp inf)
   (latch    (0 normal release inf)))
  (constraints
   ((add T-air dTa T)    (airtemp 0 airtemp))
   ((add T dTS T-source) (sourcetemp 0 sourcetemp))
   ((M+ dTA outflow)     (0 0) (inf inf) (minf minf))
   ((M+ dTS inflow)      (0 0) (inf inf) (minf minf))
   ((add outflow netflow inflow))
   ((d/dt T netflow))
   ((M+ T T-bread)       (airtemp airtemp) (inf inf) (minf minf))
   ((M+ T latch)         (airtemp normal) (inf inf))
   ((constant T-air))
   ((constant T-source)))
  (transitions
;    ((T-bread (burn inc)) -> t)
;    ((T-bread (burn std)) -> t)
    ((latch (release inc)) -> release-spring))
  (layout
    (T-air T-source T nil)
    (dTA dTS T-bread)
    (outflow inflow latch)
    (nil netflow nil))
  )

; transition -> spring

(defun release-spring (ostate)
  (create-transition-state :from-state ostate
			   :to-qde Simple-Spring-in-toaster
			   :assert '((spring-position (click nil))
				     (spring-velocity (0 nil))
				     (lever-pressure  (0 std)))))

; This will start the process with toasting, without the push-down step.

(defun Toast ()
  (let ((init (make-new-state :from-qde Double-Heat-Flow-toaster
			      :assert-values '((T-air  (airtemp std))
					       (T-source (sourcetemp std))
					       (T      (airtemp nil)))
			      :text "Container starts at air temperature.")))
    (qsim init)
    (qsim-display init)))


; This is the same heat-flow model as before, except with a landmark
; representing the temperature at which the toast burns.

(define-QDE Double-Heat-Flow-w-Burn
  (text "Toaster with heat from element heating bread and ratchet.")
  (quantity-spaces
   (T-air    (minf airtemp sourcetemp inf))
   (T        (minf airtemp sourcetemp inf))
   (T-source (minf airtemp sourcetemp inf))
   (dTA      (minf 0 inf))
   (dTS      (minf 0 inf))
   (outflow  (minf 0 inf))
   (inflow   (minf 0 inf))
   (netflow  (minf 0 inf))
   (T-bread  (minf airtemp burn inf))		; BURN landmark in (airtemp inf)
   (latch    (0 normal release inf)))
  (constraints
   ((add T-air dTa T)    (airtemp 0 airtemp))
   ((add T dTS T-source) (sourcetemp 0 sourcetemp))
   ((M+ dTA outflow)     (0 0) (inf inf) (minf minf))
   ((M+ dTS inflow)      (0 0) (inf inf) (minf minf))
   ((add outflow netflow inflow))
   ((d/dt T netflow))
   ((M+ T T-bread)       (airtemp airtemp) (inf inf) (minf minf))
   ((M+ T latch)  (airtemp normal) (inf inf))
   ((constant T-air))
   ((constant T-source)))
  (transitions
    ((T-bread (burn inc)) -> t)
    ((T-bread (burn std)) -> t)
    ((latch (release inc)) -> release-spring))
  (layout
    (T-air T-source T nil)
    (dTA dTS T-bread)
    (outflow inflow latch)
    (nil netflow nil))
  )




; (3)  Spring (not friction-dominated) popping the toast back up.

(define-QDE Simple-Spring-in-toaster
   (quantity-spaces
	      (spring-position   (minf click top 0))
	      (spring-velocity   (minf 0 inf))
	      (spring-pressure   (0 inf))
	      (lever-pressure    (minf 0))
	      (net-pressure      (minf 0 inf)))
   (constraints
     ((M- spring-position spring-pressure)    (0 0) (minf inf))
     ((ADD spring-pressure lever-pressure net-pressure))
     ((d/dt spring-position spring-velocity))
     ((d/dt spring-velocity net-pressure))
     ((constant lever-pressure)))
   (transitions
     ((spring-position (top inc)) -> bread-flies-upward))
   (text "Simple spring model")
   (layout  (spring-position spring-velocity nil)
	    (lever-pressure spring-pressure )
	    (net-pressure))
   )


(defun bread-flies-upward (ostate)
  (create-transition-state :from-state ostate
			   :to-qde Gravity-on-bread
			   :assert '((bread-position (top inc))
				     (gravity (g std)))))

(defun push-simple-spring ()
  (let ((initial
	  (make-new-state :from-qde simple-spring-in-toaster
			  :assert-values '((spring-position (click nil))
					   (spring-velocity (0 nil))
					   (lever-pressure  (0 std)))
			  :text "Start with initial velocity")))
    (qsim initial)
    (qsim-display initial)
    ))


; (4) Gravity model as the bread is thrown upward

(define-QDE Gravity-on-bread
  (text  "Throw a ball upward in constant gravity.")
  (quantity-spaces
    (bread-position   (minf click top 0 inf))
    (bread-velocity   (minf 0 inf))
    (gravity          (minf g 0)))
  (constraints
    ((d/dt bread-velocity gravity))
    ((d/dt bread-position bread-velocity))
    ((constant gravity)))
  (transitions
    ((bread-position (top dec)) -> t)
    ((bread-position (0 dec)) -> t))
  (layout (nil bread-position nil)
	  (nil bread-velocity nil)
	  (nil gravity nil))
  )



(defun throw-upward ()
  (let ((init
	  (make-new-state :from-qde Gravity-on-bread
			  :assert-values '((bread-position (top inc))
					   (gravity (g std)))
			  :text "Throw an object upward")))
    (qsim init)
    (qsim-display init)
    ))