;;; -*- Mode:Common-Lisp; Package:QSIM; Syntax:COMMON-LISP; Base:10 -*-
(in-package :qsim)

; The simple, obvious spring model gives more predictions than are 
; actually possible [Kuipers, 1985, 1986].

(define-QDE Simple-Spring
   (text "Simple spring model (predicts possible increasing and decreasing oscillation)")
   (quantity-spaces
	      (A      (minf 0 inf)      "Accelleration")
	      (V      (minf 0 inf)   "Velocity")
	      (X      (minf 0 inf)      "Position"))
   (constraints
	      ((d/dt X V))
	      ((d/dt V A))
	      ((M- A X)         (0 0) (minf inf) (inf minf)))
   (layout  (nil x nil)
	    (nil v nil)
	    (nil a nil)))


(defun push-simple-spring ()
  (let* ((sim (make-sim :state-limit 20))
	 (initial
	   (make-new-state :from-qde simple-spring
			   :assert-values '((x  (0 nil))
					    (v  ((0 inf) nil)))
			   :text "Start with initial velocity"
			   :sim sim)))
    (format *QSIM-Report*
"~%
This is the simplest model of the frictionless spring:  x'' = -M+(x).
It also provides the simplest example of spurious behaviors from 
qualitative simulation, since only the cyclic behavior is genuine,
and both the increasing and decreasing oscillations are spurious.
The spurious behaviors survive because any implicit knowledge of the
energy of the system is lost during the very coarse qualitative
descriptions of intervals.

If you use the Energy Constraint, only steady oscillations are predicted
(see examples in q-features -> energy constraint~2%")
    (qsim initial)
    (qsim-display initial)))
    

; Simple spring (1) without and (2) with HOD constraints.

(defun release-spring ()
  (let* ((sim (make-sim :HOD-constraints nil))
	 (initial
	   (make-new-state :from-qde simple-spring
			   :assert-values '((x ((0 inf) std))
					    (v (0 nil)))
			   :sim sim)))
    (qsim initial)
    (qsim-display initial)))

(defun release-spring-acc ()
  (let* ((sim (make-sim :HOD-constraints t))
	 (initial
	   (make-new-state :from-qde simple-spring
			   :assert-values '((x ((0 inf) std))
					    (v (0 nil)))
			   :sim sim)))
    (qsim initial)
    (qsim-display initial)))

; Adding energy conservation to the spring gives a single behavior,
; by making explicit the constraint that the simple spring could not
; take advantage of.

(define-QDE Spring-with-energy-conservation
   (quantity-spaces
	      (TE     (minf 0 E* inf)   "Total Energy")
	      (A      (minf 0 inf)      "Accelleration")
	      (V      (minf 0 V* inf)   "Velocity")
	      (X      (minf 0 inf)      "Position")
	      (KE     (minf 0 inf)      "Kinetic Energy")
	      (PE     (minf 0 inf)      "Potential Energy")
	      (VSQ    (minf 0 inf))
	      (XSQ    (minf 0 inf)))
   (constraints
	      ((d/dt X V))
	      ((d/dt V A))
	      ((M- A X)         (0 0) (minf inf) (inf minf))
	      ((mult V V VSQ))
	      ((mult X X XSQ))
	      ((M+ VSQ KE)      (0 0) (inf inf))
	      ((M+ XSQ PE)      (0 0) (inf inf))
              ((add KE PE TE))
	      ((constant TE)))
   (text "Spring with energy conservation")
   (layout  (x xsq pe)
	    (v vsq ke)
	    (a nil te)
	    (nil))
   )


(defun push-spring-w-energy-conservation ()
  (let* ((sim (make-sim :cycle-detection :strong))
	 (initial
	   (make-new-state :from-qde spring-with-energy-conservation
			   :assert-values '((te (E* std))
					    (x  (0 nil))
					    (v  (V* nil)))
			   :text "Push from the origin"
			   :sim sim)))
    (format *QSIM-Report*
"~2% 
If we add explicit constraints for kinetic, potential, and total energy
to the simple spring model, and assert that total energy is constant,
then it is possible to determine that the perfectly cyclic behavior is
the only possible one

Another possibility is to use the Energy Constraint.
(See examples in q-features -> energy constraint) ~2%")

    (qsim initial)
    (qsim-display initial)
    ))

; Spring with friction.
; Recent work by Fouche & Kuipers [UT-TR-90-134] made this system tractable.
; (See examples in q-features -> energy constraint).

(define-QDE spring-with-friction
  (text "Spring with friction")
  (quantity-spaces
    (x     (minf 0 inf))
    (v     (minf 0 inf))
    (a     (minf 0 inf))
    (ff    (minf 0 inf))
    (fs    (minf 0 inf))
    (f     (minf 0 inf)))
  (constraints
    ((d/dt x v))
    ((d/dt v a))
    ((m- x fs)      (0 0)     (minf inf)  (inf minf))
    ((m- v ff)      (0 0)     (minf inf)  (inf minf))
    ((add fs ff f))
    ((m+ f a)       (0 0)     (inf inf)   (minf minf)))
  (layout  (x fs nil)
	   (v ff nil)
	   (a f nil)
	   (nil nil nil)))

(defun push-spring-w-friction ()
  (let* ((sim (make-sim :state-limit 40))
	 (initial
	   (make-new-state :from-qde spring-with-friction
			   :assert-values '((x ((0 inf) nil))
					    (v (0 nil)))
			   :sim sim)))
    (format *QSIM-Report*
"~2% 
The spring with friction causes problems for the basic limit-analysis
methods of qualitative simulation.

Recent work by Fouche & Kuipers [UT-TR-90-134] made this system tractable.
(See examples in q-features -> energy constraint) ~2%")
    (qsim initial)
    (qsim-display initial)))


; These demonstrate the spring-with-friction (1) without and (2) with HOD constraints.

(defun release-spring-w-friction ()
  (let* ((sim (make-sim :HOD-constraints nil))
	 (initial
	   (make-new-state :from-qde spring-with-friction
			   :assert-values '((x ((0 inf) std))
					    (v (0 nil)))
			   :sim sim)))
    (qsim initial)
    (qsim-display initial)))

(defun release-spring-w-friction-acc ()
  (setf (qde-derived-sd2-expressions spring-with-friction) nil)
  (let* ((sim (make-sim :HOD-constraints t))
	 (initial
	   (make-new-state :from-qde spring-with-friction
			   :assert-values '((x ((0 inf) std))
					    (v (0 nil)))
			   :sim sim)))
    (qsim initial)
    (qsim-display initial)))




; Spring with friction, plus non-increasing energy.
;  =>  This demonstrates the NON-INCREASING constraint!

(define-QDE spring-with-friction+energy-loss
  (text "Spring with friction and energy loss")
  (quantity-spaces
    (x     (minf 0 x* inf))
    (v     (minf 0 v* inf))
    (a     (minf a* 0 inf))
    (ff    (minf ff* 0 inf))
    (fs    (minf fs* 0 inf))
    (KE     (minf 0 inf))
    (PE     (minf 0 inf))
    (TE     (0 inf))
    (VSQ    (minf 0 inf))
    (XSQ    (minf 0 inf)))
  (constraints
    ((d/dt x v))
    ((d/dt v a))
    ((m- x fs)      (0 0)     (minf inf)  (inf minf))
    ((m- v ff)      (0 0)     (minf inf)  (inf minf))
    ((add fs ff a))
    ((mult V V VSQ))
    ((mult X X XSQ))
    ((M+ VSQ KE)      (0 0) (inf inf))
    ((M+ XSQ PE)      (0 0) (inf inf))
    ((add KE PE TE))
    ((non-increasing TE)))
  (layout  (x fs pe)
	   (v ff ke)
	   (a f  te))
)


(defun push-spring-w-friction+energy ()
  (let ((initial
	  (make-new-state :from-qde spring-with-friction+energy-loss
			  :assert-values '((x  (x* nil))
					   (v  (0 nil))
					   (a  (a* nil))
					   (fs (fs* nil))
					   (f  (f* nil))
					   (te ((0 inf) nil))))))
    (qsim initial)
    (qsim-display initial)
    ))



