;;; -*- Syntax: Common-lisp; Package: qsim -*-
;;;     Copyright (c) 1987, Benjamin Kuipers.

(in-package 'QSIM)

(defun describe-region-trans ()
  (format *qsim-report*
"~2%The concept of `operating region transition' defines the limits of the
reasonable range of a parameter.  A given model is only valid within certain
limits and we may or may not have a model of what happens when those limits
are crossed.  Operating region limits can be established either explicitly
by defining transition functions associated with endpoints of the range
of a parameter (landmark values), or implicitly by making these endpoints
the endpoints of the quantity space for the parameter.  The Toaster and the
Home-Heater examples illustrates two slightly different ways of realizing
`operating region transitions' in QSIM.

`Operating region transition' in QSIM is accomplished basically through a
TRANSITIONS clause in the QDE.  The clause specifies `transition conditions'
which dictates when transition `occurs' (when certain parameters reach certain
values) and associated `transition functions' which create state(s) in the
`new region' from which to continue simulation.  See the examples for more
details."))

;;;-----------------------------------------------------------------------------
;;;  Model:    HOME-HEATER-MODES
;;;
;;;  Intent:   This is a model of a thermostatically (on/off) controlled home-
;;;            heating furnace.  This model demonstrates four things:
;;;
;;;            1.  Discrete-variables:
;;;                In contrast to the continuous variables declared in the
;;;                quantity-spaces clause, discrete-variables:
;;;                -- can only have landmark values (never interval values),
;;;                -- have qdir always 'std (like independent variables),
;;;                -- have an unordered qspace.
;;;
;;;            2.  Modes:
;;;                A group of constraints can be made active or inactive
;;;                depending on the values of specified variables.  An active
;;;                constraint is processed normally; an inactive constraint is
;;;                effectively invisible during simulation.  The condition in
;;;                a mode clause is usually expressed in terms of a discrete
;;;                variable (but this is not a requirement).
;;;
;;;            3.  Transition conditions:
;;;                The simplest transition condition is a test of a variable's
;;;                value to a given landmark and qdir.  It is now valid to
;;;                combine such conditions using AND, OR, and NOT, nested
;;;                to any depth, if needed.
;;;
;;;            4.  Transition functions:
;;;                The new function |create-transition-state|, with its keyword
;;;                arguments, simplifies the job of specifying what should
;;;                change and what should remain the same at a region transition.
;;;-----------------------------------------------------------------------------

(defun turn-switch-on (heater-state)
  (create-transition-state :from-state   heater-state
			   :to-qde       home-heater-modes
			   :assert       '((switch (closed std))
					   (HFin   (nil std))
					   (netHF  ((0 inf) nil)))
			   :inherit-qmag :rest
			   :inherit-qdir nil))


(defun turn-switch-off (heater-state)
  (create-transition-state :from-state   heater-state
			   :to-qde       home-heater-modes
			   :assert       '((switch (open std))
					   (HFin   (nil std))
					   (netHF  ((minf 0) nil)))
			   :inherit-qmag :rest
			   :inherit-qdir nil))



(define-QDE HOME-HEATER-MODES
  (text "Thermostatic control of home heater")
  (quantity-spaces
    (heat    (0 inf))
    (mass    (0 inf))
    (TempIn  (0 LoTemp RoomTemp HiTemp inf))
    (TempOut (0 Cold RoomTemp Hot inf))
    (TempSet (0 RoomTemp inf))
    (dTemp   (minf 0 inf))
    (error   (minf Lo 0 Hi inf))
    (R       (0 inf))
    (HFout   (minf 0 inf))
    (netHF   (minf 0 inf))
    (Power   (0 On))
    (HFin    (0 On)))
  (discrete-variables
    (switch  (open closed stuck-open stuck-closed)))
  (constraints
    (mode ((switch closed) (switch stuck-closed))
	  ((M+ HFin Power) (0 0) (On On)))
    (mode ((switch open) (switch stuck-open))
	  ((zero-std HFin)))
    ((mult TempIn Mass Heat))
    ((add TempOut dTemp TempIn)  (RoomTemp 0 RoomTemp) )
    ((add TempSet error TempIn)  (RoomTemp 0 RoomTemp) (RoomTemp Lo LoTemp) (RoomTemp Hi HiTemp))
    ((mult R HFout dTemp))
    ((add HFout netHF HFin))
    ((d/dt Heat netHF)))
   (transitions
     ((and (switch (open   std)) (error (Lo dec)))   turn-switch-on)
     ((and (switch (closed std)) (error (Hi inc)))   turn-switch-off))
  (independent HFin Power Mass TempOut TempSet R)
  (history TempIn)
  (layout (Mass Heat nil)
	  (TempOut TempIn TempSet)
	  (dTemp error R)
	  (HFout HFin netHF))
  (print-names
    (heat    "Heat content"   H)
    (mass    "Thermal mass"   M)
    (TempIn  "Temp(inside)"   Ti)
    (TempOut "Temp(outside)"  To)
    (TempSet "Temp(set)"      Ts)
    (dTemp   "dTemp(in,out)"  dT)
    (error   "error=in-set"  E)
    (R       "Heat flow resistance")
    (HFout   "Heat flow (to environment)" HFo)
    (HFin    "Heat flow (from heater)" HFi)
    (netHF   "net Heat Flow"  nHF)))


(defun home-heater ()
  (let ((start (make-initial-state Home-Heater-Modes
				   '((Heat ((0 inf) nil))
				     (TempIn (RoomTemp nil))
				     (Mass ((0 inf) std)) (TempOut (Cold std))
				     (TempSet (RoomTemp std)) (R ((0 inf) std))
				     (Power (On std)) (Switch (open std)))
				   "Room temperature; furnace off.")))
    (format *qsim-report*
"~2%This example uses `mode' clauses in the constraint set to give rise to different
models upon region transition.  Only one QDE is used for the whole mechanism.  Note
the hand-shaking in the conditions of the `transitions' clause and the conditions
in the `mode' clauses.  This is in contrast to using different QDE's for different
models altogether as in the TOASTER example.  `Discrete variables' (versus the
normally continuous variables used in QSIM) are used in this example.  See
`examples;region-transition' for more details.")
    (qsim start)
    (qsim-display start)
    t))



;;; = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = =

;;; = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = = =

; This is an attempt to build a sequence of linked mechanisms to explain the
; pop-up toaster example from Richard J. Doyle's JACK program.

; Start.

(defun toaster-scenario ()
  (let ((initial-state
	  (make-initial-state friction-dominated-spring
			      '((lever-pressure  ((minf 0) std))
				(net-pressure    ((minf 0) nil))
				(spring-position (top nil)))
			      "Pushing down toaster")))
    (declare (special *state-limit* toaster-heat-flow-model))
    (format *QSIM-Report* 
"~2%  This is the QSIM model corresponding to the Toaster example developed
in Richard Doyle's JACK program [MIT-AI PhD 1988].  Eventually, it should be possible
to make a version of JACK that compiles automatically into QSIM.  The model consists
of four QDEs linked by operating region transitions:
  (1)  Pushing the carrying rack and toast down into the toaster, against spring pressure.
  (2)  Heat source heats up the interior of the toaster, including the bread.
       Bread approaches the burn point, latch approaches the release point.
  (3)  The carrying rack and toast accellerate upward, once the latch releases.
  (4)  After the rack stops, the toast continues upward, perhaps flying out of the toaster.
See `examples;region-transition' for details of this example.
~2%")
    (qsim initial-state)
    (qsim-display initial-state :layout full-layout)
    ))

(defparameter full-layout '((lever-pressure T-air T-source T)
			    (spring-position dTA dTS T-bread bread-position)
			    (spring-pressure outflow inflow latch bread-velocity)
			    (net-pressure spring-velocity netflow gravity)))

; This is the same initialization as before, except setting up the model to 
; use the BURN landmark in T-BREAD.  This increases branching, of course.

(defun toaster ()
  (let ((*state-limit* 100)			; global QSIM control
	(toaster-heat-flow-model Double-Heat-Flow-w-Burn))
    (declare (special *state-limit* toaster-heat-flow-model))
    (toaster-scenario)
    ))

; (1) Friction-dominated spring:  press the carriage down until it clicks.

(define-QDE Friction-dominated-spring
  (text "Pushing the toaster down")
  (independent lever-pressure)
  (history     spring-position)
  (quantity-spaces
    (spring-position   (minf click top 0))
    (spring-pressure   (0 inf))
    (lever-pressure    (minf 0))
    (net-pressure      (minf 0 inf)))
  (constraints
    ((M- spring-position spring-pressure)    (0 0) (minf inf))
    ((ADD spring-pressure lever-pressure net-pressure))
    ((d/dt spring-position net-pressure)))
  (transitions
    ((spring-position (click dec)) toaster-on))
  (print-names (lever-pressure  "lever pressure"   LP)
	       (spring-position "spring position"  POS)
	       (spring-pressure "spring pressure"  PR)
	       (net-pressure    "net pressure"     NET))
  (layout (nil spring-position nil)
	  (lever-pressure spring-pressure nil)
	  (nil net-pressure))
)

; Turn on the toaster.

(defun toaster-on (ostate)
  (declare (special toaster-heat-flow-model))
  (make-transition-result ostate		; previous state
			  toaster-heat-flow-model	; new mechanism
			  '((T-air    (airtemp std))	; its initial state.
			    (T-source (sourcetemp std))
			    (T        (airtemp nil)))))


; This is the same heat-flow model as before, except with a landmark
; representing the temperature at which the toast burns.

(define-QDE Double-Heat-Flow-w-Burn
  (text "Toaster with heat from element heating bread and ratchet.")
  (quantity-spaces
   (T-air    (minf airtemp sourcetemp inf))
   (T        (minf airtemp sourcetemp inf))
   (T-source (minf airtemp sourcetemp inf))
   (dTA      (minf 0 inf))
   (dTS      (minf 0 inf))
   (outflow  (minf 0 inf))
   (inflow   (minf 0 inf))
   (netflow  (minf 0 inf))
   (T-bread  (minf airtemp burn inf))		; BURN landmark in (airtemp inf)
   (latch    (0 normal release inf)))
  (constraints
   ((add T-air dTa T)    (airtemp 0 airtemp))
   ((add T dTS T-source) (sourcetemp 0 sourcetemp))
   ((M+ dTA outflow)     (0 0) (inf inf) (minf minf))
   ((M+ dTS inflow)      (0 0) (inf inf) (minf minf))
   ((add outflow netflow inflow))
   ((d/dt T netflow))
   ((M+ T T-bread)       (airtemp airtemp) (inf inf) (minf minf))
   ((M+ T latch)  (airtemp normal) (inf inf)))
  (independent T-air T-source)
  (history T)
  (transitions
    ((T-bread (burn inc)) t)
    ((T-bread (burn std)) t)
    ((latch (release inc)) release-spring))
  (layout
    (T-air T-source T nil)
    (dTA dTS T-bread)
    (outflow inflow latch)
    (nil netflow nil))
  )

; transition -> spring

(defun release-spring (ostate)
  (make-transition-result ostate
			  Simple-Spring-in-toaster
			  '((spring-position (click nil))
			    (spring-velocity (0 nil))
			    (lever-pressure  (0 std)))))




; (3)  Spring (not friction-dominated) popping the toast back up.

(define-QDE Simple-Spring-in-toaster
   (quantity-spaces
	      (spring-position   (minf click top 0))
	      (spring-velocity   (minf 0 inf))
	      (spring-pressure   (0 inf))
	      (lever-pressure    (minf 0))
	      (net-pressure      (minf 0 inf)))
   (constraints
     ((M- spring-position spring-pressure)    (0 0) (minf inf))
     ((ADD spring-pressure lever-pressure net-pressure))
     ((d/dt spring-position spring-velocity))
     ((d/dt spring-velocity net-pressure)))
   (independent lever-pressure)
   (history spring-position spring-velocity)
   (transitions
     ((spring-position (top inc)) bread-flies-upward))
   (text "Simple spring model")
  (print-names (lever-pressure  "lever pressure"   LP)
	       (spring-position "spring position"  POS)
	       (spring-velocity "spring velocity"  V)
	       (spring-pressure "spring pressure"  PR)
	       (net-pressure    "net pressure"     NET))
   (layout  (spring-position spring-velocity nil)
	    (lever-pressure spring-pressure )
	    (net-pressure))
   )


(defun bread-flies-upward (ostate)
  (make-transition-result ostate
			  Gravity-on-bread
			  '((bread-position (top inc))
			    (gravity (g std)))))

; (4) Gravity model as the bread is thrown upward

(define-QDE Gravity-on-bread
  (text  "Throw a ball upward in constant gravity.")
  (quantity-spaces
    (bread-position   (minf click top 0 inf))
    (bread-velocity   (minf 0 inf))
    (gravity          (minf g 0)))
  (constraints
    ((d/dt bread-velocity gravity))
    ((d/dt bread-position bread-velocity)))
  (independent gravity)
  (transitions
    ((bread-position (top dec)) t)
    ((bread-position (0 dec)) t))
  (layout (nil bread-position nil)
	  (nil bread-velocity nil)
	  (nil gravity nil))
  )