;;; -*- Syntax: Common-lisp; Package: qsim -*-
(in-package :qsim)

;
; Copyright (c) 1987, Benjamin Kuipers.
;

; This builds the water balance system for Q.  The water-balance system, 
; and its behavior in SIADH, are described in detail in MIT LCS TM-280,
; and discussed in the Time-Scale Abstraction paper.

(define-QDE WATER-BALANCE
  (text "Water balance via ADH")
  (quantity-spaces
    (afp   (0 afp* inf))
    (anp   (0 ANP%% ANP* ANP% inf))
    (nfwip (0 NFWIP%% NFWIP* NFWIP% inf))
    (cnh   (0 cnh* inf))
    (ffwpu (minf 0 ffwpu* inf))
    (cnp   (0 cnp* inf))
    (cadh  (0 cadh* inf))
    (rfup  (0 rfup* inf))
    (nfpu  (0 nfpu* inf))
    (nfwop (minf 0 inf))
    (cnu   (0 cnu* inf))
    (nfnpu (0 nfnpu* inf)))
  (constraints
    ((M+ AFP CNH)            (afp* cnh*)    (0 0)  (inf inf))
    ((M+ CNH FFWPU)          (cnh* ffwpu*)  (0 0)  (inf inf))
    ((mult AFP CNP ANP)      (afp* cnp* anp*))
    ((add RFUP NFPU FFWPU)   (rfup* nfpu* ffwpu*))
    ((M+ CNP CADH)           (cnp* cadh*)   (0 0)  (inf inf))
    ((M+ CADH RFUP)          (cadh* rfup*)  (0 0)  (inf inf))
    ((d/dt AFP NFWOP))
    ((add NFPU NFWOP NFWIP)  (nfpu* 0 nfwip*))
    ((mult NFPU CNU NFNPU)   (nfpu* cnu* nfnpu*)))
  (independent nfwip anp nfnpu)
  (history afp)
  (layout
    (AFP CNP ANP)
    (CNH CADH NFNPU)
    (FFWPU RFUP CNU)
    (NFWIP NFPU NFWOP))
  (print-names
    (AFP    "amt(water,P)")
    (CNH    "c(natriuretic-hormones,P)")
    (FFWPU  "flow(water,P->U)")
    (ANP    "amt(Na,P)")
    (CNP    "c(Na,P)")
    (CADH   "c(ADH,P)")
    (RFUP   "reabs.flow(water,U->P)")
    (NFPU   "net flow(water,P->U)")
    (NFWIP  "net flow(water,in->P)")
    (NFWOP  "net flow(water,out->P)")
    (CNU    "c(Na,U)")
    (NFNPU  "net flow(Na,P->U)"))
  )

(defun high-water-intake ()
  (let* ((normal (make-initial-state water-balance
				     '((afp (afp* std))
				       (nfwip (nfwip* std))
				       (anp (anp* std))
				       (nfnpu (nfnpu* std)))))
	 (init (make-modified-state normal
				    '((afp (afp* nil))
				      (nfwip (nfwip% std))
				      (anp (anp* std))
				      (nfnpu (nfnpu* std)))
				    "Increased water intake")))
    (qsim init)
    (qsim-display init :reference-states `((normal ,normal)))
    ))

(defun low-intake ()
  (let* ((normal (make-initial-state water-balance
				     '((afp (afp* std))
				       (nfwip (nfwip* std))
				       (anp (anp* std))
				       (nfnpu (nfnpu* std)))))
	 (init (make-modified-state normal
				    '((afp (afp* nil))
				      (nfwip (nfwip%% std))
				      (anp (anp* std))
				      (nfnpu (nfnpu* std)))
				    "Decreased water intake")))
    (qsim init)
    (qsim-display init :reference-states `((normal ,normal)))
    ))


(defun high-volume ()
  (let* ((normal (make-initial-state water-balance
				     '((afp (afp* std))
				       (nfwip (nfwip* std))
				       (anp (anp* std))
				       (nfnpu (nfnpu* std)))))
	 (init (make-modified-state normal
				    '((afp ((afp* inf) nil))
				      (nfwip (nfwip* std))
				      (anp (anp* std))
				      (nfnpu (nfnpu* std)))
				    "Suddenly high water VOLUME, normal intake, normal sodium")))
    (qsim init)
    (qsim-display init :reference-states `((normal ,normal)))
    ))

; SIADH deletes the constraint between CNP and CADH, and makes CADH
; permanently high.

(define-QDE SIADH-WATER-BALANCE
  (text "Broken water balance mechanism with SIADH")
  (quantity-spaces
    (afp   (0 afp* inf))
    (anp   (0 ANP%% ANP* ANP% inf))
    (nfwip (0 NFWIP* inf))
    (cnh   (0 cnh* inf))
    (ffwpu (minf 0 ffwpu* inf))
    (cnp   (0 cnp* inf))
    (cadh  (0 cadh* CADH% inf))
    (rfup  (0 rfup* inf))
    (nfpu  (0 nfpu* inf))
    (nfwop (minf 0 inf))
    (cnu   (0 cnu* inf))
    (nfnpu (0 nfnpu* inf)))
  (constraints
    ((M+ AFP CNH)            (afp* cnh*)    (0 0)  (inf inf))
    ((M+ CNH FFWPU)          (cnh* ffwpu*)  (0 0)  (inf inf))
    ((mult AFP CNP ANP)      (afp* cnp* anp*))
    ((add RFUP NFPU FFWPU)   (rfup* nfpu* ffwpu*))
  ;  ((M+ CNP CADH)           (cnp* cadh*)   (0 0)  (inf inf))
    ((M+ CADH RFUP)          (cadh* rfup*)  (0 0)  (inf inf))
    ((d/dt AFP NFWOP))
    ((add NFPU NFWOP NFWIP)  (nfpu* 0 nfwip*))
    ((mult NFPU CNU NFNPU)   (nfpu* cnu* nfnpu*)))
  (independent nfwip anp nfnpu cadh)
  (history afp)
  (layout
    (AFP CNP ANP)
    (CNH CADH NFNPU)
    (FFWPU RFUP CNU)
    (NFWIP NFPU NFWOP))
  (print-names
    (AFP    "amt(water,P)")
    (CNH    "c(natriuretic-hormones,P)")
    (FFWPU  "flow(water,P->U)")
    (ANP    "amt(Na,P)")
    (CNP    "c(Na,P)")
    (CADH   "c(ADH,P)")
    (RFUP   "reabs.flow(water,U->P)")
    (NFPU   "net flow(water,P->U)")
    (NFWIP  "net flow(water,in->P)")
    (NFWOP  "net flow(water,out->P)")
    (CNU    "c(Na,U)")
    (NFNPU  "net flow(Na,P->U)"))
  (other					; to test abstracted constraint
    (abstracted-to ((m+ nfwip afp)))
    (no-new-landmarks nfwop))
  )

; The SIADH scenario is easily done by modifying states to produce new ones.

(defun siadh-normal-intake ()
  (let* ((normal (make-initial-state water-balance
				     '((afp (afp* std))
				       (nfwip (nfwip* std))
				       (anp (anp* std))
				       (nfnpu (nfnpu* std)))))
	 (init (make-initial-state siadh-water-balance
				   '((NFWIP (NFWIP* std))
				     (anp (anp* std))
				     (cadh (cadh% std))
				     (nfnpu (nfnpu* std))
				     (afp (afp* nil)))
				   "Normal water intake")))
    (qsim init)
    (qsim-display init :reference-states `((normal ,normal)))
    ))

; Here we simulate the response of the kidney to "sudden onset SIADH" (!).
; Then the response to water restriction, which branches three ways due to incomplete
; knowledge of the severity of water restriction.

(defun siadh-therapy ()
  (let* ((normal (make-initial-state water-balance
				     '((afp (afp* std))
				       (nfwip (nfwip* std))
				       (anp (anp* std))
				       (nfnpu (nfnpu* std)))))
	 (init (make-initial-state siadh-water-balance
				   '((NFWIP (NFWIP* std))
				     (anp (anp* std))
				     (cadh (cadh% std))
				     (nfnpu (nfnpu* std))
				     (afp (afp* nil)))
				   "Normal water intake"))
	 (siadh-final nil)
	 (therapy nil))
    (qsim init)
    (setq siadh-final (car (last (car (get-behaviors init)))))
    (setq therapy (make-modified-state siadh-final
				       `((afp (,(qvalue= (afp siadh-final)) nil))
					 (NFWIP ((0 nfwip*) std))	; low intake
					 (anp (anp* std))
					 (cadh (cadh% std))
					 (nfnpu (nfnpu* std)))
				       "Reduced water intake therapy"))
    (qsim therapy)
    (qsim-display therapy
		  :reference-states `((normal ,normal) (siadh ,siadh-final)))
    ))

; This case repeats the SIADH/water-restriction scenario, but maps backward from the
; abstracted constraint M+(nfwip,afp) to determine that there is some landmark value
; of nfwip that determines which branch is taken.

(defun siadh-therapy-learn ()
  (let* ((normal (make-initial-state water-balance
				     '((afp (afp* std))
				       (nfwip (nfwip* std))
				       (anp (anp* std))
				       (nfnpu (nfnpu* std)))))
	 (init (make-initial-state siadh-water-balance
				   '((NFWIP (NFWIP* std))
				     (anp (anp* std))
				     (cadh (cadh% std))
				     (nfnpu (nfnpu* std))
				     (afp (afp* nil)))
				   "Normal water intake"))
	 (siadh-final nil)
	 (therapy1 nil)				; learn from this one
	 (lesson nil)				;   the final state is here
	 (therapy2 nil))			; demonstrate on this one

    (format *QSIM-Report*
"~2%This demonstrates the abstracted-constraint scenario discussed in the paper
`Qualitative Simulation as Causal Explanation', by B. Kuipers, in IEEE-SMC 17(3), 1987.
The disease SIADH causes water volume (AFP) to rise in spite of a normal rate of
water intake (NFWIP).  The usual therapy is to decrease water intake, but this
yields a non-deterministic prediction.  Using an abstracted constraint, we can
resolve the non-determinism by discovering a new landmark value for NFWIP.~2%")

    (qsim init)
    (setq siadh-final (car (last (car (get-behaviors init)))))
    (setq therapy1 (make-modified-state siadh-final
				       `((afp   (,(qvalue= (afp siadh-final)) nil))
					 (NFWIP ((0 NFWIP*) std))	; low intake
					 (anp   (anp* std))
					 (cadh  (cadh% std))
					 (nfnpu (nfnpu* std)))
				       "Reduced water intake therapy; before lesson"))
    (qsim therapy1)

    (format *QSIM-Report* 
"~2%The first time through the water-restriction scenario, we branch three ways, 
apparently non-deterministically, according to where AFP ends up with respect to AFP*, 
since we don't know how severe the water restriction is.~2%")

    (qsim-display therapy1
		  :reference-states `((normal ,normal) (siadh ,siadh-final)))
    (setq lesson (select-final-state-with 'afp 'afp* therapy1))
    (if (null lesson)
	(error "No behavior from ~a has ~a=~a." therapy1 'afp 'afp*))
    (setq therapy2 (make-modified-state lesson
				       `((afp   (,(qvalue= (afp siadh-final)) nil))
					 (NFWIP ((0 NFWIP*) std))	; low intake
					 (anp   (anp* std))
					 (cadh  (cadh% std))
					 (nfnpu (nfnpu* std)))
				       "Reduced water intake therapy; with lesson"))
    (qsim therapy2)

    (format *QSIM-Report* 
"~2%From the result of the first scenario, we determine that there is a state that
restores AFP=AFP*, and save corresponding values on the abstracted constraint M+(NFWIP,AFP).
Then the second time through the scenario, the branch is no longer non-deterministic,
but is deterministic, according to where NFWIP is with respect to the landmark that
corresponds to AFP=AFP*.  Thus, we have backed up from a result we wanted, to the input
that can produce that result.~2%")

    (qsim-display therapy2
		  :reference-states `((normal ,normal) (siadh ,siadh-final)))
    ))

(defun select-final-state-with (param val init)
  (do ((L (get-behaviors init) (cdr L))
       (final-state nil))
      ((null L) nil)
    (setq final-state (car (last (car L))))
    (cond ((eql val (lmark-name (qmag (alookup param (state-qvalues final-state)))))
	   (return final-state)))))