;;; -*- Mode:Common-Lisp; Package:QSIM; Syntax:COMMON-LISP; Base:10 -*-
(in-package :qsim) ;changed DJC

;;; Simple ball example, showing off the Q2 filter.
;;;
(define-QDE q2-Gravity
  (text  "Constant gravity (no friction).")
  (quantity-spaces
    (y   (0 y* inf))
    (v   (minf vmin 0 inf))
    (a   (minf g 0)))
  (constraints
   ((constant a))
    ((d/dt v a))
    ((d/dt y v)))
  (layout (time y nil)
	  (nil v nil)
	  (nil a nil))
  (initial-ranges ((y y*) (80 100))
		  ((a g) (-9.8 -9.8))
		  ((time t0) (0 0))))


;;; Drop the ball from a known height.
;;;
(defun q2-drop-ball ()
  (let* ((sim (make-sim :q2-constraints t))
	 (initial-state
	  (make-new-state :from-qde q2-gravity
			  :assert-values '((y (y* nil))
					   (v (0 nil))
					   (a (g std)))
			  :text "Drop ball in constant gravity")))
    (format *QSIM-Report*
	    "~%Ball dropped from known height, known gravity.~
             ~%See the file EXAMPLES;q2-ball.lisp for details")

    (qsim initial-state)
    (qsim-display initial-state :show-inconsistent-successors t)
    ))


;;; This function simulates the dropping ball using Nsim to tighten the
;;; trajectory prediction.
;;; This is a rather compelling example for the use of NSIM as Q2 alone can
;;; only give general values for the time when the ball hits the ground.  
;;;
(defun nsim-drop-ball ()
  (let* ((*use-dynamic-envelopes* T)
	 (sim (make-sim :q2-constraints T))
	 (init
	   (make-new-state
	    :from-qde q2-Gravity
	    :sim sim
	    :assert-values '((y  (y* nil))	; positive initial position
			     (v  (0 nil)) 	; zero initial velocity
			     (a  (g  std)))	; constant downward gravity
	    :text "Drop an object downward (using nsim")))

    (format *QSIM-Report*
	    "~%Ball dropped from an unknown height, known gravity using Nsim.~
             ~%See the file EXAMPLES;q2-ball.lisp for details")
    (nsim-initialize init)
    (qsim init)
    (nsim init :stop 5)
    (set-numeric-layout init '((y v)))
    (set-numeric-graph-options init '((* (rangestyle nil))))
    (qsim-display init :plot-mode 'numeric-time-plot
		  :show-inconsistent-successors t)
    ))


;;; This function simulates dropping the ball from an unknown height.
;;; 
(defun drop-thing ()
  (let* ((sim (make-sim :q2-constraints T))
	 (init (make-new-state
		  :from-qde q2-Gravity
		  :assert-values
		  '((y  ((0 inf)  nil))	        ; positive initial position
		    (v  (0 nil))	        ; zero initial velocity
		    (a  (g  std)))	        ; constant downward gravity
		  :text "Drop an object downward")))
    (format *QSIM-Report*
	    "~%Ball dropped from an unknown height, known gravity.~
             ~%See the file EXAMPLES;q2-ball.lisp for details")
    (qsim init)
    (qsim-display init)
    ))



; Generalize the ball slightly to include gravity and air resistance

(define-QDE Gravity+Friction
  (text  "Constant gravity with friction.")
  (quantity-spaces
    (y   (0 inf))
    (v   (minf 0 inf))
    (a   (minf 0 inf))
    (g   (minf G* 0))
    (r   (minf 0 inf))
    )
  (constraints
    ((d/dt v a))
    ((d/dt y v))
    ((m- v r) (0 0) (inf minf) (minf inf))
    ((add g r a))
    ((constant g)))
  (layout (nil y nil)
	  (nil v r)
	  (nil a g))
  )



(defun throw-upward-r ()
  (let ((init
	  (make-new-state :from-qde Gravity+Friction
			  :assert-values
			  '((y  (0  nil))	        ; zero initial position
			    (v  ((0 inf) nil))	; positive initial velocity
			    (g  (g*  std)))	        ; constant downward gravity
			  :text "Throw an object upward (w friction)")))
    (qsim init)
    (qsim-display init)
    ))

(defun drop-thing-r ()
  (let ((init
	  (make-new-state :from-qde Gravity+Friction
			  :assert-values
			  '((y  ((0 inf)  nil))	; positive initial position
			    (v  (0 nil))	        ; zero initial velocity
			    (g  (g*  std)))	        ; constant downward gravity
			  :text "Drop an object downward (w friction)")))
    (qsim init)
    (qsim-display init)))
