;;; -*- Syntax: Common-lisp; Package: QSIM -*-
(in-package :qsim)


; Population restricted by food supply, without regeneration.

(define-QDE Population
  (text "Population restricted by food supply")
  (quantity-spaces
    (population     (0 inf)       "Population")
    (food-needed    (0 inf)       "food needed")
    (food-supply    (0 inf)       "food supply")
    (surplus        (minf 0 inf)  "surplus = need - supply")
    (change         (minf 0 inf)  "d/dt population"))
  (constraints
    ((M+ population food-needed)    (0 0) (inf inf))
    ((ADD surplus food-needed food-supply))
    ((S+ surplus change (minf minf) (0 0)))
    ((d/dt population change))
    ((constant food-supply)))
  (layout (nil population  nil)
	  (food-supply food-needed surplus)
	  (nil change))
)

; Start with stable population, then restrict food supply.


(defun population-model ()
  (let* ((normal (make-new-state :from-qde Population
				 :assert-values '((population  ((0 inf) std))
						  (food-supply ((0 inf) std))
						  (surplus     ((0 inf) std)))
				 :text "Population with surplus food."))
	 (margin (make-new-state :from-state normal
				 :inherit '(population)
				 :perturb '((food-supply -))
				 :text "Decreased food supply"))
	 (final nil)
	 (recovery nil))

    (format *QSIM-Report* 
"~2%  This demonstrates a population model where food-needed is a function of population,
but food-supply is not.  If a surplus is available, the population does not change,
but if inadequate food is available, the population decreases.  (This is modeled after
the response of optic nerve cells, which do not regenerate, to restricted blood flow in
glaucoma.)~2%")

    (qsim margin)
    (qsim-display margin
		  :reference-states `((normal ,normal)))
    (when *QSIM-Report*
      (y-or-n-p "Continue with recovered food supply? "))
    (format *QSIM-Report*
"~2%  When the food supply is returned to normal, the population does not recover.~2%")

    (setq final (car (last (longest-behavior (get-behaviors margin)))))
    (setq recovery (make-new-state :from-state final
				   :inherit '(population)
				   :assert-values `((food-supply (,(qvalue= (food-supply normal)) std)))
				   :text "Food supply returns to normal."))
    (qsim recovery)
    (qsim-display recovery
		  :reference-states `((normal ,normal) (famine ,final)))

    ))

(defun longest-behavior (L &optional (best nil))
  (cond ((null L) best)
	((> (length (car L)) (length best)) (longest-behavior (cdr L) (car L)))
	(t (longest-behavior (cdr L) best))))