;;; -*- Mode:Common-Lisp; Package:QSIM; Syntax:COMMON-LISP; Base:10 -*-
(in-package :qsim)

;;; This file contains functions for simulating two cascaded tanks using
;;; Q2-augmented QDEs.  It is similar to nsim-cascade.lisp, except that the
;;; monotonic functions are square-root based rather than linear.
;;; Two scenarios are simulated :
;;; 1. The two-tank cascade filling from empty.
;;; 2. The two-tank cascade filling, but not from empty.




;;; This models the tanks using Q2 information about capacities and flows.
;;; Each tank is given the same ranges and envelopes as were given
;;; in EXAMPLES;bathtub-w-ranges.lisp.
;;;
(define-QDE Cascaded-tanks-w-ranges
  (text "Two cascaded tanks")
  (quantity-spaces
    (inflowa    (0 ifa* inf))
    (amounta    (0 full inf))
    (outflowa   (0 inf))
    (netflowa   (minf 0 inf))
    (amountb    (0  full inf))
    (outflowb   (0 inf))
    (netflowb   (minf 0 inf))
    )
  (constraints
    ((constant inflowa))
    ((M+ amounta outflowa)            (0 0) (inf inf)   )
    ((ADD outflowa netflowa inflowa)                    )
    ((d/dt amounta netflowa))
    ((m+ amountb outflowb)            (0 0) (inf inf)   )
    ((add outflowb netflowb outflowa)                   )
    ((d/dt amountb netflowb)))
  (ignore-qdirs netflowb)
  (transitions ((amountB (full inc)) -> t))
  (m-envelopes
     ((M+ amounta outflowa)
      ;; True formula is outflow = Cda*sqrt(2*g*H)
      ;; where Cda is the coefficient of discharge times area
      ;;       H is the liquid height
      ;;       g is the accelleration due to gravity
      ;; See S.J. Mitchell, Fluid and Particle Mechanics,
      ;;     Pergamon Press 1970. p. 40.
      (expected-function (lambda (x) (* 10 (sqrt x))))
      (expected-inverse  (lambda (y) (expt (/ y 10) 2)))
      ;; Piecewise linear
;      (upper-envelope (lambda (x) x))
;      (upper-inverse  (lambda (y) y))
;      (lower-envelope (lambda (x) (if (< x 20) (/ x 2)
;				      (+ (/ x 5) 6))))
;      (lower-inverse  (lambda (y) (if (< y 10) (* y 2)
;				      (* 5 (- y 6)))))
      ;; Narrow
      (upper-envelope (lambda (x) (* 10 (sqrt x))))
      (upper-inverse  (lambda (y) (expt (/ y 10) 2)))
      (lower-envelope (lambda (x) (* 8  (sqrt x))))
      (lower-inverse  (lambda (y) (expt (/ y 8) 2)))
      ;; Wide
;      (upper-envelope (lambda (x) (* 5 (* 10 (sqrt x)))))
;      (upper-inverse  (lambda (y) (expt (/ (/ y 10) 5) 2)))
;      (lower-envelope (lambda (x) (* 5 (* 8  (sqrt x)))))
;      (lower-inverse  (lambda (y) (expt (/ (/ y 8) 5) 2)))
      )
     ((M+ amountb outflowb)
      ;; True formula is outflow = Cda*sqrt(2*g*H)
      ;; where Cda is the coefficient of discharge times area
      ;;       H is the liquid height
      ;;       g is the accelleration due to gravity
      ;; See S.J. Mitchell, Fluid and Particle Mechanics,
      ;;     Pergamon Press 1970. p. 40.
      (expected-function (lambda (x) (* 10 (sqrt x))))
      (expected-inverse  (lambda (y) (expt (/ y 10) 2)))
      ;; Piecewise linear
;      (upper-envelope (lambda (x) x))
;      (upper-inverse  (lambda (y) y))
;      (lower-envelope (lambda (x) (if (< x 20) (/ x 2)
;				      (+ (/ x 5) 6))))
;      (lower-inverse  (lambda (y) (if (< y 10) (* y 2)
;				      (* 5 (- y 6)))))
      ;; Narrow
      (upper-envelope (lambda (x) (* 10 (sqrt x))))
      (upper-inverse  (lambda (y) (expt (/ y 10) 2)))
      (lower-envelope (lambda (x) (* 8  (sqrt x))))
      (lower-inverse  (lambda (y) (expt (/ y 8) 2)))
      ;; Wide
;      (upper-envelope (lambda (x) (* 5 (* 10 (sqrt x)))))
;      (upper-inverse  (lambda (y) (expt (/ (/ y 10) 5) 2)))
;      (lower-envelope (lambda (x) (* 5 (* 8  (sqrt x)))))
;      (lower-inverse  (lambda (y) (expt (/ (/ y 8) 5) 2)))
      ))
  (initial-ranges
                  ((inflowA ifa*) (20 40))
;		  ((inflowA ifa*) (10 20))   ; For piecewise linear envs
		  ((time t0)      (0 0))
		  ((amounta full) (90 100))
		  ((amountb full) (90 100))
		  )
  (layout
    (amounta amountb outflowa)
    (neflowa netflowb outflowb)
    (time nil nil))
)

;;; Fill the cascade using quantitative information about outflow.
;;; This added information is enough to refute the overflow behaviors that
;;; result from a "plain" Qsim simulation of the system and
;;; so there are no region transitions.  As a result, there are only two
;;; behaviors predicted -- one where A reaches equilibrium first and one
;;; where they reach equilibrium simultaneously.
;;; Note: This model takes around a minute to simulate.
;;; This gives two behaviors that end with B in [5 20]
;;;
(defun q2-fill-cascade-from-empty ()
  (let* ((sim (make-sim :q2-constraints t))
	 (init (make-new-state :from-qde cascaded-tanks-w-ranges
			       :sim sim
			       :assert-values '((inflowa (ifa* std))
						(amounta (0 nil))
						(amountb (0 nil)))
			       :text  "Fill from empty")))
    (qsim init)
    (qsim-display init)
    ))


;;; Fill the cascade as before, but use Nsim to further refine the behavior
;;; over time intervals.  Note that the dynamic envelope prediction for
;;; amountb is in fact larger than that predicted by Q2.  This points out
;;; that the dynamic envelopes do not always produce tighter bounds than Q2.
;;;
(defun nsim-fill-cascade-from-empty ()
  (qsim-cleanup)
  (let* ((sim (make-sim :q2-constraints t))
	 (*use-dynamic-envelopes* T)
	 (*trace-nsim* T)
	 (*trace-function-gen* T)
	 (init (make-new-state :from-qde cascaded-tanks-w-ranges
			       :sim sim
			       :assert-values '((inflowa (ifa* std))
						(amounta (0 nil))
						(amountb (0 nil)))
			       :text "Fill from empty (using Nsim)")))
    (nsim-initialize init :simvars NIL)
    (qsim init)
    (nsim init)
    (set-numeric-layout init '((amounta amountb)))
    (set-numeric-graph-options init '((* (rangestyle nil))))
    (qsim-display init :plot-mode 'numeric-time-plot)
    ))



;;; Start with tank A full and simulate.  This predicts 7 behaviors.
;;;
(defun q2-fill-cascade ()
  (let* ((sim (make-sim :q2-constraints t :state-limit 100))
	 (init (make-new-state :from-qde cascaded-tanks-w-ranges
			       :sim sim
			       :assert-values '((inflowa (ifa* std))
						(amounta (full nil))
						(amountb (0 nil)))
			       :text  "Fill from empty")))
    (qsim init)
    (qsim-display init)
    ))


;;; Same as the previous function but use NSIM as well.  Note that behavior 1
;;; (overflow) can be refuted using NSIM.
;;;
(defun nsim-fill-cascade ()
  (qsim-cleanup)
  (let* ((sim (make-sim :q2-constraints t :state-limit 100))
	 (*use-dynamic-envelopes* T)
	 (init (make-new-state :from-qde cascaded-tanks-w-ranges
			       :sim sim
			       :assert-values '((inflowa (ifa* std))
						(amounta (full nil))
						(amountb (0 nil)))
			       :text "Fill from empty (using Nsim)")))
    (format t "~%Init is ~a" init)
    (nsim-initialize init)
    (qsim init)
    (nsim (second (state-successors init)) :stop 20 :simStep .2)
    (set-numeric-layout init '((amounta amountb)))
    (set-numeric-graph-options init '((* (rangestyle nil))))
    (qsim-display init :plot-mode 'numeric-time-plot)
    ))

