;;; -*- Syntax: Common-lisp; Package: QSIM -*-
; Copyright (c) 1987, Benjamin Kuipers.
;;;  $Id: hq-heater.lisp,v 1.2 1991/09/16 14:12:03 clancy Exp $
(in-package :qsim)

(Define-QDE HQ-HEATER-BASE
  (text "Hierarchical home heater (fast response to environment)")
  (quantity-spaces
    (heat    (0 inf))
    (mass    (0 inf))
    (TempIn  (0 inf))
    (TempOut (0 inf))
    (dTemp   (minf 0 inf))
    (R       (0 inf))
    (HFout   (minf 0 inf))
    (HFin    (minf 0 inf))			; (MaxCool 0 MaxHeat) caused a problem
    (netHF   (minf 0 inf)))
  (constraints
    ((mult TempIn Mass Heat))
    ((add TempOut dTemp TempIn))
    ((mult R HFout dTemp))
    ((add HFout netHF HFin))
    ((d/dt Heat netHF)))
  (independent Mass TempOut HFin R)
  (history TempIn)
  (layout (Mass Heat nil)
	  (TempOut TempIn)
	  (dTemp R)
	  (HFout HFin netHF))
  (print-names
    (heat    "Heat content"   H)
    (mass    "Thermal mass"   M)
    (TempIn  "Temp(inside)"   Ti)
    (TempOut "Temp(outside)"  To)
    (dTemp   "dTemp(in,out)"  dT)
    (R       "Heat flow resistance")
    (HFout   "Heat flow (to environment)" HFo)
    (HFin    "Heat flow (from heater/cooler)" HFi)
    (netHF   "net Heat Flow"  nHF))
  (other
    (define-normal
      ((heat ((0 inf) nil)) (mass ((0 inf) std)) (TempIn ((0 inf) nil))
       (TempOut ((0 inf) std)) (dTemp (0 nil)) (R ((0 inf) std))
       (HFout (0 nil)) (HFin (0 std)) (netHF (0 nil)) ))
    (normal-state nil)
    (slower hq-heater-control)
    (abstracted-to ((m+ HFin TempIn))))
  )


(defun test-fast-heater-base ()
  (let* ((normal (get-normal-state hq-heater-base))
	 (new (make-modified-state normal
				   `((TempIn   (,(qvalue= (TempIn normal)) nil))
				     (mass     (,(qvalue= (mass normal)) std))
				     (R        (,(qvalue= (R normal)) std))
				     (HFin     (,(qvalue= (HFin normal)) std))
				     (TempOut  (,(qvalue- (TempOut normal)) std)))
				   "Low outside temperature")))
    (qsim new)
    (qsim-display new
		  :reference-states `((normal ,normal)))
    ))

; The heater controller.

(define-QDE HQ-HEATER-CONTROL
  (text "Hierarchical home heater (slow controller)")
  (quantity-spaces
    (HFin    (minf 0 inf))
    (TempIn  (minf 0 inf))
    (TempSet (0 inf))
    (error   (minf 0 inf)))
  (constraints
    ((m+ HFin TempIn)            (inf inf) (minf minf))
    ((add TempIn error TempSet))
    ((d/dt HFin error)))
  (independent TempSet)
  (history TempIn)
  (layout (nil HFin nil)
	  (nil TempIn TempSet)
	  (nil error nil))
  (print-names
    (TempIn  "Temp(inside)"   Ti)
    (TempSet "Temp(set)"      Ts)
    (error   "error=in-set"  E)
    (HFin    "Heat flow (from heater)" HFi))
  (other
    (define-normal
      ((HFin (0 std)) (TempIn ((0 inf) std)) (TempSet ((0 inf) std))))
    (normal-state nil)
    (faster hq-heater-base)
    (abstracted-from
      ((m+ HFin TempIn) hq-heater-base)))
  )

(defun test-slow-controller ()
  (let* ((normal (get-normal-state hq-heater-control))
	 (new (make-modified-state normal
				   `((TempSet (,(qvalue= (TempSet normal)) std))
				     (TempIn  (,(qvalue- (TempIn normal)) nil)))
				   "Low inside temperature")))
    (qsim new)
    (qsim-display new
		  :reference-states `((normal ,normal)))
    ))

(defparameter *starting-point* nil)
(defparameter slice-vars '(TempOut TempIn TempSet HFin))

(defun test-hierarchical-heater ()
  (let ((normal (get-normal-state hq-heater-base)))
    (setq *starting-point*
	  (make-modified-state normal
			       `((TempIn   (,(qvalue= (TempIn normal)) nil))
				 (mass     (,(qvalue= (mass normal)) std))
				 (R        (,(qvalue= (R normal)) std))
				 (HFin     (,(qvalue= (HFin normal)) std))
				 (TempOut  (,(qvalue- (TempOut normal)) std)))
			       "Low outside temperature"))
    (tsa-simulation *starting-point*)
    (HQD-display *starting-point* slice-vars)
    ))




(defun low-temperature-prop ()
  (let* ((normal (make-initial-state Home-Heater-Prop
				     '((HFin (0 std)) (Heat ((0 inf) std))
				       (TempIn (RoomTemp nil))
				       (Mass (M* std)) (TempOut (RoomTemp std))
				       (TempSet (RoomTemp std)) (R (R* std)))
				     "Normal state; room temperature."))
	 (start (make-modified-state normal
				     '((HFin (0 nil)) (Heat ((0 inf) nil))
				       (TempIn (RoomTemp nil))
				       (Mass (M* std)) (TempOut (Cold std))
				       (TempSet (RoomTemp std)) (R (R* std)))
				     "Suddenly cold outdoors; furnace off.")))
    (qsim start)
    (qsim-display start
		  :reference-states `((normal ,normal)))
    t))

; Switch to the constant-heat-input model when the system reaches MaxHeat.

(defun constant-max-heating (heater-state)
  (let ((nstate (make-transition-result
		  heater-state
		  constant-home-heater
		  (switch-values heater-state '((HFin (MaxHeat std)))))))
    nstate))

(defun constant-max-cooling (heater-state)
  (let ((nstate (make-transition-result
		  heater-state
		  constant-home-heater
		  (switch-values heater-state '((HFin (MaxCool std)))))))
    nstate))


; This models the heater system with constant input, either HFin=0 for the
; broken system, or HFin=MaxHeat for the system pinned to its top end.

(define-QDE CONSTANT-HOME-HEATER
  (text "Constant input from heater/cooler")
  (quantity-spaces
    (heat    (0 inf))
    (mass    (0 M* inf))
    (TempIn  (0 RoomTemp inf))
    (TempOut (0 Cold RoomTemp Hot inf))
    (TempSet (0 RoomTemp inf))
    (dTemp   (minf 0 inf))
    (error   (minf 0 inf))
    (R       (0 R* inf))
    (HFout   (minf 0 inf))
    (HFin    (MaxCool 0 MaxHeat))
    (netHF   (minf 0 inf)))
  (constraints
    ((mult TempIn Mass Heat))
    ((add TempOut dTemp TempIn)      (RoomTemp 0 RoomTemp) )
    ((add TempSet error TempIn)      (RoomTemp 0 RoomTemp) )
    ((mult R HFout dTemp))
    ((add HFout netHF HFin))
    ((d/dt Heat netHF)))
  (independent Mass TempOut TempSet R HFin)
  (history TempIn)
  (layout (Mass Heat nil)
	  (TempOut TempIn TempSet)
	  (dTemp error R)
	  (HFout HFin netHF))
  (print-names
    (heat    "Heat content"   H)
    (mass    "Thermal mass"   M)
    (TempIn  "Temp(inside)"   Ti)
    (TempOut "Temp(outside)"  To)
    (TempSet "Temp(set)"      Ts)
    (dTemp   "dTemp(in,out)"  dT)
    (error   "error=in-set"  E)
    (R       "Heat flow resistance")
    (HFout   "Heat flow (to environment)" HFo)
    (HFin    "Heat flow (from heater/cooler)" HFi)
    (netHF   "net Heat Flow"  nHF)))

; The broken furnace starts off at RoomTemp, but with no heat input.

(defun low-temperature-broken-heater ()
  (let* ((normal (make-initial-state Constant-Home-Heater
				     '((HFin (0 std)) (Heat ((0 inf) std))
				       (TempIn (RoomTemp nil))
				       (Mass (M* std)) (TempOut (RoomTemp std))
				       (TempSet (RoomTemp std)) (R (R* std)))
				     "Normal state; room temperature."))
	 (start (make-modified-state normal
				     '((HFin (0 std)) (Heat ((0 inf) nil))
				       (TempIn (RoomTemp nil))
				       (Mass (M* std)) (TempOut (Cold std))
				       (TempSet (RoomTemp std)) (R (R* std)))
				     "Suddenly cold outdoors; furnace broken.")))
    (qsim start)
    (qsim-display start
		  :reference-states `((normal ,normal)))
    t))
