
;;; -*- Package: QSIM; Syntax: Common-lisp -*-
;
;-----------------------------------------------------------------------------
;
;
;            veins             left ventricle          arteries
;
;           |     |                |     |              |     |
;           |  Z  |                |  Y  |              |  X  |
;           |     |                |     |              |     |
;     ______|     |________________|     |______________|     |______
;    /  _______c_______________________b_____________________a_____  \
;   |  |                 --->                  --->                |  |
;   |  |                                                           |  |
;   |  |                                                           |  |
;   |  |___________________________________________________________|  |
;   \________________________________________________________________/
;                                <---
;
;
;
;   This heart model treats the three tanks at one QSIM level.
;-----------------------------------------------------------------------------

;---------------------------------------------------------------------------------
; This is the three tank heart, with the faster CNS regulatory mechanisms
; abstracted to a faster time-scale. 

(define-QDE HEART-3-TANKs

  (text "Three-tank heart model:  Artery --> vein --> heart flows")

  (quantity-spaces
    (sabc     (0 inf))
    (sab  (0 inf))
    (x     (0 inf))
    (z     (0 inf))
    (y  (0 inf))
    (pa    (0 inf))
    (pc    (0 inf))
    (pb  (0 inf))
    (dac   (0 inf))
    (dcb   (0 inf))
    (ofb     (0 inf))
    (hr      (0 inf))
    (fac    (0 inf))
    (fcb    (0 inf))
    (fba    (0 inf))
    (xp    (minf 0 inf))
    (yp (minf 0 inf))
    (zp    (minf 0 inf))
    (pap    (minf 0 inf))
    (pbp (minf 0 inf))
    (pcp    (minf 0 inf))
    (dacp    (minf 0 inf))
    (dcbp    (minf 0 inf))

  )
  (constraints
    ((ADD y x sab))
    ((ADD  sab z sabc))
    ((M+ x pa)(0 0)(inf inf))
    ((M+ z pc)(0 0)(inf inf))
    ((M+ y pb)(0 0)(inf inf))
    ((ADD dac pc pa))
    ((ADD dacp pcp pap)(0 0 0))
    ((M+  dac fac)(0 0)(inf inf))
    ((ADD pb dcb pc))
    ((ADD dcbp pbp pcp)(0 0 0))
    ((M+  dcb fcb)(0 0)(inf inf))
    ((M+ pb ofb)(0 0)(inf inf))
    ((MULT ofb hr fba))				; cardiac output (fba) = stroke volume * hr
    ((ADD zp fcb fac))
    ((ADD xp fac fba))
    ((ADD yp fba fcb))
    ((m+ pap xp)(minf minf)(0 0 )(inf inf))
    ((m+ pbp yp)(minf minf)(0 0 )(inf inf))
    ((m+ pcp zp)(minf minf)(0 0 )(inf inf))
    ((D/DT x xp))
    ((D/DT y yp))
    ((D/DT z zp))
    ((D/DT dac dacp))
    ((D/DT dcb dcbp))
  )
  (independent  sabc hr)
  (history      x z y)
  (layout
    (y     yp     fba  sab)
    (x     xp     fac  dacp)
    (z     zp     fcb  dcbp)
  )

;  (print-names
;    (sabc      "plasma volume")
;    (x      "arterial volume")
;    (z      "venous volume")
;    (y   "y")
;    (ac      "arterial compliance")
;    (vc      "venous compliance")
;    (pa     "mean arterial pressure")
;    (pc     "central venous pressure")
;    (ofb      "stroke volume")
;    (hr      "heart rate")
;    (pb   "left ventricular end-diastolic pressure")
;    (dac    "pa - pc")
;    (dcb    "pc - pb")
;    (fac     "flow: arteries to veins")
;    (fcb     "flow: veins to heart")
;    (fba     "flow: heart to arteries")
;    (zp     "delta venous volume")
;    (yp  "delta y")
;    (xp     "delta arterial volume")
;  )
  (other					; who uses these?
    (coeffs (variables xp zp)
	    (constraint (yp (minus (add xp zp)))))
    (define-normal
          ((sabc   ((0 inf) std))    ; independent parameters
           (hr     ((0 inf) std))
	   (x      ((0 inf) std))    ; some dependent parameters
	   (z      ((0 inf) std))
	   (y      ((0 inf) std))
	  )
    )

    (unreachable-values
      (y      0) (x      0)
      (z      0) (dac    0)
      (dcb    0) (fcb    0 inf))
    (normal-state  nil)
  )
)

; Modified curvature constraints. Element with "*" symbol does not have sign ambiguity.
; Element without "*" symbol has sign-ambiguity.
; All three assignments, t -> default, 0, and nil -> (minus default), are allowed.

(defun Heart-3t-hod-constraints(signs qde)
  (let((default-cast-clause     '((xp (sd1 yp))
				  (yp (minus (sd1 xp) *))
				  (zp (sd1 xp))   
				  (dacp (sd1 xp)) 
				  (dcbp (sd1 xp)))))
    (cast-clause-of-coupled-qdes signs default-cast-clause qde)
    (sd3-clause-of-coupled-qdes  signs default-cast-clause qde)))

(defun heart-3T (&rest acc-sign-specs)    

   (declare (special HEART-3-TANKs))
   (setq *perform-acc-analysis* t)
   (heart-3t-hod-constraints  acc-sign-specs heart-3-tanks)
   (let* ((normal (get-normal-state HEART-3-TANKs))
          (new    (make-modified-state normal
			      `((sabc         (,(qvalue= (sabc normal)) std))
				(hr            (,(qvalue+ (hr normal)) std))
				(x            (,(qvalue= (x normal)) nil))
				(y            (,(qvalue= (y normal)) nil))
				(z            (,(qvalue= (z normal)) nil))
			       )
			     "Heart rate up, normal volume.")))
     (qsim new)
     (qsim-display new
		   :reference-states `((normal ,normal)))
   ))

; Sample runs: with maximum 240.
; Output - (No. of simulated-states, No. of behaviors, No. terminated at queiscent-states)
; (heart-3t)                     output - ( 89     6     5)
; (heart-3t t t   t   t nil)              ( 27     7     1)
; (heart-3t t t   t nil nil)              (238    14     8)
; (heart-3t t t nil nil nil)              (172     8     2)
; (heart-3t t t   0   t   t)              ( 28     7     1)
; (heart-3t t t   t nil   t)              (241   nonterminating)
; (heart-3t t t nil nil   t)              (25      7     1)


; Revised version without aggregation: Use nq. Max 250.
; Output - (No. of simulated-states, No. of behaviors, No. terminated at queiscent-states)
; (heart-3t)                     output - ( 89     8     5)
; (heart-3t t t   t   t nil)              ( 27     4     1)
; (heart-3t t t   t nil nil)              (238    10     8)
; (heart-3t t t nil nil nil)              (172     8     2)
; (heart-3t t t   0   t   t)              ( 28     4     1)
; (heart-3t t t   t nil   t)              (241   nonterminating)
; (heart-3t t t nil nil   t)              (25      1     1)


