;;; -*- Mode:Common-Lisp; Package:QSIM; Syntax:COMMON-LISP; Base:10 -*-

;;;  $Id: bouncing-ball.lisp,v 1.3 1991/09/16 14:11:13 clancy Exp $
(in-package :qsim)
; The bouncing ball is the basic example of operating region transition,
; between the gravity portion, where accelleration is constant and
; downward, and the spring portion, where accelleration is upward and
; proportional to compression.

(define-QDE Gravity
   (text "Ball bouncing, with transition to spring and back.")
   (quantity-spaces
	      (TE     (0 inf))
	      (A      (minf g 0 inf))
	      (V      (minf 0 inf))
	      (Y      (0 inf))
	      (KE     (0 inf))
	      (PE     (0 inf))
	      (VSQ    (0 inf)))
   (constraints
	      ((d/dt Y V))
	      ((d/dt V A))
	      ((constant A))
	      ((mult V V VSQ))
	      ((M+ VSQ KE)      (0 0) (inf inf))
	      ((M+ Y PE)        (0 0) (inf inf))
              ((add KE PE TE))
	      ((constant TE)))
   (transitions
     ((y (0 dec)) -> transition-to-spring)))

(defun throw-bouncing-ball ()
  (let* ((sim (make-sim :state-limit 17
			:cycle-detection nil))
	 (initial
	   (make-new-state :from-qde Gravity
			   :assert-values '((y  (0 nil))
					    (v  ((0 inf) nil))
					    (a  (g  std))
					    (te ((0 inf) std)))
			   :text "Bouncing ball, thrown upward"
			   :sim sim)))
    
    (format *QSIM-Report*
"~2% The bouncing ball alternates between a gravity model and a spring model,
with conservation-of-energy constraints to keep them in correspondence.
The cycle-detection switch is turned off in this example to illustrate the
repetitive nature of the cycle.~2%")
						
    (qsim initial)
    (qsim-display initial :layout '((y ysq pe)
				    (v vsq ke)
				    (a nil te)
				    (nil)))))

; These are the transition functions between the spring and ball structures.

(defun transition-to-spring (ball-state)
  (create-transition-state :from-state   ball-state
			   :to-qde       Spring
			   :inherit-qmag '(y v te)))

(defun transition-to-gravity (spring-state)
  (create-transition-state :from-state   spring-state
			   :to-qde       Gravity
			   :assert       '((a (g std)))
			   :inherit-qmag '(y v te)))



; This is a version of the spring model, adapted to the bounce part of
; the bouncing ball model.

(define-QDE Spring
   (text "Spring portion of bouncing ball.")
   (quantity-spaces
	      (TE     (0 inf))
	      (A      (minf 0 inf))
	      (V      (minf 0 inf))
	      (Y      (minf 0))
	      (KE     (0 inf))
	      (PE     (0 inf))
	      (VSQ    (0 inf))
	      (YSQ    (0 inf)))
   (constraints
	      ((d/dt Y V))
	      ((d/dt V A))
	      ((M- A Y)         (0 0) (inf minf))
	      ((mult V V VSQ))
	      ((mult Y Y YSQ))
	      ((M+ VSQ KE)      (0 0) (inf inf))
	      ((M+ YSQ PE)      (0 0) (inf inf))
              ((add KE PE TE))
	      ((constant TE)))
   (transitions
     ((y (0 inc)) -> transition-to-gravity)))



; The simple bouncing ball, just reflected off the ground!
; This one handles the bounce by simply reversing the sign of velocity
; when Y=0, and thus gives no internal structure to the bounce.

(define-QDE Simple-bounce
   (text "Simple bouncing ball, reflecting off Y=0")
   (quantity-spaces
	      (y      (minf 0 inf))
	      (v      (minf 0 inf))
	      (a      (minf 0 inf))
	      (vsq    (0 inf))
	      (KE     (0 inf))
	      (PE     (0 inf))
	      (E      (0 inf)))
   (constraints
	      ((d/dt y v))
	      ((d/dt v a))
	      ((constant a))
	      ((mult v v vsq))
	      ((M+ vsq KE)      (0 0) (inf inf))
	      ((M+ y PE)        (0 0) (inf inf))
              ((add KE PE E))
	      ((constant E)))
   (transitions
     ((y (0 dec)) -> reflect-velocity))
   )

(defun reflect-velocity (ball-state)
  (create-transition-state :from-state    ball-state
			   :to-qde        simple-bounce
			   :inherit-qmag '(a E)
			   :assert       '((y (0 inc)))))


(defun throw-simple-bouncing-ball ()
  (let* ((sim (make-sim :state-limit 13
			:cycle-detection nil))
	 (initial
	   (make-new-state :from-qde simple-bounce
			   :assert-values '((y  (0 nil))
					    (v  ((0 inf) nil))
					    (a  ((minf 0) std))
					    (e  ((0 inf) std)))
			   :text "Simple bouncing ball, thrown upward"
			   :sim sim)))
    
    (format *QSIM-Report*
"~2% This model of the bouncing ball simply treats the bounce as a reflection off
the ground, with a transition back into the gravity model with a revised state.
The cycle-detection is turned off here, too, just to make the display look nice.~2%")

    (qsim initial)
    (qsim-display initial :layout '((y nil pe)
				    (v vsq ke)
				    (a nil e)
				    (nil))) ))