;;; -*- Mode:Common-Lisp; Package:QSIM; Syntax:COMMON-LISP; Base:10 -*-

(in-package :qsim)

; The Ball is the simplest second-order system, illustrating
; creation of a new landmark representing the peak of the
; trajectory.  Turn *trace-count* on while this simulates to see
; the QSIM algorithm in action as in [Kuipers, 1985, 1986].

(define-QDE Ball
  (text  "Throw a ball upward in constant gravity.")
  (quantity-spaces
    (y   (0 inf))
    (v   (minf 0 v* inf))
    (a   (minf g 0)))
  (constraints
    ((d/dt v a))
    ((d/dt y v))
    ((constant a)))
  (layout (nil y nil)
	  (nil v nil)
	  (nil a nil))
  )



(defun throw-ball ()
  (let ((initial
	  (make-new-state :from-qde Ball
			  :assert-values '((y  (0  nil))
					   (v  (v* nil))
					   (a  (g  std)))
			  :text "Throw the ball upward")))
    (qsim initial)
    (qsim-display initial)))

; Generalize the ball slightly in gravitational terms.

(define-QDE Gravity
  (text  "Constant gravity (no friction).")
  (quantity-spaces
    (y   (0 inf))
    (v   (minf 0 inf))
    (a   (minf g 0)))
  (constraints
    ((d/dt v a))
    ((d/dt y v))
    ((constant a)))
  (layout (nil y nil)
	  (nil v nil)
	  (nil a nil))
  )



(defun throw-upward ()
  (let ((init
	  (make-new-state :from-qde Gravity
			  :assert-values '((y  (0  nil))	; zero initial position
					   (v  ((0 inf) nil))	; positive initial velocity
					   (a  (g  std)))	; constant downward gravity
			  :text "Throw an object upward")))
    (qsim init)
    (qsim-display init)
    ))

(defun drop-thing ()
  (let ((init
	  (make-new-state :from-qde Gravity
			  :assert-values '((y  ((0 inf) nil))	; positive initial position
					   (v  (0 nil)) 	; zero initial velocity
					   (a  (g  std)))	; constant downward gravity
			  :text "Drop an object downward")))
    (qsim init)
    (qsim-display init)
    ))

; Generalize the ball slightly to include gravity and air resistance

(define-QDE Gravity+Friction
  (text  "Constant gravity with friction.")
  (quantity-spaces
    (y   (0 inf))
    (v   (minf 0 inf))
    (a   (minf 0 inf))
    (g   (minf G* 0))
    (r   (minf 0 inf))
    )
  (constraints
    ((d/dt v a))
    ((d/dt y v))
    ((m- v r) (0 0) (inf minf) (minf inf))
    ((add g r a))
    ((constant g)))
  (layout (nil y nil)
	  (nil v r)
	  (nil a g))
  )



(defun throw-upward-r ()
  (let ((init
	  (make-new-state :from-qde Gravity+Friction
			  :assert-values '((y  (0  nil))	; zero initial position
					   (v  ((0 inf) nil))	; positive initial velocity
					   (g  (g*  std)))	; constant downward gravity
			  :text "Throw an object upward (w friction)")))
    (qsim init)
    (qsim-display init)
    ))

(defun drop-thing-r ()
  (let ((init
	  (make-new-state :from-qde Gravity+Friction
			  :assert-values '((y  ((0 inf)  nil))	; positive initial position
					   (v  (0 nil))	; zero initial velocity
					   (g  (g*  std)))	; constant downward gravity
			  :text "Drop an object downward (w friction)")))
    (qsim init)
    (qsim-display init)
    ))