/*
 *  'genpvals' generates, for a given state, all possible successor values
 *   of all the parameters. These are stored as pval entries for each
 *  parameter.
 *
 *  This function first examines the qmag index to select the appropriate
 *  transition array and then uses the qdir and time's point/interval value
 *  to index directly to the correct set of transitions in the array.
 *  No searching is needed; the function does constant work for each param.
 */


#include  <stdio.h>
#include  "defs.h"
#define   EXTERN  extern
#include  "global.h"


/*
 *  The P-transitions and I-transitions that follow are as defined in
 *  "Qualitative Simulation" by Benjamin Kuipers, Artificial Intelligence,
 *  Vol. 29, page 300.  Each definition contains an increment for the 
 *  qmag index and a new qdir value.
 *
 *				    QS(f,t)         ==>  QS(f,t,t+1)
 *				__________________________________________
 */
#define  P1	 0, STD		/*  <l,std>         ==>  <l,std>	*/
#define  P2	 1, INC		/*  <l,std>	    ==>  <(l,l+1),inc>	*/
#define  P3	-1, DEC		/*  <l,std>	    ==>  <(l-1,l),dec>	*/
#define  P4	 1, INC		/*  <l,inc>	    ==>  <(l,l+1),inc>	*/
#define  P5	 0, INC		/*  <(l,l+1),inc>   ==>  <(l,l+1),inc>	*/
#define  P6	-1, DEC		/*  <l,dec>	    ==>  <(l-1,l),dec>	*/
#define  P7	 0, DEC		/*  <(l,l+1),dec>   ==>  <(l,l+1),dec>	*/

/*
 *				    QS(f,t,t+1)     ==>  QS(f,t+1)
 *				__________________________________________
 */
#define  I1	 0, STD		/*  <l,std>	    ==>  <l,std>	*/
#define  I2	 1, STD		/*  <(l,l+1),inc>   ==>  <l+1,std>	*/
#define  I3	 1, INC		/*  <(l,l+1),inc>   ==>  <l+1,inc>	*/
#define  I4	 0, INC		/*  <(l,l+1),inc>   ==>  <(l,l+1),inc>	*/
#define  I5	-1, STD		/*  <(l,l+1),dec>   ==>  <l,std>	*/
#define  I6	-1, DEC		/*  <(l,l+1),dec>   ==>  <l,dec>	*/
#define  I7	 0, DEC		/*  <(l,l+1),dec>   ==>  <(l,l+1),dec>	*/
#define  I8	 0, STD		/*  <(l,l+1),inc>   ==>  <l*,std>	*/
#define  I9	 0, STD		/*  <(l,l+1),dec>   ==>  <l*,std>	*/


struct transition
{
    int  npvals;		/*  # of possible values for this transition */
    struct
    {   short  qmagdiff;
	short  qdir;
    } qval[4];			/*  space for up to 4 possible values  */
};


/*
 *  The following 2-dimensional arrays are arranged as follows:
 *  -- the first index, which equals 0 or 1, reflects whether time is
 *     a point or interval;
 *  -- the second index, which equals 0, 1 or 2, reflects whether qdir
 *     is increasing (INC=0), steady (STD=1), or decreasing (DEC=2).
 *
 *  The 6 arrays apply to the following cases:
 *
 *  trans1 -- qmag is an interval adjacent to minf (the array excludes
 *	      transitions to the lower point, i.e., minf).
 *  trans2 -- qmag is an interval adjacent to inf (the array excludes
 *	      transitions to a higher point, i.e., inf).
 *  trans3 -- qmag is an interval not adjacent to minf or inf (the array
 *	      allows movement in both directions).
 *  trans4 -- qmag is a point at the lower limit of its qspace (the array
 *            excludes transitions to a lower value).
 *  trans5 -- qmag is a point at the upper limit of its qspace (the array
 *            excludes transitions to a higher value).
 *  trans6 -- qmag is a point somewhere "in the middle" of its qspace
 *	      (the array allows movement in both directions).
 *
 *  The reason for the first 3 arrays is to avoid generating divergent
 *  states (a state where some value goes to +infinity or -infinity)
 *  at the earliest possible moment rather than filtering it out at a
 *  later stage.  The last 3 arrays are necessary to avoid generating
 *  values outside the quantity space.
 */

static struct transition trans1[2][3] =
  {
    { /* ================ Time is a point =============================== */
	{ 1, P5 },				/*  <(l,l+1), inc>	*/
	{ 0 },					/*  <(l,l+1), std> note 1 */
	{ 1, P7 },				/*  <(l,l+1), dec>	*/
    },
    { /* ================ Time is an interval =========================== */
	{ 4, I2, I3, I4, I8 },			/*  <(l,l+1), inc>	*/
	{ 0 },					/*  <(l,l+1), std> note 1 */
	{ 2, I7, I9 }				/*  <(l,l+1), dec>	*/
    }
  };


static struct transition trans2[2][3] =
  {
    { /* ================ Time is a point =============================== */
	{ 1, P5 },				/*  <(l,l+1), inc>	*/
	{ 0 },					/*  <(l,l+1), std> note 1 */
	{ 1, P7 },				/*  <(l,l+1), dec>	*/
    },
    { /* ================ Time is an interval =========================== */
	{ 2, I4, I8 },				/*  <(l,l+1), inc>	*/
	{ 0 },					/*  <(l,l+1), std> note 1 */
	{ 4, I5, I6, I7, I9 }			/*  <(l,l+1), dec>	*/
    }
  };


static struct transition trans3[2][3] =
  {
    { /* ================ Time is a point =============================== */
	{ 1, P5 },				/*  <(l,l+1), inc>	*/
	{ 0 },					/*  <(l,l+1), std> note 1 */
	{ 1, P7 },				/*  <(l,l+1), dec>	*/
    },
    { /* ================ Time is an interval =========================== */
	{ 4, I2, I3, I4, I8 },			/*  <(l,l+1), inc>	*/
	{ 0 },					/*  <(l,l+1), std> note 1 */
	{ 4, I5, I6, I7, I9 }			/*  <(l,l+1), dec>	*/
    }
  };


static struct transition trans4[2][3] =
  {
    { /* ================ Time is a point ============================= */
	{ 1, P4 },				/*  <l, inc>		*/
	{ 2, P1, P2 },				/*  <l, std>		*/
	{ 0 },					/*  <l, dec>		*/
    },
    { /* ================ Time is an interval ========================= */
	{ 0 },					/*  <l, inc>		*/
	{ 1, I1 },				/*  <l, std>		*/
	{ 0 }					/*  <l, dec>		*/
    }
  };


static struct transition trans5[2][3] =
  {
    { /* ================ Time is a point ============================= */
	{ 0 },					/*  <l, inc>		*/
	{ 2, P1, P3 },				/*  <l, std>		*/
	{ 1, P6 },				/*  <l, dec>		*/
    },
    { /* ================ Time is an interval ========================= */
	{ 0 },					/*  <l, inc>		*/
	{ 1, I1 },				/*  <l, std>		*/
	{ 0 }					/*  <l, dec>		*/
    }
  };


static struct transition trans6[2][3] =
  {
    { /* ================ Time is a point =============================== */
	{ 1, P4 },				/*  <l, inc>		*/
	{ 3, P1, P2, P3 },			/*  <l, std>		*/
	{ 1, P6 },				/*  <l, dec>		*/
    },
    { /* ================ Time is an interval =========================== */
	{ 0 },					/*  <l, inc>		*/
	{ 1, I1 },				/*  <l, std>		*/
	{ 0 }					/*  <l, dec>		*/
    }
  };

/*  Note 1:  this transition invalid unless landmark creation turned off. */



genpvals()
{
    register int   i, j;
    register struct transition  *transp;

    struct param   *p;
    struct qval    *qvalp;
    int   tpi;


    pvp   = &pval[0];
    tpi   = curstatep->time & 01;
    qvalp = curstatep->qvalp;

    /*  Loop over all parameters and generate pvals for each  */
    for (i = 0;  i < nparams;  i++, qvalp++)
    {
	p = &param[i];
	p->qspacep = qvalp->qspacep;	/* (so that prpvals works right) */
	p->pvalp  = pvp;

#ifdef TRACE
	if (trace & TR_PVALS)
	    printf("genpvals: parameter %s, qmagx=%d, # landmarks=%d\n",
		p->namep, qvalp->qmagx, qvalp->qspacep->nlmarks);
#endif

	if (p->independent)
	{   /*  This is an independent parameter, so use its initial value */
	    pvp->used = pvp->bad = FALSE;
	    pvp->ncons = 0;
	    pvp->qmagx    = qvalp->qmagx;
	    (pvp++)->qdir = qvalp->qdir;
	    continue;
	}
	    
	if (interval(qvalp->qmagx))
	{

	    /*  Magnitude is an interval, so test for adjacency to minf/inf  */
	    if (qvalp->qmagx == 1)
	    {
		if (qvalp->qspacep->minf)
		    /*  qmag is adjacent to minf (-infinity)  */
		    transp = &trans1[tpi][qvalp->qdir];
		else if ((qvalp->qspacep->nlmarks == 2) &&
			 (qvalp->qspacep->inf))
		    /*  qmag is adjacent to inf (+infinity)  */
		    transp = &trans2[tpi][qvalp->qdir];
		else
		    /*  qmag is not adjacent to minf or inf  */
		    transp = &trans3[tpi][qvalp->qdir];
	    }
	    else if (qvalp->qmagx == ((qvalp->qspacep->nlmarks << 1) - 3))
	    {
		if (qvalp->qspacep->inf)
		    /*  qmag is adjacent to inf (+infinity)  */
		    transp = &trans2[tpi][qvalp->qdir];
		else
		    /*  qmag is adjacent to an upper bound other than inf  */
		    transp = &trans3[tpi][qvalp->qdir];
	    }
	    else
		/*  qmag is somewhere in the middle of its qspace  */
		transp = &trans3[tpi][qvalp->qdir];
		    
	}
	else
	{
	    /*  Magnitude is a point, so test for upper/lower limits  */
	    if (qvalp->qmagx == 0)
		    /*  qmag is at lower limit of qspace  */
		    transp = &trans4[tpi][qvalp->qdir];
	    else if (qvalp->qmagx == ((qvalp->qspacep->nlmarks - 1) << 1))
		    /*  qmag is at upper limit of qspace  */
		    transp = &trans5[tpi][qvalp->qdir];
	    else
		    /*  qmag is somewhere in the middle of its qspace  */
		    transp = &trans6[tpi][qvalp->qdir];
	}

	/*  'transp' now points to the set of legal transitions.  */
	if ((p->npvals = transp->npvals) == 0)
	{
	    printf("Invalid transition attempted on param %s\n", p->namep);
	    prstate(curstatep);
	    abort();
	}

	/*  Loop over each transition and install the pvals  */
	for (j = 0;  j < transp->npvals;  j++)
	{
	    pvp->used = pvp->bad = FALSE;
	    pvp->ncons = 0;
	    pvp->qmagx    = qvalp->qmagx + transp->qval[j].qmagdiff;
	    (pvp++)->qdir = transp->qval[j].qdir;
	}
    }

    if (pvp > pvpmax)
    {   printf("Exceeded limit of %d pvals\n", MAXPVALS);
        abort();
    }
    else if (pvp > pvphigh)  pvphigh = pvp;	/* (for execution stats) */

#ifdef TRACE
    if (trace & TR_PVALS)
	prpvals();
#endif
}
