;;; -*- Package: QSIM -*-
; copyright 1991 Benjamin Kuipers
(in-package :qsim)

;;;
;;;   Chatter Box Abstraction Example
;;;
;;;  A discussion of this model is provided below.  It is the best example
;;;  of how chatter box abstraction can help.  Without chatter box abstraction,
;;;  an intractable bvehavior tree results due to standard chatter as well as
;;;  chatter around landmarks.  Since chatter around zero occurs in the
;;;  behavior of XPP, ignore qdirs cannot be used to eliminate this chatter.
;;;
;;;  The behavior tree resulting when chatter box abstraction occurs is still
;;;  not completely tractable.  THis is due to occurrence branching.
;;;  Techniques to eliminate this branching have also been developed and
;;;  they will be distributed shortly.


; This tries to handle the van der Pol equation:  x'' + (x^2-1) x' + x = 0

; Modeling assumptions:
;  - Using U+(x) to augment x^2-1.
;
; Problems:
;  - HOD derivation can't derive any HODs, probably due to the U+.
;    Adding the auxiliary U+(x) = x^2 - 1 constraints got the HOD
;    expressions derived, but didn't help simulation.
;  - Branching seems to be due to chattering on x''  around zero.
;  - Dependencies of HOD expressions:    (chatters?)
;      sd2(U)   <- x, xp                    no
;      sd2(XPP) <- x, xp, u                 yes
;      sd2(EXP) <- x, xp, xpp, u            yes

(define-QDE van-der-Pol-Equation
  (text "van der Pol Equation:  x'' + (x^2-1) x' + x = 0")
  (quantity-spaces
   (x     (minf -1 0 +1 inf)  "x")
   (xp    (minf 0 inf)          "x'")
   (xpp   (minf 0 inf)          "x''")
   (u     (minf -1 0 inf)       "U+(x)")
   (exp   (minf 0 inf)          "U+(x) x'")
   (-exp  (minf 0 inf)          "-u x'")
   (-x    (minf 0 inf)          "-x")
   (xsq   (0 inf)               "x^2")
   (one   (0 1 inf)             "1 (constant)")

   )
  (constraints
   ((d/dt x xp))
   ((d/dt xp xpp))
   ((mult u xp exp))
   ((minus exp -exp))
   ((minus x -x))
   ((add xpp x -exp))
   ((U+ x u (0 -1))  (1 0) (-1 0) (minf inf) (inf inf))
   ((mult x x xsq))
   ((add u one xsq)   (-1 1 0))
   ((constant one))
   )
  (energy-constraint (x xp -x -exp))
  (elim-lm-chat (xpp 0))
  (layout (x xp xpp)
	  (u exp)
	  ())
  )


(defun vdPol (&key (state-limit 20) (ign nil) (clean t))
  (when clean (qsim-cleanup))
  (let* ((*perform-energy-analysis* nil)
	 (sim (make-sim :ignore-qdirs  (when ign '(xpp exp -exp))
			:no-new-landmarks '(xpp exp -exp)
			:unreachable-values '((xp minf inf) (xpp minf inf))
			:KET-constraint t
			:HOD-constraints t
			:sd3-constraints nil
			:analytic-functions-only t
			:state-limit state-limit))
	 (init (make-new-state :from-qde van-der-Pol-Equation
			       :sim sim
			       :assert-values (append  '((x  (1 nil))
							(xp (0 nil))
							(one (1 std)))
						      ;(unless ign
						;	'((xpp (nil std))))
						       )
			       :text "van der Pol Equation.")))

    (qsim init)
    (qsim-display init)
    ))
