;;; -*- Mode:Lisp; Package:QSIM; Syntax:COMMON-LISP; Base:10 -*-
(in-package :qsim)

#|

CHATTER BOX ABSTRACTION

Introduction

Chatter box abstraction is an abstraction technique designed to
eliminate chatter on the fly during simulation.  In its basic form it
does not require the user to have any knowledge about which variables
chatter.


Using Chatter Box Abstraction

To activate chatter box abstraction, set the following special variable:

       *perform-chatter-abstraction*

This special variable is automatically set when this file is loaded.


To trace the abstraction process set:

       *trace-chatter-abstraction*.

Chatter box abstraction is implemented as a global filter.  Upon
entering a potentially chattering region of the state space (this will
be defined later), chatter box abstraction will perform a limited
envisionment from the state being filtered.  A limited envisionment is
one in which the envisionment is limited to a specified region of the
state space.  States which exit this region are suspended from the
envisionment simulation.  For standard chatter box abstraction, the
region is defined as the qmag of all potentially chattering variables
and the current qval of all other variables.  Thus, it allows the qdir
of chattering variables to vary during the envisionment.  If anything
else changes, then the state is suspended.  If chatter around a
landmark is being elimininated, then the qmag of the chattering
variable is also allowed to vary.

Once the envisionment is completed, the envisionment graph is analyzed
to determine if it exhibited chatter.  If it did, then an abstract
state is created which is an abstraction of the entire envisionment
graph.  This state will be a normal state except the qdir for
chattering variables will be a list of the possible values that the
qdir could take (e.g.  (inc std dec) in most cases).  When chatter
around a landmark is eliminated, it is possible for the qmag of the
chattering variable to also be abstracted.  This results in an open
interval qmag that extends over the landmark around which the variable
chatters.  For example, if variable A has the qspace (minf 0 inf) and
it is chattering around zero, the resulting abstracted qmag for A
might be (minf inf).  The states which exited the region will then
be used to create the successors of the chattering region.


Viewing the Results

A state which has been abstracted by chatter box abstraction will be
displayed as a box in the behavior tree.  An abstracted qdir in a
behavior plot is displayed as a combination of the symbols for the
valid qdirs.  If the possible qdirs is (inc std dec) then a two headed
arrow is displayted.  An abstracted qmag (resulting when chatter
around alandmark is eliminated) is identified by a line of dots
extending over the interval of the qmag.

The behavior of the system within the envisionment can be viewed by
displaying the spanning tree of the envisionment graph for the limited
envisionment.  This is done by selecting the QSIM Extensions (E)
option from the main menu and then selecting the Chatter Box Display
option.  The user is then prompted to enter the state name of the
chatter box that is of interest.  For additional information about a
chatter box, execute (describe-state <state-name>) at the lisp prompt.


Chatter around Landmarks

The phenomenon of chatter discussed above centers on the chattering
of a variable within an open interval.  Chatter can also occur around
a landmark.  In this case, the qmag for the chattering variable will
move through the landmark and cause branching each time.  This is
coupled with chatter above and below the landmark resulting in some
really ugly branching.

Chatter around landmarks has been divided into two categories: chatter
around user specified landmarks, and chatter around QSIM introduced
landmarks.  Chatter sround user specified landmarks is eliminated by
specifying the variable and the landmark in the qde.  The following
entry is used in the qde:

   (elim-lm-chat (<var> <lm> <lm> ..) (<var> <lm> <lm> ...) ...)

For each variable, a list of landmarks aroudn which chatter occurs is
specified.  You can also add this information to the qde by calling
the function:

   (eliminate-chatter-around-lmark <qde-or-state> <var> <lm>)

an entry will be made on the QDE which will tell the code to eliminate
chatter around LM in the qspace of VAR.

To eliminate chatter around newly introduced landmarks there are one
of two choices:

   1.  Set the special variable *elim-chat-around-all-new-lms*
       to T.  This eliminates chatter around all new landmarks.

   2.  If you want to select only certain qspaces to eliminate
       chatter around new landmarks set *elim-chat-around-all-new-lms*
       to NIL and identify the variables for which chatter should be
       eliminated.  This can be specified in the elim-lm-chat slot of
       the qde by simply listing the variable as follows:

          (elim-lm-chat <var> <var>)

       The specification of these variables can be interspersed
       between the entries discussed above.

       It is also possible to call the function

	       (eliminate-chatter-around-lmark <qde-or-state> <var>)

       for each variable.  Do not include a landmark in this
       call.

When chatter around a landmark is eliminated, the limited envisionment is
performed with an extended QMAG for the variables that might chatter.
The QMAG is extended above and below the landmark around which chatter
might occur.    When chatter occurs around a landmark the display
contains dotted lines indicating the range of the qval at a particular
point. 

Large Models

When simulating large models, it is very possible that the amount of
chatter will result in very large sub-envisionment graphs.  In
general, the order of magnitude of the size of the sub-envisionment is
3 to the n where n is the number of chattering variables.  The
envisionment will frequently be smaller than this, but this can give
you an idea of how large the envisionment can get.  Due to various
other factors it can get larger than this.  The size of the
sub-envisionment graphs can lead to garbage collection problems.  To
help with this problem, the special variable

	*cleanup-chatter-env-graphs*

has been added.  When this is set to t, the envisionment graphs in the
sub-envisionment will be eliminated after they are used to create the
abstraction.  This allows the garbage collector to reclaim this space.
Obviously, you will not be able to view the results of each
envisionment independently after the simulation is completed.

If the envisionment graphs are large enough, space can be a problem
even with this special variable set.  One cause of the space useage is
the number of inconsistent states within the focused envisionment.
The special variable

	*cleanup-chatter-incon-states*

will cleanup inconsistent states during the limited envisionment.
These states are not used when abstracting the envisionment graph into
a single abstract state.  The state structures are eliminated so that
the space can be reclaimed via garbage collection.  This variable is
set to T as a default.  Only states that are marked inconsistent by
the function MARK-INCONSISTENT-STATE are cleaned up.  Thus, if a
filter marks a states as inconsistent simply by modifying the status,
it will not be cleaned-up.  This is important because states that exit
the chattering region are filtered by marking them as iconsistent.

|#


(defparameter *perform-chatter-abstraction* t)  ;; Controls when chatter
						;; abstraction is performed.
(defparameter *elim-chat-around-all-new-lms* t) ;; Causes chatter elimination filter to
						;; eliminate chatter around all
						;; newly created landmarks.
(defparameter *cleanup-chatter-env-graphs* nil) ;; When t, it will
						;; cause the sub envisionment
						;; graphs to be cleaned up after
						;; the abstraction is performed

(defparameter *cleanup-chatter-incon-states* t) ;; This variable will cleanup inconsistent states
                                                ;; in the focused envisionment during the 
                                                ;; simulation.  It destroys the state structure
                                                ;; allowing the space to be relaimed.
                                                ;; For large envisionment graphs
                                                ;; the number of inconsistent states can be
                                                ;; quite large and this can lead to space
                                                ;; problems.

(defparameter *cleanup-inconsistent-states* nil)  ;; DO NOT CHANGE THIS VARIABLE
                                                  ;; When *cleanup-chatter-incon-states* is set
                                                  ;; to T, then this variable is lexically
                                                  ;; bound within the sub-envisionment to cause
                                                  ;; the reclamation of space from inconsistent
                                                  ;; states.

(defparameter *trace-chatter-abstraction* nil)
(defparameter *trace-region-filter* nil)
(defparameter *trace-sub-envisionment* nil)

;; Taken from behavior aggregation code

(defother state abstraction-of)   ;; This slot is used when a state is
				  ;; an abstraction of a set of other
				  ;; distinct states.  It will list the
				  ;; states which are abstracted by this
				  ;; state.

(defother state abstractions)     ;; This slot contains a list of the
				  ;; states which are abstractions of
				  ;; this state.  It is the inverse
				  ;; pointer of the abstraction-of slot.



(defother sim exits-region)       ;; Keeps track of states in a 
                                  ;; sub-envisionment which exit the designated
                                  ;; region.
(defother qde elim-lm-chat)       ;; A list of lists.  Each entry is a 
                                  ;; variable followed by landmarks from the
                                  ;; original qspace.  Chatter around thse
                                  ;; landmarks is eliminated.
(defother state chatter-box-info) ;; Keeps track of information about a
                                  ;; chatter box state.


(pushnew 'chatter-box-info *state-other-dont-copy*)
(pushnew 'abstraction-of *state-other-dont-copy*)
(pushnew 'abstractions *state-other-dont-copy*)

(pushnew 'filter-from-agenda *statuses-to-filter-from-agenda*)
(pushnew 'exits-region *statuses-to-filter-from-agenda*)

(add-global-state-filter 'chatter-elim-filter)



;;;
;;;  CHATTER-BOX Structure
;;;
;;;  This structure is used to store information about a chatter box
;;;  on the state-other slot of the abstracted state.  I have placed it in
;;;  a structure so as not to clutter the state-other slot when
;;;  users are looking at it.  I have added functions state-<slot-name> so
;;;  that each of these slots can be reached directly.

(defstruct (chatter-box-info (:print-function chatter-box-info-printer))
  state
  chattering-vars   ;; a list of the variables that chattered
  env-root          ;; root of the sub-envisionment graph
  region            ;; region over which the envisionment was performed
  states-replaced   ;; states replaced by the abstract state
  env-size)         ;; number of states in the sub-envisionment

(defun chatter-box-info-printer (ch-box stream ignore &key (detail *detailed-printing*))
  (declare (ignore ignore))
  (when detail
    (format stream "~%Chatter-box:  ~a ~
                  ~%  Chattering vars    : ~a~
                  ~%  Envisionment root  : ~a~
                  ~%  Replaces state     : ~a~
                  ~%  Size of Env graph  : ~a"
	    (chatter-box-info-state ch-box)
	    (chatter-box-info-chattering-vars ch-box)
	    (chatter-box-info-env-root ch-box)
	    (chatter-box-info-states-replaced ch-box)
	    (chatter-box-info-env-size ch-box))
    (when (chatter-box-info-region ch-box)
      (format stream "~%  Region             : ")
      (pprint-at (remove-if #'(lambda (pair)
				(atom (cadadr pair)))
			    (chatter-box-info-region ch-box)) 23)))
  (unless detail
    (format stream "(Chattering Vars ~a) (Env-Root ~a) (Replaces-state ~a) (Env-size ~a) (Region ~a)"
	    (chatter-box-info-chattering-vars ch-box)
	    (chatter-box-info-env-root ch-box)
	    (chatter-box-info-states-replaced ch-box)
	    (chatter-box-info-env-size ch-box)
	    (remove-if #'(lambda (pair)
			   (atom (cadadr pair)))
		       (chatter-box-info-region ch-box)))))
    

;;;
;;;   MACROS
;;;

			       

(defmacro canonicalize-vars (vars)
  `(loop for var in ,vars
         collect (varname var) into ret-val
         finally (return ret-val)))

;;;  The following macros provide direct access to slots in a chatter-box-info
;;;  structure from the state.

(defmacro state-chattering-vars (state)
  `(and (state-chatter-box-info ,state)
    (chatter-box-info-chattering-vars
     (state-chatter-box-info ,state))))

(defmacro state-env-root (state)
  `(and (state-chatter-box-info ,state)
    (chatter-box-info-env-root
     (state-chatter-box-info ,state))))

(defmacro state-region (state)
  `(and (state-chatter-box-info ,state)
    (chatter-box-info-region
     (state-chatter-box-info ,state))))

(defmacro state-states-replaced (state)
  `(and (state-chatter-box-info ,state)
    (chatter-box-info-states-replaced
     (state-chatter-box-info ,state))))

(defmacro state-env-size (state)
  `(and (state-chatter-box-info ,state)
    (chatter-box-info-env-size (state-chatter-box-info ,state))))
	
  


(defmacro abstracted-qdir (qdir)
  `(listp ,qdir))

(defmacro original-lm (lm)
  `(null (lmark-where-defined ,lm)))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;   CLOSED INTERVALS
;;;
;;;   The following code is used to allow for closed intervals in QSIM.  This is
;;;   done by adding a third element to an interval qmag.  The third element
;;;   can be one of the following:  :oo, :cc, :co, or :oc.  THe first letter stands
;;;   for the lower boundary while the second for the upper.  "c" stands for closed
;;;   while "o" stands for open.  If the third entry does not exist or is nil,
;;;   then both boundaries are open.


;;;
;;;  The following four macros are boolean macros that determine if a particular
;;;  boundary is open or closed in a qmag.  If the qmag is a point then the
;;;  boundary is considered closed.

(defmacro upper-boundary-closed? (qmag)
  `(or (qmag-point-p ,qmag)
    (member (third ,qmag) (list :cc :oc))))

(defmacro lower-boundary-closed? (qmag)
  `(or (qmag-point-p ,qmag)
    (member (third ,qmag) (list :cc :co))))

(defmacro upper-boundary-open? (qmag)
  `(and (qmag-interval-p ,qmag)
    (or (null (third ,qmag))
     (member (third ,qmag) (list :oo :co)))))

(defmacro lower-boundary-open? (qmag)
  `(and (qmag-interval-p ,qmag)
    (or (null (third ,qmag))
     (member (third ,qmag) (list :oo :oc)))))


;;; The following four macros determine the relationship between
;;; either the upper or the lower boundaries of two qmags.


(defmacro lower-boundary-lt (qmag1 qmag2 qspace)
  `(let ((lower1 (lower-lmark ,qmag1))
	 (lower2 (lower-lmark ,qmag2)))
    (if (and (lower-boundary-closed? ,qmag1)
	     (lower-boundary-open? ,qmag2))
	(landmark-le lower1 lower2 ,qspace)
	(landmark-lt lower1 lower2 ,qspace))))

(defmacro lower-boundary-le (qmag1 qmag2 qspace)
  `(let ((lower1 (lower-lmark ,qmag1))
	 (lower2 (lower-lmark ,qmag2)))
    (if (and (lower-boundary-open? ,qmag1)
	     (lower-boundary-closed? ,qmag2))
	(landmark-lt lower1 lower2 ,qspace)
	(landmark-le lower1 lower2 ,qspace))))


(defmacro upper-boundary-gt (qmag1 qmag2 qspace)
  `(let ((upper1 (upper-lmark ,qmag1))
	 (upper2 (upper-lmark ,qmag2)))
    (if (and (upper-boundary-closed? ,qmag1)
	     (upper-boundary-open? ,qmag2))
	(landmark-ge upper1 upper2 ,qspace)
	(landmark-gt upper1 upper2 ,qspace))))

(defmacro upper-boundary-ge (qmag1 qmag2 qspace)
  `(let ((upper1 (upper-lmark ,qmag1))
	 (upper2 (upper-lmark ,qmag2)))
    (if (and (upper-boundary-open? ,qmag1)
	     (upper-boundary-closed? ,qmag2))
	(landmark-gt upper1 upper2 ,qspace)
	(landmark-ge upper1 upper2 ,qspace))))


(defmacro lm-included-in-lower (lm qmag qspace)
  `(if (lower-boundary-closed? ,qmag)
      (landmark-ge ,lm (car ,qmag) ,qspace)
      (landmark-gt ,lm (car ,qmag) ,qspace)))

(defmacro lm-included-in-upper (lm qmag qspace)
  `(if (upper-boundary-closed? ,qmag)
    (landmark-le ,lm (cadr ,qmag) ,qspace)
    (landmark-lt ,lm (cadr ,qmag) ,qspace)))


;;;
;;;  UTILITIES
;;;

(defun abs-qdir (qdir)
  "Returns T if the QDIR is abstracted."
  (and qdir (listp qdir)))

(defun abs-qmag (qmag qspace)
  (and (qmag-interval-p qmag)
       (not (lmark-equal (succ (car qmag) qspace)
			 (cadr qmag)))))

(defun varnames (vars)
  "Returns the variable names (i.e. not structures) for a list of either variables or namees."
  (loop for var in vars
	collect (varname var) into ret-val
	finally (return ret-val)))

(defun exiting-states (sim-or-state)
  "Returns a list of the states exiting a limited envisionment graph."
  (let ((sim (if (sim-p sim-or-state) sim-or-state
		 (state-sim sim-or-state))))
    (mapcar #'car (sim-exits-region sim))))



(defun lmark-in-qmag (lm qmag qspace)
  (if (atom qmag)
      (robust-lmark-equal lm qmag)
      (and (landmark-gt lm (lower-lmark qmag) qspace)
	   (landmark-lt lm (upper-lmark qmag) qspace))))


(defun qdir-equal (qd1 qd2)
  (not (set-exclusive-or (listify qd1)
			 (listify qd2))))


(defun QVAL-EQUAL (qv1 qv2)
  "True if two qvals have equal qmags and qdirs."
  (and (qdir-equal (qval-qdir qv1) (qval-qdir qv2))
       (qmag-equal (qval-qmag qv1) (qval-qmag qv2))))


(defun perform-chatter-elim-on-state? (state)
  "Returns T if the chatter-elimination filter should be run on STATE."
  (and (not (intersection (state-status state)
			  '(inconsistent quiescent)))
       (time-interval-state state)  
       ))

(defun mark-state-uninteresting (state reason)
  "Marks a state as uninteresting.  CUrrently it uses the INCONSISTENT status since this
is all QSIM works with."
  (pushnew 'inconsistent (state-status state))
  (pushnew (list 'inconsistent reason)
	   (state-status state)))

(defun chatter-vars (state)
  "Returns a list of varaibles which have been abstracted for chattering in this state by
looking at the qvals."
  (loop for (var . qval) in (cdr (state-qvalues state))
	when (listp (qdir qval))
	collect var into chvars
	finally (return chvars)))

(defun chatter-box-state? (state)
  "Returns T if STATE is a chatter box."
  (if (state-chatter-box-info state) t nil))

(defun chatter-abstraction-of (state)
  (car (alookup 'chatter-abstraction (state-abstraction-of state))))

(defun eliminate-chatter-around-lmark (qde-or-state var &optional (lm nil))
  "Used to add variables to the qde-elim-lm-chat slot.  If you want to eliminate
chatter around an existing landmark then call this function with the variable and
the landmark.  If you wish to eliminate chatter around newly created landmarks for
this variable (i.e. when *elim-chat-around-all-new-lms* is nil), then simply
call this function with just a variable and no landmark."
  (let* ((qde (if (state-p qde-or-state)
		  (state-qde qde-or-state)
		  qde-or-state))
	 (entry (car (member var (qde-elim-lm-chat qde)
			     :test #'(lambda (var elem)
				       (and (listp elem)
					    (equal var (car elem)))))))
	 (lm (if (lmark-p lm) (lmark-name lm) lm)))
    (unless (assoc var (qde-var-alist qde))
      (cerror "Continue" "~a is not a variable in the qde ~a" var qde)
      (return-from eliminate-chatter-around-lmark))
    ;;(assoc var (qde-elim-lm-chat qde))
    (cond ((and entry lm)
	   (pushnew lm (cdr entry)))
	  (lm
	   (pushnew (list var lm)
		    (qde-elim-lm-chat qde)))
	  (t (pushnew var (qde-elim-lm-chat qde))))))


(defun elim-chat-around-zero? (qde-or-state var)
  (let* ((qde (if (state-p qde-or-state)
		  (state-qde qde-or-state)
		  qde-or-state))
	 (entry (car (member var (qde-elim-lm-chat qde)
			     :test #'(lambda (var elem)
				       (and (listp elem)
					    (equal var (car elem))))))))
    (equal (cadr entry) 0)))

(defun describe-state (s)
  "This function describes the chatter abstraction information for STATE."
  (let ((state (get-state s)))
    (cond ((state-chatter-box-info state)
	   (chatter-box-info-printer (state-chatter-box-info state) t nil :detail t))
	  (t (format t "~%State ~a is not a chatter box." (state-name state))))))


(defun multiple-behaviors? (state)
  (when state
    (or (cdr (successor-states state))
	(multiple-behaviors? (car (successor-states state))))))
;;;
;;; QMAG-INCLUDES
;;;
;;;  Returns true when the value of QMAG1 includes the value of QMAG2.
;;;  The first QMAG does not have to be an interval.  If it is a
;;;  point and it is equal to QMAG2 then it will return T.

(defun qmag-includes (qmag1 qmag2 qspace)
  "Returns qmag1 if it includes qmag2"
  (and (lower-boundary-le qmag1 qmag2 qspace)
       (upper-boundary-ge qmag1 qmag2 qspace)))

(defun qdir-includes (qdir1 qdir2)
  (loop for dir in (listify qdir2)
	when (not (member dir (listify qdir1)))
	return nil
	finally (return qdir1)))
	

(defun qmags-intersect (qmag1 qmag2 qspace)
  (or (qmag-includes qmag1 qmag2 qspace)
      (qmag-includes qmag2 qmag1 qspace)))


;;;
;;;  DEVELOPMENT UTILITIES
;;;

(defun resimulate-state (state)
  "This function will run the chatter elimination filter on STATE and
then resimulate it.  It will return its successors."
  (let ((*current-sim* (state-sim state)))
    (with-bindings-from-sim *current-sim*
      (setf (state-status state) '(GF COMPLETE))
      (chatter-elim-filter state)
      (unless (and (chatter-vars state)
		   (time-interval-state state))
	(qsimulate-state state (state-sim state)))
      (successor-states state))))
  
(defun chattering-var-values (state &key (test #'abs-qdir))
  (format t "~%State ~a:" state)
  (loop for (var . qval) in (cdr (state-qvalues state))
	when (funcall test (qdir qval))
	do (format t "~%   ~a: ~a" var qval)))



  

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                                                                       ;;;
;;;                                                                       ;;;
;;;      CHATTER-ELIM-FILTER                                              ;;;
;;;                                                                       ;;;
;;;                                                                       ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun chatter-elim-filter (state)
  (when (and *perform-chatter-abstraction*
	     (perform-chatter-elim-on-state? state))
    (multiple-value-bind (chvars chvars-region integral-chat-zero)
	;; Three values are returned by FIND-POSS-CHATTER-VARS
	;;    chvars-region      An alist of variables and qvals.  These variables will be
	;;                       allowed to chatter over the region of the included qval.
	;;                       The qval is a list not a structure.
	;;    chvars             A list of the variables that might be chattering.
	;;    integral-chat-zero A list of the variables that are integrals of variables
	;;                       that are allowed to chatter around zero.  These variables
	;;                       will be allowed to exhibit qdir chatter.
	(find-poss-chatter-vars state)
      (when chvars
	(let* ((*perform-chatter-abstraction* nil)
	       (all-chvars (append chvars integral-chat-zero))
	       ;; Define the entire region for the chatter box.
	       (region (define-chatter-box state chvars-region))
	       ;; Perform sub-envisionment for each set of chattering variables
	       (sub-env-graph (perform-sub-envisionment state region all-chvars))
	       ;; Create a a function that determines if two states are equivalent
	       ;; with respect to the region.
	       (state-reg-equiv-func (create-region-equiv-function region))
	       ;; Create a single abstract state summarizing the sub-envisionment.
	       (abs-state 
		;; If chatter around landmarks is eliminated, it is possible that the
		;; sub-envisionment will abstract the behavior of a variable around a landmark
		;; when no chatter occurs.  This results in an abstracted QMAG without an
		;; abstracted QDIR.  This checks for this and will prune the envisionment
		;; graph to eliminate this abstraction.
		(loop for abs-state = (create-chatter-abstraction-state sub-env-graph state)
		      for over-abs = (get-over-abstracted-vars abs-state all-chvars)
		      when (null over-abs) return abs-state
		      do (correct-for-over-abstraction sub-env-graph abs-state over-abs state))))
	  (when abs-state
	    (let* ((succs (build-chatter-successor-states
			   state (sim-exits-region (state-sim sub-env-graph))
			   all-chvars state-reg-equiv-func))
		   ;; A list of the variables that exhibited chatter
		   (actual-chvars (chatter-vars abs-state)))
	      ;; If there are chattering variables, insert the abstract state with its
	      ;; successors into the tree and modify the agenda accordingly.
	      (when actual-chvars
		(record-chatter-box-info abs-state sub-env-graph region state actual-chvars)
		(reset-q2-info-in-succs succs state)
		(insert-state-and-succs-in-tree state abs-state succs)
		(mark-state-uninteresting state "Chatter-eliminated"))
	      ;; For large models it may be desireable to get rid of the
	      ;; envisionment graph.
	      (when *cleanup-chatter-env-graphs*
		(sim-cleanup (state-sim sub-env-graph)))))
	  (trace-chatter-elim-filter state sub-env-graph (when abs-state
							   (successor-states abs-state))
				     abs-state)
	  ))))
  state)




(defun record-chatter-box-info (abs-state env-root region states-replaced chvars)
  "Records information about the chatyter box int eh chatter box info structure."
  (setf (state-chatter-box-info abs-state)
	(make-chatter-box-info :state abs-state
			       :env-root env-root
			       :region region
			       :states-replaced states-replaced
			       :chattering-vars chvars
			       :env-size (when env-root
					   (sim-state-count (state-sim env-root))))))




(defun reset-q2-info-in-succs (succs state)
  "Resets Q2 info in the successors of the chatter box since it was eliminated
to save space during the envisionment."
  (loop for s in succs
	do
	(setf (state-bindings s)
	      (copy-tree (state-bindings state)))
	(setf (state-eqn-index s)
	      (copy-eqn-index (state-eqn-index state)))))

	  
(defun define-chatter-box (state chvars-region)
  (canonicalize-qvals (mapcar #'(lambda (qval-pair)
				  (or (assoc (car qval-pair) chvars-region)
				      qval-pair))
			      (cdr (state-qvalues state)))
		      state))



(defun find-poss-chatter-vars (state)
  "This function identifies the state space over which the sub-envisionment will be
performed.  It determines which are variables are able to chatter and whether or not
they might chatter around a landmark."
  (let ((chvars (when (get-chvars-classes (state-qde state))
		  (reduce #'union (get-chvars-classes (state-qde state)))))
	chvars-region chvars-list integrals-of-zero-chat)
    (loop for var in chvars
	  for qval = (qval var state)
	  for qdir = (qdir qval)
	  for qmag = (qmag qval)
	  for qspace = (qspace var state)
	  do (let (upper-lm lower-lm)
	       ;; Variables can only chatter when they are changing in an open interval.
	       (when (and (qmag-interval-p qmag)  
			  (or (member qdir '(inc dec))
			      (listp qdir)))
		 ;; If the one of the end points of the interval upon which the variables
		 ;; lies is a landmark around which chatter is to be eliminated, then
		 ;; determine and upper and a lower bound for the variable.  This bound will
		 ;; be the next landmark in each direction around which chatter is not
		 ;; eliminated.
		 (setf lower-lm
		       (find-if #'(lambda (lm)
				    (not (chatter-lm lm var state)))
				(member (car qmag) (reverse qspace))))
		 (setf upper-lm
		       (find-if #'(lambda (lm)
				    (not (chatter-lm lm var state)))
				(member (cadr qmag) qspace)))
		 (pushnew var chvars-list)
		 (push (list var
			     (list (list lower-lm upper-lm)
				   '(inc std dec)))
		       chvars-region))))
    ;; Identify variables whose integrals are being allowed to chatter around 
    ;; zero.  These variables must also be allowed to chatter.  
    (loop for con in (qde-constraints (state-qde state))
	  for varnames = (constraint-varnames con)
	  for deriv-var = (cadr varnames)
	  for qspace = (qspace deriv-var state)
	  when (equal (constraint-type con)
		      (contype-from-name 'd/dt))
	  do (let ((entry (cadr (assoc deriv-var chvars-region))))
	       (when (and (car entry)
			  ;; Determine if zero is within the range that the variables
			  ;; is allowed to take.
			  (or (landmark-lt (caar entry) *zero-lmark* qspace)
			      (landmark-gt (cadar entry) *zero-lmark* qspace)))
		 (pushnew (car varnames) integrals-of-zero-chat)
		 (push (list (car varnames)
			     (list (qmag (qval (car varnames) state)) '(inc std dec)))
		       chvars-region)
		 )))
    (values chvars-list (nreverse chvars-region) integrals-of-zero-chat)))



(defun chatter-lm (lm var state)
  "This function determines if LM is a landmark around which chatter should be eliminated.
Chatter can be eliminated around newly introduced landmarks with a special variable
*elim-chat-around-all-new-lms* or around existing landmarks by specifying it in
the qde."
  (let ((elim-lm-chat (qde-elim-lm-chat (state-qde state))))
    (or (and (not (original-lm lm))
	     (or *elim-chat-around-all-new-lms*
		 (member var elim-lm-chat)))
	(elim-chat-orig-lm lm elim-lm-chat var))))



(defun elim-chat-orig-lm (lm elim-lm-chat var)
  (let ((lms (car (member var elim-lm-chat
			  :test #'(lambda (var entry)
				    (or (equal var entry)
					(and (listp entry)
					     (equal var (car entry)))))))))
    (and (listp lms)
	 (member (lmark-name lm) lms))))

	 
(defun pred-at-lm (var lm pred)
  "Returns T if the PRED state has value LM for VAR or if its qmag is a qmag abstraction
that includes LM."
  (when pred
    (let ((pred-qmag (qmag (qval var pred)))
	  (pred-qspace (qspace var pred)))
      (or (equal pred-qmag lm)
	  (and (member lm (intervening-lmarks (lower-lmark pred-qmag)
					      (upper-lmark pred-qmag)
					      pred-qspace)))))))

		      


(defun intervening-lmarks (lower-lm upper-lm qspace)
  "Returns all of the landmarks in QSPACE that are between LOWER-LM and UPPER-LM."
  (loop for lm in (cdr (member lower-lm qspace :test #'lmark-equal))
	if (lmark-equal lm upper-lm)
	return interv-lms
	else
	collect lm into interv-lms))

	       
(defun get-chvars-classes (qde)
  "This function gets the chattering variable classes for the QDE.  If they have not been
calculated then it calls the function to calculate them."
  (unless (qde-derived-chvar-classes qde)
    (identify-chatter qde)
    (mapcar #'varnames (qde-derived-chvar-classes qde)))
  (mapcar #'varnames (qde-derived-chvar-classes qde)))



;;;
;;; INSERT-STATE-AND-SUCCS-IN-TREE
;;;

(defun insert-state-and-succs-in-tree (orig-state abs-state succs)
  "This function will insert ABS-STATE in the tree, add SUCCS to its
successors slto and add the successors to the agenda.  It also will
run the global filters on the successors."
  (replace-state-in-tree orig-state abs-state)
  (pushnew 'complete (state-status abs-state))
  (pushnew 'GF (state-status abs-state))
  (when succs
    (setf (state-successors abs-state)
	  (cons 'successors succs)))
  (mapcar #'(lambda (succ)
	      (set-justification abs-state succ)
	      (setf (state-predecessors succ) (list abs-state))
	      (push 'complete (state-status succ))
	      )
	  succs)
  ;; Run the global filters on both the abstract state and the new successors
  (let ((new-states (cons abs-state succs))
	(*perform-chatter-abstraction* nil))
    (dolist (ns new-states)
      (apply-global-state-analysis ns)
      (apply-global-state-filters  ns)
      (pushnew 'GF (state-status ns))))
  (pushnew abs-state (sim-states (state-sim orig-state)))
  (add-states-to-agenda succs (state-sim orig-state))
  succs
  )


  
(defun set-justification (pred state)
  "Sets the justification slot of STATE to be the correct value given PRED as a
predecessor."
  (setf (state-justification state)
	(cond ((incomplete-p pred)
	       (list 'one-of-several-completions-of pred))
	      ((cdr (successor-states pred))
	       (list 'one-of-several-successors-of pred (successor-states pred)))
	      (t (list 'unique-successor-of pred)))))

(defun replace-state-in-tree (cur-state new-state)
  (let ((pred (state-predecessor cur-state)))
    (when (null pred) (cerror "Continue" "State ~a does not have a predecessor." cur-state))
    (setf (state-successors pred)
	  (substitute new-state cur-state (state-successors pred)))
    (setf (state-time new-state)
	  (state-time cur-state))
    (setf (state-justification new-state)
	  (if (cdr (successor-states pred))
	      (list 'one-of-several-successors-of pred (successor-states pred))
	      (list 'unique-successor-of pred)))))
    


;;;
;;;  BUILD-CHATTER-SUCCESSOR-STATES
;;;

(defun build-chatter-successor-states (pred succs chvars reg-equiv-states)
  "This function will receive a list of states that exited the chattering region
during the limited envisionment (SUCCS).  PRED is the abstract state describing the chattering
region.  SUCCS will be used to create the states that are successors of PRED.  States
that are equivalent with respect to the chattering variables are combined together."
  ;;  THe following steps are taken:
  ;;    -  Partition the states with respect to the chatter box region.
  ;;    -  Create an abstract state for each of these partitions.
  ;;    -  Identify variables that have been over-abstracted.  These are states with
  ;;       abstracted QMAGS but not abstracted QDIRs.
  ;;    -  Repartition each partition to eliminate these over-abstractions.
  ;;    -  Create an abstract state for each resulting abstraction.
  ;;  Return a list of the successor states.
  (let* ((init-partitions (partition succs :test reg-equiv-states))
	 (init-states (mapcar #'create-abstract-state-from-list init-partitions))
	 (over-abstracted-var-lists (mapcar #'(lambda (state) (get-over-abstracted-vars
							       state chvars))
					    init-states))
	 (re-partitions (mapcar #'(lambda (partition var-list)
				    (if var-list
					(partition partition
						   :test #'(lambda (s1 s2)
							     (states-equal-for-vars s1 s2 var-list)))
					(list partition)))
				init-partitions over-abstracted-var-lists))
	 (succ-states (mapcan #'(lambda (re-partition init-state)
				  (if (cdr re-partition)
				      (mapcar #'create-abstract-state-from-list re-partition)
				      (list init-state)))
			      re-partitions init-states))
	 (final-partitions (apply #'append re-partitions)))
    (unless (eq (length final-partitions) (length succ-states))
      (cerror "Continue with Simulation" "Something is wrong in the partitioning code in ~
                   build-chatter-successor-states."))
    (mapcar #'(lambda (new-state partition)
		(change-time new-state pred)
		(when (cdr partition)
		  (record-chatter-box-info new-state nil nil partition
					   (chatter-vars new-state)))
		new-state)
	    succ-states final-partitions)))



(defun get-over-abstracted-vars (state chvars)
  "Gets a list of the varibales in STATE that have an abstracted QMAG without an
abstracted QDIR.  Only those variables in chvars are checked."
  (when state
    (remove-if-not #'(lambda (var)
		       (over-abstracted-var? state var))
		   chvars)))

(defun states-equal-for-vars (s1 s2 vars)
  "Returns T when S1 and S2 are equal with respect to the variables in VARS."
  (loop for var in vars
	when (not (qval-equal (qval var s1)
			      (qval var s2)))
	return nil
	finally (return t)))
  

(defun change-time (state pred)
  "Changes the time of STATE so that it is appropriately set to be a successor of PRED."
  (let* ((old-time (state-time pred))
	 (new-time (if (qpointp old-time)
		       (list (qmag old-time) (succ (qmag old-time) (time-qspace state)))
		       (cadr (qmag old-time)))))
    (setf (qmag (state-time state))
	  new-time)
    (when (and (listp new-time) (equal (cadr new-time) *inf-lmark*))
      (create-new-landmark-for-time state))
    (state-time state)
    ))




;;;
;;;  CHECK-AND-CORRECT-FOR-OVER-ABSTRACTION
;;;
;;;  Determines if the sub-envisionment over-abstracted a variable when trying to
;;;  eliminate chatter around a landmark.  This occurs when a the sub envisionment
;;;  is designed to eliminate chatter around alandmark and it doesn't occur.  It
;;;  is possible that the resulting abstract state will have an abstracted QMAG
;;;  without an abstracted QDIR for the variable.  When this occurs, the envisionment
;;;  is pruned to eliminate those states whose QMAG changes for the over abstracted
;;;  variable.

(defun correct-for-over-abstraction (sub-env-graph abs-state over-abstracted-vars orig-state)
  "This function checks to see if the abstract state has abstracted the qmag of a variable
when teh qdir does not chatter.  When this occurs, the abstraction must be undone.  This
is done by traversing the envisionment graph and restricting the region of those
over-abstracted variables."
  (when abs-state
    (let* ((var-qvalues (mapcar #'(lambda (var)
				    (cons var (qval var orig-state)))
				over-abstracted-vars)))
      (when over-abstracted-vars
	;;	(warn "Chatter abstraction envisionment graph over-abstracted the variables ~
	;;             ~%~a while filtering ~a. ~
	;;             ~%The root of the sub-envisionment graph is ~a. This is being corrected.  ~
	;;             ~%Please send this model to clancy@cs.utexas.edu as an example of this phenomenon.~
	;;             ~%Thank you." over-abstracted-vars orig-state sub-env-graph)
	(setf (sim-exits-region (state-sim sub-env-graph)) nil)
	;;  TO DO  The abstract state must also be modified to relfect the elimination
	;;         of this abstraction.
	(mod-env-graph-for-over-abstraction sub-env-graph var-qvalues)))))
  
(defun over-abstracted-var? (state var)
  "Returns T if VAR has an abstracted qmag and not an abstracted qdir in STATE."
  (let ((qval (qval var state))
	(qspace (qspace var state)))
    (and (not (abstracted-qdir (qdir qval)))
	 (abstracted-qmag (qmag qval) qspace))))

(defun abstracted-qmag (qmag qspace)
  (and (qmag-interval-p qmag)
       (not (robust-lmark-equal (succ (car qmag) qspace)
				(cadr qmag)))))


(defun mod-env-graph-for-over-abstraction (state var-qvalues)
  "This function will modify the envisionment graph for variables which have an abstracted
qmag and not an abstracted qdir.  It will traverse the envisionment graph.  States whose
qvalues do not match the values in the alist VAR-QVALUES will be marked as exiting the
region.  A new record of the states exiting the region is recorded.  This function is
called recursively."
  (when state
    (cond ((member 'exits-region (state-status state))
	   (pushnew state (sim-exits-region (state-sim state))))
	  ((loop for (var . qval) in var-qvalues
		 unless (qval-equal qval (qval var state))
		 return t)
	   (pushnew state (sim-exits-region (state-sim state)))
	   (pushnew 'exits-region (state-status state))
	   (mapcar #'(lambda (succ)
		       (mark-state-uninteresting succ
						 "Predecessor pruned due to over abstraction."))
		   (successor-states state)))
	  (t (mapcar #'(lambda (succ)
			 (mod-env-graph-for-over-abstraction succ var-qvalues))
		     (successor-states state))))))
		    




;;;
;;;  ABSTRACT-INITIAL-STATE-COMPLETIONS
;;;
;;;  This function will combine completions of an incomplete state.  Completions
;;;  will be combined if they differ only in the qdir of variables within intervals
;;;  that are potential chatter candidates.

(defun abstract-initial-state-completions (init-state)
  (when *perform-chatter-abstraction*
    (let ((completions (when (equal (car (state-successors init-state))				  
				    'completions)
			 (successor-states init-state)))
	  (chvars (find-poss-chatter-vars init-state)))
      (when (and completions chvars)
	(let* ((partitions (npartition completions :test #'(lambda (s1 s2)
							     (state-chvar-equivalent s1 s2 chvars))))
	       (new-states
		(mapcar #'(lambda (partition)
			    (let ((new-state (when (cdr partition)
					       (create-abstract-state-from-list partition))))
			      (when new-state
				(record-chatter-box-info new-state nil nil partition
							 (chatter-vars new-state)))
			      (or new-state (car partition))))
			partitions)))
	  (when (not (eq (length completions) (length new-states)))
	    (format *qsim-report* "~%Combining completions of the initial state ~a to help ~
                                   eliminate chatter." init-state))
	  (setf (cdr (state-successors init-state))
		new-states)
	  (mapcar #'(lambda (state)
		      (set-justification init-state state)
		      (pushnew 'complete (state-status state)))
		  new-states))
	))))

(defun state-chvar-equivalent (s1 s2 chvars)
  "Returns T if the two states are equivalent with the exception of the
qdir and possibly qmag of the CHVARS."
  (and (eq (state-qde s1) (state-qde s2))
       (loop for (var . qv1) in (cdr (state-qvalues s1))
	     for (nil . qv2) in (cdr (state-qvalues s2))
	     unless (qval-chvar-equivalent qv1 qv2 (member var chvars))
	     return nil
	     finally (return t))))


(defun qval-chvar-equivalent (qv1 qv2 chatter-var)
  "True if the two qvals are equivalent or if they have different qdirs and the var is a
member of CHVARS then the qmags should be intervals and equivalent."
  (and (qmag-equivalent (qmag qv1) (qmag qv2))
       (or (equal (qdir qv1) (qdir qv2))
	   (and chatter-var
		;; Either the qmag is an interval or if it is a landmark then it is a
		;; landmark created by qsim
		(or (interval-p (qmag qv1))
		    (lmark-where-defined (qmag qv1)))
		))))






;;;
;;;  CREATE-CHATTER-ABSTRACTION-STATE
;;;

(defun create-chatter-abstraction-state (env-root orig-state)
  "Creates a single state which abstracts the envisionment graph rooted at ENV-ROOT."
  (when (multiple-behaviors? env-root)
    (let* ((abs-state (create-abstract-state-from-list
		       (get-all-states
			env-root
			:successor-function
			#'(lambda (state)
			    (remove-if #'(lambda (elem)
					   (member 'exits-region (state-status elem)))
				       (get-successors state)))))))
      (cond ((chatter-vars abs-state)
	     (pushnew (list 'chatter-abstraction env-root)
		      (state-abstraction-of abs-state))
	     abs-state)
	    ((state-chattering-vars orig-state)
	     (cerror "Continue" "A state which already had abstracted qdirs ~a resulted in an ~
                               ~% envisionment graph ~a without any chattering variables.  The ~
                               ~% abstraction needs to be undone.  Please send this example to ~
                               ~% clancy@cs.utexas.edu."
		     orig-state env-root)
	     nil)))))



;;;
;;; CREATE-ABSTRACT-STATE-FROM-LIST
;;;

(defun create-abstract-state-from-list (states)
  "This function will receive a list of states and it will create a state which
abstracts these states."
  (let* ((new-qvals (create-abs-qvalues states (copy-qvalues (state-qvalues (car states))))))
    (new-state-from-old-state (car states) new-qvals)))

(defun abs-qvalues? (qvalues)
  (loop for (nil . qval) in qvalues
	when (and (qdir qval)
		  (listp (qdir qval)))
	return t))


(defun check-for-abs-qmag-only (state)
  (loop for (var . qval) in (cdr (state-qvalues state))
	for qspace in (cdr (state-qspaces state))
	for qmag = (qmag qval)
	when (and (not (listp (qdir qval)))
		  (abs-qmag qmag qspace))
	do (cerror "CONTINUE SIMULATION"
		   "The abstract state ~a has an abstracted qmag for the variable ~a, but the ~
                    qdir is not abstracted."
		   state var)
	and
	return nil
	finally (return t)))

(defun create-abs-qvalues (states abs-qvalues)
  (mapcar #'(lambda (state)
	      (loop for (nil . qval) in (state-qvalues state)
		    for (nil . abs-qval) in abs-qvalues
		    for qspace in (state-qspaces state)
		    for abs-qdir = (listify (qdir abs-qval))
		    when (and (qdir qval)
			      (not (member (qdir qval) abs-qdir)))
		    do (setf (qdir abs-qval) (cons (qdir qval) abs-qdir))
		    do (expand-qmag-if-needed abs-qval (qmag qval) (cdr qspace))))
	  states)
  abs-qvalues)


(defun expand-qval-if-needed (abs-qval qval qspace)
  (expand-qmag-if-needed abs-qval (qmag qval) qspace)
  (unless (qdir-includes (qdir abs-qval) (qdir qval))
    (setf (qdir abs-qval)
	  (union (listify (qdir abs-qval))
		 (listify (qdir qval))))))


(defun expand-qmag-if-needed (abs-qval qmag qspace)
  "This function receives and abstract QVAL and a new qmag.  If QMAG is not
included in the abstract qval, then the qmag of the abstract qval is expanded
to include the new QMAG."
  (let ((abs-qmag (qmag abs-qval)))
    (unless (qmag-includes abs-qmag qmag qspace)
      (let ((abs-qmag (if (qmag-point-p abs-qmag)
			  (list abs-qmag abs-qmag :cc)
			  abs-qmag)))
	(multiple-value-bind (upper-lm upper-closed)
	    (when (upper-boundary-gt qmag abs-qmag qspace)
	      (values (upper-lmark qmag)
		      (upper-boundary-closed? qmag)))
	  (multiple-value-bind (lower-lm lower-closed)
	      (when (lower-boundary-lt qmag abs-qmag qspace)
		(values (lower-lmark qmag)
			(lower-boundary-closed? qmag)))
	    (when upper-lm
	      (set-upper-boundary abs-qmag :lm upper-lm :closed upper-closed))
	    (when lower-lm
	      (set-lower-boundary abs-qmag :lm lower-lm :closed lower-closed))
	    (when (or lower-lm upper-lm)
	      (setf (qmag abs-qval) abs-qmag)
	      abs-qval)))))))




(defun set-lower-boundary (qmag &key (lm nil) (closed nil))
  (let ((new-int-type (cond ((and closed
				  (upper-boundary-closed? qmag))
			     :cc)
			    (closed
			     :co)
			    ((lower-boundary-open? qmag)
			     (third qmag))
			    ((upper-boundary-open? qmag)
			     :oo)
			    ((upper-boundary-closed? qmag)
			     :oc))))
    (when lm (setf (car qmag) lm))
    (when new-int-type
      (setf (cddr qmag) (list new-int-type)))
    qmag))

(defun set-upper-boundary (qmag &key (lm nil) (closed nil))
  (let ((new-int-type (cond ((and closed
				  (lower-boundary-closed? qmag))
			     :cc)
			    (closed
			     :oc)
			    ((upper-boundary-open? qmag)
			     (third qmag))
			    ((lower-boundary-open? qmag)
			     :oo)
			    ((lower-boundary-closed? qmag)
			     :co))))
    (when lm (setf (cadr qmag) lm))
    (when new-int-type
      (setf (cddr qmag) (list new-int-type)))
    qmag))
			




;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                                                                       ;;;
;;;  PERFORM-SUB-ENVISIONMENT                                             ;;;
;;;                                                                       ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;



(defun perform-sub-envisionment (state region chvars)
  "This function will perform a sub-envisionment.  It creates an initial state from
STATE.  It also will defined a new global-state-filter.  This global state
filter ensures that the envisionment stays within a given region of the simulation.
This region is defined by the current qvalue of all of the variables except those
in the chvars list.  These can have any qdir."
  (with-envisioning
      (let* ((sim (make-sim :state-limit nil :q2-constraints nil
			    :time-limit nil))
	     (*initial-state* nil)
	     (*check-quantitative-ranges* nil)
	     (*perform-behavior-aggregation* nil)
	     (*cleanup-inconsistent-states* *cleanup-chatter-incon-states*)
	     (qvalues (sub-env-qvalues state))
	     (init-state (make-new-state :from-state state
					  :sim sim
					  :assert-values qvalues
					  :text (format nil "Total envisionment to ~
                                                   abstract chatter from ~a" state)))
	     (chatter-box-filter (create-region-filter-function region))
	     (*xedge-hash-table-size* (calc-hash-size-for-env
				       (length chvars)))
	     (*treat-abs-qdir-as-ign* nil)
	     (*global-state-filters* (append (remove 'cross-edge-filter *global-state-filters*)
					     (when region (list chatter-box-filter))
					     '(cross-edge-filter))))
	(declare (special *treat-abs-qdir-as-ign* *cleanup-inconsistent-states*
			  *perform-behavior-aggregation* *xedge-hash-table-size*))

	;; Remove bindings and eqn-index from the state.  This keeps the state
	;; structure smaller. BKay
	(remove-q2-info init-state)
	
	(when (incomplete-p init-state)
	  (prune-non-matching-completions init-state chvars state))
	(trace-sub-envisionment chvars state init-state)

	(qsim init-state)
	;; Signals an error if no chatter is observed in the resulting envisionment
	;; when chatter existed in the initial state.  This problem must be looked at.
	(when (and (chatter-vars state)
		   (<= (length (get-behaviors init-state)) 1))
	  (cerror "Continue" "Chatter not observed in state ~a by sub-envisionment ~a."
		  state init-state))
	init-state)))

;;; Remove the bindings and eqn-index from a state and its incomplete
;;; successors.
;;;
(defun remove-q2-info (state)
  (setf (state-bindings state) nil)
  (setf (state-eqn-index state) nil)
  (when (incomplete-p state)
    (loop for s in (cdr (state-successors state))
	  do
	  (setf (state-bindings s) nil)
	  (setf (state-eqn-index s) nil))))

(defun calc-hash-size-for-env (num-vars)
  (let ((init-size (expt num-vars 3)))
    (if (> init-size 500)
	500
	init-size))) 

(defun prune-non-matching-completions (init-state chvars orig-state)
  "This function eliminates completions that use qdirs that are not in the
qdir abstractions for the original state."
  (let ((chvars-and-qdirs (mapcan #'(lambda (var)
				      (when (abs-qdir (qdir (qval var orig-state)))
					(list (list var (qdir (qval var orig-state))))))
				  chvars)))
    (when chvars-and-qdirs
      (setf (cdr (state-successors init-state))
	    (remove-if #'(lambda (completion)
			   (loop for (var abs-qdir) in chvars-and-qdirs
				 when (not (member (qdir (qval var completion)) abs-qdir))
				 do (return t)))
		       (cdr (state-successors init-state)))))))
		      
(defun sub-env-qvalues (state)
  "This function will return the qvalues for the initial state of the sub envisionment.
If a QDIR is abstracted then NIL is returned for the qdir."
  (mapcar #'(lambda (pair)
	      (let* ((var (car pair))
		     (qval (cdr pair))
		     (qmag (qval-qmag qval))
		     (qdir (unless (abs-qdir (qval-qdir qval))
			     (qdir qval))))
		(list var (list (name-of-qmag qmag) qdir))))
	  (cdr (state-qvalues state))))

(defun remove-lm (state var lm)
  (let* ((qval (qval var state))
	 (qspace (assoc var (state-qspaces state))))
    (unless (robust-lmark-equal lm (qmag qval))
      (cerror "Continue" "Cannot remove ~a from ~a for ~a" lm state  var))
    (setf (qmag qval)
	  (lmark-where-defined lm))
    (setf (cdr qspace)
	  (loop for cur-lm in (cdr qspace)
		unless (lmark-equal lm cur-lm)
		collect cur-lm into new-qspace
		finally (return new-qspace)))))
	
;;;
;;; Function: CREATE-REGION-FILTER-FUNCTION
;;;
;;;  Input: REG - a list of qvalues.  Each qvalue is viewed as an acceptable region
;;;               for that variable to remain within.
;;;
;;;  Returns: A function which can be used as a global state filter that will
;;;    filter out states which extend outside the region.


(defun create-region-filter-function (reg)
  "Returns a function that takes a state as an argument.  This function
will return the state if it is within the REGION.  Otherwise, it
returns nil and marks the state as inconsistent."
  #'(lambda (state)
      (let* ((region (canonicalize-qvals reg state))
	     qvals-exiting-region)
	(loop for (var . qval) in (cdr (state-qvalues state))
	      for qspace in (cdr (state-qspaces state))
	      for qval-region = (cadr (assoc var region))
	      when (qval-not-in-region qval qval-region (cdr qspace))
	      do (pushnew (list var qval qval-region) qvals-exiting-region))
	(when qvals-exiting-region
	  (pushnew 'exits-region (state-status state))
	  (pushnew state (sim-exits-region (state-sim state))))
	(trace-region-filter state region qvals-exiting-region)
	state)))

(defun qval-not-in-region (qval qval-region qspace)
  (cond ((null qval-region))
	((or (not (qmag-in-region (qmag qval)
				  (car qval-region) qspace))
	     (not (qdir-in-region (qdir qval)
				  (cadr qval-region)))))))


(defun create-region-equiv-function (region)
  "Returns a function that will determine if two states are equivalent with respect
to REGION.  They are equivlanent with respect to the region if the distinctions between
the states lie within the region.  Those portions of the state that exit the region
must be the same."
  #'(lambda (s1 s2)
      (loop for (var . qv1) in (cdr (state-qvalues s1))
	    for (nil . qv2) in (cdr (state-qvalues s2))
	    for (reg-var reg-qv) in region
	    for qspace in (cdr (state-qspaces s1))
	    for qv1-not-in-region = (qval-not-in-region qv1 reg-qv (cdr qspace))
	    for qv2-not-in-region = (qval-not-in-region qv2 reg-qv (cdr qspace))
	    unless (equal reg-var var)
	    do (cerror "Conitnue" "The variables are not equal")
	    do (cond ((and qv1-not-in-region
			   qv2-not-in-region
			   (qval-equal qv1 qv2)))
		     ((and (not qv1-not-in-region)
			   (not qv2-not-in-region)))
		     (t (return nil)))
	    finally (return t))))
				 

(defun qmag-in-region (qmag qmag-region qspace &key (inclusive nil))
  (let ((comparison-function (if inclusive #'landmark-le #'landmark-lt)))
    (cond ((atom qmag-region)
	   (robust-lmark-equal qmag qmag-region))
	  ((atom qmag)
	   (and (funcall comparison-function (car qmag-region) qmag qspace)
		(funcall comparison-function qmag (cadr qmag-region) qspace)))
	  (t (and (funcall #'landmark-le (car qmag-region) (car qmag) qspace)
		  (funcall #'landmark-le (cadr qmag) (cadr qmag-region) qspace))))))

(defun qdir-in-region (qdir region)
  (cond ((atom region)
	 (equal qdir region))
	((member qdir region))
	(t (and (listp qdir)
		(null (set-difference qdir region))))))

  
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                                                                       ;;;
;;;                                                                       ;;;
;;;   DISPLAY CODE                                                        ;;;
;;;                                                                       ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


(defun chatter-box-display (&rest rest)
  (declare (ignore rest))
  (let* ((chatter-state (get-chatter-box-to-display))
	 (disp-state (when chatter-state (chatter-abstraction-of chatter-state))))
    (when disp-state
      (format *qsim-report* "~%~%Displaying the envisionment graph for the portion of the behavior~
                    ~%tree abstracted by state ~a.  This state exhibits chatter in variables ~a~%"
	      chatter-state (chatter-vars chatter-state))
      (if (and (state-p disp-state)
	       (state-name disp-state))
	  (qsim-display disp-state)
	  (format *qsim-report* "~%Cannot display this graph.  It was destroyed after the ~
                  envisionment.")))))

(defun get-chatter-box-to-display ()
  (loop for chatter-state = (get-valid-state)
	if (or (null chatter-state)
	       (chatter-vars chatter-state))
	return chatter-state
	else
	do (format *qsim-report* "~%State ~a is not a chatter box.  Please enter a new state ~
		               enter NIL to exit." chatter-state)))

(eval-when (eval load)
  (add-to-qsim-extension-menu 'C "Display Envisionment Graph for a Chatter Box"
			      'chatter-box-display))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;                                                                       ;;;
;;;                                                                       ;;;
;;;   STRUCTURE CLEANUP  CODE                                             ;;;
;;;                                                                       ;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


;;;  This code is used to cleanup state and SIM structures.
;;;  Eventually it should be incorporated into the QSIM baseline.
;;;  Here it is very useful in case you have sun envisionments that
;;;  are very large.  It can be used to clean-up the garbage made from
;;;  a particular simulation.

;;;; File: structures.lisp (although maybe it can go elsewhere).

;;; Cleanup a state by removing all data hanging off the state slots.
;;; This function is needed because structures in Lucid (at least) are
;;; not stored as either a list or an array, so struct-cleanup doesn't work.
;;;
(defun state-cleanup (s)
  (let ((refinements (mapcan #'cdr (state-abstraction-of s))))
    ;;  Clean-up any refinements of this state and their tree.
    (mapcar #'(lambda (refinement)
		(if (equal (state-sim refinement)
			   (state-sim s))
		    (tree-cleanup refinement)
		    (sim-cleanup (state-sim refinement))))
	    refinements)
    ;; Reset *initial-state* if this is the initial-state.
    (when (equal s *initial-state*)
      (setf *initial-state* nil))
    ;; Unintern the symbol name
    (unintern (state-name s))
    (setf (state-qde s) nil)
    (setf (state-qvalues s) nil)
    (setf (state-qspaces s) nil)
    (setf (state-cvalues s) nil)
    (setf (state-status s) nil)
    (setf (state-justification s) nil)
    (setf (state-successors s) nil)
    (setf (state-text s) nil)
    (setf (state-time-label s) nil)
    (setf (state-other s) nil)
    (setf (state-misc s) nil)
    (setf (state-name s) nil)
    (struct-cleanup s)))
  
(defun tree-cleanup (state)
  (let ((*show-inconsistent-successors* t)
	(*traverse-xedges* nil))
    (rec-tree-cleanup state)))

(defun rec-tree-cleanup (state)
  (when state
    (mapcar #'rec-tree-cleanup (successor-states state))
    (state-cleanup state)))
	  

(defun sim-cleanup (sim)
  (when (equal *current-sim* sim)
    (setf *current-sim* nil))
  (tree-cleanup (sim-state sim))
  (when (sim-xedge-state-table sim)
    (clrhash (sim-xedge-state-table sim)))
  (unintern (sim-name sim))
  (setf (sim-qde sim) nil)
  (setf (sim-state sim) nil)
  (setf (sim-display-block sim) nil)
  (setf (sim-time-limit sim) nil)
  (setf (sim-state-limit sim) nil)
  (setf (sim-state-count sim) nil)
  (setf (sim-agenda sim) nil)
  (setf (sim-xedge-state-table sim) nil)
  (setf (sim-ignore-qdirs sim) nil)
  (setf (sim-no-new-landmarks sim) nil)
  (setf (sim-phase-planes sim) nil)
  (setf (sim-unreachable-values sim) nil)
  (setf (sim-other-variables sim) nil)
  (setf (sim-other sim) nil))
  






;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;   TRACE FUNCTIONS



(defun trace-chatter-elim-filter (state root succs abs-state)
  (when *trace-chatter-abstraction*
    (format *Qsim-Report* "~%Envisionment Simulations while filtering ~a" state)
    (format *Qsim-Report* "~%   Chattering Vars     : ~a~
                           ~%   Root of env graph   : ~a~
                           ~%   Exiting States      : ~a~
                           ~%   Abstract State      : ~a~
                           ~%   Numb states in env  : ~a~%"
	    (if abs-state (state-chattering-vars abs-state) "NONE")
	    root succs abs-state
	    (if abs-state (state-env-size abs-state))
	    )))



(defun trace-sub-envisionment (chvars state init-state)
  (when *trace-sub-envisionment*
    (format *Qsim-Report* "~%Performing a subenvisionment.~
                           ~%   Chattering Vars:           ~a~
                           ~%   Orig State:                ~a~
                           ~%   Root state:                ~a."
	    chvars state init-state)))


(defun trace-region-filter (state region qvals)
  (when *trace-region-filter*
    (format *Qsim-report* "~%State ~a ~a the region filter. ~
                           ~%    REGION: ~a."
	    state (if qvals "FAILED" "PASSED") region)
    (when qvals (format *qsim-report* "~%    VARS: ~a" qvals))))





