;;; -*- Mode:Lisp; Package:QSIM; Syntax:COMMON-LISP; Base:10 -*-

(in-package 'qsim)

;; QDE's for the motor design example of Kiriyama, Tomiyama, and Yoshikawa
;; given in QR-91.

;; CC model

(define-component-interface
  2-terminal-shaft "Two terminal motor shaft" mechanical-rotation
  (terminals t1 t2)
  (quantity-spaces
    (defaults
      (velocity                       motor-velocity-qspace)
      (force                          angular-force-qspace)
      ((mechanical-translation force) motor-lateral-force-qspace)
      ((electrical current)           motor-current-qspace))))


(define-component-implementation
  1 2-terminal-shaft "Two terminal motor shaft"
  (terminal-variables (t1 (F-ang1 force)
                          (V1     velocity)
                          (F-lat1 (mechanical-translation force))
                          (I1     (electrical current)))
                      (t2 (F-ang2 force)
                          (V2     velocity)
                          (F-lat2 (mechanical-translation force))
                          (I2     (electrical current))))
  (component-variables (X         displacement)
                       (F-ang-sum force)
                       (Cum-F-ang force)
                       (Cum-F-lat (mechanical-translation force)))
  (mode-variables
    (position
      (positive <- (or (X ((0 X180+) nil))
                       (X (X180+ dec))
                       (X (X180+ std))))
      (negative <- (or (X ((X180- 0) nil))
                       (X (X180- inc))
                       (X (X180- std))))
      (:discontinuous-transition <- (X (X180+ inc))
                                 negative (X (X180- inc)))
      (:discontinuous-transition <- (X (X180- dec))
                                 positive (X (X180+ dec)))))
  (constraints
    ((d/dt X V1))
    ((d/dt V1 Cum-F-ang))
    ((add F-ang1 F-ang2 F-ang-sum) (F- F- F-) (0 0 0) (F+ F+ F+))
    ((minus Cum-F-ang F-ang-sum)   (F- F+) (0 0) (F+ F-))
    ((minus I1 I2)                 (Imax- Imax+) (0 0) (Imax+ Imax-))
    ((equal V1 V2)                 (0 0) (V* V*))
    ((constant Cum-F-lat 0))
    ((position positive) -> ((S- V1 I1 (0 Imax+) (V* 0))))
    ((position negative) -> ((S- V1 I2 (0 Imax+) (V* 0))))))


(define-component-implementation
  3 motor
  "Double magnet, double rotor"
  (quantity-spaces
    (defaults
      ((mechanical-rotation force)    angular-force-qspace)
      ((mechanical-rotation velocity) motor-velocity-qspace)))
  (mode-variables
    (position
      (positive <- (or ((shaft X) ((0 X180+) nil))
                       ((shaft X) (X180+ dec))
                       ((shaft X) (X180+ std))))
      (negative <- (or ((shaft X) ((X180- 0) nil))
                       ((shaft X) (X180- inc))
                       ((shaft X) (X180- std))))
      (:discontinuous-transition <- ((shaft X) (X180+ inc))
                                 negative ((shaft X) (X180- inc)))
      (:discontinuous-transition <- ((shaft X) (X180- dec))
                                 positive ((shaft X) (X180+ dec)))))
  (components
    (magnet1 magnet)
    (magnet2 magnet)
    (rotor1  2-field-rotor (no-new-landmarks F-lat F-ang Orientation)
             (ignore-qdir F-ang))
    (rotor2  2-field-rotor (no-new-landmarks F-lat F-ang Orientation)
             (ignore-qdir F-ang))
    (shaft   (2-terminal-shaft (impl 1))
             (no-new-landmarks F-lat1 F-lat2 F-ang1 F-ang2
                               F-ang-sum Cum-F-ang)
             (ignore-qdir F-ang1 F-ang2 F-ang-sum Cum-F-ang)
             (quantity-spaces (X position-90-qspace))))
  (constraints
    ((position positive)
      ->
      ((U- (shaft X) (rotor1 Orientation) (X90+ Omax+)) 
                                          (0 0) (X180+ 0))
      ((U- (shaft X) (rotor2 Orientation) (X90+ Omax+))
                                          (0 0) (X180+ 0)))
    ((position negative)
      ->
      ((U+ (shaft X) (rotor1 Orientation) (X90- Omax-))
                                          (X180- 0) (0 0))
      ((U+ (shaft X) (rotor2 Orientation) (X90- Omax-))
                                          (X180- 0) (0 0))))
  (connections (c1 (rotor1 magnet+) (rotor2 magnet-) (magnet1 north))
               (c2 (rotor1 magnet-) (rotor2 magnet+) (magnet2 south))
               (c3 (rotor1 shaft) (shaft t1))
               (c4 (rotor2 shaft) (shaft t2))))



(defun motor-3-pos-0-vel-0 ()
  (let* ((initial-values '(((shaft V1)  (0 nil)) ((shaft X)  (0 nil))))
	 (models (build-qde '(motor (impl 3)) :initial-values initial-values)))
    (unless (listp models) (setf models (list models)))
    (format t "~&~% Models for (MOTOR (impl 3)) with initial velocity 0, initial position 0:")
    (dolist (qde models) (format t "~&  ~A" (qde-name qde)))
    (dolist (qde models)
      (format t "~%~& Simulating ~A" (qde-name qde))
      (motor-sim qde initial-values "Position = 0, velocity = 0 (Dead point)"))))

(defun motor-3-pos-180-vel-0 ()
  (let* ((initial-values '(((shaft V1)  (0 nil)) ((shaft X)  (X180+ nil))))
	 (models (build-qde '(motor (impl 3)) :initial-values initial-values)))
    (unless (listp models) (setf models (list models)))
    (format t "~&~% Models for (MOTOR (impl 3)) with initial velocity 0, initial position X180+:")
    (dolist (qde models) (format t "~&  ~A" (qde-name qde)))
    (dolist (qde models)
      (format t "~%~& Simulating ~A" (qde-name qde))
      (motor-sim qde initial-values "Position = X180+, velocity = 0 (Dead point)"))))

(defun motor-3-pos-180--vel-0 ()
  (let* ((initial-values '(((shaft V1)  (0 nil)) ((shaft X)  (X180- nil))))
	 (models (build-qde '(motor (impl 3)) :initial-values initial-values)))
    (unless (listp models) (setf models (list models)))
    (format t "~&~% Models for (MOTOR (impl 3)) with initial velocity 0, initial position X180-:")
    (dolist (qde models) (format t "~&  ~A" (qde-name qde)))
    (dolist (qde models)
      (format t "~%~& Simulating ~A" (qde-name qde))
      (motor-sim qde initial-values "Position = X180-, velocity = 0 (Dead point)"))))

(defun motor-3-pos-X-vel-0 ()
  (let* ((initial-values '(((shaft V1)  (0 nil)) ((shaft X)  (X90+ nil))))
	 (models (build-qde '(motor (impl 3)) :initial-values initial-values)))
    (unless (listp models) (setf models (list models)))
    (format t "~&~% Models for (MOTOR (impl 3)) with initial velocity 0, initial position X90+:")
    (dolist (qde models) (format t "~&  ~A" (qde-name qde)))
    (dolist (qde models)
      (format t "~%~& Simulating ~A" (qde-name qde))
      (motor-sim qde initial-values "Position = X90+, velocity = 0" 150))))

(defun motor-3-pos-0-vel-V* ()
  (let* ((initial-values '(((shaft V1)  (V* nil)) ((shaft X)  (0 nil))))
	 (models (build-qde '(motor (impl 3)) :initial-values initial-values)))
    (unless (listp models) (setf models (list models)))
    (format t "~&~% Models for (MOTOR (impl 3)) with initial velocity V*, initial position 0:")
    (dolist (qde models) (format t "~&  ~A" (qde-name qde)))
    (dolist (qde models)
      (format t "~%~& Simulating ~A" (qde-name qde))
      (motor-sim qde initial-values "Position = 0, velocity = V*"))))

(defun motor-3-pos-+-vel-V* ()
  (let* ((initial-values '(((shaft V1)  (V* nil)) ((shaft X)  ((0 X180+) nil))))
	 (models (build-qde '(motor (impl 3)) :initial-values initial-values)))
    (unless (listp models) (setf models (list models)))
    (format t "~&~% Models for (MOTOR (impl 3)) with initial velocity V*, initial position positive:")
    (dolist (qde models) (format t "~&  ~A" (qde-name qde)))
    (dolist (qde models)
      (format t "~%~& Simulating ~A" (qde-name qde))
      (motor-sim qde initial-values "Position = +, velocity = V*"))))
