;;; -*- Mode:Common-Lisp; Package:QSIM; Syntax:COMMON-LISP; Base:10 -*-

(in-package 'qsim)

;; QDE's for the motor design example of Kiriyama, Tomiyama, and Yoshikawa
;; given in QR-91.

;; CC model

(define-quantity-space position-X-qspace (X180- X- 0 X+ X180+)
  (conservation-correspondences (X180- X180+) (X- X+)))

(define-quantity-space position-X-positive-qspace (0 X+ X180+)
  (parent position-X-qspace))

(define-quantity-space position-X-negative-qspace (X180- X- 0)
  (parent position-X-qspace))

(define-quantity-space positive-qspace (0 inf))

(define-quantity-space polarity-qspace (South 0 North)
  (conservation-correspondences (South North)))

(define-quantity-space angular-force-qspace (F- 0 F+)
  (conservation-correspondences (F- F+)))

(define-quantity-space orientation-qspace (Omax- 0 Omax+)
  (conservation-correspondences (Omax- Omax+)))

(define-quantity-space motor-current-qspace (Imax- 0 Imax+)
  (conservation-correspondences (Imax- Imax+)))

(define-quantity-space motor-velocity-qspace (minf 0 V* inf))


(define-component-interface
  rotor "Rotor for electro-mechanical motor" mechanical-rotation
  (terminals shaft magnet))

(define-component-implementation
  1 rotor ""
  (terminal-variables
    (shaft  (F-ang force)
            (V     velocity)
            (F-lat (mechanical-translation force))
            (I     (electrical current)))
    (magnet (F-mag (magnetic force))))
  (component-variables
    (Polarity    (magnetic force) (quantity-space polarity-qspace))
    (Repulsion   force (quantity-space angular-force-qspace))
    (PotentialF  force (quantity-space angular-force-qspace))
    (Orientation displacement (quantity-space orientation-qspace)))
  (constraints
    ((M+ I Polarity) (Imax- South) (0 0) (Imax+ North))
    ((mult F-mag Polarity Repulsion)     (0 0 0) (South North F-)
               (North South F-) (South South F+) (North North F+))
    ((minus Repulsion PotentialF)        (F- F+) (0 0) (F+ F-))
    ((mult PotentialF Orientation F-ang) (F- Omax- F+) (F- Omax+ F-)
                                 (0 0 0) (F+ Omax- F-) (F+ Omax+ F+))
    ((U- V F-lat (0 0))                  (minf minf) (inf minf))))


(define-component-interface
  magnet "Magnet for electro-mechanical motor" magnetic
  (terminals north south))

(define-component-implementation
  1 magnet ""
  (terminal-variables
    (north (F-north force (quantity-space polarity-qspace)))
    (south (F-south force (quantity-space polarity-qspace))))
  (constraints ((constant F-north north))
               ((constant F-south south))))


(define-component-interface
  one-terminal-shaft
  "Single terminal shaft for electro-mechanical motor"
  mechanical-rotation
  (terminals t))

(define-component-implementation
  1 one-terminal-shaft ""
  (terminal-variables
    (t (F-ang force)
       (V     velocity (quantity-space motor-velocity-qspace))
       (F-lat (mechanical-translation force))
       (I     (electrical current))))
  (component-variables
    (X         displacement)
    (Cum-F-ang force (quantity-space angular-force-qspace))
    (KE        energy))
  (constraints ((d/dt X V))
               ((d/dt V Cum-F-ang))
               ((minus Cum-F-ang F-ang) (F- F+) (0 0) (F+ F-))
               ((U+ V KE (0 0))         (minf inf) (inf inf))))



(define-component-interface
  motor "Electro-mechanical motor" mechanical
  (quantity-spaces
    (defaults ((magnetic force)            polarity-qspace)
              ((electrical current)        motor-current-qspace)
              ((mechanical-rotation force) angular-force-qspace))))


(define-component-implementation
  1 motor
  "Single magnet, single rotor"
  (quantity-spaces
    (default
      ((mechanical-rotation velocity) motor-velocity-qspace)))
  (component-variables (PE energy (quantity-space (0 PE+ PE*)))
                       (TE energy))
  (mode-variables
    (position
      (positive <- (or ((shaft X) ((0 X180+) nil))
                       ((shaft X) (X180+ dec))
                       ((shaft X) (X180+ std))))
      (negative <- (or ((shaft X) ((X180- 0) nil))
                       ((shaft X) (X180- inc))
                       ((shaft X) (X180- std))))
      (:discontinuous-transition <- ((shaft X) (X180+ inc))
                                 negative ((shaft X) (X180- inc)))
      (:discontinuous-transition <- ((shaft X) (X180- dec))
                                 positive ((shaft X) (X180+ dec)))))
  (components (magnet magnet)
              (rotor (rotor (impl 1)) (no-new-landmarks F-lat F-ang)
                     (ignore-qdir F-ang)
                     (quantity-spaces (X position-X-qspace)))
              (shaft (one-terminal-shaft (impl 1))
                     (no-new-landmarks F-lat F-ang Cum-F-ang)
                     (ignore-qdir F-ang Cum-F-ang)
                     (quantity-spaces (X position-X-qspace))))
  (constraints
    ((ADD PE (shaft KE) TE)              (0 0 0))
    ((constant TE))
    ((constant (shaft I) Imax+))
    ((position positive)
      ->
      ((U- (shaft X) (rotor Orientation) (X+ Omax+)) (0 0) (X180+ 0))
      ((S- (shaft X) PE (0 PE*) (X180+ 0)) (X+ PE+)))
    ((position negative)
      ->
      ((U+ (shaft X) (rotor Orientation) (X- Omax-)) (X180- 0) (0 0))
      ((S+ (shaft X) PE (X180- 0) (0 PE*)) (X- PE+))))
  (connections (c1 (rotor magnet) (magnet north))
               (c2 (rotor shaft)  (shaft t))))


(defun motor-sim (qde initial-values text &optional (state-limit 50))
  (qsim-cleanup)
  (let* ((sim (make-sim :fire-transitions :first
			:cycle-detection :standard
			:state-limit state-limit))
	 (initial-state (make-new-state :from-qde qde
					:assert-values (CC-name qde :alist initial-values)
					:text text
					:sim sim)))
      (qsim initial-state)
      (qsim-display initial-state)))


(defun motor-1-pos-0-vel-0 ()
  (let* ((initial-values '(((shaft V)  (0 nil)) ((shaft X)  (0 nil))))
	 (models (build-qde '(motor (impl 1)) :initial-values initial-values)))
    (unless (listp models) (setf models (list models)))
    (format t "~&~% Models for (MOTOR (impl 1)) with initial velocity 0, initial position 0:")
    (dolist (qde models) (format t "~&  ~A" (qde-name qde)))
    (dolist (qde models)
      (format t "~%~& Simulating ~A" (qde-name qde))
      (motor-sim qde initial-values "Position = 0, velocity = 0 (Dead point)"))))

(defun motor-1-pos-180-vel-0 ()
  (let* ((initial-values '(((shaft V)  (0 nil)) ((shaft X)  (X180+ nil))))
	 (models (build-qde '(motor (impl 1)) :initial-values initial-values)))
    (unless (listp models) (setf models (list models)))
    (format t "~&~% Models for (MOTOR (impl 1)) with initial velocity 0, initial position X180+:")
    (dolist (qde models) (format t "~&  ~A" (qde-name qde)))
    (dolist (qde models)
      (format t "~%~& Simulating ~A" (qde-name qde))
      (motor-sim qde initial-values "Position = X180+, velocity = 0 (Dead point)"))))

(defun motor-1-pos-180--vel-0 ()
  (let* ((initial-values '(((shaft V)  (0 nil)) ((shaft X)  (X180- nil))))
	 (models (build-qde '(motor (impl 1)) :initial-values initial-values)))
    (unless (listp models) (setf models (list models)))
    (format t "~&~% Models for (MOTOR (impl 1)) with initial velocity 0, initial position X180-:")
    (dolist (qde models) (format t "~&  ~A" (qde-name qde)))
    (dolist (qde models)
      (format t "~%~& Simulating ~A" (qde-name qde))
      (motor-sim qde initial-values "Position = X180-, velocity = 0 (Dead point)"))))

(defun motor-1-pos-X-vel-0 ()
  (let* ((initial-values '(((shaft V)  (0 nil)) ((shaft X)  (X+ nil))))
	 (models (build-qde '(motor (impl 1)) :initial-values initial-values)))
    (unless (listp models) (setf models (list models)))
    (format t "~&~% Models for (MOTOR (impl 1)) with initial velocity 0, initial position X+:")
    (dolist (qde models) (format t "~&  ~A" (qde-name qde)))
    (dolist (qde models)
      (format t "~%~& Simulating ~A" (qde-name qde))
      (motor-sim qde initial-values "Position = X+, velocity = 0"))))

(defun motor-1-pos-vel-0 ()
  (let* ((initial-values '(((shaft V)  (0 nil)) ((shaft X)  ((0 X180+) nil))))
	 (models (build-qde '(motor (impl 1)) :initial-values initial-values)))
    (unless (listp models) (setf models (list models)))
    (format t "~&~% Models for (MOTOR (impl 1)) with initial velocity 0, initial position > 0:")
    (dolist (qde models) (format t "~&  ~A" (qde-name qde)))
    (dolist (qde models)
      (format t "~%~& Simulating ~A" (qde-name qde))
      (motor-sim qde initial-values "Position > 0, velocity = 0" 450))))
