;; -*- Mode: LISP; Syntax: Common-lisp; Package: QSIM; Base: 10 -*-

(in-package 'qsim)

;;
;; Dynamic mode examples
;;

;;
;; Home heater with Thermostat
;;


(define-quantity-space heater-temperature-qspace (0 lo hi H* inf))

(define-quantity-space heater-voltage-qspace (0 V*))


(define-component-interface heater
  "Heat source" thermal
  (terminals control out))

(define-component-implementation 1 heater
  "Heater implementation"
  (terminal-variables
    (control (V  (electrical voltage)))
    (out     (T  temperature)
	     (HF heat-flow)))
  (component-variables
    (sourceT temperature)
    (deltaT  temperature (quantity-space (minf 0 inf)) no-new-landmarks))
  (constraints
    ((ADD sourceT deltaT T) (H* 0 H*))
    ((CONSTANT sourceT H*))
    ((MULT V deltaT HF))))


(define-component-interface heater-thermostat
  "Heater thermostat" thermal
  (terminals sensor control))

(define-component-implementation 1 heater-thermostat
  "Heater thermostat"
  (terminal-variables
    (sensor (T  temperature)
	    (HF heat-flow))
    (control (V (electrical voltage))))
  (mode-variables
    (mode (off <- (T ((hi inf) nil)))
	  (on  <- (T ((0 lo)   nil)))))
  (constraints
    ((CONSTANT HF 0))    ; Heat flow into thermostat negligible
    ((mode off) 
     -> ((CONSTANT V 0)))
    ((mode on)
     -> ((CONSTANT V V*)))))


(define-component-interface 1-terminal-thermal-mass
  "Thermal mass, one terminal" thermal
  (terminals t1))

(define-component-implementation 1 1-terminal-thermal-mass
  "Thermal mass, 1 terminal"
  (terminal-variables
    (t1 (Temp1 temperature)
	(HF1   heat-flow)))
  (component-variables
    (Heat    entropy)
    (Temp    temperature))
  (constraints
    ((EQUAL Temp Temp1) (Lo Lo) (Hi Hi) (H* H*) (inf inf))
    ((d/dt Heat HF1))
    ((M+ Heat Temp) (0 0) (inf inf))))


(define-component-interface 2-terminal-thermal-mass
  "Thermal mass, two terminals" thermal
  (terminals t1 t2))

(define-component-implementation 1 2-terminal-thermal-mass
  "Thermal mass, 2 terminals"
  (terminal-variables
    (t1 (Temp1 temperature)
	(HF1   heat-flow))
    (t2 (Temp2 temperature)
	(HF2   heat-flow)))
  (component-variables
    (netFlow heat-flow)
    (Heat    entropy)
    (Temp    temperature))
  (constraints
    ((EQUAL Temp Temp1) (Lo Lo) (Hi Hi) (H* H*) (inf inf))
    ((EQUAL Temp Temp2) (Lo Lo) (Hi Hi) (H* H*) (inf inf))
    ((d/dt Heat netFlow))
    ((M+ Heat Temp) (0 0) (inf inf))
    ((ADD HF1 HF2 netFlow))))

(define-component-interface thermal-sink
  "Thermal sink, constant temperature, infinite mass"
  thermal
  (terminals in))

(define-component-implementation 1 thermal-sink
  "Thermal sink"
  (terminal-variables
    (in (Tin temperature)
	(HF  heat-flow)))
  (component-variables
    (Temp   temperature)
    (deltaT temperature (quantity-space (minf 0 inf)) no-new-landmarks))
  (constraints
    ((constant Temp 0))
    ((ADD Temp deltaT Tin))
    ((M+ HF deltaT) (minf minf) (0 0) (inf inf))))


(define-component-interface home-heater
  "Home heater model" thermal)

(define-component-implementation 1 home-heater
  "Home heater composed of heater, thermostat, thermal mass (home)."
  (quantity-spaces
    (defaults (temperature heater-temperature-qspace)
              ((electrical voltage) heater-voltage-qspace)))
  (components (Heater     heater (display HF))
	      (Thermostat heater-thermostat (display V mode))
	      (Room       1-terminal-thermal-mass (display Heat Temp HF1)))
  (connections (control (Thermostat control) (Heater control))
	       (sensor (Thermostat sensor) (Heater out) (Room t1))))


(define-component-implementation 2 home-heater
  "Home heater composed of heater, thermostat, thermal mass (home), thermal sink (outside)."
  (quantity-spaces
    (defaults (temperature heater-temperature-qspace)
              ((electrical voltage) heater-voltage-qspace)))
  (components (Heater     heater (display HF))
	      (Thermostat heater-thermostat (display V mode))
	      (Room       2-terminal-thermal-mass (display Heat Temp HF1 HF2))
	      (Outside    thermal-sink))
  (connections (control (Thermostat control) (Heater control))
	       (sensor (Thermostat sensor) (Heater out) (Room t1))
	       ((Room t2) (Outside in))))

(setf HH-0-Lo   '(((Room Temp) ((0 Lo)   nil))))
(setf HH-Lo     '(((Room Temp) (Lo       nil))))
(setf HH-Lo-Hi  '(((Room Temp) ((Lo Hi)  nil))))
(setf HH-Hi     '(((Room Temp) (Hi       nil))))
(setf HH-Hi-H*  '(((Room Temp) ((Hi H*)  nil))))
(setf HH-H*     '(((Room Temp) (H*       nil))))
(setf HH-Hi-inf '(((Room Temp) ((Hi inf) nil))))


(defun HH (qdes inits)
  (dolist (qde qdes)
    (format t "~&~% Home heater with thermostat mode: ~S~%   Initial values: ~S"
	    (qde-name qde) inits)
    (let* ((initial-values (CC-name qde :alist inits))
	   (sim (make-sim :fire-transitions :first
			  :cycle-detection  :standard))
	   (initial-state (make-new-state :from-qde qde
					  :assert-values initial-values
					  :text (format nil "Home heater initialized with ~S"
							inits)
					  :sim sim)))
      (qsim initial-state)
      (qsim-display initial-state)
      )))
