;; -*- Mode: LISP; Syntax: Common-lisp; Package: QSIM; Base: 10 -*-

(in-package 'qsim)

(defun DIGEST-QUANTITY-SPACE-SPEC (source spec)
  "Given a quantity space spec and the source of the spec, return the corresponding structure."
  (let ((new-qspace nil))
    (when spec
      (if (and (= (length spec) 1)        ; Spec = a symbol, return the symbol.
	       (symbolp (first spec)))
	  (setf new-qspace (car spec))
	(progn
	  (setf new-qspace
		(make-cc-quantity-space :name source
					:landmark-list (first spec)))
	  (when (assoc 'parent spec)
	    (setf (cc-quantity-space-parent-qspace new-qspace)
		  (digest-quantity-space-spec nil (cdr (assoc 'parent spec)))))
	  (setf (cc-quantity-space-correspondences new-qspace)
		(cdr (assoc 'conservation-correspondences spec))))))
    new-qspace))


(defun DIGEST-QUANTITY-SPACE-DEFAULTS (source qspace-defaults-spec domain)
  "Given a quantity-spaces clause, process the quantity space specs."
  (mapcar #'(lambda (default-spec)
	      (cons (car default-spec)
		    (mapcar #'(lambda (type-qspace-spec)
				(cons (if (symbolp (car type-qspace-spec))
					(list domain (car type-qspace-spec))
					(car type-qspace-spec))
				      (digest-quantity-space-spec source (cdr type-qspace-spec))))
			    (cdr default-spec))))
	  qspace-defaults-spec))

(defun FIND-NAMED-QSPACE (qspace-name)
  "Given a name (symbol) for a quantity space, find it on the quantity space list and return it."
  (if (find qspace-name *quantity-spaces* :key 'cc-quantity-space-name :test 'eq)
      (find qspace-name *quantity-spaces* :key 'cc-quantity-space-name :test 'eq)
    (progn
      (format *qsim-report* "~&~% Unknown quantity space ~S.  Assuming base qspace (minf 0 inf)."
	      qspace-name)
      *base-quantity-space*)))


(defun FIND-DEFAULT-QSPACE-OR-PARENT (variable-type qspace-or-parent)
  "Search the stack of qspace defaults for the first relevant definition, and return that definition."
  (let ((default-qs (do* ((specs *quantity-space-defaults* (cdr specs))
			  (qs (cdr (assoc variable-type 
					  (cdr (assoc qspace-or-parent (car specs)))
					  :test #'equal))
			      (cdr (assoc variable-type 
					  (cdr (assoc qspace-or-parent (car specs)))
					  :test #'equal))))
			 ((or (null specs) qs)
			  qs)
			 )))
    (if default-qs
	(if (symbolp default-qs)
	    (find-named-qspace default-qs)
	  default-qs)
      *base-quantity-space*)))


(defun COMPLETE-QUANTITY-SPACE-SPEC (qs type-for-default)
  "Complete the quantity space specification for a component-variable."
  (cond ((null qs)
	 (find-default-qspace-or-parent type-for-default 'defaults))
	((symbolp qs)
	 (find-named-qspace qs))
	((typep qs 'cc-quantity-space)
	 ;; Locate parent qspace, if not defined and not *base-quantity-space*
	 (if (null (cc-quantity-space-parent-qspace qs))
	     (setf (cc-quantity-space-parent-qspace qs)
		   (find-default-qspace-or-parent type-for-default 'hierarchical-parents))
	   (when (symbolp (cc-quantity-space-parent-qspace qs))
		 (setf (cc-quantity-space-parent-qspace qs) 
		       (find-named-qspace (cc-quantity-space-parent-qspace qs)))))
	 qs)
	(t (format *qsim-report* "~&~% Unknown object in quantity space completion: ~S" qs)
	   (find-default-qspace-or-parent type-for-default 'defaults))))


(defun DIGEST-VARIABLE-TYPE-SPEC (var type-spec component-type)
  "Interpret the variable type spec in the context of the default component domain."
  (let ((domain-type-map
	  (if (listp type-spec)
	      ;; Type specification assumed to be of the form ( <domain> <type> )
	      (let ((domain-types (find (first type-spec) *cc-domains* :key #'first)))
		(if domain-types
		    (progn (setf (component-variable-domain var) (first type-spec))
			   (cdr domain-types))
		  (progn (format *qsim-report* "~&~% Unknown domain ~S for variable ~S of component type ~S."
				 (first type-spec) (component-variable-name var) component-type)
			 *generic-variable-types*)))
	    (progn (setf (component-variable-domain var) *component-domain*)
		   *domain-specific-variable-types*)))
	(type-name (if (listp type-spec) (second type-spec) type-spec))
	(type-map nil))
    (setf (component-variable-type var)
	  (if (setf type-map (find type-name domain-type-map :key #'second))
	      (first type-map)
	    (if (setf type-map (find type-name *generic-variable-types* :key #'second))
		(first type-map)
	      (format *qsim-report* "~&~% Unknown type ~S for variable ~S of component type ~S"
		      type-name (component-variable-name var) component-type))))))
	
(defun DIGEST-VARIABLE-SPEC (component-type component-instance-name variable-spec other-data)
  "Create a component variable instance from a variable spec."
  (let ((new-component-variable (make-component-variable :component-name component-instance-name
							 :name (car variable-spec)
							 :name-stack *name-stack*))
	(quantity-space-data (cdr (assoc (car variable-spec) 
					 (cdr (assoc 'quantity-spaces other-data :test #'eq))
					 :test #'eq)))
	(ignore-qdir-variables (cdr (assoc 'ignore-qdir other-data :test #'eq)))
	(display-variables (cdr (assoc 'display other-data :test #'eq)))
	(no-new-landmarks-variables (cdr (assoc 'no-new-landmarks other-data :test #'eq)))
	;; The landmarks and initable options are maintained to support old CC models.
	(landmark-data (cadr (assoc (car variable-spec) 
				    (cdr (assoc 'landmarks other-data :test #'eq))
				    :test #'eq)))
	(initable-variables (cdr (assoc 'initable other-data :test #'eq))))

    ;; Add this variable to the name tree.
    (nconc *name-tree* (list (cons (component-variable-name new-component-variable)
				   new-component-variable)))
    
    ;; Extract the type specifier for this variable.
    (digest-variable-type-spec new-component-variable (second variable-spec) component-type)
    
    ;; Extract optional variable facets.
    (dolist (variable-facet (cddr variable-spec))
      (cond ((listp variable-facet)
	     (case (car variable-facet)
	       (quantity-space 
		 (setf (component-variable-quantity-space new-component-variable)
		       (complete-quantity-space-spec 
			 (digest-quantity-space-spec nil (cdr variable-facet))
			 (if (symbolp (second variable-spec))
			   (list *component-domain* (second variable-spec))
			   (second variable-spec)))))
	       ((landmark-symbol lm-symbol)
		(setf (component-variable-lm-symbol new-component-variable) 
		      (second variable-facet)))
	       (t (format *qsim-report* "~&~% Unknown variable facet ~S for variable ~S of component type ~S"
			  (car variable-facet) (car variable-spec) component-type))))
	    ((symbolp variable-facet)
	     (case variable-facet
	       (independent (setf (component-variable-independent new-component-variable) t))
	       (dependent)
	       (display (setf (component-variable-display new-component-variable) t))
	       (ignore-qdir (setf (component-variable-ignore-qdir new-component-variable) t))
	       (no-new-landmarks (setf (component-variable-new-landmarks new-component-variable) t))
	       (t (format *qsim-report* "~&~% Unknown variable facet ~S for variable ~S of component type ~S"
			  variable-facet  (car variable-spec) component-type))))
	    (t (format *qsim-report* "~&~% Unknown variable facet ~S for variable ~S of component type ~S"
		       variable-facet (car variable-spec) component-type))))
    (when (null (component-variable-quantity-space new-component-variable))
      (setf (component-variable-quantity-space new-component-variable)
	    (complete-quantity-space-spec
	      (digest-quantity-space-spec
		nil (cond (quantity-space-data quantity-space-data)
			  (landmark-data (list (if (listp landmark-data)
						      landmark-data
						      (list '0 landmark-data 'inf))))
			  (t nil)))
	      (if (symbolp (second variable-spec))
	        (list *component-domain* (second variable-spec))
		(second variable-spec)))))
    (setf (component-variable-initable new-component-variable)
	  (member (car variable-spec) initable-variables :test #'eq))
    (setf (component-variable-ignore-qdir new-component-variable)
	  (member (car variable-spec) ignore-qdir-variables :test #'eq))
    (when (member (car variable-spec) display-variables :test #'eq)
      (setf (component-variable-display new-component-variable) t))
    (when (member (car variable-spec) no-new-landmarks-variables :test #'eq)
      (setf (component-variable-new-landmarks new-component-variable) nil))

    new-component-variable))

(defmacro FIND-GLOBAL-COMPONENT-TYPE-DEFAULT (type-name)
  `(find ,type-name *global-configurations* :key 'car))

(defun FIND-MODE-DEFAULTS (instance-name instance-type implementation-name)
  (let ((for-clause (find instance-name (first *local-configurations*) :key 'car)))
    (if for-clause
      (when (or (null (component-type-ref-impl-name for-clause))
		(eq (component-type-ref-impl-name for-clause) implementation-name))
	(component-type-ref-modes for-clause))
      (component-type-ref-modes (find-global-component-type-default instance-type)))))

#||
      (dolist (global-config *global-configurations*)
	(when (and (eq (component-type-ref-type-name global-config) instance-type)
		   (or (null (component-type-ref-impl-name global-config))
		       (eq (component-type-ref-impl-name global-config) implementation-name)))
	  (return (component-type-ref-modes global-config)))))))
||#    
	
(defun RESOLVE-COMPONENT-MODES (local-instance-name instance-type implementation resolved-modes)
  ;; Examine local, then global configuration specifications in an attempt to resolve
  ;; any unresolved mode variables.  Resolution may not be possible at this time.
  (let ((mode-defaults (find-mode-defaults local-instance-name instance-type 
					 (component-implementation-name implementation)))
	(mode-variables (mapcar 'first 
				(component-implementation-mode-variables implementation))))
    (dolist (mode-variable-default mode-defaults)
      (unless (member (first mode-variable-default) resolved-modes :key 'first)
	(if (member (first mode-variable-default) mode-variables)
	  (setf resolved-modes (cons mode-variable-default resolved-modes))
	  (progn 
	    (format *qsim-report*
		    "~&~% ~S is not a mode variable for component instance ~S, (type ~S)."
		    (first mode-variable-default) local-instance-name instance-type)))))
    resolved-modes))


(defun RESOLVE-COMPONENT-TYPE-REFERENCE 
  (local-instance-name component-instance-name component-type-ref &optional (instantiation nil))
  "Given a component-type reference of the form TYPE-NAME or (TYPE-NAME (IMPL name) (MODE name)), return the appropriate component-implementation structure, or nil."
  ;; If component-type-ref is a symbol, it specifies either a component interface name or
  ;; a configuration name.  The list of component interfaces is searched first.  If found,
  ;; the first implementation is used (if none was specified).  If more than one imple-
  ;; mentation exists, the user is notified that a default implementation has been used.
  ;; If no component interface exists by this name, the list of configurations is searched
  ;; for a configuration by the specified name.  If found, the configuration will reference
  ;; a component interface and possibly an implementation.  If only an interface is specified,
  ;; an implementation is assumed as before.  Modes are resolved in a similar manner.
  ;; 
  (prog
    ((instance-type (component-type-ref-type-name component-type-ref))
     (instance-impl (component-type-ref-impl-name component-type-ref))
     (instance-mode (component-type-ref-modes component-type-ref))
     (component-interface nil)
     (component-definition nil)
     (configuration nil)
     (for-clause nil))

    (when (null (setf component-interface (find instance-type *component-definitions* 
						:key 'component-interface-name)))
      ;; May be a configuration name.  Check the list of configurations.
      (unless (setf configuration (find instance-type *configurations*
					:key 'configuration-name))
	(format *qsim-report* "~&~% Unknown component type ~S, declaration ignored." instance-type)
	(return (values nil nil)))
      (when instantiation
	(setf (instantiated-cc-model-configuration-name instantiation)
	      (configuration-name configuration)))
      ;; Configuration.  Determine the associated component interface.
      (setf component-interface (configuration-component configuration))
      (when (typep component-interface 'component-implementation)
	(setf component-definition component-interface)
	(setf component-interface (component-implementation-interface component-definition))
	(when instance-impl
	  (format *qsim-report* "~& ** Configuration ~S specifies implementation ~S for component type ~S and overrides implementation ~S"
		  (configuration-name configuration)
		  (component-implementation-name component-definition)
		  (component-interface-name component-interface)
		  instance-impl))
	(setf instance-impl (component-implementation-name component-definition))))

    (when (null (component-interface-implementations component-interface))
      (format *qsim-report* "~& ** No implementations have been specified for component interface ~S. **"
	      instance-type)
      (return (values nil nil)))

    ;; If an implementation name supplied, process it.
    (when instance-impl
      (when (setf component-definition 
		  (find instance-impl (component-interface-implementations component-interface)
			:key 'component-implementation-name))
	(when (component-implementation-mode-variables component-definition)
	  (setf instance-mode
		(resolve-component-modes local-instance-name instance-type
					 component-definition instance-mode)))
	(return (values component-definition instance-mode)))

      ;; May be a configuration name.  Check the list of configurations.
      (unless (and (setf configuration (find instance-impl *configurations*
					     :key 'configuration-name))
		   (eq component-interface (configuration-component-interface configuration)))
	(format *qsim-report* "~& ** Unknown implementation ~S of component-interface ~S specified for component ~S. **"
		instance-impl instance-type component-instance-name)
	(return (values nil nil)))
      ;; A configuration.  If implementation specified, return it.
      (when (configuration-component-implementation configuration)
	(return (values (configuration-component-implementation configuration)
			(resolve-component-modes
			  local-instance-name
			  (component-interface-name (configuration-component-interface configuration))
			  (configuration-component-implementation configuration)
			  instance-mode)))))

    ;; No specific implementation specified.  Check the local and global configurations for
    ;; implementation information.  If none found, select default implementation.
    (when (setf for-clause (find local-instance-name (first *local-configurations*) :key 'car))
      (when (setf component-definition
		  (find (setf instance-impl (component-type-ref-impl-name for-clause))
			(component-interface-implementations component-interface)
			:key 'component-implementation-name))
	(when (component-implementation-mode-variables component-definition)
	  (setf instance-mode (resolve-component-modes local-instance-name instance-type
						       component-definition instance-mode)))
	(return (values component-definition instance-mode))))
    (when (setf for-clause (find-global-component-type-default instance-type))
      (when (setf component-definition
		  (find (setf instance-impl (component-type-ref-impl-name for-clause))
			(component-interface-implementations component-interface)
			:key 'component-implementation-name))
	(when (component-implementation-mode-variables component-definition)
	  (setf instance-mode (resolve-component-modes local-instance-name instance-type
						       component-definition instance-mode)))
	(return (values component-definition instance-mode))))

    ;; Return the default implementation.
    (progn (setf component-definition 
		 (first (component-interface-implementations component-interface)))
	   (when (> (length (component-interface-implementations component-interface)) 1)
	     (format *qsim-report* "~& ** Implementation ~S assumed for component ~S. **"
		     (component-implementation-name component-definition)
		     component-instance-name)))

    (return (values component-definition nil))))


(defun RESOLVE-EMBEDDED-VARIABLE-REFERENCE (variable-reference component-variable-alist)
  (when *model-macro-trace-flag*
    (format *qsim-report* "~&  Attempting to perform structure substitution for variable reference ~S"
	    variable-reference))
  (or (cdr (assoc variable-reference component-variable-alist :test #'eq))
      (car (find-all-matches-in-tree (if (symbolp variable-reference)
				       (list variable-reference)
				       variable-reference)
				     *name-tree*))))


(defun DIGEST-CONSTRAINT-SPEC (component constraint-spec component-variable-alist)
  "Substitute symbolic variable references with the corresponding component variable structure instances."
  (let ((constraint-type (find (caar constraint-spec) *known-constraint-types* :key #'contype-name)))
    (if constraint-type
      (let ((arg-count (contype-nargs constraint-type)))
	(mapl #'(lambda (variable-reference-list)
		  (when (> arg-count 0)
		    (let* ((variable-reference (car variable-reference-list))
			   (component-variable-structure-instance (resolve-embedded-variable-reference
								    variable-reference
								    component-variable-alist)))
		      (setf arg-count (1- arg-count))
		      (if component-variable-structure-instance
			(setf (car variable-reference-list) component-variable-structure-instance)
			(format *qsim-report* 
				"~&~% Unknown variable name ~S in constraint ~S of component type (~S (impl ~S))"
				variable-reference constraint-spec 
				(component-interface-name
				  (component-implementation-interface component))
				(component-implementation-name component)
				)))))
	      (cdar constraint-spec)))
      (format *qsim-report* "~&~% Unknown constraint type ~S of component type (~S (impl ~S))"
	      (caar constraint-spec) 
	      (component-interface-name
		(component-implementation-interface component))
	      (component-implementation-name component))))
  constraint-spec)


(defun DIGEST-MODE-CONDITION (component mode-condition component-variable-alist)
  "Substitute symbolic variable references with the corresponding component variable structure instances."
  (case (first mode-condition)
    ((and or not)
     (dolist (clause (cdr mode-condition))
       (digest-mode-condition component clause component-variable-alist)))
    (otherwise  ; Assumed to be a variable reference.
      (let* ((variable-reference (first mode-condition))
	     (cv (resolve-embedded-variable-reference variable-reference component-variable-alist)))
	(if cv
	  (setf (first mode-condition) cv)
	  (format *qsim-report* "~&~% Unknown variable name ~S in mode condition ~S of component type (~S (impl ~S))"
		  variable-reference mode-condition
		  (component-interface-name (component-implementation-interface component))
		  (component-implementation-name component)
		  )))))
  mode-condition)


(defun SET-OPTIONS-FOR-MODEL-VARIABLE (model-variable)
  "Given a model variable, set options (initable, idnore-qdir, new-landmarks)."
  (dolist (cv (model-variable-component-variables model-variable) nil)
    (when (component-variable-independent cv)
      (setf (model-variable-independent model-variable) t))
    (when (component-variable-display cv)
      (setf (model-variable-display model-variable) t))
    (when (component-variable-initable cv)
      (setf (model-variable-initable model-variable) t))
    (when (component-variable-ignore-qdir cv)
      (setf (model-variable-ignore-qdir model-variable) t))
    (when (null (component-variable-new-landmarks cv))
      (setf (model-variable-new-landmarks model-variable) nil))
    (setf (component-variable-model-variable cv) model-variable)))

(defun CONNECTION-SPEC-WITH-FULL-COMPONENT-NAMES (connection-spec instance-name)
  (mapcar #'(lambda (terminal-spec)
	      (if (symbolp terminal-spec)
		terminal-spec
		(cons (intern (concatenate 'string (string instance-name)
					   "_" (string (first terminal-spec))))
		      (cdr terminal-spec))))
	  connection-spec))


(defun PROCESS-TERMINAL-SPECIFICATIONS (definition instance-name other-data)
  ;; Process terminal specifications
  (let ((terminal-variable-alist nil)
	(component-variable-alist nil))
    (dolist (terminal-spec (component-implementation-terminal-variables definition))
      (push (cons (car terminal-spec)
		  (mapcar #'(lambda (variable-spec)
			      (let ((new-component-variable
				      (digest-variable-spec (component-implementation-type definition)
							    instance-name
							    variable-spec other-data)))
				(push (cons (car variable-spec) new-component-variable)
				      component-variable-alist)
				new-component-variable))
			  (cdr terminal-spec)))
	    terminal-variable-alist))
    (values terminal-variable-alist component-variable-alist)))


(defun PROCESS-COMPONENT-VARIABLE-SPECIFICATIONS (definition instance-name other-data)
  ;; Process component variable specifications.  An alist of local variable name and
  ;; component-variable instance is returned.
  (let ((component-variable-alist nil))
    (dolist (component-variable-spec 
	      (component-implementation-component-variables definition))
      (let ((new-component-variable
	      (digest-variable-spec (component-implementation-type definition) 
				    instance-name
				    component-variable-spec
				    other-data)))
	(push (cons (car component-variable-spec) new-component-variable)
	      component-variable-alist)
	(push (make-model-variable
		:type (component-variable-type new-component-variable)
		:domain (component-variable-domain new-component-variable)
		:quantity-space (component-variable-quantity-space new-component-variable)
		:component-variables (list new-component-variable))
	      *model-variables*)
	(set-options-for-model-variable (car *model-variables*))
	(setf (model-variable-name (car *model-variables*))
	      (intern (concatenate 'string (string instance-name) "."
				   (string (first component-variable-spec)))))
	))
    component-variable-alist))


(defun GENERATE-TRANSITION-EXPRESSIONS (mode)
  ;; Generate the transition expressions for each value of the mode variable MODE.
  ;; These expressions will be placed in the transitions clause of QDE's.
  (dolist (value-condition (mode-variable-condition-alist mode))
    (let ((expressions nil)
	  (mode-value (first value-condition))
	  (condition (second value-condition)))
      ;; Just do simple condititions for now (no ANDs, ORs, or NOTs yet).
      (let ((condition-variable (first condition))
	    (qmag (first (second condition)))
	    (qdir (second (second condition))))
	(push (cons mode-value 
		    (if (listp qmag)
		      (list (list condition-variable (list (first qmag) 'inc))
			    (list condition-variable (list (second qmag) 'dec)))
		      (list (list condition-variable (list qmag (if (eq qdir 'ign) nil qdir))))))
	      (mode-variable-transition-expressions mode))))))


(defun PROCESS-MODE-CONDITION-SPECS (condition-specs definition component-variable-alist)
  ;; Digest mode conditions, including any :discontinuous-transistion specs.
  (mapcar #'(lambda (value-condition)
	      (let ((digested-condition (list (first value-condition)
					      (digest-mode-condition definition
								     (copy-tree (third value-condition))
								     component-variable-alist))))
		(when (cdddr value-condition)
		  (nconc digested-condition
			 (cons (fourth value-condition)
			       (mapcar #'(lambda (v) 
					   (list (resolve-embedded-variable-reference 
						   (first v)
						   component-variable-alist)
						 (second v)))
				       (cddddr value-condition)))))
		digested-condition))
	  condition-specs))


(defun PROCESS-MODE-VARIABLE-SPECIFICATIONS (definition instance-name other-data component-variable-alist)
  ;; Process mode variable specifications.  An alist of local variable name and
  ;; component-variable instance is returned.
  (dolist (mode-variable-spec (component-implementation-mode-variables definition))
    (let ((new-component-variable (make-component-variable :component-name instance-name
							   :name (first mode-variable-spec)
							   :name-stack *name-stack*
							   :type 'mode
							   :quantity-space (cdr mode-variable-spec))))
      ;; Add this variable to the name tree.
      (nconc *name-tree* (list (cons (first mode-variable-spec) new-component-variable)))
      
      (push (cons (first mode-variable-spec) new-component-variable)
	    component-variable-alist)
      (push (make-mode-variable
	      :name (intern (concatenate 'string (string instance-name) "."
					 (string (first mode-variable-spec))))
	      :display-text (format nil "~S"
				    (reverse (cons (component-variable-name new-component-variable)
						   (component-variable-name-stack new-component-variable))))
	      :display (find (first mode-variable-spec) (cdr (assoc 'display other-data)))
	      :static (symbolp (second mode-variable-spec))
	      :quantity-space (delete :discontinuous-transition
				      (mapcar #'(lambda (value-form)
						  (if (symbolp value-form)
						      value-form
						    (first value-form)))
					      (cdr mode-variable-spec)))
	      :condition-alist (when (listp (second mode-variable-spec))
				 (process-mode-condition-specs (cdr mode-variable-spec)
							       definition
							       component-variable-alist)))
	    *mode-variables*)
      (setf (component-variable-model-variable new-component-variable) (car *mode-variables*))
      (generate-transition-expressions (car *mode-variables*))))
    component-variable-alist)

(defun PUSH-CONSTRAINT (constraint &optional (index nil))
  ;; Add the constraint to *CONSTRAINTS*, indexed by INDEX.
  (let ((indexed-list (assoc index *constraints* :test #'equal)))
    (if indexed-list
      (nconc indexed-list (list constraint))
      (push (list index constraint) *constraints*))))


(defun PROCESS-CONSTRAINT-SPECIFICATIONS 
  (component selected-modes component-instance-name component-variable-alist)
  ;; Digest constraints specifications.  Constraints are placed on an alist indexed by
  ;; the assumed modes (a qualified variable name and a value).  Common constraints have
  ;; index NIL.
  ;; If SELECTED-MODES is non-nil, the constraints selected by the specified mode
  ;; variable/value pairs are added to the common constraint list, and the (qualified)
  ;; mode variable name and its value are pushed onto the list *MODE-ASSUMPTIONS*.  
  (let ((common-constraints nil)
	(mode-indexed-constraints nil)
	(mode-variables nil))

    ;; Extract mode variables from COMPONENT-VARIABLE-ALIST.
    (dolist (cv component-variable-alist)
      (when (eq (component-variable-type (cdr cv)) 'mode)
	(push (cdr cv) mode-variables)))


    ;; Extract individual constraints.
    (dolist (constraint-spec (component-implementation-constraints component))
      (if (find (first (first constraint-spec)) mode-variables :key 'component-variable-name)
	(push constraint-spec mode-indexed-constraints)
	;; Could check to see if second of list is "->", indicating a mode group.
	(push constraint-spec common-constraints)))
    (setf common-constraints (nreverse common-constraints))

    ;; If SELECTED-MODES nonNIL, check for correctness.
    (when selected-modes
      (dolist (mode-spec selected-modes)
	(let ((var (find (first mode-spec) mode-variables :key 'component-variable-name)))
	  (if var
	    (push (list (mode-variable-name (component-variable-model-variable var))
			(second mode-spec))
		  *mode-assumptions*)
	    (format *qsim-report* "~& ** Unknown mode ~S specified for component ~S. **"
		    mode-spec component-instance-name)))))

    (dolist (constraint-spec common-constraints)
      (push-constraint (digest-constraint-spec component
					       (copy-tree constraint-spec)
					       component-variable-alist)))

    (dolist (mode-group mode-indexed-constraints)
      (let* ((index (list (mode-variable-name
			    (component-variable-model-variable
			      (find (first (first mode-group)) mode-variables 
				    :key 'component-variable-name)))
			  (second (first mode-group)))))
	(dolist (constraint-spec (cddr mode-group))
	  (push-constraint (digest-constraint-spec component 
						   (copy-tree constraint-spec) component-variable-alist)
			   index))))
    ))


(defun PROCESS-DANGLING-SUBCOMPONENT-TERMINALS (component-terminal-alist connection-specs)
  ;; Identify subcomponent terminals that are not connected to anything, and process them.
  (dolist (comp-terminal-alist component-terminal-alist)
    (let ((component-name (car comp-terminal-alist))
	  (terminal-alist (cdr comp-terminal-alist)))
      (dolist (term-alist terminal-alist)
	(let ((terminal (car term-alist))
	      (terminal-referenced-p nil))
	  (dolist (connection-spec connection-specs)
	    (dolist (terminal-spec connection-spec)
	      (when (and (listp terminal-spec)
			 (eql component-name (first terminal-spec))
			 (eql terminal (second terminal-spec)))
		(setf terminal-referenced-p t))))
	  (unless terminal-referenced-p
	    ;; Process this subcomponent terminal, as it is unconnected.
	    (when (cdr term-alist)
              (generate-connection-model-variables terminal (cdr term-alist) component-name)))
	  )))))


(defun COLLECT-TERMINAL-VARIABLES-TO-RETURN (instance-name 
					      terminal-variable-alist
					      terminal-connection-alist
					      component-terminal-alist)
  ;; Collect terminal variables to be returned to higher level.
  ;; terminal-variable-alist is a list of lists whose CARs are a terminal name and
  ;;    whose CDRs are a list of variables defined at that terminal.
  ;; terminal-connection-alist is a list of lists whose CAR are a terminal name and
  ;;    whose CDRs are the connection spec referencing that terminal.
  ;; component-terminal-alist is a list of lists whose CARs are a component instance
  ;;    name and whose CDRs are terminal-variable-alists.
  (let ((tv-alist
	  (mapcar #'(lambda (alist-item)
		      (let ((variables nil)
			    (terminal (car alist-item))
			    (connection-spec (cdr alist-item)))
			(dolist (terminal-spec connection-spec)
			  (when (listp terminal-spec) ; Only (sub)component contribute variables
			    (setf variables
				  (nconc variables
					 (cdr (assoc (second terminal-spec)
						     (cdr (assoc (intern 
								   (concatenate 
								     'string
								     (string instance-name)
								     "_"
								     (string (first terminal-spec))))
								 component-terminal-alist))))))))
			(cons terminal variables)))
		  terminal-connection-alist)))
    (when (and terminal-variable-alist tv-alist)
      ;; Merge these two lists.  Association is based on terminal name.
      (dolist (term-var-list terminal-variable-alist)
	(let ((candidate (assoc (car term-var-list) tv-alist)))
	  (if candidate
	      (nconc candidate (cdr term-var-list))
	    (nconc tv-alist (list term-var-list))))))
    (cons instance-name (if tv-alist tv-alist terminal-variable-alist))))

(defun INTERPRET-COMPONENT-REFERENCE-INTERNAL
       (component-instance-name component-definition instance-mode other-data)
  "Given a component declaration (instance name, type, and landmark info), create the component-variables and constraints associated with the specified component type."
  (let ((component-variable-alist nil)
	(terminal-variable-alist nil)
	(component-terminal-alist nil)
	(terminal-connection-alist nil)
	(terminal nil))

    ;; Process terminal specifications
    (multiple-value-setq (terminal-variable-alist component-variable-alist)
      (process-terminal-specifications component-definition component-instance-name other-data))

    ;; Process component variable specifications
    (setf component-variable-alist
	  (nconc component-variable-alist
		 (process-component-variable-specifications component-definition 
							    component-instance-name
							    other-data)))

    ;; Process component specifications
    (dolist (component-spec (component-implementation-components component-definition))
      (push (interpret-component-reference 
	      (first component-spec)
	      (intern (concatenate 'string (string component-instance-name)
				   "_" (string (first component-spec))))
	      (second component-spec) (cddr component-spec))
	    component-terminal-alist))

    ;; Process mode variable specifications
    (setf component-variable-alist
	  (process-mode-variable-specifications component-definition 
						component-instance-name
						other-data
						component-variable-alist))

    ;; Process constraint specifications
    (process-constraint-specifications 
      component-definition instance-mode component-instance-name component-variable-alist)

    ;; Process connection specifications
    ;; Connections not involving terminals of the composed component have model variables
    ;; generated at this time.  Connections involving terminals of the composed component
    ;; are returned to the higher level for handling.
    (dolist (connection-spec (component-implementation-connections component-definition))
      (if (setf terminal 
		(some #'(lambda (terminal-spec)
			  (and (symbolp terminal-spec)
			       (member terminal-spec 
				       (component-implementation-terminals component-definition))
			       terminal-spec))
		      connection-spec))
	  (push (cons terminal connection-spec) terminal-connection-alist)
	  (interpret-component-connection
	    component-instance-name
	    (connection-spec-with-full-component-names connection-spec component-instance-name)
	    component-terminal-alist)))

    ;; Process subcomponent terminals that are dangling.  Presumably, they will be
    ;; "clamped" with some initial value by the modeler.  A terminal variable involved
    ;; in the connection from the subcomponent will be used to access the associated
    ;; model variables for initialization purposes.
    (process-dangling-subcomponent-terminals
      component-terminal-alist 
      (mapcar #'(lambda (connection-spec)
		  (connection-spec-with-full-component-names connection-spec 
							     component-instance-name))
	      (component-implementation-connections component-definition)))

    ;; Collect terminal variables to be returned to higher level.
    (collect-terminal-variables-to-return component-instance-name 
					      terminal-variable-alist
					      terminal-connection-alist
					      component-terminal-alist)
    ))

(defun GET-CONFIGURATION (component-type-ref)
  ;; Given component-type-ref, return any explicit or implied configuration name.
  (let ((component (find (component-type-ref-type-name component-type-ref)
			 *component-definitions* :key 'component-interface-name))
	(impl-name (or (component-type-ref-impl-name component-type-ref)
		       (let ((default-clause (find-global-component-type-default
					       (component-type-ref-type-name component-type-ref))))
			 (component-type-ref-impl-name default-clause)))))
    (if (listp component-type-ref)
      (unless (member impl-name (component-interface-implementations component)
		      :key 'component-implementation-name)
	;; Assume that impl-name is a configuration.
	(find impl-name *configurations* :key 'configuration-name))
      (if component
	(when (and impl-name (null (member impl-name (component-interface-implementations component)
					   :key 'component-implementation-name)))
	  (find impl-name *configurations* :key 'configuration-name))
	(find component-type-ref *configurations* :key 'configuration-name)))))


(defun BIND-CONFIGS (component-type-ref config-info-type)
  "Extract local configuration information per Component-Type-Reference."
  ;; If component-type-reference is a configuration name or specifies an implementation name
  ;; which is a configuration, some local (instance specific) configuration information may be
  ;; specified.
  (let ((config (get-configuration component-type-ref))
	(local-configs nil))
    (when config
      ;; Collect local configuration info (i.e. specified via instance name)
      (mapc #'(lambda (for-clause)
		(if (member (first for-clause) *component-definitions* :key 'component-interface-name)
		    (when (eq config-info-type 'global) (push for-clause local-configs))
		    (when (eq config-info-type 'local) (push for-clause local-configs))))
	    (configuration-for-clauses config))
      local-configs)))


(defun INTERPRET-COMPONENT-REFERENCE (local-instance-name component-instance-name component-type-ref other-data)
  "Given a component declaration (instance name, type, and landmark info), determine the component implementation type (primitive or composed) and handle accordingly."
  (let ((new-name-tree-entry (list local-instance-name)))
    (push local-instance-name *name-stack*)
    (nconc *name-tree* (list new-name-tree-entry))
    (let ((*name-tree* new-name-tree-entry)
	  (return-value nil))
      (setf return-value
	    (multiple-value-bind (component-definition mode)
	          (resolve-component-type-reference local-instance-name component-instance-name
						    component-type-ref)
	      (let ((comp-if (component-implementation-interface component-definition)))
		(nconc *name-tree* (list (component-info-name comp-if)))
		(let ((*quantity-space-defaults*
			(cons (component-info-quantity-space-info component-definition)
			      (cons (component-info-quantity-space-info comp-if)
				    *quantity-space-defaults*)))
		      (*local-configurations* (cons (bind-configs component-type-ref 'local)
						    *local-configurations*))
		      (*global-configurations* (nconc (bind-configs component-type-ref 'global)
						      *global-configurations*))
		      (*domain-specific-variable-types*
			(let ((type-map (find (component-interface-domain comp-if) *cc-domains* :key #'first)))
			  (if type-map
			    (cdr type-map)
			    (progn (unless (component-interface-domain comp-if)
				     (format *qsim-report* 
					     "~& ** Unrecognized domain name ~A for component type ~A. **"
					     (component-interface-domain comp-if)
					     (component-interface-name comp-if)))
				   *generic-variable-types*))))
		      (*component-domain*
			(when (find (component-interface-domain comp-if) *cc-domains* :key #'first)
			  (component-interface-domain comp-if))))
		  (interpret-component-reference-internal component-instance-name 
							  component-definition mode other-data)))))
      (pop *name-stack*)
      return-value)))

(defun GENERATE-COMPONENT-AND-VARIABLE-NAME-STRING (component-variable)
  (concatenate 'string (string (component-variable-component-name component-variable))
	       "." (string (component-variable-name component-variable))))

(defun RESOLVE-QSPACE-REFERENCE (qs)
  "If a quantity space refernce is a symbol, resolve it."
  (when qs 
    (if (symbolp qs) (find-named-qspace qs) qs)))

(defun PARENT-QSPACE-P (qspace1 qspace2)
  "Determine if QSPACE1 is equal to or the parent of QSPACE2."
  (do ((qs (resolve-qspace-reference qspace2)
	   (resolve-qspace-reference (cc-quantity-space-parent-qspace qs))))
      ((or (eq qspace1 qs)
	   (null qs))
       qs)))

(defun FIND-COMMON-PARENT-QSPACE (qspace1 qspace2)
  "Determine the common parent of two quantity spaces."
  (do ((qs (resolve-qspace-reference qspace1)
	   (resolve-qspace-reference (cc-quantity-space-parent-qspace qs))))
      ((or (parent-qspace-p qs qspace2)
	   (null qs))
       qs)))

(defun MERGE-QUANTITY-SPACES (qspace1 qspace2)
  "Merge two quantity spaces.  If no common parent is available, report as a problem."
  (let ((parent (find-common-parent-qspace qspace1 qspace2)))
    (if (null parent)
	(progn
	  (break "qspace merge problem")
	  (format *qsim-report* "~&~% Unable to merge quantity spaces ~S and ~S." qspace1 qspace2)
	  *base-quantity-space*)
      (cond ((eq parent qspace1) qspace1)
	    ((eq parent qspace2) qspace2)
	    (t (let ((new-qspace (make-cc-quantity-space :name 'merge :parent-qspace parent))
		     (new-landmark-list nil))
		 (mapc #'(lambda (lm) (when (or (member lm (cc-quantity-space-landmark-list qspace1))
						(member lm (cc-quantity-space-landmark-list qspace2)))
					(push lm new-landmark-list)))
		       (cc-quantity-space-landmark-list parent))
		 (setf (cc-quantity-space-landmark-list new-qspace) (nreverse new-landmark-list))
		 new-qspace))))))

(defun GENERATE-CONNECTION-SHARED-MODEL-VARIABLE
       (shared-component-variables connection-name component-prefix variable-type)
  "Given a list of component variables, create the model variable which represents effort at their connection."
  (let ((name-tree-entries nil))
    (let* ((domain (component-variable-domain (first shared-component-variables)))
	   (new-effort-model-variable 
	     (make-model-variable :type 'effort
				  :domain domain
				  :name (intern (concatenate 'string 
							     (string component-prefix)
							     "." (string variable-type) "_"
							     (string domain)
							     "@" (string connection-name))))))
      
      (push new-effort-model-variable *model-variables*)
      (setf (model-variable-component-variables new-effort-model-variable) shared-component-variables)
      (set-options-for-model-variable new-effort-model-variable)
      (when *trace-connection-processing*
	(format *qsim-report* "~& Model variable ~A generated for shared component variables:"
		(model-variable-name new-effort-model-variable)))
      
      (dolist (ccv shared-component-variables)
	(when *trace-connection-processing*
	      (format *qsim-report* " ~A" (generate-component-and-variable-name-string ccv)))
	(unless (eq (component-variable-quantity-space ccv) *base-quantity-space*)
	  ;; Merge this quantity space into that already defined for the model variable.
	  (setf (model-variable-quantity-space new-effort-model-variable)
		(if (null (model-variable-quantity-space new-effort-model-variable))
		  (component-variable-quantity-space ccv)
		  (merge-quantity-spaces (model-variable-quantity-space new-effort-model-variable)
					 (component-variable-quantity-space ccv))))))
      (when (null (model-variable-quantity-space new-effort-model-variable))
	    (setf (model-variable-quantity-space new-effort-model-variable) *base-quantity-space*))
      
      (nconc *name-tree* 
	     (list (list domain 'domain (cons variable-type (car shared-component-variables)))))
      (push (cons domain (car shared-component-variables)) name-tree-entries)
      
      (setf shared-component-variables (delete domain shared-component-variables
					       :key 'component-variable-domain)))
    (let ((default (find *component-domain* name-tree-entries :key #'first)))
      (if default
	(push (cons 'default (cdr default)) name-tree-entries)
	(push (cons 'default (cdr (car name-tree-entries))) name-tree-entries)))
    (nconc *name-tree*
	   (list (nconc (list variable-type 'type) name-tree-entries)))))

(defun CORRESPONDENCE-IN-QUANTITY-SPACES-P (correspondence qspaces)
  ;; Determine whether a correspondence should be included in a constraint by examining the landmarks of
  ;; the correspondence and the quantity spaces of interest.  For example, the correspondences
  ;; (INF MINF), (0 0) and (MINF INF) can be checked for the MINUS constraint.
  (if (null correspondence)
      t
    (and (member (car correspondence) (car qspaces) :test #'eq)
	 (correspondence-in-quantity-spaces-p (cdr correspondence) (cdr qspaces)))))



(defun CORRESPONDENCES-FOR-FLOW-MINUS-CONSTRAINT (model-var-1 model-var-2)
  (let* ((correspondences nil)
	 (parent (find-common-parent-qspace (model-variable-quantity-space model-var-1)
					    (model-variable-quantity-space model-var-2)))
	 (candidate-correspondences (cc-quantity-space-correspondences
				     (if parent parent *base-quantity-space*))))
    (do ((candidate (pop candidate-correspondences) (pop candidate-correspondences)))
	((null candidate))
      (when (eql (length candidate) 2)       ; Only want correspondences between two variables here.
	(when (correspondence-in-quantity-spaces-p
	        candidate
		(list (cc-quantity-space-landmark-list (model-variable-quantity-space model-var-1))
		      (cc-quantity-space-landmark-list (model-variable-quantity-space model-var-2))))
	  (push candidate correspondences))
	(when (not (eq (first candidate) (second candidate)))
	  (when (correspondence-in-quantity-spaces-p
	          (reverse candidate)
		  (list (cc-quantity-space-landmark-list (model-variable-quantity-space model-var-1))
			(cc-quantity-space-landmark-list (model-variable-quantity-space model-var-2))))
	    (push (list (second candidate) (first candidate)) correspondences)))))
    correspondences))

(defun FIND-PARENT-QSPACE-OF-VARIABLES (current-parent variable-list)
  "Find the common qspace parent of a list of variables."
  (if (null variable-list)
      current-parent
    (let ((next-qspace (component-variable-quantity-space (car variable-list))))
      (find-parent-qspace-of-variables (find-common-parent-qspace current-parent next-qspace)
				       (cdr variable-list)))))

(defun CANDIDATE-CONSERVATION-CORRESPONDENCES (count qspace)
  "Return a list of conservation correspondences of length COUNT."
  (let ((correspondences nil))
    (mapc #'(lambda (correspondence) (when (eql (length correspondence) count)
				       ;; Eventually, make all convolutions of correspondence
				       (push correspondence correspondences)
				       (push (reverse correspondence) correspondences)))
	  (cc-quantity-space-correspondences qspace))
    correspondences))

(defun LIST-OF-ZEROES (COUNT)
  (if (<= count 0) nil (cons 0 (list-of-zeroes (1- count))))) 

(defun CORRESPONDENCE-CONVOLUTION (selected-variables selections-remaining remaining-variables)
  "Generation the convolution product of correspondences (recursively)."
  ;; Returns two values, the list of non-zero correspondence values (not yet used)
  ;; and the partial correspondences generated so far.
  (let ((correspondences nil))
    ;; If all selections have been made, generate correspondences, possibly with
    ;; additional zeroes for remaining (unselected) variables.
    (if (eql selections-remaining 0)
	(let ((conservation-correspondences
	        (candidate-conservation-correspondences
		  (length selected-variables)
		  (find-parent-qspace-of-variables (component-variable-quantity-space
						     (car selected-variables))
						   (cdr selected-variables)))))
	  (if (null conservation-correspondences)
	      (values nil nil)
	    (let ((variable-qspaces (mapcar #'(lambda (variable)
						(cc-quantity-space-landmark-list
						  (component-variable-quantity-space variable)))
					    selected-variables)))
						      
	      (mapc #'(lambda (correspondence)
			(when (correspondence-in-quantity-spaces-p correspondence variable-qspaces)
			  (push correspondence correspondences)))
		    conservation-correspondences)
	      (values correspondences 
		      (let ((remaining (list-of-zeroes (length remaining-variables))))
			(mapcar #'(lambda (correspondence) remaining) correspondences))))))
      ;; Collect correspondences including this variable (CAR of remaining variables).
      (let ((returned-non-zero-values nil))
	(multiple-value-bind (non-zero-correspondence-values partial-correspondences)
 	    (correspondence-convolution (cons (car remaining-variables) selected-variables)
					(1- selections-remaining)
					(cdr remaining-variables))
	  (setf correspondences (mapcar #'(lambda (non-zero-values partial-correspondence)
					    (cons (car non-zero-values) partial-correspondence))
					non-zero-correspondence-values
					partial-correspondences))
	  (setf returned-non-zero-values (mapcar 'cdr non-zero-correspondence-values)))
	;; If this variable need not be selected (i.e. selections-remaining >= remaining-variables)
	;; recurse assuming this variable will be 0 in correspondences.
	(when (< selections-remaining (length remaining-variables)) 
	  (multiple-value-bind (non-zero-correspondence-values partial-correspondences)
	      (correspondence-convolution selected-variables
					  selections-remaining
					  (cdr remaining-variables))
	    (setf correspondences 
		  (nconc (mapcar #'(lambda (correspondence) (cons 0 correspondence))
				 partial-correspondences)
			 correspondences))
	    (setf returned-non-zero-values
		  (nconc non-zero-correspondence-values returned-non-zero-values))))
	(values returned-non-zero-values correspondences)))))



(defun CORRESPONDENCES-FOR-FLOW-SUM-ZERO-CONSTRAINT (variable-list)
  "Generate corresponding values for SUM-ZERO constraint."
  ;; Generate correspondences from two element conservation correspondences.  All variable
  ;; pairs of the constraint are considered, and other variables (of a correspondence) have
  ;; value 0.
  (let ((correspondences nil))
    (do ((count 2 (1+ count)))
	((> count (length variable-list))
	 correspondences)
      (multiple-value-bind (ignore correspondence-list)
          (correspondence-convolution nil count variable-list)
	(setf correspondences (nconc correspondences correspondence-list))))))
    

(defun GENERATE-CONNECTION-SUM-ZERO-MODEL-VARIABLES-AND-CONSTRAINTS (sum-zero-component-variables connection-name)
  "Given a list of component variables associated with flow, generate the associated model variables and constraints."
  (dolist (ccv sum-zero-component-variables)
    (push (make-model-variable :type 'flow :independent (component-variable-independent ccv)
			       :domain (component-variable-domain ccv)
			       :quantity-space (component-variable-quantity-space ccv)
			       :component-variables (list ccv)
			       :display (component-variable-display ccv)
			       :name (intern 
				       (concatenate 'string
						    (string (component-variable-component-name ccv))
						    "." (string (component-variable-name ccv))
						    "@" (string connection-name))))
	  *model-variables*)
    (set-options-for-model-variable (car *model-variables*)))

  (when (or *trace-KCL-constraint-application* *trace-connection-processing*)
    (format *qsim-report* "~& Applying KCL constraint to variables:")
    (dolist (cv sum-zero-component-variables)
      (format *qsim-report* " ~A" 
	      (model-variable-name (component-variable-model-variable cv)))))
  
  (cond ((= (length sum-zero-component-variables) 2)
	 (let ((new-constraint
		 (nconc (list (list 'MINUS
				    (model-variable-name (component-variable-model-variable
							   (first sum-zero-component-variables)))
				    (model-variable-name (component-variable-model-variable
							   (second sum-zero-component-variables)))))
			(correspondences-for-flow-minus-constraint
			  (component-variable-model-variable (first sum-zero-component-variables))
			  (component-variable-model-variable (second sum-zero-component-variables))))))
	   (push-constraint new-constraint)
	   (when *trace-KCL-constraint-application*
	     (format *qsim-report* "~&  Constraint generated from KCL: ~S" new-constraint))))
	((> (length sum-zero-component-variables) 2)
	 (let ((new-constraint
		 (nconc (list (cons 'SUM-ZERO
				    (mapcar #'(lambda (cv) (model-variable-name 
							     (component-variable-model-variable cv)))
					    sum-zero-component-variables)))
			(correspondences-for-flow-sum-zero-constraint sum-zero-component-variables))))
	   (push-constraint new-constraint)
	   (when *trace-KCL-constraint-application*
	     (format *qsim-report* "~&  Constraint generated from KCL: ~S" new-constraint))))
	;; Note that a single sum-zero variable will have its model variable created,
	;; but no conservation constraint need be generated (i.e. the sum-zero variable
	;; is not constrained at the connection, since corresponding sum-zero variables
	;; at the connection do not exist).
	))


(defun GENERATE-CONNECTION-MODEL-VARIABLES (connection-name connected-variables component-prefix)
  (let ((new-name-tree-entry (list connection-name 'connection)))
    (nconc *name-tree* (list new-name-tree-entry))
    (let ((*name-tree* new-name-tree-entry))

      (loop
	(when (null connected-variables)
	  (return))
	(let ((domain (component-variable-domain (first connected-variables)))
	      (effort-variables nil)
	      (flow-variables nil))
	  (mapc #'(lambda (v) (when (eql domain (component-variable-domain v))
				(case (component-variable-type v)
				  (effort (push v effort-variables))
				  (flow (push v flow-variables)))))
		connected-variables)
	  (setf connected-variables (remove domain connected-variables
					    :key 'component-variable-domain))

	  (when *trace-connection-processing*
            (format *qsim-report* "~& For domain ~A variables are: " domain)
	    (when effort-variables
              (dolist (cv effort-variables)
		(format *qsim-report* " ~A" (component-variable-name cv))))
	    (when flow-variables
	      (dolist (cv flow-variables)
	        (format *qsim-report* " ~A" (component-variable-name cv)))))
  
	  (case (cdr (assoc domain *domain-connection-junction-types*))
	    (0 (when effort-variables
		 (generate-connection-shared-model-variable effort-variables
							    connection-name
							    component-prefix
							    'effort))
	       (when flow-variables
		 (generate-connection-sum-zero-model-variables-and-constraints flow-variables
									       connection-name)))
	    (1 (when flow-variables
		 (generate-connection-shared-model-variable flow-variables
							    connection-name
							    component-prefix
							    'flow))
	       (when effort-variables
		 (generate-connection-sum-zero-model-variables-and-constraints effort-variables
									       connection-name)))
	    (otherwise (format *qsim-report* "~& Unknown junction type ~S for domain ~S"
			       (cdr (assoc domain *domain-connection-junction-types*))
			       domain)))
	  )))))
	    
							       
(defun INTERPRET-COMPONENT-CONNECTION (component-prefix connection-spec component-terminal-alist)
  "Interpret a connection specification, creating the required model variables."
  (when *trace-connection-processing*  
    (format *qsim-report* "~&Interpreting connection ~S" connection-spec))
  ;; Connections involving terminals of the component being defined are not handled here.
  ;; Hence, connection specs arriving here have only terminal references of the type
  ;; (instance-name terminal-name).  A connection name may or may not have been specified.
  ;; If a connection name is specified, it is added to the name tree with associated
  ;; effort variable, using both the domain specific and generic types (e.g.
  ;; effort and voltage).  If none was specified, a unique name is generated, but no
  ;; information is added to the name tree.
  (when (consp (first connection-spec))
    (setf connection-spec (cons (gensym "C-") connection-spec)))
  (let ((connected-component-variables nil))
    (dolist (terminal-spec (cdr connection-spec))
      (let* ((component-name (car terminal-spec))
	     (component-terminals-cons (assoc component-name component-terminal-alist :test #'eq))
	     (terminal-name (cadr terminal-spec))
	     (terminal-variables-cons (when component-terminals-cons
					(assoc terminal-name (cdr component-terminals-cons) :test #'eq))))
	(if terminal-variables-cons
	    (setq connected-component-variables (append (cdr terminal-variables-cons)
						    connected-component-variables))
	    (format *qsim-report* "~& ** Node spec ~S unknown, and excluded from connection ~S. **"
		    terminal-spec (first connection-spec)))))
					
    ;; Identify and create Model-Variables.
    ;; Model variables created here are for Effort and Flow.  Efforts are made equivalent by
    ;; associating them with the same model variable.  Flow variables each have a model variable
    ;; generated, and the conservation constraint is applied to the model variables.
    (when connected-component-variables
      (generate-connection-model-variables (first connection-spec)
					   connected-component-variables
					   component-prefix))
    ))

(defun CC-VARIABLE-QUANTITY-SPACE (mv)
  (typecase mv
    (model-variable 
      (if (model-variable-quantity-space mv)
	(cc-quantity-space-landmark-list (model-variable-quantity-space mv))
	(list 'minf 0 'inf)))
    (mode-variable
      (mode-variable-quantity-space mv))))


(defun COLLECT-QUANTITY-SPACE-DECLARATIONS ()
  (mapcar #'(lambda (mv) (list (model-variable-name mv) (cc-variable-quantity-space mv)))
	  *model-variables*))


(defun COLLECT-DISCRETE-VARIABLES ()
  (mapcar #'(lambda (mv) (list (mode-variable-name mv) (mode-variable-quantity-space mv)))
	  *mode-variables*))


(defun COLLECT-INDEPENDENT-VARIABLES ()
  (let ((independent-variables nil))
    (dolist (mv *model-variables*) (when (model-variable-independent mv)
				     (push (model-variable-name mv) independent-variables)))
    independent-variables))


(defun COLLECT-DEPENDENT-VARIABLES ()
  (let ((dependent-variables nil))
    (dolist (mv *model-variables*) (unless (model-variable-independent mv)
				     (push (model-variable-name mv) dependent-variables)))
    dependent-variables))


(defun PROCESS-CONSTRAINTS ()
  (dolist (constraint-group *constraints*)
    (dolist (constraint (cdr constraint-group))
      (mapl #'(lambda (cv-list)
		(when (component-variable-p (car cv-list))
		  (setf (car cv-list)
			(model-variable-name (component-variable-model-variable (car cv-list))))))
	    (cdr (first constraint))))))


(defun PROCESS-MODE-CONDITIONS-INTERNAL (condition)
  (case (first condition)
    ((and or not)
      (dolist (clause (cdr condition)) (process-mode-conditions-internal clause)))
    (otherwise
      (when (typep (first condition) 'component-variable)
	(setf (first condition) (component-variable-model-variable (first condition)))))))


(defun PROCESS-MODE-CONDITIONS ()
  (let ((condition nil))
    (dolist (mv *mode-variables*)
      (dolist (var-condition-pair (mode-variable-condition-alist mv))
	(process-mode-conditions-internal (second var-condition-pair))
	;; Process any user declared asserts.
	(when (cddr var-condition-pair)
	  (dolist (assert (cdddr var-condition-pair))
	    (when (typep (first assert) 'component-variable)
	      (setf (first assert)
		    (model-variable-name (component-variable-model-variable (first assert))))
	      )))))))


(defun COLLECT-CONSTRAINTS (&optional (index nil))
  (let ((constraints (cdr (assoc index *constraints* :test #'equal))))
    (if index
      (cons (list (cons 'CONSTANT index)) constraints)
      constraints)))


(defun COLLECT-PRINT-NAMES ()
  (let ((print-info-list nil)
	(cv nil))
    (dolist (mv *model-variables*)
      (setf cv (find-if #'component-variable-display (model-variable-component-variables mv)))
      (unless cv (setf cv (car (model-variable-component-variables mv))))
      (push (list (model-variable-name mv)
		  (format nil "~S" (reverse (cons (component-variable-name cv)
						  (component-variable-name-stack cv))))
		  (component-variable-lm-symbol cv))
	    print-info-list))
    print-info-list))


(defun COLLECT-LAYOUT-DESCRIPTION ()
  (let ((print-list nil))
    (dolist (mv *model-variables*) 
      (when (model-variable-display mv) (push (model-variable-name mv) print-list)))
    (dolist (mv *mode-variables*) 
      (when (mode-variable-display mv) (push (mode-variable-name mv) print-list)))
    (do ((temp print-list (nthcdr 3 temp))
	 (layout-rows nil))
	((null temp) (nreverse layout-rows))
      (if (> (length temp) 3)
	  (push (list (first temp) (second temp) (third temp)) layout-rows)
	  (push temp layout-rows)))))


(defun COLLECT-IGNORE-QDIR-VARIABLES ()
  (let ((ignore-qdir-variables nil))
    (dolist (mv *model-variables*) (when (model-variable-ignore-qdir mv)
				     (push (model-variable-name mv) ignore-qdir-variables)))
    ignore-qdir-variables))


(defun COLLECT-NO-NEW-LANDMARKS-VARIABLES ()
  (let ((no-new-landmarks-variables nil))
    (dolist (mv *model-variables*) (when (null (model-variable-new-landmarks mv))
				     (push (model-variable-name mv) no-new-landmarks-variables)))
    no-new-landmarks-variables))


(defun PROCESS-TRANSITION-CONDITIONS (condition)
  (when condition
    (cons (if (listp (car condition))
	    (process-transition-conditions (car condition))
	    (if (eq (type-of (car condition)) 'component-variable)
	      (let ((mv (component-variable-model-variable (car condition))))
		(case (type-of mv)
		  (model-variable (model-variable-name mv))
		  (mode-variable  (mode-variable-name mv))))
	      (car condition)))
	  (process-transition-conditions (cdr condition)))))
	      

(defun ENUMERATE-MODE-COMBINATIONS (unresolved-mode-variables)
  ;; Enumerate all combinations for mode variable values.
  (if (null unresolved-mode-variables)
    nil
    (let ((other-combinations (enumerate-mode-combinations (cdr unresolved-mode-variables)))
	  (current-mode-variable (car unresolved-mode-variables)))
      (mapcar #'(lambda (mode-variable-value)
		  (cons (list (mode-variable-name current-mode-variable) mode-variable-value)
			other-combinations))
	      (mode-variable-quantity-space current-mode-variable)))))


(defun MODE-ASSUMPTION-STRING (mode-assumption)
  (concatenate 'string "_WITH_" (string (first mode-assumption))
	               "=" (string (second mode-assumption))))

(defun GENERATE-QDE (text mode-assumptions model-info)
  ;; Generate the QDE for the specified mode assumptions.
  (let ((*model-variables* (instantiated-cc-model-model-variables model-info))
	(*mode-variables* (instantiated-cc-model-mode-variables model-info))
	(*constraints* (instantiated-cc-model-constraints model-info))
	(*name-tree* (instantiated-cc-model-name-tree model-info))
	(mode-based-name
	  (intern (apply #'concatenate 'string (string (instantiated-cc-model-name model-info))
			 (if mode-assumptions
			   (mapcar 'mode-assumption-string mode-assumptions)
			   (list "")))))
	(qde nil)
	(clauses nil)
	)
    (setf qde
	  (make-qde :name        mode-based-name
		    :independent (collect-independent-variables)
		    :text        (list text)
		    :layout      (collect-layout-description)
		    :other       (list (cons 'ignore-qdirs
					     (collect-ignore-qdir-variables))
				       (cons 'no-new-landmarks
					     (collect-no-new-landmarks-variables))
				       (cons 'cc-info model-info)
				       (cons 'cc-mode-assumptions mode-assumptions))))
    (setf clauses
	  (list (cons 'quantity-spaces (collect-quantity-space-declarations))
		(cons 'discrete-variables (collect-discrete-variables))
		(cons 'constraints 
		      (let ((constraints (collect-constraints)))
			(when (and *trace-mode-processing* *mode-variables*)
			  (format *qsim-report* "~& Mode processing for QDE ~S" 
				  mode-based-name))
			(dolist (mode mode-assumptions)
			  (when *trace-mode-processing*
			    (format *qsim-report* "~& For mode ~S add constraints:"
				    mode)
			    (dolist (c (collect-constraints mode))
			      (format *qsim-report* "  ~%~S" c)))
			  (setf constraints (append (collect-constraints mode)
						    constraints)))
			constraints))
		(cons 'print-names (collect-print-names))))
    (check-syntax qde clauses)

    ;; If dynamic modes are involved, add a transition clause.
    (if (notevery 'mode-variable-static *mode-variables*)
      (initialize-constraint-network qde (cons '(transitions (cc-transition-detection-function 
							       ->
							       cc-transition-function))
					       clauses))
      (initialize-constraint-network qde clauses))
						  
    (set mode-based-name qde)
    (push qde (instantiated-cc-model-qdes model-info))
    qde))

(defun RETRIEVE-QDES-FOR-MODES (possible-mode-values selected-mode-values model-info)
  (if (null possible-mode-values)
    ;; Retrieve QDE for SELECT-MODE-VALUES.
    (let ((new-qde (find-if #'(lambda (qde)
				(mode-values-match? 
				  (cdr (assoc 'cc-mode-assumptions (qde-other qde)))
				  selected-mode-values))
			    (instantiated-cc-model-qdes model-info))))
      (unless new-qde
	(setf new-qde (generate-qde (instantiated-cc-model-text model-info)
				    selected-mode-values
				    model-info)))
      (list new-qde))
    (let ((next-mode (pop possible-mode-values)))
      (if (eql (length next-mode) 2)
	(retrieve-qdes-for-modes possible-mode-values
				 (cons next-mode selected-mode-values)
				 model-info)
	(mapcan #'(lambda (mode-value)
		    (retrieve-qdes-for-modes possible-mode-values
					     (cons (list (first next-mode) mode-value)
						   selected-mode-values)
					     model-info))
		(cdr next-mode))))))


(defun QSPACE-RELATIVE-LT (qspace lm1 lm2)
  (let ((pos1 (position lm1 qspace))
	(pos2 (position lm2 qspace)))
    (and pos1 pos2 (< pos1 pos2))))


(defun QSPACE-RELATIVE-LE (qspace lm1 lm2)
  (let ((pos1 (position lm1 qspace))
	(pos2 (position lm2 qspace)))
    (and pos1 pos2 (<= pos1 pos2))))


(defun MODE-CONDITION-SATISFIED? (condition var-values)
  ;; Determine whether CONDITION is specified by values.  If at least one of the variables
  ;; does not appear in the VAR-VALUES list, then CONDITION is not specified.  Returns
  ;; one of 'yes, 'no, or 'unknown.  This distinction is necessary for negated clauses
  ;; since no known value is different than a value outside of range.
  (case (first condition)
    (and (if (every #'(lambda (conjunct) (eq (mode-condition-satisfied? conjunct var-values) 'yes))
		    (cdr condition))
	   'yes
	   (if (some #'(lambda (conjunct) (eq (mode-condition-satisfied? conjunct var-values)
					      'unknown))
		     (cdr condition))
	     'unknown
	     'no)))
    (or (if (some #'(lambda (disjunct) (eq (mode-condition-satisfied? disjunct var-values) 'yes))
		  (cdr condition))
	  'yes
	  (if (some #'(lambda (disjunct)
			(eq (mode-condition-satisfied? disjunct var-values) 'unknown))
		    (cdr condition))
	    'unknown
	    'no)))
    (not (case (mode-condition-satisfied? (second condition) var-values)
	   (no      'yes)
	   (yes     'no)
	   (unknown 'unknown)))
    (otherwise
      (let* ((var (first condition))
	     (var-name (model-variable-name var))
	     (asserted-value (second (assoc var-name var-values))))
	(if asserted-value
	  (let ((qmag (first asserted-value))
		(qdir (second asserted-value))
		(condition-qmag (first (second condition)))
		(condition-qdir (second (second condition)))
		(qspace (cc-variable-quantity-space var)))
	    (if (if (listp qmag)
		  ;; Qmag is an interval.
		  (unless (symbolp condition-qmag)
		    (and (qspace-relative-le qspace (first condition-qmag) (first qmag))
			 (qspace-relative-le qspace (second qmag) (second condition-qmag))
			 (or (eq condition-qdir 'nil)
			     (eq condition-qdir qdir))))
		  ;; Qmag is a single landmark.
		  (if (symbolp condition-qmag)
		      (and (eq qmag condition-qmag)
			   (or (eq condition-qdir 'nil)
			       (eq condition-qdir qdir)))
		    (and (qspace-relative-lt qspace (first condition-qmag) qmag)
			 (qspace-relative-lt qspace qmag (second condition-qmag))
			 (or (eq condition-qdir 'nil)
			     (eq condition-qdir qdir)))))
	      'yes
	      'no))
	  'unknown)))))

(defun RESOLVE-MODE-VARIABLES (unresolved-mode-variables qspaces initial-values)
  ;; Resolve mode variables based on INITIAL-VALUES and mode conditions.
  (let ((resolved-modes nil))
    (dolist (mv unresolved-mode-variables)
      (if (mode-variable-static mv)
	;; Resolve static mode.
	(let ((asserted-mode-value (assoc (mode-variable-name mv) initial-values)))
	  (when asserted-mode-value
	    (if (member (first (second asserted-mode-value)) (mode-variable-quantity-space mv))
	      (push (list (mode-variable-name mv) (first (second asserted-mode-value)))
		    resolved-modes)
	      (let ((default-mode-value (or (second (assoc (mode-variable-name mv) declared-modes))
					    (first (mode-variable-quantity-space mv)))))
		(format *qsim-report* "~& Asserted value ~S not in value list of mode variable ~S."
			(first (second asserted-mode-value)) (mode-variable-name mv))
		(format *qsim-report* " Value (~S std) assumed." default-mode-value)
		(push (list (mode-variable-name mv) default-mode-value) resolved-modes)))))
	;; Resolve dynamic mode.
	(let ((mode-values-asserted nil))
	  (dolist (condition (mode-variable-condition-alist mv))
	    (when (eq (mode-condition-satisfied? (second condition) initial-values) 'yes)
	      (push (first condition) mode-values-asserted)))
	  (when mode-values-asserted
	    (if (eql (length mode-values-asserted) 1)
	      (push (cons (mode-variable-name mv) mode-values-asserted) resolved-modes)
	      ;; Report multiple assignments to user.
	      (progn (format *qsim-report* 
			     "~& Initial values imply multiple values ~S for mode variable ~S"
			     mode-values-asserted (mode-variable-name mv))
		     (push (list (mode-variable-name mv) mode-values-asserted) resolved-modes)))))))
    resolved-modes))


(defun SELECT-INITIAL-QDES (initial-values resolved-modes model-info)
  ;; Given initial values, select the (moded) QDEs consistent with these values.
  ;; If modes cannot be determined from the INITIAL-VALUES, build a state and 
  ;; propagate values.  If additional modes can be determined, add associated
  ;; constraints and repeat until (1) all modes have been determined or (2) no
  ;; new modes are determined.
  (setf initial-values 
	(mapcar #'(lambda (iv) 
		    (cons (translate-cc-name-internal 
			    (car iv) (instantiated-cc-model-name-tree model-info))
			  (cdr iv)))
		initial-values))
  ;; Add a check here to see if any resolved mode has an asserted value in initial-values.
  ;; If so, signal an error to the user.
  (let ((unresolved-mode-variables (copy-list *mode-variables*))
	(variable-qspaces (collect-quantity-space-declarations)))
    (dolist (resolved-mode resolved-modes)
      (setf unresolved-mode-variables (delete (first resolved-mode) unresolved-mode-variables
					      :key 'mode-variable-name)))
    (let ((new-mode-assignments (resolve-mode-variables unresolved-mode-variables 
							variable-qspaces
							initial-values)))
      (when new-mode-assignments
	(setf resolved-modes (append new-mode-assignments resolved-modes))
	(dolist (assignment new-mode-assignments)
	  (setf unresolved-mode-variables (delete (first assignment) unresolved-mode-variables
						  :key 'mode-variable-name)))))

    (loop
      (when (null unresolved-mode-variables)
	(return nil))
      
      ;; Create a partial QDE and state, propagate, and try to resolve modes again
      (let ((mode-value-alist nil)
	    (partial-qde (make-qde :name        'partial-qde
				   :independent (collect-independent-variables)
				   :other       (list (cons 'ignore-qdirs
							    (collect-ignore-qdir-variables)))))
	    (clauses (list (cons 'quantity-spaces variable-qspaces)))
	    (constraints (collect-constraints))
	    (partial-state nil)
	    (sim (make-sim)))
	;; Add constraints from resolved modes.
	(dolist (resolved-mode resolved-modes)
	  (when (symbolp (second resolved-mode))
	    (setf constraints (append (collect-constraints resolved-mode) constraints))))
        (push (cons 'constraints constraints) clauses)
	;; Finish QDE construction.
	(check-syntax partial-qde clauses)
	(initialize-constraint-network partial-qde clauses)
	(setq *current-sim* sim)
	(with-bindings-from-sim sim
	  (setq *current-qde* partial-qde)
	  (initialize-network partial-qde)
	  ;; Make a state
	  (setf partial-state
		(new-state-from-qde partial-qde
				    (convert-user-values initial-values partial-qde)
				    "Partial state")))
	(setf (sim-qde sim) partial-qde)
	(setf (sim-state sim) partial-state)
	(initialize-SIM-from-QDE sim)
	(set-ignore-qdirs partial-state (sim-ignore-qdirs sim))
	;; Propagate values.
	(let ((current-state partial-state))
	  (declare (special current-state))         ; Make visible to cfilter, etc.
	  (constraint-net-for-state partial-state)  ; Install the constraint network
	  (with-bindings-from-sim sim (propagation partial-state)))
	;; Retrieve values and update initial-value list.
	(setf initial-values nil)
	(dolist (var-qval-cons (cdr (state-qvalues partial-state)))
	  (when (and (qval-qmag (cdr var-qval-cons))
		     (or (typep (qval-qmag (cdr var-qval-cons)) 'lmark)
			 (and (first (qval-qmag (cdr var-qval-cons)))
			      (second (qval-qmag (cdr var-qval-cons))))))
	    (let ((qmag (qval-qmag (cdr var-qval-cons)))
		  (qdir (qval-qdir (cdr var-qval-cons))))
	      (push (list (car var-qval-cons) 
			  (if (listp qmag)
			      (list (list (lmark-name (first qmag)) (lmark-name (second qmag))) qdir)
			    (list (lmark-name qmag) qdir)))
		    initial-values))))
	)
      ;; Try to resolve modes again.  If any, repeat.  If none, exit.
      (let ((new-mode-assignments (resolve-mode-variables unresolved-mode-variables
							  variable-qspaces
							  initial-values)))
	(if new-mode-assignments
	  (progn (setf resolved-modes (append new-mode-assignments resolved-modes))
		 (dolist (assignment new-mode-assignments)
		   (setf unresolved-mode-variables (delete (first assignment) unresolved-mode-variables
							   :key 'mode-variable-name))))
	  (return nil)))
      )
    ;; If modes remain unresolved, put all possible values on the resolved list,
    ;; and generate initial QDE list.
    (when unresolved-mode-variables
      (dolist (mv unresolved-mode-variables)
	(push (cons (mode-variable-name mv) (mode-variable-quantity-space mv)) resolved-modes)))
    ;; Select the initial QDEs.
    (retrieve-qdes-for-modes resolved-modes nil model-info)
    ))


(defun DISPLAY-MODEL-STATS (inst)
  ;; Display model stats, variable qspaces and equivalence classes
  (format *qsim-report* "~&~% Model stats: ~D variables, ~D mode variables, ~D constraints."
	  (length (instantiated-cc-model-model-variables inst))
	  (length (instantiated-cc-model-mode-variables inst))
	  (length (cdr (assoc nil (instantiated-cc-model-constraints inst)))))
  (when (and *trace-mode-processing*
	     (> (length (instantiated-cc-model-constraints inst)) 1))
    (format *qsim-report* "~% Modes indexing constraints:~%   ")
    (dolist (constraint-group (instantiated-cc-model-constraints inst))
      (when (first constraint-group)
	(format *qsim-report* " ~S" (first constraint-group)))))
  (format *qsim-report* 
	  "~% Model variable quantity spaces:~
           ~%   Hierarchical name~40TQuantity Space~80TInternal CC name")
  (let ((equivalence-class-model-variables nil)
	(other-model-variables nil))
    (mapc #'(lambda (mv) (if (> (length (model-variable-component-variables mv)) 1)
			   (push mv equivalence-class-model-variables)
			   (push mv other-model-variables)))
	  (instantiated-cc-model-model-variables inst))
    (do ((mv (pop other-model-variables) (pop other-model-variables))
	 (cv nil))
	((null mv))
	(setf cv (car (model-variable-component-variables mv)))
	(format *qsim-report* "~&   ~S ~A ~40T~S~80T~S"
		(reverse (cons (component-variable-name cv)
			       (component-variable-name-stack cv)))
		(if (model-variable-display mv) "[d]" " ")
		(cc-quantity-space-landmark-list (model-variable-quantity-space mv))
		(model-variable-name mv)))
    (when (instantiated-cc-model-mode-variables inst)
      (dolist (mv (instantiated-cc-model-mode-variables inst))
	(format *qsim-report* "~&   ~A ~A ~40T~S~80T~S"
		(mode-variable-display-text mv)
		(if (mode-variable-display mv) "[d]" " ")
		(mode-variable-quantity-space mv)
		(mode-variable-name mv))))
    (format *qsim-report* "~&  Effort variable equivalence classes:")
    (dolist (mv equivalence-class-model-variables)
      (format *qsim-report* "~&   Quantity space: ~S~60T~S"
	      (cc-quantity-space-landmark-list (model-variable-quantity-space mv))
	      (model-variable-name mv))
      (dolist (cv (model-variable-component-variables mv))
	(format *qsim-report* "~&     ~S ~A"
		(reverse (cons (component-variable-name cv)
			       (component-variable-name-stack cv)))
		(if (component-variable-display cv) "[d]" " "))))))

(defun build-qde (component-type-ref &key (initial-values nil) (display-stats nil))
  (let ((instantiation     (make-instantiated-cc-model))
	(model-name        nil)
	(brief-model-name  nil)
	(descriptive-text  nil)
	(*model-variables* nil)
	(*mode-variables*  nil)
	(*mode-assumptions* nil)
	(*constraints*     nil)
	(*name-stack*      nil)
	(qdes              nil))

    (declare (special *model-variables* *mode-variables* *mode-assumptions*
		      *constraints* *name-stack*))

    (multiple-value-bind (m mode)
        (resolve-component-type-reference nil 'top-level component-type-ref instantiation)
      (when m
	;; Set fields of instantiated-cc-model structure.
	(setf (instantiated-cc-model-implementation-name instantiation)
	      (component-implementation-name m))
	(setf (instantiated-cc-model-interface-name instantiation)
	      (component-interface-name (component-implementation-interface m)))
	(setf (instantiated-cc-model-mode instantiation) mode)

	(setf model-name 
	      (intern (concatenate 'string 
				   (string (princ-to-string (component-interface-name
							      (component-implementation-interface m))))
				   "_" (string (princ-to-string (component-implementation-name m))))))
	(setf brief-model-name (string (princ-to-string (component-interface-name
							  (component-implementation-interface m)))))
	;; Add configuration name, if it is one.
	(when (and (symbolp component-type-ref)
		   (not (eq component-type-ref (component-interface-name
						 (component-implementation-interface m)))))
	  (setf model-name (intern (concatenate 'string (string component-type-ref) "_"
						(string model-name)))))
	
	(setf descriptive-text (component-implementation-text m))
	(setf *name-tree* (list 'cc-name-tree))
	
	(interpret-component-reference (component-interface-name (component-implementation-interface m))
				       brief-model-name component-type-ref nil)
	
	(setq *model-variables* (nreverse *model-variables*))
	(process-name-tree (setf *name-tree* (second *name-tree*)))
	(process-constraints)
	(process-mode-conditions)

	;; Set fields in instantiated-cc-model structure
	(setf (instantiated-cc-model-name instantiation) model-name
	      (instantiated-cc-model-model-variables instantiation) *model-variables*
	      (instantiated-cc-model-mode-variables instantiation) *mode-variables*
	      (instantiated-cc-model-constraints instantiation) *constraints*
	      (instantiated-cc-model-name-tree instantiation) *name-tree*)

	(when display-stats
	  (display-model-stats instantiation))
	
	(setf qdes (if *mode-variables*
		     ;; Generate moded QDEs
		     (select-initial-qdes initial-values *mode-assumptions* instantiation)
		     ;; Generate the single QDE for this model.
		     (generate-qde (list descriptive-text) nil instantiation)))
	(if (and (listp qdes) 
		 (eql (length qdes) 1))
	  (first qdes)
	  qdes)))))

(defvar *NAME-TREE-MATCHES* nil)


(defun FIND-ALL-MATCHES-INTERNAL1 (name tree)
  (when (listp tree)
    (mapcar #'(lambda (subtree) 
		(if (eq (car name) (car subtree))
		    (if (listp (cdr subtree))  ; Skip over component type if there
			(find-all-matches-internal2 (cdr name) (cddr subtree))
		      (find-all-matches-internal2 (cdr name) (cdr subtree)))
		  (if (listp (cdr subtree))    ; Skip over component type if there
		      (find-all-matches-internal1 name (cddr subtree))
		    (find-all-matches-internal1 name (cdr subtree)))))
	    tree)))


(defun FIND-ALL-MATCHES-INTERNAL2 (name tree)
  (if (null name)
      (if (listp tree)
	  (let ((default (find 'default tree :key #'first)))
	    (when default (push (cdr default) *name-tree-matches*)))
	(push tree *name-tree-matches*))
    (when (listp tree)
      (mapcar #'(lambda (subtree) 
		  (when (eq (car name) (car subtree))
			(if (listp (cdr subtree)) ; Skip over component type, if there
			    (find-all-matches-internal2 (cdr name) (cddr subtree))
			  (find-all-matches-internal2 (cdr name) (cdr subtree)))))
	      tree))))


(defun FIND-ALL-MATCHES-IN-TREE (name name-tree)
  "Given a hierarchical name, find all occurences in the name tree."
  (let ((*name-tree-matches* nil))
    (find-all-matches-internal1 name (list name-tree))
    *name-tree-matches*))


(defun TRANSLATE-CC-NAME-INTERNAL (name name-tree)
  "Given a list of hierarchical names of CC model variables, find the CC generated names."
  (let ((matches (find-all-matches-in-tree name name-tree)))
    (cond ((null matches)
	   (format *qsim-report* "~&~% No name match for hierarchical name: ~S" name)
	   name)
	  ((eql (length matches) 1)
	   (first matches))
	  (t (format *qsim-report* "~&~% Ambiguous hierarchical name: ~S~% Possible matches: ~S"
		     name matches)
	     (first matches)))))

(defmacro TRANSLATE-GET-NAME-TREE (qde)
  `(instantiated-cc-model-name-tree (cdr (assoc 'cc-info (qde-other ,qde)))))


(defun TRANSLATE-CC-NAME (qde name)
  "Given a hierarchical name of a CC model variable, find the CC generated names."
  (let ((name-tree (translate-get-name-tree qde)))
    (if (null name-tree)
	(progn (format *qsim-report* "~&~% No CC name tree available for translation.  Name returned as is.")
	       name)
      (translate-cc-name-internal name name-tree))))


(defun TRANSLATE-CC-NAME-LIST (qde name-list)
  "Given a list of hierarchical names of CC model variables, find the CC generated names."
  (let ((name-tree (translate-get-name-tree qde))
	(internal-names nil))
    (if (null name-tree)
	(progn (format *qsim-report* "~&~% No CC name tree available for translation.  List of names returned as is.")
	       name-list)
      (do ((name (pop name-list) (pop name-list)))
	  ((null name)
	   (nreverse internal-names))
	(push (translate-cc-name-internal name name-tree)
	      internal-names)))))


(defun TRANSLATE-CC-NAME-ALIST (qde name-alist)
  "Given an alist with hierarchical names as CARs, find the CC generated names and return a new alist."
  (let ((name-tree (translate-get-name-tree qde))
	(new-alist nil))
    (if (null name-tree)
	(progn (format *qsim-report* "~&~% No CC name tree available for translation.  A-list of names returned as is.")
	       name-alist)
      (progn
	(dolist (item name-alist)
	  (push (cons (translate-cc-name-internal (car item) name-tree)
		      (cdr item))
		new-alist))
	(nreverse new-alist)))))


(defun CC-NAME (qde &key (name nil) (list nil) (alist nil))
  "Translate hierarchical CC names into the internal name per QDE."
  (cond (name  (translate-cc-name       qde name))
	(list  (translate-cc-name-list  qde list))
	(alist (translate-cc-name-alist qde alist))))


(defun PROCESS-NAME-TREE (tree)
  (if (listp (cdr tree))
    (mapcar #'process-name-tree (cddr tree))
    (let* ((cv (cdr tree))
	   (mv (component-variable-model-variable cv)))
      (unless mv
	(setf mv 
	      (make-model-variable :type (component-variable-type cv)
				   :domain (component-variable-domain cv)
				   :independent (component-variable-independent cv)
				   :quantity-space (component-variable-quantity-space cv)
				   :component-variables (list cv)
				   :display (component-variable-display cv)
				   :name (intern 
					   (concatenate 'string
							(string (component-variable-component-name cv))
							"." (string (component-variable-name cv))))))
	(push mv *model-variables*))
      (case (type-of mv)
	(model-variable
	  (set-options-for-model-variable mv)
	  (setf (cdr tree) (model-variable-name mv)))
	(mode-variable
	  (setf (cdr tree) (mode-variable-name mv)))))))



(defun MERGE-NAME-TREES (name-tree1 name-tree2)
  "Merge two name trees."
  (if (and (listp (cdr name-tree1))
	   (listp (cdr name-tree2)))
      (if (and (eq (first name-tree1) (first name-tree2))
	       (eq (second name-tree1) (second name-tree2)))
	  (let ((matching-subtree nil)
		(result-tree nil))
	    (dolist (subtree (cddr name-tree2))
	      (if (setf matching-subtree (assoc (car subtree) (cddr name-tree1)))
		  (progn
		    (setf result-tree (nconc result-tree
					     (merge-name-trees matching-subtree subtree)))
		    (setf name-tree1 (remove matching-subtree name-tree1)))
		(push subtree result-tree)))
	    (when (cddr name-tree1)
		  (setf result-tree (nconc result-tree (cddr name-tree1))))
	    (list (nconc (list (first name-tree1) (second name-tree1)) result-tree)))
	(list (list 'new-tree 'merge name-tree1 name-tree2)))
    (if (equal name-tree1 name-tree2)
	(list name-tree1)
      (list name-tree1 name-tree2))))
      

;; Routines to resolve variable references made in design specs.  These references
;; are of the form:  (<component type designation> <variable designation>)
;; where <component type designation> can be either a symbol or a list of symbols.
;; When a symbol list is used, the component types must appear in the hierarchy in
;; the order specified by the list, but need not be adjacent in the hierarchy.
;; Similarly, <variable designation> can be either a symbol or a list of symbols.
;; When a symbol list is used for <variable designation>, the symbols should name
;; component instance names and a variable name that 1) are adjacent in the 
;; hierarchy and 2) begin immediately under a component specified by the component
;; type designation.  That is to say, the variable designation should uniquely
;; name a variable.

(defun FIND-COMPONENT-NAME-TREES (component-spec name-tree path)
  "Return a list of pointers into the name tree that correspond to components of type COMPONENT-SPEC."
  (if (null component-spec)
      (list (cons nil name-tree))
    (let ((name-tree-list nil))
      (when (listp (cdr name-tree))
	(if (eql (car component-spec) (second name-tree))
	    (if (null (cdr component-spec))
		(push (cons (reverse (cons (first name-tree) path)) name-tree) name-tree-list)
	      (dolist (subtree (cddr name-tree))
		(setf name-tree-list (nconc name-tree-list 
					    (find-component-name-trees (cdr component-spec)
								       subtree
								       (cons (car name-tree) path))))))
	  (dolist (subtree (cddr name-tree))
	    (setf name-tree-list (nconc name-tree-list
					(find-component-name-trees component-spec subtree
								   (cons (car name-tree) path)))))))
    name-tree-list)))
    

(defun FIND-COMPONENT-VARIABLES (component-spec variable-specs name-tree)
  "Return a list of variable sets per COMPONENT-SPEC."
  ;; A variable set is a list containing the hierarchical and digested CC names for 
  ;; each variable.  These two names are the first and second elements of a list,
  ;; respectively.
  (let ((component-name-trees 
	 (find-component-name-trees (if (listp component-spec)
					component-spec
				      (list component-spec))
				    name-tree nil))
	(component-variable-sets nil))
    (dolist (path-and-name-tree component-name-trees)
      (let ((variable-names nil)
	    (path (car path-and-name-tree))
	    (name-tree (cdr path-and-name-tree)))
	(dolist (variable-spec variable-specs)
	  (unless (listp variable-spec) (setf variable-spec (list variable-spec)))
	  (let ((matches (find-all-matches-in-tree variable-spec name-tree))
		(hierarchical-name (append path variable-spec)))
	    (cond ((null matches)
		   (format *qsim-report* "~&~% Unable to find variable ~S under an instance of component spec ~S"
			   variable-spec component-spec)
		   (push nil variable-names))
		  ((eql (length matches) 1)
		   (push (list hierarchical-name (car matches)) variable-names))
		  (t (format *qsim-report* "~&~% Ambiguous hierarchical name ~S under component ~S~% Possible matches: ~S"
			     variable-spec component-spec matches)
		     (push (list hierarchical-name (car matches)) variable-names)))))
	(push (nreverse variable-names) component-variable-sets)))
    component-variable-sets))


(defun MODE-VALUES-MATCH? (mode-value-list1 mode-value-list2)
  (do ((mode-value (pop mode-value-list1) (pop mode-value-list1))
       (matching-mode-variable nil))
      ((null mode-value) t)
    (if (setf matching-mode-variable
	      (find (car mode-value) mode-value-list2 :key #'first))
      (unless (eq (second mode-value) (second matching-mode-variable))
	(return nil))
      (return nil))))


(defun FIND-HISTORY-VARIABLES (qde)
  ;; Returns a list of variables that have explicit derivatives in the QDE.
  (let ((history-variables nil))
    (dolist (constraint (qde-constraints qde))
      (when (eq (contype-name (constraint-type constraint)) 'D/DT)
	(push (variable-name (first (constraint-variables constraint))) history-variables)))
    history-variables))


#||
(defun FIND-CONSTANT-VARIABLES (qde)
  ;; Returns a list of variable/value pairs for variables involved in a CONSTANT
  ;; constraint that have a value specified.
  (let ((assert-variables nil))
    (dolist (constraint (qde-constraints qde))
      (when (and (eq (contype-name (constraint-type constraint)) 'CONSTANT)
		 (constraint-bend-points constraint))
	(push (list (variable-name (first (constraint-variables constraint)))
		    (list (first (constraint-bend-points constraint)) 'std))
	      assert-variables)))
    assert-variables))
||#

    
(defmacro DETECTION--QSPACE-RELATIVE-LT (lm1 lm2 qspace)
  `(< (position ,lm1 ,qspace :key 'lmark-name)
      (position ,lm2 ,qspace :key 'lmark-name)))


(defmacro DETECTION--QSPACE-RELATIVE-LE (lm1 lm2 qspace)
  `(or (eql ,lm1 ,lm2) (detection--qspace-relative-lt ,lm1 ,lm2 ,qspace)))


(defun TRANSITION-DETECTION--CONDITION-SATISFIED? (condition qvalues qspaces)
  ;; Determine if the mode condition CONDITION is satisfied, based on
  ;; QVALUES and QSPACES.
  (case (first condition)
    (and (every #'(lambda (conjunct)
                    (transition-detection--condition-satisfied?
                      conjunct qvalues qspaces))
                (cdr condition)))
    (or (some #'(lambda (disjunct)
                  (transition-detection--condition-satisfied?
                    disjunct qvalues qspaces))
              (cdr condition)))
    (not (null (transition-detection--condition-satisfied?
                 (second condition) qvalues qspaces)))
    (otherwise ;; A variable/qvalue pair.
      (let* ((variable-name (model-variable-name (first condition)))
             (condition-qmag (first (second condition)))
             (condition-qdir (second (second condition)))
             (qval (cdr (assoc variable-name qvalues)))
             (qmag (qval-qmag qval))
             (qdir (qval-qdir qval))
             (qspace (cdr (assoc variable-name qspaces)))
            )
	;; Verify that condition-qmag is represented in the corresponding qspace.
	(if (null (or (and (listp condition-qmag)
			   (member (first condition-qmag) qspace :key 'lmark-name)
			   (member (second condition-qmag) qspace :key 'lmark-name))
		      (member condition-qmag qspace :key 'lmark-name)))
	  ;; Report error, condition qmag not represented in variable qspace
	  (format *qsim-report* 
		  "~&~% Transition condition qmag ~S cannot be expressed in qspace ~S of variable ~S"
		  condition-qmag qspace variable-name)
	  (if (listp condition-qmag)
	      (if (listp qmag)
		  (and (detection--qspace-relative-le (first condition-qmag)
						      (lmark-name (first qmag))
						      qspace)
		       (detection--qspace-relative-le (lmark-name (second qmag))
						      (second condition-qmag)
						      qspace))
		(or (and (detection--qspace-relative-lt (first condition-qmag)
							(lmark-name qmag)
							qspace)
			 (detection--qspace-relative-lt (lmark-name qmag)
							(second condition-qmag)
							qspace))
		    (and (eql (lmark-name qmag) (first condition-qmag))
			 (eq qdir 'inc))
		    (and (eql (lmark-name qmag) (second condition-qmag))
			 (eq qdir 'dec))))
	    (and (null (listp qmag))
		 (eql condition-qmag (lmark-name qmag))
		 (or (null condition-qdir) (eq condition-qdir qdir)))))
        ))))


(defun CC-TRANSITION-DETECTION-FUNCTION (state)
  ;; Determine if a mode transition is in order, based on mode conditions.
  (let* ((qde (state-qde state))
	 (current-modes (cdr (assoc 'cc-mode-assumptions (qde-other qde))))
	 (mode-variables (instantiated-cc-model-mode-variables
			   (cdr (assoc 'cc-info (qde-other qde)))))
	 (qvalues (state-qvalues state))
	 (qspaces (state-qspaces state))
	 (new-mode-values nil)
	 (stop nil)
	 )
    (dolist (mv mode-variables)
      (let ((new-values nil))
        (dolist (value-condition-pair (mode-variable-condition-alist mv))
          (unless (eql (first value-condition-pair)
                       (second (assoc (mode-variable-name mv) current-modes)))
            ;; Test this mode condition.
            (let ((new-value (first value-condition-pair))
                  (condition (second value-condition-pair)))
              (when (transition-detection--condition-satisfied?
                      condition qvalues qspaces)
		(push (if (eq new-value :discontinuous-transition)
			(cddr value-condition-pair) ;; Save new value and asserts.
			new-value)
		      new-values)
		(when (or (eq new-value 'stop)
			  (eq new-value :stop))
		      (setf stop t))))))
        (when new-values
          (push (cons (mode-variable-name mv) new-values) new-mode-values))
        ))
    ;; If new-mode-values is non-NIL, a transition is in order.  Detected
    ;; mode values need to be transmitted to the transition QDE builder.
    ;; If any mode value is STOP, then signal this to QSIM.
    (cond (stop (values nil nil))
	  (new-mode-values
	    (setf *cc-transition-new-mode-values* new-mode-values)
	    (values t nil))
	  (t (values nil t)))
    ))



(defun CC-TRANSITION-FUNCTION (state)
  ;; Given the mode transitions in *CC-TRANSITION-NEW-MODE-VALUES*, select the
  ;; next QDE for simulation.
  ;; For now, 
  ;; - modes not explicitly named in a transition are assumed to remain
  ;;   the same.  Eventually, will have to generate the partial QDE and see if
  ;;   other modes can be inferred.

  (let ((new-modes (mapcar #'(lambda (mode-value-list) 
			       (cons (first mode-value-list)
				     (mapcar #'(lambda (val) (if (listp val) (first val) val))
					     (cdr mode-value-list))))
			   *cc-transition-new-mode-values*))
	(new-qdes nil)
	(current-qde (state-qde state))
	(new-states nil))
    (dolist (previous-mode (cdr (assoc 'cc-mode-assumptions (qde-other current-qde))))
      (unless (member (first previous-mode) new-modes :key #'first)
	(push previous-mode new-modes)))
    ;; Find QDEs for the new set of modes.
    (setf new-qdes 
	  (retrieve-qdes-for-modes new-modes nil
				   (cdr (assoc 'cc-info (qde-other current-qde)))))

    (when *trace-mode-processing*
      (format *qsim-report* "~&~% New QDEs after transitions ~S are:" new-qdes)
      (dolist (qde new-qdes) (format *qsim-report* "~%   ~S" qde)))

    ;; Make new states from new QDEs and initial values.
    (dolist (new-qde new-qdes)
      (let ((inherit-qmags (find-history-variables (state-qde state)))
;	    (asserts (nconc (mapcar #'(lambda (var-val-pair) (list (first var-val-pair)
;								   (list (second var-val-pair) 'std)))
;				    (cdr (assoc 'cc-mode-assumptions (qde-other new-qde))))
;			    (find-constant-variables new-qde)))
	    (mode-asserts (mapcar #'(lambda (var-val-pair) (list (first var-val-pair)
								 (list (second var-val-pair) 'std)))
				  (cdr (assoc 'cc-mode-assumptions (qde-other new-qde)))))
	    (user-asserts nil))

	;; Find any user asserted values.
	(dolist (mode-assert-pair mode-asserts)
	  (let ((mode-value-list (assoc (first mode-assert-pair) *cc-transition-new-mode-values*)))
	    (dolist (val (cdr mode-value-list))
	      (when (and (listp val) (eq (first (second mode-assert-pair)) (first val)))
		(setf user-asserts (append (cdr val) user-asserts))
		;; Remove this variable from inherit-qmags, if it is there.
		(dolist (assert (cdr val))
		  (setf inherit-qmags (remove (car assert) inherit-qmags)))
		))))

	(when *trace-mode-processing*
	      (format *qsim-report* "~& Variables inheriting qmag: ~S~% Mode asserts: ~S~% User asserts: ~S~%"
		      inherit-qmags mode-asserts user-asserts))
	(push (create-transition-state :from-state   state
				       :to-qde       new-qde
				       :assert       (append user-asserts mode-asserts)
				       :inherit-qmag inherit-qmags)
	      new-states)))
    (when (> (length new-states) 1)
      (setf new-states (nreverse new-states))
      (format *qsim-report* "~&~% CC mode transition resulted in more than one next QDE.~
                               ~%   QDE ~S assumed."
	      (car new-qdes)))
    (car new-states)))
